# axes.rb, copyright (c) 2006 by Vincent Fourmond: 
# A module to deal with axes customization.
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'CTioga/utils'
require 'CTioga/layout'
require 'CTioga/log'

module CTioga

  # A module to be included by the main PlotMaker instance. It
  # deals with various aspects of axes manipulations.
  module Axes

    LabelSpecification = {
      /x(label)?/i => :xlabel,
      /y(label)?/i => :ylabel,
      /t(itle)?/i => :title,
    }

    LabelSide = {
      /bottom/i => Tioga::FigureConstants::BOTTOM,
      /top/i => Tioga::FigureConstants::TOP,
      /left/i => Tioga::FigureConstants::LEFT,
      /right/i => Tioga::FigureConstants::RIGHT,
    }

    Alignment = {
      /B/ => Tioga::FigureConstants::ALIGNED_AT_BASELINE,
      /baseline/i => Tioga::FigureConstants::ALIGNED_AT_BASELINE,
      /b/ => Tioga::FigureConstants::ALIGNED_AT_BOTTOM,
      /bottom/i => Tioga::FigureConstants::ALIGNED_AT_BOTTOM,
      /t(op)?/i => Tioga::FigureConstants::ALIGNED_AT_TOP,
    }

    Justification = {
      /c(enter)?/i => Tioga::FigureConstants::CENTERED,
      /l(eft)?/i => Tioga::FigureConstants::LEFT_JUSTIFIED,
      /r(ight)?/i => Tioga::FigureConstants::RIGHT_JUSTIFIED,
    }
    
    # Whether to have X or Y log axes
    attr_accessor :x_log, :y_log
    # Scaling of data axes
    attr_accessor :x_factor, :y_factor
    # Decimal separator
    attr_accessor :decimal_separator
    
    # Initializes variables pertaining to axes
    def init_axes
      @x_log = false
      @y_log = false
      @x_factor = false
      @y_factor = false
      @decimal_separator = false
    end

    def which_label(w)
      return Utils::interpret_arg(w, LabelSpecification) {false}
    end

    # Runs a block safely with a label specification 
    def run_with_label(w)
      w = which_label(w)
      if w
        yield w
      else
        error "The label specification #{w} was not understood, ignoring"
      end
    end

    # Prepare the option parser for axes options
    def axes_options(parser)

      parser.separator "\nVarious axis stuff:"
      parser.on("--xfact FACT",
                "Multiply all x values by FACT") do |f|
        @x_factor = safe_float(f)
      end
      parser.on("--yfact FACT",
                "Multiply all y values by FACT") do |f|
        @y_factor = safe_float(f)
      end
      parser.on("--[no-]xlog",
                "Uses logarithmic scale for X axis") do |v|
        @x_log = v
        add_elem_funcall(:xaxis_log_values=, v)
      end
      parser.on("--[no-]ylog",
                "Uses logarithmic scale for Y axis") do |v|
        @y_log = v
        add_elem_funcall(:yaxis_log_values=, v)
      end
      parser.on("--comma",
                "Uses a comma for the decimal separator") do 
        @decimal_separator = ','
      end
      parser.on("--decimal SEP",
                "Uses SEP for the decimal separator") do |s|
        @decimal_separator = s
      end

      parser.separator "\nLabels and titles"
      parser.on("-x","--[no-]xlabel [LABEL]",
                "Label of the x axis") do |l|
        current_object.xlabel.label = l
      end
      parser.on("-y","--[no-]ylabel [LABEL]",
                 "Label of the y axis") do |l|
        current_object.ylabel.label = l
      end
      parser.on('-t', "--[no-]title [TITLE]",
                 "Sets the title of the plot") do |l|
        current_object.title.label = l
      end
      parser.on("--side WHAT ALIGN",
                "Sets the side for the WHAT label, ",
                "where WHAT = x(label),y(label) or t(itle)") do |w|
        run_with_label(w) do |w|
          a = Utils::interpret_arg(@args.shift, LabelSide) {false}
          current_object.send(w).side = a if a
        end
      end

      parser.on("--lcolor WHAT COLOR",
                "Sets the color for the WHAT label, ",
                "where WHAT = x(label),y(label) or t(itle)") do |w|
        run_with_label(w) do |w|
          a = CTioga.get_tioga_color(@args.shift)
          current_object.send(w).color = a if a
        end
      end

      parser.on("--position WHAT WHERE",
                "Sets the position for the WHAT label, ",
                "where WHAT = x(label),y(label) or t(itle)",
                "and WHERE a number between 0 and 1, 0.5 = centered") do |w|
        run_with_label(w) do |w|
          a = safe_float(@args.shift)
          current_object.send(w).position = a if a
        end
      end

      parser.on("--angle WHAT ANGLE",
                "Sets the angle for the WHAT label, ",
                "where WHAT = x(label),y(label) or t(itle).") do |w|
        run_with_label(w) do |w|
          a = safe_float(@args.shift)
          current_object.send(w).angle = a if a
        end
      end

      parser.on("--scale WHAT SCALE",
                "Sets the scale for the WHAT label, ",
                "where WHAT = x(label),y(label) or t(itle).") do |w|
        run_with_label(w) do |w|
          a = safe_float(@args.shift)
          current_object.send(w).scale = a if a
        end
      end

      parser.on("--shift WHAT SHIFT",
                "Sets the shift for the WHAT label, ",
                "where WHAT = x(label),y(label) or t(itle).",
                "The shift is the distance of the label from the plot") do |w|
        run_with_label(w) do |w|
          a = safe_float(@args.shift)
          current_object.send(w).shift = a if a
        end
      end

      parser.on("--align WHAT ALIGN",
                "Sets the 'vertical' alignment for the WHAT label, ",
                "where WHAT = x(label),y(label) or t(itle).") do |w|
        run_with_label(w) do |w|
          a = Utils::interpret_arg(@args.shift, Alignment) {false}
          current_object.send(w).alignment = a if a
        end
      end

      parser.on("--just WHAT JUST",
                "Sets the 'horizontal' alignment for the WHAT label, ",
                "where WHAT = x(label),y(label) or t(itle).") do |w|
        run_with_label(w) do |w|
          a = Utils::interpret_arg(@args.shift, Justification) {false}
          current_object.send(w).alignment = a if a
        end
      end

    end
    
  end

  # A class that describes the various attributes of a label,
  # either for the X or Y axis, or even the plot title.
  class Label

    include Log

    # Include the figure constants, else it gets really painful...
    include Tioga::FigureConstants
    
    # The various attributes
    Attributes = [:alignment, :angle, :color, :justification,
                  :position, :scale, :shift, :side, :visible]
    
    attr_accessor *Attributes

    # Which axis ?
    attr_accessor :which

    # The actual text to e displayed
    attr_accessor :label

    # Creates a label specification for the _which_ axis.
    # _other_values_ is a hash that can be used to provide default
    # values for the various Attributes.
    #
    # _which_ can be :xlabel, :ylabel or :title
    def initialize(which, label = nil, other_values = {})
      @which = which
      @label = label
      for key,val in other_values
        self.send("#{key}=", val)
      end
    end

    # Shows the given axis on the figure _t_
    def show(t)
      # We don't do anything unless we have a label to display !
      return unless @label
      # We wrap the call in a context so we don't pollute subpictures
      t.context do 
        for attr in Attributes
          if instance_variables.include?("@#{attr}")
            val = instance_variable_get("@#{attr}")
            t.send("#{which}_#{attr}=", val)
          end
        end
        t.send("show_#{which}", @label)
      end
    end

    # Guesses the extension of the label on the side of the plot.
    # The output is only garanteed when called with _t_ exactly in the
    # context that will be used for the plot. An external _scale_ factor
    # can be given in the case we have more information about the actual
    # overall scale. Of course, just multiplying the result by it does
    # give the same thing.
    #
    # The value returned is an array left,right, top, bottom
    # containing only zeroes but for the place where the axis
    # does extend.
    def extension(t, scale = 1)
      ret_val = [0,0,0,0]
      if @label
        ext = (@scale || tioga_value(t,:scale)) * 
          (1 + (@shift || tioga_value(t,:shift))) * 
        t.default_text_scale * t.default_font_size * scale
        # Set to 0 if label doesn't extend.
        ext = 0 if ext < 0
        case (@side || tioga_value(t,:side)) 
        when LEFT
          ret_val[0] = ext
        when BOTTOM
          ret_val[3] = ext
        when TOP
          ret_val[2] = ext
        when RIGHT
          ret_val[1] = ext
        end
      end
      debug "Axis #{self.inspect} extension #{ret_val.inspect}"
      return ret_val
    end
  
  
    # Returns the value of the _what_ attribute
    # corresponding to the value of which.
    def tioga_value(t, what)
      return t.send("#{@which}_#{what}")
    end
  end

  # A class that holds the information about how to plot the edges.
  # And the size they potentially take.
  class EdgesAndAxes

    # A class to deal with axis attributes
    class Axis

      include Log

      # Include the figure constants, else it gets really painful...
      include Tioga::FigureConstants
    
      # The various attributes
      Attributes = [:line_width, :loc, :log_values, 
                    :type, :numeric_label_shift ]
      
      attr_accessor *Attributes

      # Which axis ?
      attr_accessor :which

      # _which_ can be :x or :y
      def initialize(which)
        @which = which
        case which
        when :x
          @loc = BOTTOM
        when :y
          @loc = LEFT
        end
      end
      
      # Setup attributes for the axis.
      def setup(t)
        for attr in Attributes
          if instance_variables.include?("@#{attr}")
            val = instance_variable_get("@#{attr}")
            t.send("#{which}axis_#{attr}=", val)
          end
        end
      end
      
      def extension(t, scale = 1)
        ext = 0
        f = [0,0,0,0]
        # First, we estimate the size:
        case @type
        when AXIS_WITH_MAJOR_TICKS_AND_NUMERIC_LABELS, 
          AXIS_WITH_TICKS_AND_NUMERIC_LABELS
          ext = (1 + (@numeric_label_shift ||
                      t.send("#{@which}axis_numeric_label_shift"))) * 
            t.send("#{@which}axis_numeric_label_scale") * 
            t.default_text_scale * t.default_font_size
        else
          ext = 0
        end
        if @loc && @loc < 4 && @loc >= 0  # That is, left,right, top, bottom
          f[@loc] = ext
        end
        return f
      end
    
    end

    include Tioga::FigureConstants

    include Log

    Edges = [:left, :right, :top, :bottom]


    # A hash containing the edge type for each edge.
    attr_accessor :edge_specs

    # The default values for the edges
    Defaults = {
      :left => AXIS_WITH_TICKS_AND_NUMERIC_LABELS,
      :right => AXIS_WITH_TICKS_ONLY,
      :top => AXIS_WITH_TICKS_ONLY,
      :bottom => AXIS_WITH_TICKS_AND_NUMERIC_LABELS
    }

    # The axes
    attr_accessor :xaxis, :yaxis

    def initialize
      @edge_specs = Defaults.dup
      @xaxis = Axis.new(:x)
      @xaxis.type = AXIS_WITH_TICKS_AND_NUMERIC_LABELS
      @yaxis = Axis.new(:y)
      @yaxis.type = AXIS_WITH_TICKS_AND_NUMERIC_LABELS
    end

    def set_edges(value, *which)
      which = Edges if which.empty?
      for edge in which
        @edge_specs[edge] = value
      end
    end


    # Disable the given edges, or all if no arguments
    def disable(*which)
      set_edges(AXIS_HIDDEN, *which)
    end

    # Show only one line for the given edges.
    def line_only(*which)
      set_edges(AXIS_LINE_ONLY, *which)
    end
    

    # Sets up the edges for the given plot. Should probably be called
    # just before the exit of the show_plot_with_legend block argument
    def setup(t)
      # Send axes informations.
      @xaxis.setup(t)
      @yaxis.setup(t)
      for key,val in @edge_specs
        if Edges.include?(key)
          t.send("#{key}_edge_type=", val)
        else
          warn "Invalid edge: #{key}"
        end
      end
    end

    # Disable all display of a given axis. The optional value if
    # the value
    def disable_axis_display(axis, residual = AXIS_WITH_TICKS_ONLY)
      case axis
      when :x
        @xaxis.type = residual
        set_edges(residual, :top,:bottom)
      when :y
        @yaxis.type = residual
        set_edges(residual, :left,:right)
      end
    end

    # This does not take ticks into account yet. No need !
    def extension(t, scale = 1)
      f = [0,0,0,0]
      Dimension.update_extensions(f, @xaxis.extension(t))
      Dimension.update_extensions(f, @yaxis.extension(t))
      debug "Edges and axis #{self.inspect} extension #{f.inspect}"
      return f
    end
        
    
  end
end
    
