# descriptions.rb : The basics of the MetaBuilder system
# Copyright (C) 2006 Vincent Fourmond

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

module MetaBuilder


  # The Description class is a meta-information class that records several
  # informations about the class:
  #
  # * a basic name, code-like, which is used mainly for internal
  #   purposes;
  # * a long name, more explanatory, in proper English (or any
  #   other language. By the way, it should definitely be
  #   translated in a production environment);
  # * a description itself, some small text describing the nature
  #   of the class;
  # * a list of all the Parameters that are important for the class,
  #   and enough to recreate the state of an object.
  #   
  # This class is fairly general, and can be subclassed to fit specific
  # needs. An example is in the SciYAG/Backend system, where the description
  # of a backend is derived from Description.
  #
  # To make use of the Description system for a class, use the following:
  #
  #   class SomeClass
  #     extend  MetaBuilder::DescriptionExtend
  #     include MetaBuilder::DescriptionInclude
  #     
  #     describe 'someclass', 'Some nice class', <<EOD
  #     The description of a nice class
  #     EOD
  #
  #   end
  #
  # Descriptions can be used in two completely different manners:
  #
  # * you can use a single Description to facilitate the interface
  #   with the user to query|save parameters;
  # * you can use the Description system to describe a whole set of
  #   classes providing similar functions and sharing a base ancestor,
  #   such as series of input|output plugins, database formats,
  #   themes... In this case, the Description system can act as a real
  #   plugin factory, recording every new subclass of the base one
  #   and providing facilities to prompt the user.
  #
  # Please note that if you want to use the facilities to dynamically
  # create objects at run-time, the classes used by describe
  # *should not need any parameter for #initialize*.
  # 
  #
  # Description provides the following facilities:
  # 
  # * a #save_state system, returning a hash containing the current
  #   parameters of an object; it's state can be returned to using
  #   #restore_state. Please note that the parameters will
  #   be set in the same order as they appear in the class. Moreover,
  #   if you want the #restore_state to work properly, you must ensure
  #   that setting all the parameters restore the state, i.e. you didn't
  #   forget anything.
  # * facilities to prepare OptionParser to work on a described class (see
  #   #option_parser_fill, #option_parser_banner, #option_parser_options)
  # * facilities to create widgets, dialog boxes and so on to query and
  #   modify the contents of a described class instance.


  class Description
    # The Class to instantiate. 
    attr_accessor :object_class
    
    # The name of the class (short, code-like)
    attr_accessor :name
    
    # (text) description !
    attr_accessor :description
    
    # Long name, the one for public display
    attr_accessor :long_name

    # The parameter list. The parameters are added when they are found
    # in the class description, and will be used in the order found
    # in this list to recreate the state; beware if one parameter is
    # depending on another one.
    attr_reader   :param_list

    # A hash index on the (short) name and the symbols of the parameters,
    # for quick access. None of those should overlap.
    attr_reader   :param_hash
    
    
    # The list of groups
    attr_accessor :group_list

    # A hash indexing the groups by their name
    attr_accessor :group_hash
    
    # Initializes a Description
    def initialize(cls, name, long_name, description = "")
      @object_class = cls
      @name = name
      @long_name = long_name
      @description = description
      @param_list = []
      @param_hash = {}
      @init_param_list = []

      @group_list = []
      @group_hash = {}

      # There is not default group.
      @current_group = nil
    end


    # Adds a group to the list, and switches to it
    def add_group(group)
      @group_list << group
      @group_hash[group.name] = group
      @current_group = group
    end

    # Switches to the named group
    def switch_to_group(name)
      @current_group = @group_hash.fetch(name)
    end
    
    # Adds a new parameter to the description
    def add_param(param)
      @param_list << param

      # Update the parameter hash, for easy access.
      @param_hash[param.reader_symbol] = param
      @param_hash[param.writer_symbol] = param
      @param_hash[param.name] = param

      # Update the current group, if necessary
      @current_group.add_parameter(param) unless @current_group.nil?
    end

    # Small set of functions dealing with OptionParsers

    # Fills an OptionParser with all the parameters the of the class.
    # _instance_ is the instance of the class we want to
    # parametrize, _parser_ is an OptionParser, and if _uniquify_ is set,
    # the description name is prefixed to the parameter name.
    def option_parser_fill(parser, instance, uniquify = true, groups = false)
      option_parser_banner(parser, instance)
      option_parser_options(parser, instance, uniquify, groups)
    end
    
    # Fills a parser with options for all parameters. _parser_, _instance_
    # and _uniquify_ have the same meaning as in #option_parser_option.
    # If _grouping_ is set to _true_, the options are organized according
    # to group listing.
    def option_parser_options(parser, instance, uniquify = true, 
                              grouping = false)
      if grouping
        for group in @group_list
          parser.separator group.long_name
          for param in group.parameter_list
            option_parser_option(parser, param, instance, uniquify)
          end
        end
      else
        for param in @param_list
          option_parser_option(parser, param, instance, uniquify)
        end
      end
    end

    # Adds an option for the parameter _param_ to the OptionParser _parser_.
    # The corresponding parameter will be set in _instance_. If _uniquify_
    # is set, the description name if prefixed to the option.
    # _param_ can be either a Parameter or a string|symbol registered
    # in #param_hash.
    def option_parser_option(parser, param, instance, uniquify = true)
      raise "The instance is not of the right class" unless
        instance.is_a? @object_class
      if not param.is_a?(Parameter)
        param = param_hash.fetch(param)
      end
      if uniquify
        param_name = "#{@name}-#{param.name}"
      else
        param_name = "#{param.name}"
      end
      param.option_parser_option(parser, param_name, instance)
    end

    # Subclass Description and reimplement this method to have your own banner.
    def banner(instance)
      return false
    end

    # Ouputs an OptionParser banner based on the return value
    # of the #banner method.
    def option_parser_banner(parser, instance) 
      if b = banner(instance)
        parser.separator b
      end
    end
    
    # Creates a hash representing the state of an instance of the
    # described class at a given time. It can be fed to #restore_state
    # and #recreate_state to restore the state to an already existing
    # instance or to create a new one from scratch.
    def save_state(instance)
      ret = {}
      for param in @param_list
        ret[param.name] = param.get(instance)
      end
      return ret
    end

    # Puts the _target_ into the _state_ returned by #save_state.
    # The parameters are set in the order in which they are found
    # in the class declaration.
    def restore_state(state, target)
      for param in @param_list
        if state.has_key?(param.name)
          param.set(target, state[param.name])
        end
      end
      return target
    end

    # Creates a new (default) instance of the #object_class associated
    # with this Description. Although this method takes additional
    # parameters that are fed to the new method of the class, its
    # use is strongly discouraged.
    def instantiate(*a)
      return @object_class.new(*a)
    end

    # Creates
    def recreate_state(state)
      obj = instantiate
      restore_state(state, obj)
      return obj
    end
    
  end

  # This module should be used in conjunction with DescriptionExtend to
  # provide Descriptions to a class or a family of classes:
  #
  #  class NiceClass
  #    extend  DescriptionExtend
  #    include DescriptionInclude
  #    # real code now
  #  end
  #
  # The DescriptionInclude in itself provide some small facilities: a mehod
  # to access directly the #description of an object (and its #long_name),
  # a series of methods
  # to set|get parameters based on their name (#parameter, #get_param,
  # #get_param_raw, #set_param, #set_param_raw) and some methods to fill
  # an OptionParser (#option_parser_fill, #option_parser_options,
  # #option_parser_banner)

  module DescriptionInclude

    # Returns the description associated with the class.
    def description
      return self.class.description
    end
    
    
    # Returns the long name of the class
    def long_name
      return description.long_name
    end

    # Small functions to deal with the Parameter objects associated with
    # the class. They are absolutely not necessary to actually access the
    # data, but can come in really useful at some points.

    # Returns the Parameter object associated with the named parameter
    def parameter(param)
      return description.param_hash.fetch(param)
    end

    # Query the value of a parameter, returns a String
    def get_param(p)
      return parameter(p).get(self)
    end

    # Query the value of a parameter, returns the actual value
    def get_param_raw(p)
      return parameter(p).get_raw(self)
    end
    
    # Sets the value of a paramete according to a String
    def set_param(param, str)
      return parameter(param).set(self, str)
    end

    # Sets directly the value of a parameter
    def set_param_raw(param, val)
      return parameter(param).set_raw(self, val)
    end

    # OptionParser related methods.

    # Fills an OptionParser with their parameters. Most probably, the
    # default implementation should do for most cases. _uniquify_ asks
    # if we should try to make the command-line options as unique as
    # reasonable to do. If _groups_ is true, organize parameters
    # in groups.
    def option_parser_fill(parser, uniquify = true, groups = false)
      description.option_parser_fill(parser, self, 
                                     uniquify, groups)
    end
    
    # Provides only a banner for the current class to the OptionParser
    # _parser_.
    def option_parser_banner(parser)
      description.parser_banner(parser, self)
    end

    # Prepares an option parser for this instance. See
    # Description#option_parser_options
    def option_parser_options(parser, uniquify = true, groups = false)
      description.option_parser_options(parser, self, uniquify, groups)
    end

    # SaveState and the like:

    # Calls Description#save_state on this object
    def save_state
      return description.save_state(self)
    end

    # Restores a previously saved state. (Well, it doesn't need to
    # be previously saved, you can make it up if you like).
    def restore_state(state)
      description.restore_state(state, self)
    end

  end
    
  # This module should be used with +extend+ to provide the class with
  # descriptions functionnalities. You also need to +include+
  # DescriptionInclude. Please not that all the *instance* methods
  # defined here will become *class* methods in the class you extend.
  #
  # This module defines several methods to add a description (#describe) to a
  # class, to add parameters (#param, #param_noaccess) and to import
  # parameters from parents (#inherit_parameters).
  #
  # Factories can be created using the #craete_factory statement.
  # This makes the
  # current class the factory repository for all the subclasses. It creates
  # a factory class method returning the base factory.
  # You can use #register_class to register the current class into the
  # base factory class.
  module DescriptionExtend

    # The functions for factory handling.

    # Makes this class the factory class for all subclasses.
    # It creates four class methods: base_factory, that always
    # points to the closest factory in the hierarchy
    # and three methods used internally.
    #
    # If _auto_ is true, the subclasses are all automatically
    # registered to the factory. If _register_self_ is true
    # the class itself is registered. It is probably not a good
    # idea, so it is off by default.
    def create_factory(auto = true, register_self = false)
      cls = self
      # we create a temporary module so that we can use
      # define_method with a block and extend this class with it
      mod = Module.new
      mod.send(:define_method, :factory) do
        return cls
      end
      mod.send(:define_method, :private_description_list) do
        return @registered_descriptions
      end
      mod.send(:define_method, :private_description_hash) do
        return @registered_descriptions_hash
      end
      # Creates an accessor for the factory class
      mod.send(:define_method, :private_auto_register) do 
        @auto_register_subclasses
      end
      self.extend(mod)

      # Creates the necessary arrays|hashes to handle the registered
      # classes:
      @registered_descriptions = []
      @registered_descriptions_hash = {}

      @auto_register_subclasses = auto
    end

    # Checks if the class has a factory
    def has_factory?
      return self.respond_to?(:factory)
    end

    # Returns the base description if there is one, or nil if there isn't
    def base_description
      if has_factory?
        return factory.description
      else
        return nil
      end
    end

    # Returns the description list of the factory. Raises an exception if
    # there is no factory
    def factory_description_list
      raise "Must have a factory" unless has_factory?
      return factory.private_description_list
    end

    # Returns the description hash of the factory. Raises an exception if
    # there is no factory
    def factory_description_hash
      raise "Must have a factory" unless has_factory?
      return factory.private_description_hash
    end

    # Returns the factory description with the given name
    def factory_description(name)
      raise "Must have a factory" unless has_factory?
      return factory_description_hash.fetch(name)
    end

    # Returns the Class object associated with the given
    # name in the factory
    def factory_class(name)
      return factory_description(name).object_class
    end

    # Registers the given description to the factory. If no description
    # is given, the current class is registered.
    def register_class(desc = nil)
      raise "One of the superclasses should have a 'factory' statement" unless
        has_factory?
      desc = description if desc.nil?
      factory_description_list << desc
      factory_description_hash[desc.name] = desc
    end
    
    # Returns the Description of the class.
    def description
      return @description
    end
    
    # Sets the description of the class. It is probably way better to use
    # #describe, or write your own class method in the base class in the
    # case of a family of classes.
    def set_description(desc)
      @description = desc
      if has_factory? and factory.private_auto_register
        register_class
      end
    end
    

    # Registers an new parameter, with the following properties:
    # * _writer_ is the name of the method used to write that parameter;
    # * _reader_ the name of the method that returns its current value;
    # * _name_ is a short code-like name of the parameter (typically
    #   lowercase);
    # * _long_name_ is a more descriptive name, properly capitalized and
    #   localized if possible;
    # * _type_ is it's type. Please see the MetaBuilder::ParameterType
    #   for a better description of what is a type;
    # * _desc_ is a proper (short) description of the parameter,
    #   something that would fit on a What's this box, for instance.
    # * _attrs_ are optional parameters that may come useful, see
    #   Parameter#attributes documentation.
    #
    # You might want to use the  #param_reader, #param_writer, and
    # #param_accessor facilities that create the respective accessors
    # in addition. A typical example would be:
    #
    #  param :set_size, :size, 'size', "Size", {:type => :integer},
    #  "The size !!"
    # 
    def param(writer, reader, name, long_name, type, 
              desc = "", attrs = {})
      raise "Use describe first" if description.nil? 
      param = MetaBuilder::Parameter.new(name, writer, reader,
                                         long_name, 
                                         type, desc, attrs)
      description.add_param(param)
      return param
    end
    
    # The same as #param, but creates a attr_reader in addition
    def param_reader(writer, reader, name, long_name, type, 
              desc = "", attrs = {})
      p = param(writer, reader, name, long_name, type, desc, attrs)
      attr_reader reader
      return p
    end

    # The same as #param, except that _writer_ is made from _symbol_ by
    # appending a = at the end. An attr_writer is created for the _symbol_.
    def param_writer(symbol, name, long_name, type, 
                     desc = "", attrs = {})
      writer = (symbol.to_s + '=').to_sym
      p = param(writer, symbol, name, long_name, type, desc, attrs)
      attr_writer symbol
      return p
    end

    # The same as #param_writer, except that an attr_writer is created
    # for the _symbol_ instead of only a attr_writer. The most useful of
    # the four methods. Typical use:
    #
    #  param_accessor :name, 'name', "Object name", {:type => :string},
    #  "The name of the object"
    def param_accessor(symbol, name, long_name, type, 
                       desc = "", attrs = {})
      writer = (symbol.to_s + '=').to_sym
      p = param(writer, symbol, name, long_name, type, desc, attrs)
      attr_accessor symbol
      return p
    end
    
    # Creates a description attached to the current class. It needs to be
    # used before anything else.
    def describe(name, longname = name, desc = "")
      d = Description.new(self, name, longname, desc)
      set_description(d)
    end
      
    
    # Imports the given parameters directly from the parent class.
    # This function is quite naive and will not look further than
    # the direct #superclass.
    def inherit_parameters(*names)
      if self.superclass.respond_to?(:description)
        parents_params = self.superclass.description.param_hash
        for n in names
          if parents_params.key?(n)
            description.add_param(parents_params[n])
          else
            warn "Param #{n} not found"
          end
        end
      else
        warn "The parent class has no description"
      end
    end
    
    # Switches to the given ParameterGroup. If it doesn't exist, it is
    # created with the given parameters.
    def group(name, long_name = name, desc = nil)
      if not description.group_hash.has_key?(name)
        group = ParameterGroup.new(name, long_name, desc)
        description.add_group(group)
      end
      description.switch_to_group(name)
    end
    
  end
  
end

