/*
 *  methods for encryption/security mechanisms for cryptmount
 *  $Revision: 161 $, $Date: 2007-05-05 22:53:24 +0100 (Sat, 05 May 2007) $
 *  Copyright 2007 RW Penney
 */

/*
    This file is part of cryptmount

    cryptmount is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exemption, permission is granted to link cryptmount
    with the OpenSSL project's "OpenSSL" library and distribute
    the linked code without invoking clause 2(b) of the GNU GPL version 2.

    cryptmount is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with cryptmount; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <config.h>

#include <ctype.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "armour.h"
#include "blowfish.h"
#include "cryptmount.h"
#include "utils.h"
#ifdef TESTING
#  include "cmtesting.h"
#endif



/*
 *  ==== built-in sha1/blowfish key-management routines ====
 */


/*
 *  keyfile format is:
 *      char magic[7]="cm-blti";
 *      char version;
 *      uint16{LSB-first} keylength;
 *      char salt[kmblti_saltlen];
 *      [64-bit block][64-bit block][...];
 *      uint64{LSB-first} xor-checksum of key
 */

static const char kmblti_magstr[]="cm-blti";
static const char kmblti_version = (char)0;
static const size_t kmblti_maglen = 7;  /* = strlen(kmblti_magstr) */
enum {
    kmblti_saltlen = 10
};


static cm_bf_ctxt_t *kmblti_initcipher(const unsigned char *salt,
        const char *pass, size_t passlen, uint32_t iv[2])
{   cm_bf_ctxt_t *ctxt;
    cm_sha1_ctxt_t *md;
    unsigned char *ckey=NULL;
    size_t ckeysz;
    int i;

    /* generate cipher key by sha1-hashing password: */
    md = cm_sha1_init();
    for (i=16; i--; ) {
        cm_sha1_block(md, (const unsigned char*)pass, passlen);
        cm_sha1_block(md, salt, (size_t)kmblti_saltlen);
    }
    iv[0] = md->H[0];
    iv[1] = md->H[3];
    cm_sha1_block(md, salt, (size_t)kmblti_saltlen);
    cm_sha1_final(md, &ckey, &ckeysz);
    cm_sha1_free(md);

    /* initialized Blowfish cipher with hashed password: */
    ctxt = cm_bf_init(ckey, ckeysz);
    sec_free((void*)ckey);

    return ctxt;
}


static int kmblti_init_algs(void)
{
    /* nothing needed */
    return 0;
}


static int kmblti_free_algs(void)
{
    /* nothing needed */
    return 0;
}


static void kmblti_mk_default(keyinfo_t *keyinfo)
{
    if (keyinfo == NULL) return;

    if (keyinfo->digestalg == NULL) {
        keyinfo->digestalg = cm_strdup("sha1");
    }

    if (keyinfo->cipheralg == NULL) {
        keyinfo->cipheralg = cm_strdup("blowfish-cbc");
    }
}


static int kmblti_is_compat(const keyinfo_t *keyinfo, FILE *fp_key)
{   char buff[32];

    if (keyinfo->format != NULL) {
        return (strcmp(keyinfo->format, "builtin") == 0);
    } else {
        if (fp_key != NULL) {
            /* check header of existing key-file: */
            buff[0] = '\0';
            (void)fread((void*)buff, kmblti_maglen, (size_t)1, fp_key);
            return (strncmp(buff, kmblti_magstr, kmblti_maglen) == 0);
        }
    }

    /* be prepared to act as default key-manager: */
    return 1;
}


static int kmblti_needs_pw(const keyinfo_t *keyinf)
{
    return 1;
}


static int kmblti_get_key(const char *ident, const keyinfo_t *keyinfo,
            unsigned char **key, int *keylen, FILE *fp_key)
    /* extract key from sha1/blowfish encrypted file */
{   cm_bf_ctxt_t *ctxt;
    enum { BUFFSZ = 512 };
    unsigned char hbuff[kmblti_maglen+4], salt[kmblti_saltlen],
                *buff=NULL, *bptr;
    uint32_t iv[2], cv[2], pl[2];
    char *passwd=NULL;
    uint32_t chksum, chksum0;
    int cnt, eflag=ERR_NOERROR;

    *key = NULL; *keylen = 0;

#ifdef TESTING
    passwd = (char*)malloc((size_t)1024);
    strncpy(passwd, (test_ctxtptr->argpassword[0] != NULL ? test_ctxtptr->argpassword[0] : ""),
            1024);
#else
    if (km_get_passwd(ident, &passwd, 0, 0) != ERR_NOERROR) goto bail_out;
#endif

    /* read key header: */
    hbuff[0] = '\0';
    (void)fread((void*)hbuff, (size_t)1, kmblti_maglen, fp_key);
    if (strncmp((const char*)hbuff, kmblti_magstr, kmblti_maglen) != 0) {
        fprintf(stderr, "bad keyfile format (builtin)\n");
        eflag = ERR_BADFILE;
        goto bail_out;
    }
    fread((void*)hbuff, (size_t)1, (size_t)1, fp_key);
    if (hbuff[0] != '\0') {
        fprintf(stderr, "bad keyfile version [%d]\n", (int)hbuff[0]);
        eflag = ERR_BADFILE;
        goto bail_out;
    }
    fread((void*)hbuff, (size_t)2, (size_t)1, fp_key);
    *keylen = ((unsigned)hbuff[0]) | (((unsigned)hbuff[1]) << 8);

    /* read salt from keyfile: */
    fread((void*)salt, (size_t)1, sizeof(salt), fp_key);

    /* read encrypted key from keyfile: */
    ctxt = kmblti_initcipher(salt, passwd, strlen(passwd), iv);
    cnt = km_aug_keysz((unsigned)*keylen, 8u) / 8;
    buff = sec_realloc(buff, (size_t)(cnt * 8));
    fread((void*)buff, (size_t)8, (size_t)cnt, fp_key);
    bptr = buff;
    while (cnt--) {
        cv[0] = (((uint32_t)bptr[7]) << 24) | (((uint32_t)bptr[6]) << 16)
                | (((uint32_t)bptr[5]) << 8) | ((uint32_t)bptr[4]);
        cv[1] = (((uint32_t)bptr[3]) << 24) | (((uint32_t)bptr[2]) << 16)
                | (((uint32_t)bptr[1]) << 8) | ((uint32_t)bptr[0]);

        /* apply cipher block-chaining: */
        pl[0] = cv[0];
        pl[1] = cv[1];
        cm_bf_decipher(ctxt, pl, pl+1);
        pl[0] ^= iv[0];
        pl[1] ^= iv[1];
        iv[0] = cv[0];
        iv[1] = cv[1];

        bptr[7] = (unsigned char)((pl[0] >> 24) & 0xff);
        bptr[6] = (unsigned char)((pl[0] >> 16) & 0xff);
        bptr[5] = (unsigned char)((pl[0] >> 8) & 0xff);
        bptr[4] = (unsigned char)(pl[0] & 0xff);
        bptr[3] = (unsigned char)((pl[1] >> 24) & 0xff);
        bptr[2] = (unsigned char)((pl[1] >> 16) & 0xff);
        bptr[1] = (unsigned char)((pl[1] >> 8) & 0xff);
        bptr[0] = (unsigned char)(pl[1] & 0xff);

        bptr += 8;
    }
    cm_bf_free(ctxt);

    /* verify checksum: */
    if (!km_aug_verify(buff, (unsigned)*keylen, &chksum0, &chksum)) {
        fprintf(stderr, "checksum mismatch in keyfile (builtin, %x != %x)\n",
            (unsigned)chksum, (unsigned)chksum0);
        eflag = ERR_BADFILE;
    }

    if (keyinfo->maxlen > 0 && *keylen > keyinfo->maxlen) {
        *keylen = keyinfo->maxlen;
    }
    *key = (unsigned char*)sec_realloc((void*)*key, (size_t)*keylen);
    memcpy(*key, buff, (size_t)*keylen);

    if (ferror(fp_key) != 0) {
        fprintf(stderr, _("key-extraction failed for \"%s\"\n"),
                keyinfo->filename);
        eflag = ERR_BADDECRYPT;
    }

  bail_out:

    if (buff != NULL) sec_free(buff);

    return eflag;
}


static int kmblti_put_key(const char *ident, const keyinfo_t *keyinfo,
            const unsigned char *key, const int keylen, FILE *fp_key)
    /* store key in sha1/blowfish encrypted file */
{   cm_bf_ctxt_t *ctxt;
    unsigned char salt[kmblti_saltlen], hbuff[2], *buff=NULL, *bptr;
    uint32_t iv[2], cv[2];
    char *passwd=NULL;
    size_t buffsz, wrcnt;
    int cnt, eflag=ERR_NOERROR;

#ifdef TESTING
    passwd = sec_realloc((void*)passwd, (size_t)1024);
    strncpy(passwd, (test_ctxtptr->argpassword[1] != NULL ? test_ctxtptr->argpassword[1] : ""),
            1024);
#else
    if (km_get_passwd(ident, &passwd, 1, 1) != ERR_NOERROR) goto bail_out;
#endif

    /* write key header: */
    wrcnt = 0;
    wrcnt += fwrite((const void*)kmblti_magstr, kmblti_maglen, (size_t)1, fp_key);
    wrcnt += fwrite((const void*)&kmblti_version, (size_t)1, (size_t)1, fp_key);
    hbuff[0] = (keylen & 0xff); hbuff[1] = (keylen & 0xff00) >> 8;
    wrcnt += fwrite((const void*)hbuff, (size_t)2, (size_t)1, fp_key);

    /* generate salt & record in key-file: */
    get_randkey(salt, sizeof(salt));
    wrcnt += fwrite((const void*)salt, sizeof(salt), (size_t)1, fp_key);

    /* augment key with simple checksum: */
    buff = km_aug_key(key, (unsigned)keylen, 8u, &buffsz);

    /* write encrypted key into keyfile: */
    ctxt = kmblti_initcipher(salt, passwd, strlen(passwd), iv);
    cnt = buffsz / 8;
    bptr = buff;
    while (cnt--) {
        cv[0] = (((uint32_t)bptr[7]) << 24) | (((uint32_t)bptr[6]) << 16)
                | (((uint32_t)bptr[5]) << 8) | ((uint32_t)bptr[4]);
        cv[1] = (((uint32_t)bptr[3]) << 24) | (((uint32_t)bptr[2]) << 16)
                | (((uint32_t)bptr[1]) << 8) | ((uint32_t)bptr[0]);

        /* apply cipher block-chaining: */
        cv[0] ^= iv[0];
        cv[1] ^= iv[1];
        cm_bf_encipher(ctxt, cv, cv+1);
        iv[0] = cv[0];
        iv[1] = cv[1];

        bptr[7] = (unsigned char)((cv[0] >> 24) & 0xff);
        bptr[6] = (unsigned char)((cv[0] >> 16) & 0xff);
        bptr[5] = (unsigned char)((cv[0] >> 8) & 0xff);
        bptr[4] = (unsigned char)(cv[0] & 0xff);
        bptr[3] = (unsigned char)((cv[1] >> 24) & 0xff);
        bptr[2] = (unsigned char)((cv[1] >> 16) & 0xff);
        bptr[1] = (unsigned char)((cv[1] >> 8) & 0xff);
        bptr[0] = (unsigned char)(cv[1] & 0xff);

        bptr += 8;
    }
    wrcnt += fwrite((const void*)buff, buffsz, (size_t)1, fp_key);
    cm_bf_free(ctxt);

    if (wrcnt != 5) {
        fprintf(stderr, _("failed to create new key file\n"));
        eflag = ERR_BADFILE;
        goto bail_out;
    }

  bail_out:
    if (buff != NULL) sec_free(buff);
    if (passwd != NULL) sec_free(passwd);

    return eflag;
}


static void *kmblti_md_prepare()
{   cm_sha1_ctxt_t *mdcontext;

    mdcontext = cm_sha1_init();

    return (void*)mdcontext;
}


static void kmblti_md_block(void *state, unsigned char *buff, size_t len)
{   cm_sha1_ctxt_t *ctxt=(cm_sha1_ctxt_t*)state;

    /* apply built-in SHA1 hash to data-block: */
    cm_sha1_block(ctxt, buff, len);
}


static void kmblti_md_final(void *state, unsigned char **mdval, size_t *mdlen)
{   cm_sha1_ctxt_t *ctxt=(cm_sha1_ctxt_t*)state;

    cm_sha1_final(ctxt, mdval, mdlen);
}


static void kmblti_md_release(void *state)
{   cm_sha1_ctxt_t *ctxt=(cm_sha1_ctxt_t*)state;

    cm_sha1_free(ctxt);
}


keymanager_t keymgr_blti = {
    "builtin", 0,   kmblti_init_algs, kmblti_free_algs,
                    kmblti_mk_default, kmblti_is_compat, kmblti_needs_pw,
                    kmblti_get_key, kmblti_put_key,
                    kmblti_md_prepare, kmblti_md_block,
                    kmblti_md_final, kmblti_md_release,
    NULL
#ifdef TESTING
    , NULL, NULL
#endif
};



keymanager_t *kmblti_gethandle(void)
{
    return &keymgr_blti;
}

/*
 *  (C)Copyright 2007, RW Penney
 */
