//  crm_str_funcs.c  - Controllable Regex Mutilator,  version v1.0
//  Copyright 2001-2004  William S. Yerazunis, all rights reserved.
//  
//  This software is licensed to the public under the Free Software
//  Foundation's GNU GPL, version 2.  You may obtain a copy of the
//  GPL by visiting the Free Software Foundations web site at
//  www.fsf.org, and a copy is included in this distribution.  
//
//  Other licenses may be negotiated; contact the 
//  author for details.  
//
//  include some standard files
#include "crm114_sysincludes.h"

//  include any local crm114 configuration file
#include "crm114_config.h"

//  include the crm114 data structures file
#include "crm114_structs.h"

//  and include the routine declarations file
#include "crm114.h"

//    the command line argc, argv
extern int prog_argc;
extern char **prog_argv;

//    the auxilliary input buffer (for WINDOW input)
extern char *newinputbuf;

//    the globals used when we need a big buffer  - allocated once, used 
//    wherever needed.  These are sized to the same size as the data window.
extern char *inbuf;
extern char *outbuf;
extern char *tempbuf;


//     strnhash - generate the hash of a string of length N
//     goals - fast, works well with short vars includng 
//     letter pairs and palindromes, not crypto strong, generates
//     hashes that tend toward relative primality against common
//     hash table lengths (so taking the output of this function
//     modulo the hash table length gives a relatively uniform distribution
//
//     In timing tests, this hash function can hash over 10 megabytes
//     per second (using as text the full 2.4.9 linux kernel source)
//     hashing individual whitespace-delimited tokens, on a Transmeta
//     666 MHz.

/*****
long strnhash (char *str, long len)
{
  long i;
  long hval;
  char *hstr;
  char chtmp;

  // initialize hval
  hval= len;

  hstr = (char *) &hval;

  //  for each character in the incoming text:

  for ( i = 0; i < len; i++)
    {
      //    xor in the current byte against each byte of hval
      //    (which alone gaurantees that every bit of input will have
      //    an effect on the output)
      //hstr[0] = (hstr[0] & ( ~ str[i] ) ) | ((~ hstr [0]) & str[i]);
      //hstr[1] = (hstr[1] & ( ~ str[i] ) ) | ((~ hstr [1]) & str[i]);
      //hstr[2] = (hstr[2] & ( ~ str[i] ) ) | ((~ hstr [2]) & str[i]);
      //hstr[3] = (hstr[3] & ( ~ str[i] ) ) | ((~ hstr [3]) & str[i]);

      hstr[0] ^= str[i];
      hstr[1] ^= str[i];
      hstr[2] ^= str[i];
      hstr[3] ^= str[i];

      //    add some bits out of the middle as low order bits.
      hval = hval + (( hval >> 12) & 0x0000ffff) ;
		     
      //     swap bytes 0 with 3 
      chtmp = hstr [0];
      hstr[0] = hstr[3];
      hstr [3] = chtmp;

      //    rotate hval 3 bits to the left (thereby making the
      //    3rd msb of the above mess the hsb of the output hash)
      hval = (hval << 3 ) + (hval >> 29);
    }
  return (hval);
}
****/

// This is a more portable hash function, compatible with the original.
// It should return the same value both on 32 and 64 bit architectures.
// The return type was changed to unsigned long hashes, and the other
// parts of the code updated accordingly.
// -- Fidelis

unsigned long strnhash (char *str, long len)
{
  long i;
  // unsigned long hval;
  long hval;
  unsigned long tmp;

  // initialize hval
  hval= len;

  //  for each character in the incoming text:
  for ( i = 0; i < len; i++)
    {
      //    xor in the current byte against each byte of hval
      //    (which alone gaurantees that every bit of input will have
      //    an effect on the output)

      tmp = str[i];
      tmp = tmp | (tmp << 8) | (tmp << 16) | (tmp << 24);
      hval ^= tmp;

      //    add some bits out of the middle as low order bits.
      hval = hval + (( hval >> 12) & 0x0000ffff) ;

      //     swap most and min significative bytes 
      tmp = (hval << 24) | ((hval >> 24) & 0xff);
      hval &= 0x00ffff00;           // zero most and min significative bytes of hval
      hval |= tmp;                  // OR with swapped bytes

      //    rotate hval 3 bits to the left (thereby making the
      //    3rd msb of the above mess the hsb of the output hash)
      hval = (hval << 3) + (hval >> 29);
    }
  return (hval);
}

////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////
//
//    Cached mmap stuff.  Adapted from Win32 compatibility code from
//    Barry Jaspan.  Altered to not reveal the difference between a
//    mapped file pointer and one of Barry's 'map' structs.  In this
//    code (unlike Barry's patches), all that is ever seen are
//    pointers to memory (i.e. crm_mmap and crm_munmap have the same
//    API and semantics as with the libc mmap() and munmap() calls),
//    no structs are ever seen by the callers of this code.
//
//     Bugs in the POSIX code are my fault.  Bugs in the WIN32 code are
//     either mine or his.  So there.
//

///////////////////////////////////////////////////////////////////////////
//
//     This code section (from this line to the line below that states
//     that it is the end of the dual-licensed code section) is
//     copyright and owned by William S. Yerazunis.  In return for
//     addition of significant derivative work, Barry Jaspan is hereby
//     granted a full unlimited license to use this code section,
//     including license to relicense under other licenses.
//
////////////////////////////////////////////////////////////////////////////


//     An mmap cell.  This is how we cache.
//
typedef struct prototype_crm_mmap_cell 
{
  char *name;
  long size;
  void *addr;
  int unmap_count;         //  counter - unmap this after UNMAP_COUNT_MAX
  struct prototype_crm_mmap_cell *next, *prev;
#ifdef POSIX
    int fd;
#endif
#ifdef WIN32
    HANDLE fd, mapping;
#endif
} CRM_MMAP_CELL;

CRM_MMAP_CELL *cache = NULL;
char one_byte;


//////////////////////////////////////
//
//     Force an unmap (don't look at the unmap_count, just do it)
//     Watch out tho- this takes a CRM_MMAP_CELL, not a *ptr, so don't
//     call it from anywhere except inside this file.
//
void crm_unmap_file_internal ( CRM_MMAP_CELL *map)
{

#ifdef POSIX
  msync (map->addr, map->size, MS_SYNC);
  munmap (map->addr, map->size);

     //    Because mmap/munmap doesn't set atime, nor set the "modified"
     //    flag, some network filesystems will fail to mark the file as
     //    modified and so their cacheing will make a mistake.
     //
     //    The fix is to do a trivial read/write on the .css ile, to force
     //    the filesystem to repropagate it's caches.
     //
  {
    FEATURE_HEADER_STRUCT foo;
    lseek (map->fd, 0, SEEK_SET);
    read (map->fd, &foo, sizeof(foo));
    lseek (map->fd, 0, SEEK_SET);
    write (map->fd, &foo, sizeof(foo));
  }
  close (map->fd);
#endif


#ifdef WIN32
    FlushViewOfFile(map->addr, 0);
    UnmapViewOfFile(map->addr);
    CloseHandle(map->mapping);
    CloseHandle(map->fd);
#endif

}

//////////////////////////////////////////////////////
//
//      This is the wrapper around the "traditional" file unmap, but
//      does cacheing.  It keeps count of unmappings and only unmaps
//      when it needs to.
//
void crm_munmap_file (void *addr)
{
  CRM_MMAP_CELL *p;

  //     step 1- search the mmap cache to see if we actually have this 
  //     mmapped
  //   
  p = cache;
  while ( p != NULL && p->addr != addr)
    p = p->next;

  if ( ! p )
    {
      nonfatalerror ("Internal fault - this code has tried to unmap memory "
		     "that it never mapped in the first place.  ",
		     "Please file a bug report. ");
      return;
    }
    
  //   Step 2: we have the mmap cell of interest.  Do the right thing.
  //
  p->unmap_count = (p->unmap_count) + 1;
  if (p->unmap_count > UNMAP_COUNT_MAX) 
    {
      crm_unmap_file_internal (p);
      //
      //    File now unmapped, take the mmap_cell out of the cache
      //    list as well.
      //
      if (p->prev != NULL)
	p->prev->next = p->next;
      else
	cache = p->next;
      if (p->next != NULL)
	p->next->prev = p->prev;
      free(p->name);
      free(p);
    }
}


/////////////////////////////////////////////////////////
//
//           Force an Unmap on every mmapped memory area we know about
void crm_munmap_all()
{
  while (cache != NULL) {
    cache->unmap_count = UNMAP_COUNT_MAX + 1;
    crm_munmap_file (cache->addr);
  }
}


//////////////////////////////////////////////////////////
//
//           MMap a file in (or get the map from the cache, if possible)
//             (length is how many bytes to get mapped, remember!)
//
void *crm_mmap_file (void *addr, char *filename, long length )
{
  CRM_MMAP_CELL *p;
  long pagesize = 0;
  
  pagesize = 0;
  //    Search for the file - if it's already mmaped, just return it.
  for (p = cache; p != NULL; p = p->next) 
    {
      if (strcmp(p->name, filename) == 0) 
	{
	  return (p->addr);
	}
    }
  
  //     No, not found in our cache chain.  We need to add a cell, and
  //     mmap the file.
  //  
  p = (void *) malloc( sizeof ( CRM_MMAP_CELL) );
  if (p == NULL)
    {
      untrappableerror(" Unable to malloc enough memory for mmap cache.  ",
		       " This is unrecoverable.  Sorry.");
      return NULL;
    }
  p->name = strdup(filename);
  p->size = length;
  
#ifdef POSIX
  
  p->fd = open (filename, O_RDWR);
  if (p->fd < 0) 
    {
      free(p->name);
      free(p);
      return NULL;
    }
  p->addr = mmap (NULL, 
		  length, 
		  PROT_READ + PROT_WRITE,
		  MAP_SHARED,
		  p->fd,
		  0);
  if (p->addr == MAP_FAILED) 
    {
      close(p->fd);
      free(p->name);
      free(p);
      return NULL;
    }
  
  
#endif       
#ifdef WIN32
  p->fd = CreateFile(filename, GENERIC_READ | GENERIC_WRITE, 0,
		     NULL, OPEN_EXISTING, 0, NULL);
  if (p->fd == INVALID_HANDLE_VALUE) 
    {
      free(p->name);
      free(p);
      return NULL;
    }
  p->mapping = CreateFileMapping(p->fd, 
				 NULL, 
				 PAGE_READWRITE, 0, length,
				 NULL);
  if (p->mapping == NULL) 
    {
      CloseHandle(p->fd);
      free(p->name);
      free(p);
      return NULL;
    }
  p->addr = MapViewOfFile(p->mapping, FILE_MAP_WRITE, 0, 0, 0);
  if (p->addr == NULL) 
    {
      CloseHandle(p->mapping);
      CloseHandle(p->fd);
      free(p->name);
      free(p);
      return NULL;
    }
  
  {
    SYSTEM_INFO info;
    GetSystemInfo(&info);
    pagesize = info.dwPageSize;
  }
  
  //  Jaspan-san says force-loading every page is a good thing
  //  under Windows.  I know it's a bad thing under Linux,
  //  so we'll only do it under Windows.
  {
    char *addr = (char *) p->addr;
    long i;
    for (i = 0; i < p->size; i += pagesize)
      one_byte = addr[i];
  }
#endif
	       
  //   Now, insert this fresh mmap into the cache list
  //
  p->unmap_count = 0;
  p->prev = NULL;
  p->next = cache;
  if (cache != NULL)
    cache->prev = p;
  cache = p;
  return p->addr;
}

#ifdef WIN32
clock_t times(TMS_STRUCT *buf)
{
  FILETIME create, exit, kern, user;
  if (GetProcessTimes(GetCurrentProcess(), &create, &exit, &kern, &user)) 
    {
      buf->tms_utime = user.dwLowDateTime;
      buf->tms_stime = kern.dwLowDateTime;
      buf->tms_cutime = 0;
      buf->tms_cstime = 0;
      return GetTickCount();
    }
  return -1;
}
#endif

///////////////////////////////////////////////////////////////////////
//
//         End of section of code dual-licensed to Yerazunis and Jaspan
//
///////////////////////////////////////////////////////////////////////
