/* mkcommon.c -- functions shared by the mkxxx tools
   Copyright (C) 2000-2003 Jens Granseuer

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>

#ifdef HAVE_LIBZ
# include <zlib.h>
#endif

#include "SDL.h"
#include "SDL_endian.h"

static int compress_data( unsigned char *in, unsigned long inlen, unsigned char **out, unsigned long *outlen );

struct PicHeader {
  unsigned short width;
  unsigned short height;
  unsigned char bpp;           // currently only 8 (= 256 colors) supported
  unsigned char flags;
  unsigned long packed_size;
  unsigned short colors;       // if bpp == 8 number of colors in palette else not included in file
  unsigned char ckey_r;        // transparent color if flags has PH_TRANSPARENT set
  unsigned char ckey_g;        //                   else not included in file
  unsigned char ckey_b;
};

#define PH_TRANSPARENT  0x01

int save_image_data( const char *filename, int trans, SDL_RWops *out ) {
  int rc = 0, i, j;
  unsigned long size;
  unsigned char *packbuf, *upackbuf;

  SDL_Surface *bmp = SDL_LoadBMP( filename );
  if ( bmp ) {
    if ( bmp->format->BitsPerPixel == 8 ) {
      SDL_Palette *palette = bmp->format->palette;
      struct PicHeader pict = { bmp->w, bmp->h, 8, (trans ? PH_TRANSPARENT : 0),
                                0, palette->ncolors, palette->colors[0].r,
                                palette->colors[0].g, palette->colors[0].b };

      size = bmp->w * bmp->h;
      upackbuf = (unsigned char *)malloc( size );

      /* copy surface lines to buffer */
      for ( i = 0; i < bmp->h; ++i ) {
        unsigned char *pix = (Uint8 *)bmp->pixels + i * bmp->pitch;
        for ( j = 0; j < bmp->w; ++j ) {
          upackbuf[j + i * bmp->w] = *pix++;
        }
      }

      rc = compress_data( upackbuf, size, &packbuf, &pict.packed_size );
      if ( rc ) {
        fprintf( stderr, "Error compressing data\n" );
        free( upackbuf );
        SDL_RWclose( out );
        return rc;
      }

      SDL_WriteLE16( out, pict.width );
      SDL_WriteLE16( out, pict.height );
      SDL_RWwrite( out, &pict.bpp, 1, 1 );
      SDL_RWwrite( out, &pict.flags, 1, 1 );
      SDL_WriteLE32( out, pict.packed_size );
      SDL_WriteLE16( out, pict.colors );   /* only for <=256 colors (bpp = 8) */

      if ( pict.flags & PH_TRANSPARENT ) {
        // transparent color first
        SDL_RWwrite( out, &pict.ckey_r, 1, 1 );
        SDL_RWwrite( out, &pict.ckey_g, 1, 1 );
        SDL_RWwrite( out, &pict.ckey_b, 1, 1 );
      }

      for ( i=0; i<palette->ncolors; ++i ) {
        SDL_RWwrite( out, &palette->colors[i].r, 1, 1 );
        SDL_RWwrite( out, &palette->colors[i].g, 1, 1 );
        SDL_RWwrite( out, &palette->colors[i].b, 1, 1 );
      }

      SDL_RWwrite( out, packbuf, 1, pict.packed_size );
      if ( pict.packed_size != size ) free( packbuf );
    } else {
      fprintf(stderr, "Couldn't find palette\n" );
      rc = -1;
    }
    SDL_FreeSurface( bmp );
  } else {
    fprintf(stderr, "Couldn't open BMP %s\n", filename );
    rc = -1;
  }
  return rc;
}

static int compress_data( unsigned char *in, unsigned long inlen, unsigned char **out, unsigned long *outlen ) {
  int rc = 0;

#ifdef HAVE_LIBZ
  unsigned long bufsize = inlen + (inlen/1000) + 13;
  unsigned char *buf = (unsigned char *)malloc( bufsize );
#endif

  *out = in;
  *outlen = inlen;

#ifdef HAVE_LIBZ
  if ( buf ) {
    rc = compress2( buf, &bufsize, in, inlen, Z_BEST_COMPRESSION );
    if ( rc != Z_OK ) free( buf );
    else {
      *out = buf;
      *outlen = bufsize;
    }
  }
#endif

  return rc;
}

