/* $Id: visir_pfits.c,v 1.46 2012/02/02 10:22:05 jtaylor Exp $
 *
 * This file is part of the VISIR Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jtaylor $
 * $Date: 2012/02/02 10:22:05 $
 * $Revision: 1.46 $
 * $Name: visir-3_5_1 $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Defines
 -----------------------------------------------------------------------------*/

#define visir_pfits_get_double(KEY) irplib_pfits_get_double(self, KEY)
#define visir_pfits_get_int(KEY)    irplib_pfits_get_int(self, KEY)
#define visir_pfits_get_string(KEY) irplib_pfits_get_string(self, KEY)


/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <string.h>
#include <assert.h>
#include <math.h>
#include <cpl.h>

#include "irplib_pfits.h"

#include "visir_utils.h"
#include "visir_pfits.h"


/*-----------------------------------------------------------------------------
                        Private function prototypes
 -----------------------------------------------------------------------------*/

static const char * visir_pfits_get_slitname(const cpl_propertylist *);
static double visir_pfits_get_slitwidth_raw(const cpl_propertylist *);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup visir_pfits    FITS header protected access
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    The airmass start
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_airmass_start(const cpl_propertylist * self)
{
    return visir_pfits_get_double("ESO TEL AIRM START");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The airmass end
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_airmass_end(const cpl_propertylist * self)
{
    return visir_pfits_get_double("ESO TEL AIRM END");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The alpha angle
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_alpha(const cpl_propertylist * self)
{
    return visir_pfits_get_double("ESO TEL TARG OFFSETALPHA");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The arcfile   
  @param    self       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char * visir_pfits_get_arcfile(const cpl_propertylist * self)
{
    return visir_pfits_get_string("ARCFILE");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The chopping direction
  @param    self       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char * visir_pfits_get_chopnod_dir(const cpl_propertylist * self)
{
    return visir_pfits_get_string(VISIR_PFITS_STRING_CHOPNOD_DIR);
}


/*----------------------------------------------------------------------------*/
/**
  @brief    The chopping throw
  @param    self       property list to read from
  @return   Chopping throw
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_chop_throw(const cpl_propertylist * self)
{
    return visir_pfits_get_double(VISIR_PFITS_DOUBLE_CHOP_THROW);
}


/*----------------------------------------------------------------------------*/
/**
  @brief    The chopping throw in pixels
  @param    self       property list to read from
  @return   Chopping throw in pixels
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_chop_pthrow(const cpl_propertylist * self)
{
    const double throw = visir_pfits_get_chop_throw(self);
    const char * sscale = visir_pfits_get_pixscale(self);
    const double pscale       = sscale ? atof(sscale) : 0.0;
    return pscale > 0.0 ? throw / pscale : 0.0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    The number of chopping cycles
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int visir_pfits_get_chop_ncycles(const cpl_propertylist * self)
{
    return visir_pfits_get_int("ESO DET CHOP NCYCLES");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The cumulative offset in X
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_cumoffsetx(const cpl_propertylist * self)
{
    return visir_pfits_get_double("ESO SEQ CUMOFFSETX");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The cumulative offset in Y
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_cumoffsety(const cpl_propertylist * self)
{
    return visir_pfits_get_double("ESO SEQ CUMOFFSETY");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The date of observation 
  @param    self       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char * visir_pfits_get_date_obs(const cpl_propertylist * self)
{
    return visir_pfits_get_string("DATE-OBS");
}

/*----------------------------------------------------------------------------*/
/** 
  @brief    The DELTA keyword in a VISIR header
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_delta(const cpl_propertylist * self)
{
    return visir_pfits_get_double("ESO TEL TARG OFFSETDELTA");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The DEC
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_dec(const cpl_propertylist * self)
{
    return visir_pfits_get_double("DEC");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The DIT
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_dit(const cpl_propertylist * self)
{
    return visir_pfits_get_double("ESO DET DIT");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The number of the current exposition
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int visir_pfits_get_expno(const cpl_propertylist * self)
{
    return visir_pfits_get_int("ESO TPL EXPNO");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The filter
  @param    self       property list to read from
  @return   pointer to statically allocated character string or NULL
 */
/*----------------------------------------------------------------------------*/
const char * visir_pfits_get_filter(const cpl_propertylist * self)
{

    /* Get the instrument mode */
    const char * val   = visir_pfits_get_insmode(self);

    
    if (val == NULL) return NULL ;

    /* Imaging mode */
    if (!strcmp(val, "IMG"   ))
        return visir_pfits_get_string("ESO INS FILT1 NAME");
    /* Spectro mode */
    if (!strcmp(val, "SPC"   ))
        return visir_pfits_get_string("ESO INS FILT2 NAME");
    /* Spectro mode */
    if (!strcmp(val, "SPCIMG"))
        return visir_pfits_get_string("ESO INS FILT2 NAME");

    cpl_ensure(0, CPL_ERROR_UNSUPPORTED_MODE, NULL);

}

/*----------------------------------------------------------------------------*/
/**
  @brief    The focus
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_focus(const cpl_propertylist * self)
{
    return visir_pfits_get_double("ESO TEL FOCU LEN");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The exposure time
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_exptime(const cpl_propertylist * self)
{
    return visir_pfits_get_double("EXPTIME");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The frame type
  @param    self       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char * visir_pfits_get_frame_type(const cpl_propertylist * self)
{
    return visir_pfits_get_string("ESO DET FRAM TYPE");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The name of grating 1
  @param    self       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char * visir_pfits_get_grat1_name(const cpl_propertylist * self)
{
    return visir_pfits_get_string("ESO INS GRAT1 NAME");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The mode
  @param    self       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char * visir_pfits_get_insmode(const cpl_propertylist * self)
{
    return visir_pfits_get_string("ESO INS MODE");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The instrument
  @param    self       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char * visir_pfits_get_instrument(const cpl_propertylist * self)
{
    return visir_pfits_get_string("INSTRUME");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The MJD-OBS
  @param    self       property list to read from
  @return   the requested value 
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_mjdobs(const cpl_propertylist * self)
{
    return visir_pfits_get_double("MJD-OBS");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The mode name
  @param    self       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char * visir_pfits_get_mode(const cpl_propertylist * self)
{
    return visir_pfits_get_string("ESO DET MODE NAME");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The INS.MONOC1.POS
  @param    self       property list to read from
  @return   the requested value 
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_monoc_pos(const cpl_propertylist * self)
{
    return visir_pfits_get_double("ESO INS MONOC1 POS");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The NDIT keyword
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int visir_pfits_get_ndit(const cpl_propertylist * self)
{   
    return visir_pfits_get_int("ESO DET NDIT");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The NAXIS3 key
  @param    self       property list to read from
  @return   the requested value
 */ 
/*----------------------------------------------------------------------------*/
int visir_pfits_get_naxis3(const cpl_propertylist * self)
{
    return visir_pfits_get_int("NAXIS3");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The number of exposures
  @param    self       property list to read from
  @return   the requested value
 */ 
/*----------------------------------------------------------------------------*/
int visir_pfits_get_numbexp(const cpl_propertylist * self) 
{
    return visir_pfits_get_int("ESO TPL NEXP");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The OBS ID
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int visir_pfits_get_obs_id(const cpl_propertylist * self)
{
    return visir_pfits_get_int("ESO OBS ID");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The nodding position
  @param    self       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char * visir_pfits_get_nodpos(const cpl_propertylist * self)
{
    return visir_pfits_get_string("ESO SEQ NODPOS");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The pixel scale
  @param    self       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char *  visir_pfits_get_pixscale(const cpl_propertylist * self)
{
    return visir_pfits_get_string("ESO INS PFOV");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The pixel spacing
  @param    self       property list to read from
  @return   the requested value 
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_pixspace(const cpl_propertylist * self)
{
    return visir_pfits_get_double("ESO DET CHIP PXSPACE");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The RA
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_ra(const cpl_propertylist * self)
{
    return visir_pfits_get_double("RA");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The slit width in Arcseconds
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_slitwidth(const cpl_propertylist * self)
{
    const char * slitname = visir_pfits_get_slitname(self);
    double slitwidth = visir_pfits_get_slitwidth_raw(self);


    if (!cpl_error_get_code()) {

        static const char * slit_name[] = {"CU_0.12", "CU_0.14", "CU_0.16",
                                           "CU_0.20", "CU_0.25", "CU_0.30",
                                           "CU_0.40", "CU_0.50", "CU_0.60",
                                           "CU_0.70", "CU_0.80", "CU_1.00",
                                           "CU_1.20", "CU_1.60", "ST_0.14",
                                           "ST_0.16", "ST_0.20", "ST_0.30",
                                           "ST_0.40", "ST_0.50", "ST_0.60"};

        static double slit_width[] = {0.12, 0.14, 0.16,
                                      0.20, 0.25, 0.30,
                                      0.40, 0.50, 0.60,
                                      0.70, 0.80, 1.00,
                                      1.20, 1.60, 0.14,
                                      0.16, 0.20, 0.30,
                                      0.40, 0.50, 0.60};

        static const int slit_nums = (int)(sizeof(slit_width)/sizeof(double));

        int i;

        for (i=0; i < slit_nums; i++)
            if (strstr(slitname, slit_name[i])) break;

        if (i < slit_nums && fabs(slit_width[i] - slitwidth) < 0.001) {
            /* The Slit width is supposed to be in Arcseconds - as written in
               the comment of the FITS card.
               However, if the Slit width matches the Slit name then the Slit
               width is actually in unit millimeter and is converted */

            cpl_msg_warning(cpl_func,"The Width of slit %s is written in the "
                            "FITS card in units mm. Converting to ArcSecs: "
                            "%g => %g", slitname, slitwidth, slitwidth * 2.5);
            slitwidth *= 2.5;
        }
    }

    return slitwidth;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The std star name 
  @param    self       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char * visir_pfits_get_starname(const cpl_propertylist * self)
{
    return visir_pfits_get_string("ESO OBS TARG NAME");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The spectral resolution
  @param    self       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char * visir_pfits_get_resol(const cpl_propertylist * self)
{
    return visir_pfits_get_string("ESO INS RESOL");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The telescope (M1) temperature [Celcius]
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_temp(const cpl_propertylist * self)
{
    return visir_pfits_get_double("ESO TEL TH M1 TEMP");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The VOLT1.DCTA9
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_volt1dcta9(const cpl_propertylist * self)
{
    return visir_pfits_get_double("ESO DET VOLT1 DCTA9");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The VOLT1.DCTB9
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_volt1dctb9(const cpl_propertylist * self)
{
    return visir_pfits_get_double("ESO DET VOLT1 DCTB9");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The VOLT2.DCTA9
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_volt2dcta9(const cpl_propertylist * self)
{
    return visir_pfits_get_double("ESO DET VOLT2 DCTA9");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The VOLT2.DCTB9
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_volt2dctb9(const cpl_propertylist * self)
{
    return visir_pfits_get_double("ESO DET VOLT2 DCTB9");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The central wavelength
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double visir_pfits_get_wlen(const cpl_propertylist * self)
{
    return visir_pfits_get_double("ESO INS GRAT1 WLEN");
}

/**@}*/

/*----------------------------------------------------------------------------*/
/**
  @brief    The slit width
  @param    self       property list to read from
  @return   the requested value
  @note  The unit is Arcseconds (or for old files mm)
 */
/*----------------------------------------------------------------------------*/
static double visir_pfits_get_slitwidth_raw(const cpl_propertylist * self)
{
    return visir_pfits_get_double("ESO INS SLIT1 WID");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The slit name
  @param    self       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
static const char * visir_pfits_get_slitname(const cpl_propertylist * self)
{
    return visir_pfits_get_string("ESO INS SLIT1 NAME");
}
