/* $Id: visir_util_inputs.c,v 1.56 2010/07/22 11:52:28 llundin Exp $
 *
 * This file is part of the VISIR Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA
 */

/*
 * $Author: llundin $
 * $Date: 2010/07/22 11:52:28 $
 * $Revision: 1.56 $
 * $Name: visir-3_5_1 $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "visir_recipe.h"



/*-----------------------------------------------------------------------------
                                Defines
 -----------------------------------------------------------------------------*/

#define RECIPE_STRING   "visir_util_inputs"

/*-----------------------------------------------------------------------------
                            Private Functions prototypes
 -----------------------------------------------------------------------------*/

static cpl_error_code visir_util_inputs_save(cpl_frameset *,
                                             const cpl_parameterlist *,
                                             const cpl_imagelist *);

VISIR_RECIPE_DEFINE(visir_util_inputs,
                    VISIR_PARAM_NODPOS | VISIR_PARAM_AUTOBPM |  
                    VISIR_PARAM_STRIPITE | VISIR_PARAM_STRIPMOR |
                    VISIR_PARAM_GLITCH | VISIR_PARAM_PURGE,
                    "Conversion of raw images to nodded images",
                    "The files listed in the Set Of Frames (sof-file) "
                    "must be tagged:\n"
                    "VISIR-raw-file.fits " VISIR_UTIL_INPUTS_RAW
                    "\nAdditionally, a bad pixel map\n"
                    "may be added to the Set Of Frames "
                    "with tag: " VISIR_CALIB_BPM "."
                    "\nAdditionally, a flat field\n"
                    "may be added to the Set Of Frames "
                    "with tag: " VISIR_CALIB_FLAT ".\n"
                    "\nThe product(s) will have a FITS card\n"
                    "'HIERARCH ESO PRO CATG' with a value of:\n"
                    VISIR_UTIL_INPUTS_COMBINED_PROCATG);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup visir_util_inputs   Conversion of raw images to nodded images
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                Functions code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    The recipe data reduction part is implemented here 
  @param    parlist     the parameters list
  @param    framelist   the frames list
  @return   0 iff everything is ok
 */
/*----------------------------------------------------------------------------*/
static int visir_util_inputs(cpl_frameset            * framelist,
                             const cpl_parameterlist * parlist)
{
    irplib_framelist * allframes = NULL;
    irplib_framelist * rawframes = NULL;
    const char      *   badpix;
    const char      *   flat;
    cpl_imagelist   *   nodded = NULL;
    

    /* Identify the RAW and CALIB frames in the input frameset */
    skip_if (visir_dfs_set_groups(framelist));

    /* Objects observation */
    allframes = irplib_framelist_cast(framelist);
    skip_if(allframes == NULL);
    rawframes = irplib_framelist_extract(allframes, VISIR_UTIL_INPUTS_RAW);
    skip_if (rawframes == NULL);
    
    /* Bad pixels calibration file */
    badpix = irplib_frameset_find_file(framelist, VISIR_CALIB_BPM);

    /* Flatfield calibration file */
    flat = irplib_frameset_find_file(framelist, VISIR_CALIB_FLAT);

    skip_if(irplib_framelist_load_propertylist_all(rawframes, 0,
                                                   visir_property_regexp,
                                                   CPL_FALSE));

    /* Combine the frames */
    cpl_msg_info(cpl_func, "Construct the nodded images");
    nodded = visir_inputs_combine(RECIPE_STRING, parlist, rawframes, badpix, flat,
                                  NULL, CPL_FALSE, 0.0, 0);
    if (nodded == NULL) {
        cpl_msg_error(cpl_func, "Cannot combine the input frames");
        skip_if(1);
    }

    /* Save the nodded images */
    cpl_msg_info(cpl_func, "Save the results");
    skip_if (visir_util_inputs_save(framelist, parlist, nodded));

    end_skip;

    irplib_framelist_delete(allframes);
    irplib_framelist_delete(rawframes);

    cpl_imagelist_delete(nodded);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Save the visir_util_inputs recipe products on disk
  @param    set     the input frame set
  @param    parlist the input list of parameters
  @param    nodded  the nodded cube
  @return   0 iff everything is ok
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code visir_util_inputs_save(cpl_frameset            * set,
                                             const cpl_parameterlist * parlist,
                                             const cpl_imagelist     * nodded)
{

    int    i;
    char * filename = NULL;


    bug_if (nodded == NULL);
   
    /* Save all the images of the image set */
    for (i=0 ; i < cpl_imagelist_get_size(nodded) ; i++) {
        cpl_free(filename);
        filename = cpl_sprintf(RECIPE_STRING "%d" CPL_DFS_FITS, i+1);
        skip_if (irplib_dfs_save_image(set, parlist, set,
                                   cpl_imagelist_get_const(nodded, i),
                                   CPL_BPP_IEEE_FLOAT, RECIPE_STRING,
                                   VISIR_UTIL_INPUTS_COMBINED_PROCATG,
                                   NULL, NULL, visir_pipe_id, filename));
    }

    end_skip;

    cpl_free(filename);

    return cpl_error_get_code();
}
