/* $Id: visir_util_clip.c,v 1.24 2012/02/02 10:27:28 jtaylor Exp $
 *
 * This file is part of the VISIR Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jtaylor $
 * $Date: 2012/02/02 10:27:28 $
 * $Revision: 1.24 $
 * $Name: visir-3_5_1 $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "visir_recipe.h"

#include <string.h>

/*-----------------------------------------------------------------------------
                                Defines
 -----------------------------------------------------------------------------*/

#define RECIPE_STRING   "visir_util_clip"

#ifndef VISIR_UTIL_CLIP_KEEPFRAC
#define VISIR_UTIL_CLIP_KEEPFRAC 0.9
#endif
#ifndef VISIR_UTIL_CLIP_KAPPA
#define VISIR_UTIL_CLIP_KAPPA 3.0
#endif

#ifndef VISIR_UTIL_CLIP_MAXITE
#define VISIR_UTIL_CLIP_MAXITE 3
#endif

#define VISIR_UTIL_CLIP_QC_MAP_MAX    "ESO QC CONTRIBUTION MAX"
#define VISIR_UTIL_CLIP_QC_MAP_MEAN   "ESO QC CONTRIBUTION MEAN"
#define VISIR_UTIL_CLIP_QC_MAP_MEDIAN "ESO QC CONTRIBUTION MEDIAN"

/*-----------------------------------------------------------------------------
                            Private Functions prototypes
 -----------------------------------------------------------------------------*/

static
cpl_error_code visir_util_clip_kappa_sigma_double(cpl_imagelist *,
                                                  cpl_imagelist *,
                                                  double, double, int,
                                                  const int *);
static
cpl_error_code visir_util_clip_kappa_sigma_float(cpl_imagelist *,
                                                 cpl_imagelist *,
                                                 double, double, int,
                                                 const int *);
static
cpl_error_code visir_util_clip_kappa_sigma_int(cpl_imagelist *,
                                               cpl_imagelist *,
                                               double, double, int,
                                               const int *);

static cpl_error_code visir_util_clip_one(cpl_frameset *,
                                          irplib_framelist *,
                                          irplib_framelist *,
                                          int, cpl_boolean,
                                          const cpl_parameterlist *);

static cpl_error_code visir_util_clip_kappa_sigma(cpl_imagelist *,
                                                  cpl_imagelist *,
                                                  const cpl_parameterlist *,
                                                  const int * shifts);


cpl_recipe_define(visir_util_clip, VISIR_BINARY_VERSION,                
                  "Lars Lundin", PACKAGE_BUGREPORT, "2011",
                  "Kappa-sigma clipping of outliers for each pixel",
                  "The files listed in the Set Of Frames (sof-file) "
                  "must be tagged pair-wise:\n"
                  "VISIR-raw-file.fits " VISIR_UTIL_INPUTS_RAW "\n"
                  "VISIR-bpm-file.fits " VISIR_CALIB_BPM "\n"
                  "\nThe product(s) will have a FITS card\n"
                  "'HIERARCH ESO PRO CATG' with a value of:\n"
                  VISIR_IMG_CLIPPED_PROCATG "\n"
                  "The outliers are marked as rejected in the matching\n"
                  "bad pixel map.");

/*----------------------------------------------------------------------------*/
/**
 * @defgroup visir_util_clip   Clipping of outlier pixels
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                Functions code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief    Fill the recipe parameterlist
  @param    self  The parameterlist
  @return   0 iff everything is ok

 */
/*----------------------------------------------------------------------------*/
static
cpl_error_code visir_util_clip_fill_parameterlist(cpl_parameterlist * self)
{

    const char * context = PACKAGE "." RECIPE_STRING;
    cpl_error_code err;

    cpl_ensure_code(self, CPL_ERROR_NULL_INPUT);

    /* Fill the parameters list */

    /* --keepfrac */
    err = irplib_parameterlist_set_double(self, PACKAGE, RECIPE_STRING,
                                          "keepfrac", VISIR_UTIL_CLIP_KEEPFRAC,
                                          NULL, context, "The fraction of "
                                          "pixels to keep for the initial"
                                          "median");
    cpl_ensure_code(!err, err);


    /* --kappa */
    err = irplib_parameterlist_set_double(self, PACKAGE, RECIPE_STRING,
                                          "kappa", VISIR_UTIL_CLIP_KAPPA,
                                          NULL, context, "Clip outside "
                                          "+ or - kappa * sigma "
                                          "(the standard deviation)");
    cpl_ensure_code(!err, err);


    /* --maxite */
    err = irplib_parameterlist_set_int(self, PACKAGE, RECIPE_STRING,
                                       "maxite", VISIR_UTIL_CLIP_MAXITE, NULL,
                                       context, "Max number of kappa-sigma "
                                       "clipping iterations");

    /* --shift-beams */
    err = irplib_parameterlist_set_bool(self, PACKAGE, RECIPE_STRING,
                                        "shift-beams", CPL_TRUE, NULL,
                                        context, "Account for movements of the "
                                        "object defined in CRPIX[12]");
    cpl_ensure_code(!err, err);

    /* --error */
    err = irplib_parameterlist_set_bool(self, PACKAGE, RECIPE_STRING,
                                        "error", CPL_TRUE, NULL,
                                        context, "Output clipped standard "
                                        "deviation as error map");
    cpl_ensure_code(!err, err);

    /* --badimage */
    err = irplib_parameterlist_set_double(self, PACKAGE, RECIPE_STRING,
                                          "badimage", 0.2, NULL,
                                          context, "If percentage of clipped "
                                          "pixels above this value the whole "
                                          "image is considered bad");
    cpl_ensure_code(!err, err);

    return CPL_ERROR_NONE;

}

/*----------------------------------------------------------------------------*/
/**
  @brief    The recipe data reduction part is implemented here 
  @param    framelist   the frames list
  @param    parlist     the parameters list
  @return   0 iff everything is ok
 */
/*----------------------------------------------------------------------------*/
static int visir_util_clip(cpl_frameset            * framelist,
                           const cpl_parameterlist * parlist)
{
    cpl_errorstate     cleanstate = cpl_errorstate_get();
    cpl_error_code     didfail = CPL_ERROR_NONE;
    irplib_framelist * allframes = NULL;
    irplib_framelist * rawframes = NULL;
    irplib_framelist * bpmframes = NULL;
    cpl_size           i, n, nbad = 0;
    

    /* Identify the RAW and CALIB frames in the input frameset */
    skip_if (visir_dfs_set_groups(framelist));

    /* Objects observation */
    allframes = irplib_framelist_cast(framelist);
    skip_if(allframes == NULL);
    rawframes = irplib_framelist_extract(allframes, VISIR_UTIL_INPUTS_RAW);
    skip_if (rawframes == NULL);
    bpmframes = irplib_framelist_extract(allframes, VISIR_CALIB_BPM);
    if (bpmframes == NULL)
        cpl_errorstate_set(cleanstate);
    else
        nbad = irplib_framelist_get_size(bpmframes);
    
    n = irplib_framelist_get_size(rawframes);
    error_if(nbad != n && nbad != 1 && nbad != 0, CPL_ERROR_INCOMPATIBLE_INPUT,
             "%d raw-frames <=> %d bpm frames", n, nbad);

#ifdef _OPENMP
#pragma omp parallel for private(i)
#endif
    for (i = 0; i < n; i++) {
        if (!didfail) {

            /* The total number of iterations must be pre-determined for the
               parallelism to work. In case of an error we can therefore not
               break, so instead we skip immediately to the next iteration.
               FIXME: This check on didfail does not guarantee that only one
               iteration can cause an error to be dumped, but it is not
               worse than checking on a thread-local state, e.g. errori. */

            if (visir_util_clip_one(framelist, rawframes, bpmframes, i,
                                    nbad == 1, parlist)) {
                const cpl_error_code errori = cpl_error_set_where(cpl_func);
#ifdef _OPENMP
                /* Cannot access these errors after the join,
                   so dump them now. :-(((((((((((((((((((( */
                cpl_errorstate_dump(cleanstate, CPL_FALSE, NULL);
                cpl_errorstate_set(cleanstate);
#pragma omp critical(visir_util_clip)
#endif
                didfail = errori;
            }
        }
    }

    error_if(didfail, didfail, "Failed to clip %d frame(s)", n);

    end_skip;

    irplib_framelist_delete(allframes);
    irplib_framelist_delete(rawframes);
    irplib_framelist_delete(bpmframes);

    return cpl_error_get_code();
}

static inline int round_to_int(const double x)
{
    return x >= 0 ? (int)(x + 0.5) : (int)(x - 0.5);
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Process one frame
  @param    framelist   The frameset to append products to
  @param    rawframes   The raw frames list (will load the propertylist)
  @param    bpmframes   The bpm frames list (will load the propertylist)
  @param    i           The frame to process (0 for first)
  @param    bshared     CPL_TRUE iff the bpm is shared among all raw frames
  @param    parlist     The parameters list
  @return   CPL_ERROR_NONE iff OK, otherwise the relevant CPL error code.
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code visir_util_clip_one(cpl_frameset * framelist,
                                          irplib_framelist * rawframes,
                                          irplib_framelist * bpmframes,
                                          int i, cpl_boolean bshared,
                                          const cpl_parameterlist * parlist)
{

    cpl_frameset  * products   = cpl_frameset_new();
    cpl_frameset  * usedframes = cpl_frameset_new();
    const cpl_frame * frame;

    char * bpmname = cpl_sprintf(RECIPE_STRING "_bpm_%03d" CPL_DFS_FITS, i);
    char * mapname = cpl_sprintf(RECIPE_STRING "_map_%03d" CPL_DFS_FITS, i);
    char * errname = cpl_sprintf(RECIPE_STRING "_error_%03d" CPL_DFS_FITS, i);

    const int            n = irplib_framelist_get_size(rawframes);
    cpl_frameset       * rawone = cpl_frameset_new();
    cpl_frameset       * bpmone = cpl_frameset_new();
    cpl_frame * rawframe =
        cpl_frame_duplicate(irplib_framelist_get_const(rawframes, i));
    cpl_frame * bpmframe = bpmframes ?
        cpl_frame_duplicate(irplib_framelist_get_const(bpmframes,
                                                       bshared ? 0 : i))
        : NULL;
    const cpl_error_code errr = cpl_frameset_insert(rawone, rawframe);
    const cpl_error_code errb = bpmframe ?
        cpl_frameset_insert(bpmone, bpmframe) : CPL_ERROR_NONE;

    const cpl_boolean berr = irplib_parameterlist_get_bool(parlist, PACKAGE,
                                                           RECIPE_STRING,
                                                           "error");

    const cpl_boolean bshifts = irplib_parameterlist_get_bool(parlist, PACKAGE,
                                                              RECIPE_STRING,
                                                              "shift-beams");

    const double badimage = irplib_parameterlist_get_double(parlist, PACKAGE,
                                                            RECIPE_STRING,
                                                            "badimage");

    cpl_imagelist * devlist = cpl_imagelist_new();
    cpl_imagelist * rawlist =
        cpl_imagelist_load_frameset(rawone, CPL_TYPE_UNSPECIFIED, 0, -1);
    cpl_imagelist * bpmlist = bpmframe ?
        cpl_imagelist_load_frameset(bpmone, CPL_TYPE_INT, 0, -1) : NULL;

#ifdef CPL_BPP_IEEE_FLOAT
/* FIXME: Use just CPL_TYPE_UNSPECIFIED once CPL 5.3.1 is no longer supported */
    const cpl_type_bpp stdev_save_type = CPL_TYPE_UNSPECIFIED;
#else
    const cpl_type_bpp stdev_save_type = cpl_image_get_type
        (cpl_imagelist_get_const(rawlist, 0)) == CPL_TYPE_DOUBLE
        ? CPL_BPP_IEEE_DOUBLE : CPL_BPP_IEEE_FLOAT;
#endif

    cpl_mask      * bpm    = NULL;
    cpl_image     * newbpmimg = NULL;
    cpl_image     * map = NULL;
    cpl_propertylist * qclist = cpl_propertylist_new();
    cpl_propertylist * xtlist = cpl_propertylist_new();

    const int       m = rawlist ? cpl_imagelist_get_size(rawlist) : 0;
    const int       nbpm = bpmlist ? cpl_imagelist_get_size(bpmlist) : 0;
    int             j;
    int             mapmax, mapmean, mapmedian;
    int *           shifts = cpl_calloc(m * 2, sizeof(int));

    skip_if(0);

    for (int e = 0; bshifts && e < m; e++) {
        const cpl_propertylist * plist;
        double crpix1, crpix2;
        double fx, fy;
        cpl_errorstate prestate = cpl_errorstate_get();

        irplib_framelist_load_propertylist(rawframes, i, e + 1, "^("
                                           IRPLIB_PFITS_WCS_REGEXP ")$",
                                           CPL_FALSE);
        plist = irplib_framelist_get_propertylist_const(rawframes, i);
        if (!cpl_propertylist_has(plist, "CRPIX1") ||
            !cpl_propertylist_has(plist, "CRPIX2")) {
            cpl_errorstate_set(prestate);
            break;
        }

        crpix1 = irplib_pfits_get_double(plist, "CRPIX1");
        crpix2 = irplib_pfits_get_double(plist, "CRPIX2");

        if (e == 0) {
            shifts[0] = 0;
            shifts[1] = 0;
            fx = crpix1;
            fy = crpix2;
        } else {
            shifts[e * 2]     = round_to_int(-(fx - crpix1));
            shifts[e * 2 + 1] = round_to_int(-(fy - crpix2));
        }
        cpl_msg_debug(cpl_func, "CRPIX shifts %d %d, %f %f", shifts[e * 2],
                      shifts[e * 2 + 1], crpix1 - fx, crpix2 - fy);
    }

    skip_if(rawlist == NULL);

    bug_if(errr);
    bug_if(errb);

    error_if(nbpm != 1 && nbpm != m && nbpm != 0, CPL_ERROR_INCOMPATIBLE_INPUT,
             "Frame-pair %d/%d: %d image(s) <=> %d bad pixel map(s)", 1+i, n,
             m, (int)cpl_imagelist_get_size(bpmlist));

    bug_if(cpl_frameset_insert(usedframes, cpl_frame_duplicate(rawframe)));
    if (bpmframe)
        bug_if(cpl_frameset_insert(usedframes, cpl_frame_duplicate(bpmframe)));

    for (j = 0; bpmframe && j < m; j++) {
        const cpl_image * bpmimg =
            cpl_imagelist_get_const(bpmlist, nbpm > 1 ? j : 0);
        cpl_image * rawimg       = cpl_imagelist_get(rawlist, j);

        cpl_mask_delete(bpm);
        bpm = cpl_mask_threshold_image_create(bpmimg, 0.5, FLT_MAX);

        bug_if(cpl_image_reject_from_mask(rawimg, bpm));

    }


    skip_if(visir_util_clip_kappa_sigma(rawlist, devlist, parlist, shifts));
    /* reject whole image if too many pixels clipped,
     * assumes number of rejected frames low, so no recomputing of stddev */
    {
        cpl_image * img = cpl_imagelist_get(rawlist, 0);
        const int total_size =
            cpl_image_get_size_x(img) * cpl_image_get_size_y(img);
        cpl_binary * _allbad = cpl_malloc(sizeof(cpl_binary) * total_size);
        cpl_mask * allbad;

        memset(_allbad, CPL_BINARY_1, sizeof(cpl_binary) * total_size);
        allbad = cpl_mask_wrap(cpl_image_get_size_x(img),
                               cpl_image_get_size_y(img), _allbad);

        for (int i = 0; i < cpl_imagelist_get_size(rawlist); i++) {
            cpl_mask * mask = cpl_image_get_bpm(cpl_imagelist_get(rawlist, i));
            img = cpl_imagelist_get(rawlist, i);
            if (badimage < (double)cpl_mask_count(mask) / total_size)
                cpl_image_reject_from_mask(img, allbad);
        }

        cpl_mask_delete(allbad);
    }

    /* Must have two st.dev images */
    bug_if(cpl_imagelist_get_size(devlist) != 2);

    map = cpl_image_new_from_accepted(rawlist);
    mapmax    = cpl_image_get_max(map);
    mapmean    = cpl_image_get_mean(map);
    mapmedian = cpl_image_get_median(map);
    bug_if(cpl_propertylist_append_int(qclist, VISIR_UTIL_CLIP_QC_MAP_MAX,
                                       mapmax));
    bug_if(cpl_propertylist_append_int(qclist, VISIR_UTIL_CLIP_QC_MAP_MEAN,
                                       mapmean));
    bug_if(cpl_propertylist_append_int(qclist, VISIR_UTIL_CLIP_QC_MAP_MEDIAN,
                                       mapmedian));
    bug_if(cpl_propertylist_set_comment(qclist, VISIR_UTIL_CLIP_QC_MAP_MAX,
                                        "The maximum contribution on a pixel"));
    bug_if(cpl_propertylist_set_comment(qclist, VISIR_UTIL_CLIP_QC_MAP_MEAN,
                                        "The mean contribution on a pixel"));
    bug_if(cpl_propertylist_set_comment(qclist, VISIR_UTIL_CLIP_QC_MAP_MEDIAN,
                                        "The median contribution on a pixel"));

#ifdef VISIR_MIME_CUBE_SUPPORT
    /* Pending closure of DFS 10467 */
#error "Save data as a cube in the primary data unit"
#else
    /* Do not save data in the primary data unit (DFS10475) */
    if (bpmframe)
        skip_if(irplib_dfs_save_propertylist(products, parlist, usedframes,
                                             RECIPE_STRING,
                                             VISIR_IMG_CLIPPED_PROCATG, qclist,
                                             NULL, visir_pipe_id, bpmname));

    if (berr)
        skip_if(irplib_dfs_save_propertylist(products, parlist, usedframes,
                                             RECIPE_STRING,
                                             "ERROR_MAP", qclist,
                                             NULL, visir_pipe_id, errname));

    for (j = 0; j < m; j++) {
        cpl_image * rawimg       = cpl_imagelist_get(rawlist, j);

        if (bpmframe) {
            const cpl_mask * newbpm  = cpl_image_get_bpm_const(rawimg);

            cpl_image_delete(newbpmimg);
            newbpmimg = cpl_image_new_from_mask(newbpm);

            skip_if(cpl_image_save(newbpmimg, bpmname, CPL_BPP_8_UNSIGNED,
                                   NULL, CPL_IO_EXTEND));
        }

        if (berr)
            skip_if(cpl_image_save(cpl_imagelist_get_const(devlist, 1), errname,
                                   stdev_save_type, NULL, CPL_IO_EXTEND));
    }
#endif

    skip_if(irplib_dfs_save_image(products, parlist, usedframes,
                                  map, m < 256 ? CPL_BPP_8_UNSIGNED
                                  : (m < 65536 ? CPL_BPP_16_UNSIGNED
                                     : CPL_BPP_32_SIGNED), RECIPE_STRING,
                                  VISIR_IMG_CLIPPED_MAP_PROCATG, qclist,
                                  NULL, visir_pipe_id, mapname));

    bug_if(cpl_propertylist_append_string(xtlist, "EXTNAME", "NO CLIP STANDARD "
                                          "DEVIATION MAP"));

    skip_if(cpl_image_save(cpl_imagelist_get_const(devlist, 0), mapname,
                           stdev_save_type, xtlist, CPL_IO_EXTEND));

    bug_if(cpl_propertylist_update_string(xtlist, "EXTNAME", "CLIPPED STANDARD "
                                          "DEVIATION MAP"));

    skip_if(cpl_image_save(cpl_imagelist_get_const(devlist, 1), mapname,
                           stdev_save_type, xtlist, CPL_IO_EXTEND));

    for (frame = cpl_frameset_get_first_const(products);
         frame != NULL;
         frame = cpl_frameset_get_next_const(products)) {
        cpl_frame * copy = cpl_frame_duplicate(frame);
        cpl_error_code error;

#ifdef _OPENMP
#pragma omp critical(visir_util_clip_one)
#endif
        error = cpl_frameset_insert(framelist, copy);

        if (error) break;
    }

    bug_if(frame != NULL);


    end_skip;

    cpl_free(bpmname);
    cpl_free(mapname);
    cpl_free(errname);
    cpl_free(shifts);

    cpl_frameset_delete(rawone);
    cpl_frameset_delete(bpmone);
    cpl_frameset_delete(usedframes);
    cpl_frameset_delete(products);

    cpl_propertylist_delete(qclist);
    cpl_propertylist_delete(xtlist);

    cpl_mask_delete(bpm);
    cpl_image_delete(newbpmimg);
    cpl_image_delete(map);
    cpl_imagelist_delete(rawlist);
    cpl_imagelist_delete(bpmlist);
    cpl_imagelist_delete(devlist);

    return cpl_error_get_code();

}


/*----------------------------------------------------------------------------*/
/**
  @brief    Use kappa-sigma clipping to reject outlier pixels
  @param    self      The imagelist to process
  @param    devlist   Append stdev maps to this list
  @param    parlist   The recipe parameters
  @return   CPL_ERROR_NONE iff OK, otherwise the relevant CPL error code.
 */
/*----------------------------------------------------------------------------*/
static
cpl_error_code visir_util_clip_kappa_sigma(cpl_imagelist * self,
                                           cpl_imagelist * devlist,
                                           const cpl_parameterlist * parlist,
                                           const int * shifts)
{

    const double keepfrac = irplib_parameterlist_get_double(parlist, PACKAGE,
                                                            RECIPE_STRING,
                                                            "keepfrac");
    const double kappa    = irplib_parameterlist_get_double(parlist, PACKAGE,
                                                            RECIPE_STRING,
                                                            "kappa");
    const int    maxite   = irplib_parameterlist_get_int(parlist, PACKAGE,
                                                         RECIPE_STRING,
                                                         "maxite");
    const cpl_image * img = cpl_imagelist_get_const(self, 0);

    switch (cpl_image_get_type(img)) {
    case CPL_TYPE_DOUBLE:
        skip_if(visir_util_clip_kappa_sigma_double(self, devlist, keepfrac,
                                                   kappa, maxite, shifts));
        break;
    case CPL_TYPE_FLOAT:
        skip_if(visir_util_clip_kappa_sigma_float(self, devlist, keepfrac,
                                                  kappa, maxite, shifts));
        break;
    case CPL_TYPE_INT:
        skip_if(visir_util_clip_kappa_sigma_int(self, devlist, keepfrac,
                                                kappa, maxite, shifts));
        break;
    default:
        bug_if( 1 );
    }

    end_skip;

    return cpl_error_get_code();

}


/* These macros are needed for support of the different pixel types */

#define CONCAT(a,b) a ## _ ## b
#define CONCAT2X(a,b) CONCAT(a,b)

#define PIXEL_TYPE double
#define STDEV_TYPE CPL_TYPE_DOUBLE
#define PIXEL_TYPE_CPL CPL_TYPE_DOUBLE
#include "visir_util_clip_body.c"
#undef PIXEL_TYPE
#undef STDEV_TYPE
#undef PIXEL_TYPE_CPL

#define PIXEL_TYPE float
#define PIXEL_TYPE_CPL CPL_TYPE_FLOAT
#define STDEV_TYPE CPL_TYPE_FLOAT
#include "visir_util_clip_body.c"
#undef PIXEL_TYPE
#undef STDEV_TYPE
#undef PIXEL_TYPE_CPL

#define PIXEL_TYPE int
#define PIXEL_TYPE_CPL CPL_TYPE_INT
#define STDEV_TYPE CPL_TYPE_FLOAT
#include "visir_util_clip_body.c"
#undef PIXEL_TYPE
#undef STDEV_TYPE
#undef PIXEL_TYPE_CPL
