/* $Id: hdrl_image-test.c,v 1.5 2013-10-23 09:13:56 jtaylor Exp $
 *
 * This file is part of the HDRL
 * Copyright (C) 2013 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: jtaylor $
 * $Date: 2013-10-23 09:13:56 $
 * $Revision: 1.5 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/

#include "hdrl_image.h"
#include "hdrl_test.h"
#include "hdrl_imagelist.h"
#include "hdrl_buffer.h"
#include "hdrl_types.h"

#include <cpl.h>

#include <math.h>
#include <float.h>
#include <stdlib.h>
#include <stdio.h>

#ifndef ARRAY_LEN
#define ARRAY_LEN(a) sizeof((a))/sizeof((a)[0])
#endif

/*----------------------------------------------------------------------------*/
/**
 * @defgroup hdrl_image_test   
            Testing of hdrl_image module
 */
/*----------------------------------------------------------------------------*/

static void test_basic(void)
{
    hdrl_image * img;
    cpl_image * cimg, * cerr;

    hdrl_image_delete(NULL);

    /* empty creation */

    img = hdrl_image_new(5, 5);
    cpl_test_nonnull(img);
    cpl_test_error(CPL_ERROR_NONE);
    hdrl_image_delete(img);

    img = hdrl_image_new(0, 5);
    cpl_test_null(img);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    img = hdrl_image_new(5, 0);
    cpl_test_null(img);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    img = hdrl_image_new(0, 0);
    cpl_test_null(img);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    img = hdrl_image_create(NULL, NULL);
    cpl_test_null(img);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* creation with cpl image */

    cimg = cpl_image_new(5, 6, CPL_TYPE_DOUBLE);
    cerr = cpl_image_new(5, 6, CPL_TYPE_DOUBLE);

    img = hdrl_image_create(NULL, cerr);
    cpl_test_null(img);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    img = hdrl_image_create(cimg, NULL);
    cpl_test_nonnull(img);
    hdrl_image_delete(img);

    img = hdrl_image_create(cimg, cerr);
    cpl_test_nonnull(img);
    hdrl_image_delete(img);

    /* with bpm */

    cpl_image_reject(cimg, 2, 3);
    img = hdrl_image_create(cimg, cerr);
    cpl_test_nonnull(img);
    cpl_test_eq(hdrl_image_count_rejected(img), 1);
    hdrl_image_delete(img);

    cpl_image_reject(cerr, 2, 3);
    img = hdrl_image_create(cimg, cerr);
    cpl_test_eq(hdrl_image_count_rejected(img), 1);
    cpl_test_nonnull(img);
    hdrl_image_delete(img);

    /* incompatible bpm (emits warning) */
    cpl_image_reject(cerr, 2, 4);
    img = hdrl_image_create(cimg, cerr);
    cpl_test_eq(hdrl_image_count_rejected(img), 1);
    cpl_test_nonnull(img);
    hdrl_image_delete(img);

    cpl_image_accept_all(cimg);
    img = hdrl_image_create(cimg, cerr);
    cpl_test_eq(hdrl_image_count_rejected(img), 0);
    cpl_test_nonnull(img);
    hdrl_image_delete(img);

    cpl_image_delete(cerr);
    cerr = cpl_image_new(2, 6, CPL_TYPE_DOUBLE);
    img = hdrl_image_create(cimg, cerr);
    cpl_test_null(img);
    cpl_test_error(CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_image_delete(cerr);
    cerr = cpl_image_new(5, 2, CPL_TYPE_DOUBLE);
    img = hdrl_image_create(cimg, cerr);
    cpl_test_null(img);
    cpl_test_error(CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_image_delete(cimg);
    cpl_image_delete(cerr);
}

static void test_extract(void)
{
    cpl_size nx = 5;
    cpl_size ny = 13;
    hdrl_image * himg = hdrl_image_new(nx, ny);
    hdrl_image * ex;
    hdrl_image_add_scalar(himg, 1., 1.);

    ex = hdrl_image_extract(himg, 1, 1, nx, ny);
    cpl_test_nonnull(ex);
    hdrl_test_image_abs(himg, ex, 0);
    hdrl_image_delete(ex);

    ex = hdrl_image_extract(himg, 1, 1, 0, 0);
    cpl_test_nonnull(ex);
    hdrl_test_image_abs(himg, ex, 0);
    hdrl_image_delete(ex);

    ex = hdrl_image_extract(himg, 0, 0, 0, 0);
    cpl_test_nonnull(ex);
    cpl_test_eq(hdrl_image_get_size_x(ex), 1);
    cpl_test_eq(hdrl_image_get_size_y(ex), 1);
    hdrl_image_delete(ex);

    ex = hdrl_image_extract(himg, 2, 2, -1, -1);
    cpl_test_nonnull(ex);
    cpl_test_eq(hdrl_image_get_size_x(ex), nx - 2);
    cpl_test_eq(hdrl_image_get_size_y(ex), ny - 2);
    hdrl_image_delete(ex);

    ex = hdrl_image_extract(himg, 2, 2, -1, 2 * ny);
    cpl_test_null(ex);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    ex = hdrl_image_extract(himg, 2, 2, -1, -2 * ny);
    cpl_test_null(ex);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    ex = hdrl_image_extract(himg, 2, 2, -2 * nx, -2);
    cpl_test_null(ex);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    ex = hdrl_image_extract(himg, 2, -2 * ny, -2, -2);
    cpl_test_null(ex);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    ex = hdrl_image_extract(himg, -2 * nx, -2, -2, -2);
    cpl_test_null(ex);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    hdrl_image_delete(himg);
}

static void test_power(void)
{
    hdrl_image * himg = hdrl_image_new(1, 1);
    hdrl_image_set_pixel(himg, 1, 1, 2., 0.5);
    hdrl_image_power(himg, 2);
    hdrl_error_t e = 0.;
    cpl_test_rel(hdrl_image_get_pixel(himg, 1, 1, &e, NULL), 4., DBL_EPSILON);
    cpl_test_rel(e, 2.0, DBL_EPSILON);

    hdrl_image_set_pixel(himg, 1, 1, 2., 0.5);
    hdrl_image_power(himg, 4);
    cpl_test_rel(hdrl_image_get_pixel(himg, 1, 1, &e, NULL), 16, DBL_EPSILON);
    cpl_test_rel(e, 16.0, DBL_EPSILON);

    hdrl_image_set_pixel(himg, 1, 1, 2., 0.5);
    hdrl_image_power(himg, -1);
    cpl_test_rel(hdrl_image_get_pixel(himg, 1, 1, &e, NULL), 0.5, DBL_EPSILON);
    cpl_test_rel(e, 0.125, DBL_EPSILON);

    hdrl_image_set_pixel(himg, 1, 1, 2., 0.5);
    hdrl_image_power(himg, -2);
    cpl_test_rel(hdrl_image_get_pixel(himg, 1, 1, &e, NULL), 0.25, DBL_EPSILON);
    cpl_test_rel(e, 0.125, DBL_EPSILON); /* yes the same as ^-1 */

    hdrl_image_set_pixel(himg, 1, 1, 2., 0.5);
    hdrl_image_power(himg, -4);
    cpl_test_rel(hdrl_image_get_pixel(himg, 1, 1, &e, NULL), 0.0625, DBL_EPSILON);
    cpl_test_rel(e, 0.0625, DBL_EPSILON);

    hdrl_image_delete(himg);
}

static void test_insert(void)
{
    hdrl_image * dst = hdrl_image_new(50, 50);
    hdrl_image * dst2 = hdrl_image_new(50, 50);
    cpl_image * im1 = cpl_image_new(50, 50, HDRL_TYPE_DATA);
    cpl_image * im2 = cpl_image_new(50, 50, HDRL_TYPE_ERROR);
    hdrl_image * him = hdrl_image_create(im1, im2);

    hdrl_image_copy(dst2, him, 1, 1);
    hdrl_image_insert(dst, im1, im2, 1, 1);
    hdrl_test_image_abs(dst, dst2, 0);

    hdrl_image_insert(dst, NULL, im2, 1, 1);
    hdrl_image_insert(dst, im1, NULL, 1, 1);
    hdrl_image_insert(dst, NULL, NULL, 1, 1);
    hdrl_test_image_abs(dst, dst2, 0);

    hdrl_image_delete(dst);
    hdrl_image_delete(dst2);
    hdrl_image_delete(him);
    cpl_image_delete(im1);
    cpl_image_delete(im2);
}

static void test_reduce(void)
{
    size_t nx = 53, ny = 2310;
    double md, me;
    hdrl_image * a = hdrl_image_new(nx, ny);
    hdrl_image * b = hdrl_image_new(nx, ny);
    hdrl_image * c = hdrl_image_new(nx, ny);
    hdrl_imagelist * hl = hdrl_imagelist_new();
    hdrl_image_add_scalar(a, 5., 3.2);
    hdrl_image_add_scalar(b, 7., 1.2);
    hdrl_image_add_scalar(b, -3., .2);

    hdrl_image_get_mean(a, &md, &me);
    cpl_test_abs(md, 5., HDRL_EPS_DATA);
    cpl_test_abs(me, 3.2 / sqrt(nx * ny), HDRL_EPS_DATA * nx * ny);

    hdrl_imagelist_set(hl, a, 0);
    hdrl_imagelist_set(hl, b, 1);
    hdrl_imagelist_set(hl, c, 2);
    hdrl_image * r;
    cpl_image * con;

    /* must be equivalent */
    hdrl_imagelist_collapse_mean(hl, &r, &con);
    hdrl_image_add_image(a, b);
    hdrl_image_add_image(a, c);
    hdrl_image_div_scalar(a, 3., 0.);
    hdrl_test_image_abs(a, r, HDRL_EPS_DATA);

    hdrl_image_delete(r);
    cpl_image_delete(con);
    hdrl_imagelist_delete(hl);
}


static inline void test_buffer(void)
{
    hdrl_buffer * buf = hdrl_buffer_new();
    hdrl_imagelist * hl = hdrl_imagelist_new();
    for (size_t i = 0; i < 900; i++) {
        hdrl_image * img =hdrl_image_new_from_buffer(500, 500, buf);
        hdrl_imagelist_set(hl, img, i);
        hdrl_image_add_scalar(img, 1,1);
    }
    cpl_msg_warning(cpl_func, "alloc done");

    hdrl_image * m;
    cpl_image * c;
    hdrl_imagelist_collapse_mean(hl, &m, &c);
    hdrl_image_delete(m);
    cpl_image_delete(c);
    cpl_msg_warning(cpl_func, "collapse done");

    hdrl_imagelist_delete(hl);
    hdrl_buffer_delete(buf);
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of hdrl_image
 **/
/*----------------------------------------------------------------------------*/
int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_basic();
    test_power();
    test_insert();
    test_extract();
    test_reduce();
    //test_buffer();

    return cpl_test_end(0);
}
