/* $Id: hdrl_imagelist_view.c,v 1.8 2013-10-23 09:13:56 jtaylor Exp $
 *
 * This file is part of the HDRL
 * Copyright (C) 2013 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: jtaylor $
 * $Date: 2013-10-23 09:13:56 $
 * $Revision: 1.8 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "hdrl_imagelist.h"
#include "hdrl_imagelist_defs.h"
#include "hdrl_imagelist_view.h"
#include "hdrl_image_defs.h"
#include "hdrl_image.h"

#include <cpl.h>
#include <assert.h>

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                            Static Prototypes
 -----------------------------------------------------------------------------*/

static void hdrl_image_view_delete(hdrl_image * view);

/* TODO: internal representation of cpl_image in 6.3 required for mask views,
   can be removed in 6.4 which has cpl_image_set_bpm */
struct _cpl_image_ {
    /* Size of the image in x and y */
    cpl_size            nx, ny;
    /* Type of the pixels used for the cpl_image */
    cpl_type            type;
    /* Pointer to pixel buffer as a 1d buffer */
    void            *   pixels;
    /* Bad Pixels mask */
    cpl_mask        *   bpm;
};


/**@{*/

/*-----------------------------------------------------------------------------
                            Function codes
 -----------------------------------------------------------------------------*/

static hdrl_image *
hdrl_image_row_view_create(hdrl_image * himg,
                           cpl_size ly,
                           cpl_size uy)
{
    cpl_image * oimg = hdrl_image_get_image(himg);
    cpl_image * oerr = hdrl_image_get_error(himg);
    const size_t dsz = cpl_type_get_sizeof(cpl_image_get_type(oimg));
    const size_t esz = cpl_type_get_sizeof(cpl_image_get_type(oerr));
    const cpl_size nx = hdrl_image_get_size_x(himg);
    char * d = cpl_image_get_data(oimg);
    char * e = cpl_image_get_data(oerr);
    size_t offset = (ly - 1) * nx;
    cpl_size nny = uy - ly + 1;
    cpl_image * img = cpl_image_wrap(nx, nny, cpl_image_get_type(oimg),
                                     d + offset * dsz);
    cpl_image * err = cpl_image_wrap(nx, nny, cpl_image_get_type(oerr),
                                     e + offset * esz);

    /* we must create a mask in the original in order for views to work */
    cpl_mask * omask = hdrl_image_get_mask(himg);
    cpl_mask * mask = cpl_mask_wrap(nx, nny, cpl_mask_get_data(omask) + offset);
    /* TODO replace with 6.4 cpl_image_set_bpm */
    ((struct _cpl_image_ *)img)->bpm = mask;

    return hdrl_image_wrap(img, err, (hdrl_free*)&hdrl_image_view_delete);
}

static void hdrl_image_view_delete(hdrl_image * view)
{
    if (view) {
        hdrl_image * himg = (hdrl_image*)view;
        cpl_mask_unwrap(cpl_image_unset_bpm(hdrl_image_get_image(himg)));
        cpl_image_unwrap(himg->image);
        cpl_image_unwrap(hdrl_image_get_error(himg));
        hdrl_image_unwrap(himg);
    }
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief create view of subset of image rows in an imagelist
 *
 * @param hl source imagelist to create view of
 * @param ly lower y row to view (FITS convention)
 * @param uy upper y row to view (FITS convention)
 *
 * @return imagelist view
 *
 * The imagelist will contain the same number of images but they will have
 * uy - ly + 1 rows. Modifiying the view will modify the selected rows of the
 * original imagelist.
 * The view can be deleted with hdrl_imagelist_delete.
 * Deleting the original imagelist will invalidate all views, they must not be
 * used anymore besides being deleted.
 */
/* ---------------------------------------------------------------------------*/
hdrl_imagelist * hdrl_imagelist_row_view(
    hdrl_imagelist * hl,
    cpl_size ly,
    cpl_size uy)
{
    cpl_ensure(hl, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(uy >= ly, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(ly > 0, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);
    size_t n = hdrl_imagelist_get_size(hl);
    cpl_ensure(n > 0, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);
    cpl_ensure(uy <= hdrl_image_get_size_y(hdrl_imagelist_get_const(hl, 0)),
               CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);

    hdrl_imagelist * viewlist = hdrl_imagelist_new();
    for (size_t i = 0; i < n; i++) {
        hdrl_image * img = hdrl_imagelist_get(hl, i);
        hdrl_image * view = hdrl_image_row_view_create(img, ly, uy);
        hdrl_imagelist_set(viewlist, view, i);
    }

    return viewlist;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief create view of subset of images in an imagelist
 *
 * @param hl source imagelist to create view of
 * @param lz lower image index to view, starting from 0
 * @param uz upper bound exclusive
 *
 * @return imagelist view
 *
 * The imagelist will contain uz - lz images of the same size as the original
 * imagelist. Modifiying the view will modify the selected rows of the
 * original imagelist.
 * The view must be deleted with hdrl_imagelist_delete.
 * Deleting the original imagelist will invalidate all views, they must not be
 * used anymore besides being deleted.
 */
/* ---------------------------------------------------------------------------*/
hdrl_imagelist * hdrl_imagelist_image_view(
    hdrl_imagelist * hl,
    cpl_size lz,
    cpl_size uz)
{
    cpl_ensure(hl, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(uz > lz, CPL_ERROR_ILLEGAL_INPUT, NULL);
    size_t n = hdrl_imagelist_get_size(hl);
    cpl_ensure(lz >= 0, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);
    cpl_ensure(uz <= (cpl_size)n, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);

    hdrl_imagelist * viewlist = hdrl_imagelist_new();
    for (size_t i = lz; i < (size_t)uz; i++) {
        hdrl_image * img = hdrl_imagelist_get(hl, i);
        size_t ny = hdrl_image_get_size_y(img);
        hdrl_image * view = hdrl_image_row_view_create(img, 1, ny);
        hdrl_imagelist_set(viewlist, view, i - lz);
    }

    return viewlist;
}
/**@}*/
