/*
 * cowbell
 * Copyright (c) 2005 Brad Taylor
 *
 * This file is part of cowbell.
 *
 * cowbell is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * cowbell is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with cowbell; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Threading;
using System.Collections;
using System.Globalization;

namespace Cowbell.Base
{
	public class Utils
	{
		/* public methods */
	
		/**
		 * Calculates the LD for two given strings: s, t; returning the
		 * number of insertions, deletions or substitutions needed to
		 * transform s into t
		 * @param s the first string to match
		 * @param t the second string to match
		 * @return the Levenshtien Distance on s and t
		 */
		public static int LevenshteinDistance (string s, string t)
		{
			int[,] d;
			char s_i, t_j;
			int n, m, i, j, cost;

			n = s.Length;
			m = t.Length;

			if (n == 0)
				return m;

			if (m == 0)
				return n;

			d = new int[n+1, m+1];

			for (i = 0; i <= n; i++)
				d[i, 0] = i;

			for (j = 0; j <= m; j++)
				d[0, j] = j;

			for (i = 1; i <= n; i++)
			{
				s_i = s[i - 1];

				for (j = 1; j <= m; j++)
				{
					t_j = t[j - 1];

					if (s_i == t_j)
						cost = 0;
					else
						cost = 1;

					d[i, j] = Min (d[i-1, j]+1, d[i, j-1]+1, d[i-1, j-1] + cost);
				}
			}

			return d[n, m];
		}

		public static int Min (int a, int b, int c)
		{
			int temp;

			temp = a;
			
			if (b < temp)
				temp = b;
			if (c < temp)
				temp = c;

			return temp;
		}

		public static string HumanReadableTimeSpan (TimeSpan span)
		{
			ArrayList list = new ArrayList ();
			if (span.Days > 0) {
				list.Add (String.Format (Catalog.GetPluralString ("{0} day", "{0} days", span.Days), span.Days));
			}

			if (span.Hours > 0) {
				list.Add (String.Format (Catalog.GetPluralString ("{0} hour", "{0} hours", span.Hours), span.Hours));
			}

			if (list.Count < 2 && span.Minutes > 0) {
				list.Add (String.Format (Catalog.GetPluralString ("{0} minute", "{0} minutes", span.Minutes), span.Minutes));
			}

			if (list.Count < 2 && span.Seconds > 0) {
				list.Add (String.Format (Catalog.GetPluralString ("{0} second", "{0} seconds", span.Seconds), span.Seconds));
			}

			return String.Join (", ", (string[])list.ToArray (typeof (string))); 
		}

		public static string ToTitleCase (string str)
		{
			CultureInfo c = Thread.CurrentThread.CurrentCulture;
			TextInfo t = c.TextInfo;

			return t.ToTitleCase (str);
		}

		public static string ToSmartTitleCase (string str)
		{
			if (Thread.CurrentThread.CurrentCulture.TwoLetterISOLanguageName != "en") {
				return ToTitleCase (str);
			}

			string[] tokens = ToTitleCase (str).Split (' ');
			for (int i = 0; i < tokens.Length; i++)
			{
				// Always capitalize the first and last word
				if (i == 0 || i == tokens.Length - 1) {
					continue;
				}

				string token = tokens[i];
				// Do not capitalize articles (a, an, the)
				// Do not capitalize coordinate conjunctions
				// (and, but, for, nor, or)
				if (token == "And" || token == "But" || token == "For"
				    || token == "Nor" || token == "Or" || token == "A"
				    || token == "An" || token == "The") {
					tokens[i] = token.ToLower ();
					continue;
				}

				// Don't capitalize prepositions of four or fewer letters
				if (token == "At" || token == "But" || token == "By"
				    || token == "For" || token == "From" || token == "In"
				    || token == "Into" || token == "Like" || token == "Near"
				    || token == "Of" || token == "Off" || token == "On"
				    || token == "Onto" || token == "Over" || token == "Past"
				    || token == "Till" || token == "To" || token == "Until"
				    || token == "Up" || token == "Upon" || token == "With") {
					tokens[i] = token.ToLower ();
					continue;
				}
			}
			return String.Join (" ", tokens);
		}
	}
}
