/*
$Id: courierpasswd.c,v 1.2 2005/10/15 22:13:10 astjean Exp $

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

Copyright (c) Andrew St. Jean <andrew@arda.homeunix.net> 2002-2005

*/

#include "courierpasswd.h"

static const char *short_options = "s:cC:hvV";

static struct option long_options[] = {
	{ "service", required_argument, NULL, 's' },
	{ "changepwd", no_argument, NULL, 'c' },
	{ "cramtype", required_argument, NULL, 'C' },
	{ "help", no_argument, NULL, 'h' },
	{ "verbose", no_argument, NULL, 'v' },
	{ "version", no_argument, NULL, 'V' },
	{ "stdin", no_argument, NULL, OPT_STDIN },
	{ "stderr", no_argument, NULL, OPT_STDERR },
	{ NULL, 0, NULL, 0 }
};

const char *usage =
"Usage: " PACKAGE "  [ OPTION [ OPTION ] ] [ prog... ]\n"
"\n"
"Perform one of two operations using the Courier\n"
"authentication library\n"
"1. Authenticate the specified user and optionally run\n"
"   the program indicated by 'prog'.\n"
"2. Change the password of the specified user. This\n"
"   operation is not available when using CRAM.\n"
"\n"
"Options:\n"
"  -s, --service=SERVICE\tspecify the service name to use\n"
"\t\t\tdefault value is 'login'\n"
"  -c, --changepwd\ttells courierpasswd to change the\n"
"\t\t\tspecified user's password\n"
"  -C, --cramtype=TYPE\ttells courierpasswd the type of\n"
"\t\t\tCRAM that will be used for authentication\n"
"\t\t\tvalue can be 'md5' or 'sha1'\n"
"\t\t\tthis option will be ignored if the -c option is\n"
"\t\t\talso present\n"
"  --stdin\t\tread authentication tokens from stdin\n"
"\t\t\tthree variations are possible:\n"
"\t\t\tif the -c option is not used\n"
"\t\t\tusername\\0oldpassword\\0\n"
"\t\t\tusername\\0challenge\\0response\\0\n"
"\t\t\tor if the -c option is used\n"
"\t\t\tusername\\0oldpassword\\0newpassword\\0\n"
"  --stderr\t\tsend logging information to stderr\n"
"\t\t\tinstead of syslog\n"
"  -h, --help\t\tdisplay this help and exit\n"
"  -v, --verbose\t\tturn on verbose output\n"
"  -V, --version\t\tdisplay version information and exit\n";

int opt_stderr = 0;
int opt_verbose = 0;

#define MAX_LEN_SERVICE		129
#define BUFSIZE			513

char upp[BUFSIZE];
static int input_fd = 3;
char *username = NULL;
char *oldpwd = NULL;
char *newpwd = NULL;
char response[BUFSIZE];
char *b64_challenge = NULL;
char *b64_response = NULL;

static int
callback_func(struct authinfo *a, void *dummy) {
	        return(0);
}

static int
callback_pre(struct authinfo *a, void *min_uid) {

	struct passwd *pw;

	if (a->sysuserid != NULL){
		if(a->sysuserid < (uid_t *)min_uid) return(-1);
	} else if (a->sysusername) {
		pw = getpwnam(a->sysusername);
		if (!pw){
			logging(LOG_ERR, "getpwnam failed: %s", strerror(errno));
			/* perror("getpwnam"); */
			return(-2);
		}

		if (pw->pw_uid < (uid_t)min_uid) return(-1);
	}

	return(0);
}

int
findcramtype(const char *cram, char *authtype_cram){
	int n=0;

	if (!strncmp(cram, "md5", 3)){
		n = strlcpy(authtype_cram, AUTHTYPE_CRAMMD5, sizeof(AUTHTYPE_CRAMMD5)+1);
	} else if (!strncmp(cram, "sha1", 4)){
		n = strlcpy(authtype_cram, AUTHTYPE_CRAMSHA1, sizeof(AUTHTYPE_CRAMSHA1)+1);
	}

	return(n);
}

size_t
validate_str(const char *data){
	return(strspn(data, SAFE_CHARS));
}

void
end(int exit_code) {

        memset(upp, 0x00, sizeof(upp));
        end_logging();
        exit(exit_code);
}

int main(int argc, char **argv) {
FILE	*tokens;
int	upplen, i, rc;
size_t	n;
char	*sp;
char	service[MAX_LEN_SERVICE];
char	cram[MAX_LEN_SERVICE];
char	authtype_cram[BUFSIZE];
int	changepwd = 0;

	*service = *cram = *authtype_cram = 0;

	/* set the default value for service */
	strlcpy(service, "login", 6);
	/* set default value for cram */
	strlcpy(cram, "md5", 4);


	init_logging();

	/* process the command line options */
	opterr = 0;
	while (1){
		int option_index = 0;
		int c = getopt_long(argc, argv, short_options, long_options, &option_index);

		if (c == -1) break;

		switch (c){
		case 's':
			n = validate_str(optarg);
			if (optarg[n]){
				logging(LOG_ERR, "illegal character in service name: %c", optarg[n]);
				end(2);
			}
			n = strlcpy(service, optarg, sizeof(service));
			if (n >= sizeof(service)){
				logging(LOG_ERR, "service name too long (max %d characters)", MAX_LEN_SERVICE - 1);
				end(2);
			}
			break;
		case 'c':
			changepwd = 1;
			break;
		case 'C':
			n = validate_str(optarg);
			if (optarg[n]){
				logging(LOG_ERR, "illegal character in CRAM type: %c", optarg[n]);
				end(2);
			}
			n = strlcpy(cram, optarg, sizeof(cram));
			if (n >= sizeof(cram)){
				logging(LOG_ERR, "CRAM type too long (max %d characters)", MAX_LEN_SERVICE - 1);
				end(2);
			}
			/* convert the string to lower case. */
		        for (sp = cram; isalpha(*sp); sp++) *sp = tolower(*sp);
			n = findcramtype(cram, authtype_cram);
			if (!n){
				logging(LOG_ERR, "unknown CRAM type: %s", cram);
				end(2);
			}
			break;
		case OPT_STDIN:
			input_fd = 0;
			break;
		case OPT_STDERR:
			opt_stderr = 1;
			break;
		case 'h':
			puts(usage);
			exit(0);
		case 'v':
			opt_verbose = 1;
			break;
		case 'V':
			puts(PACKAGE " "  VERSION);
			exit(0);
		case '?':
			puts("Invalid command-line, see --help");
			exit(2);
		}
	}

	/* convert the CRAM string to lower case. */
	for (sp = cram; isalpha(*sp); sp++) *sp = tolower(*sp);
	/* do we recognise the CRAM type specified? */
	n = findcramtype(cram, authtype_cram);
	if (!n){
		logging(LOG_ERR, "unknown CRAM type: %s", cram);
		end(2);
	}

	/* read authentication tokens from file descriptor */
	tokens = fdopen(input_fd, "r");
	if (tokens == NULL) {
		logging(LOG_ERR, "Error opening fd %d: %s", input_fd, strerror(errno));
		end(2);
	}

	upplen = fread(upp, 1, BUFSIZE, tokens);
	if (upplen == 0) {
		logging(LOG_ERR, "Failure reading authentication tokens: zero bytes read");
		end(2);
	}
	
	i = 0;
	/* get username */
	username = upp + i;
	while (upp[i++]) {
		if (i >= upplen) {
			logging(LOG_ERR, "Failure while reading username");
			end(2);
		}
	}

	verbose("Username is: %s", username);

	/* get authentication token */
	oldpwd = upp + i;
	while (upp[i++]) {
		if (i >= upplen) {
			logging(LOG_ERR, "Failure reading authentication tokens");
			end(2);
		}
	}

	/* if there's a third token, get it */
	if (i < (upplen - 1)) {
		newpwd = upp + i;
		while (upp[i++]) {
			if (i >= upplen) {
				logging(LOG_ERR,
					"Failure reading authentication tokens");
				end(2);
			}
		}
	}

	if (newpwd == NULL) {
		verbose("Password is: %s", oldpwd);
	} else {
		verbose(changepwd ? "Old password is: %s" : "Challenge is: %s", oldpwd);
		verbose(changepwd ? "New password is: %s" : "Response is: %s", newpwd);
	}
	
	if (!changepwd) {
		/* check the password */
		char *authdata;

		if (newpwd != NULL) {
			base64_encode(oldpwd, strlen(oldpwd), &b64_challenge);
			/* prepend username */
			sprintf(response, "%s %s", username, newpwd);
			base64_encode(response, strlen(response), &b64_response);
			authdata = XMALLOC(char, strlen(b64_challenge)+strlen(b64_response)+3);
			sprintf(authdata, "%s\n%s\n", b64_challenge, b64_response);
			rc = auth_generic(service, authtype_cram, authdata, callback_func, 0);
		} else {
			authdata = XMALLOC(char, strlen(username)+strlen(oldpwd)+3);
			sprintf(authdata, "%s\n%s\n", username, oldpwd);
			rc = auth_generic(service, AUTHTYPE_LOGIN, authdata, callback_func, 0);

		}

		XFREE(authdata);

		if (rc != 0) {
			if (errno != EPERM) {
				logging(LOG_ERR, "Temporary authentication failure for user %s: %s", username, strerror(errno));
				end(111);
			}

			logging(LOG_ERR, "Authentication failure for user %s", username);
                        end(1);
		} else {

			logging(LOG_INFO, "Authenticated for user %s", username);
		}

	} else {

		/* is the uid restricted? */
		rc = auth_getuserinfo(service, username, callback_pre, (uid_t *)MIN_UID);

		if (rc != 0){
			logging(LOG_ERR, "Attempt to change password for the restricted or non-existant user %s", username);
			end(1);
		}

		/* change the password */
		rc = auth_passwd(service, username, oldpwd, newpwd);

		if (rc == 0){
			logging(LOG_INFO, "Password changed for %s", username);
			end(0);
		} else {
			logging(LOG_ERR, "Password change failed for %s: %s", username, strerror(errno));
			end(1);
		}
	}

	if (optind < argc) {
		verbose("Executing %s", argv[optind]);

		execvp(argv[optind], argv + optind);
		logging(LOG_ERR, "Failed to execute %s: %s", argv[optind], strerror(errno));
		end(2);
	}

	end(0);
	return(0);
}
