#!/usr/bin/env python
#
#   ConVirt   -  Copyright (c) 2008 'The ConVirt Team', Convirture Corp.
#   ======
#
# ConVirt is a Xen management tool with a GTK based graphical interface
# that allows for performing the standard set of domain operations
# (start, stop, pause, kill, shutdown, reboot, snapshot, etc...). It
# also attempts to simplify certain aspects such as the creation of
# domains, as well as making the consoles available directly within the
# tool's user interface.
#
#
# This software is subject to the GNU General Public License (GPL)
# and for details, please consult it at:
#
#    http://www.fsf.org/licensing/licenses/gpl.txt
#
import types
import os, sys, re
import gtk, gtk.glade, gobject
import xmlrpclib
import constants
from ManagedNode import ManagedNode, NodeException
from NodeProxy import Node
from XenNode import XenNode, XenConfig
from utils import XMConfig, is_host_remote, randomMAC, fetch_isp, show_url
import urllib, urlparse
import xml.parsers.expat
import time
import traceback
from Groups import ServerGroup
from ImageStore import ImageStore
from htmltextview import HtmlTextView
import webbrowser

class CreateDialog:
    """ Class that handles events from create dom dialog"""

    # keep track so that signals get connected once. We may need to find
    # better solution for this.
    initialized = False 
    
    def __init__(self, wtree, image_store):
        """ Constructor"""
        self.dialog      = wtree.get_widget("CreateDialog")
        self.domName     = wtree.get_widget("DomName")
        self.domMemory   = wtree.get_widget("DomMemory")
        self.domDiskSize = wtree.get_widget("DomDiskSize")
        self.domDiskPath = wtree.get_widget("DomDiskPath")
        self.useLVM      = wtree.get_widget("UseLVM")
        self.LVGList    = wtree.get_widget("LVGList")
        self.LVGLabel   = wtree.get_widget("LVGLabel")
        self.domImage   = wtree.get_widget("DomImage")

        # the image store.
        self.image_store = image_store
        self.availableImages = self.image_store.list()

        
        # given to it in the show method
        self.managed_node = None
        
        
        # setup handlers

        if not CreateDialog.initialized:
            wtree.signal_connect("on_createDialog_cancel_button_clicked",
                                 self.on_cancel_button_clicked)
            wtree.signal_connect("on_createDialog_ok_button_clicked",
                                 self.on_ok_button_clicked)
            wtree.signal_connect("on_createDialog_DomDiskDirButton_clicked",
                                 self.on_disk_dir_button_clicked)
            wtree.signal_connect("on_createDialog_UseLVM_toggled",
                                 self.on_useLVM_toggled)
            
            # populate the image list
            for image in self.availableImages:
                self.domImage.append_text(image)

            if len(self.availableImages) > 0:
                self.domImage.set_active(0)

            CreateDialog.initialized = True

        #self.init()


    def init(self):
        self.domName.set_text("")
        self.domMemory.set_text("256")
        self.domDiskSize.set_text("3")
        if (self.managed_node.config.get(XMConfig.PATHS,constants.prop_disks_dir) is not None):
            self.domDiskPath.set_text(self.managed_node.config.get(XMConfig.PATHS,constants.prop_disks_dir))
        else:
            self.domDiskPath.set_text('')

        if self.availableImages is None:
            return
        # set the default
        if self.image_store.get_default_image() is not None:
            lower_default = self.image_store.get_default_image().lower()
            if lower_default in self.availableImages:
                pos = self.availableImages.index(lower_default)
                if pos >= 0:
                    self.domImage.set_active(pos)
                elif len(self.availableImages) > 0:
                    self.domImage.set_active(0)
        elif len(self.availableImages) > 0:
            self.domImage.set_active(0)
        
    
    def show(self, managed_node):
        """ Displays the create Dom dialog"""
        self.managed_node = managed_node

        self.init()

        # if no lvm then disable the checkbox
        #if managed_node.lvm_proxy == None:
        self.useLVM.set_sensitive(managed_node.isLVMEnabled)
            

        self.dialog.show() 

    def on_cancel_button_clicked(self, widget):
       """  Cancel creation of dom """
       self.dialog.hide()

    def on_ok_button_clicked(self, widget):
        """  Create Dom """

        names = self.managed_node.get_dom_names()
        if self.domName.get_text() in names:
            #dom = self.managed_node.get_doms()[self.domName]
            if self.managed_node.isResident(self.domName.get_text()):
                showmsg("Running VM with the same name exists.")
                return False
        
        # ask permission to overwrite if the domname
        # has already been quickcreated.
        if os.path.isfile(os.path.join(self.managed_node.config.get(XMConfig.PATHS,constants.prop_xenconf_dir),self.domName.get_text())):
            # domain conf file exists
            if confirmation('"%s" already exists: delete and replace?' % self.domName.get_text()):
                # permission to overwrite granted. cleanup exisiting files.
                cleanupQCDomain(self.managed_node,
                                self.domName.get_text(),
                                self.domDiskPath.get_text())
            else:
                # permission not granted. abort domain creation
                showmsg('Aborting VM Creation')
                self.dialog.hide()
                return
            
        msg = ''
        try:
            diskpath =  self.domDiskPath.get_text()
            if self.useLVM.get_active():
                diskpath = self.LVGList.child.get_text()

            quickcreateDom(self.managed_node,
                           self.domName.get_text(),
                           self.domMemory.get_text(),
                           diskpath,
                           int(float(self.domDiskSize.get_text())*1024),
                           self.useLVM.get_active(),
                           image_store = self.image_store,
                           image_name  = self.domImage.get_active_text())
        except (IOError, OSError), err:
            traceback.print_exc()
            msg = 'FAILED: '+str(err)
        except Exception, err:
            traceback.print_exc()
            msg = 'FAILED: '+str(err)
        else:
            msg = 'Creation Successful'

        showmsg(msg)
        self.dialog.hide()
        
    def on_disk_dir_button_clicked(self, widget):
        (res, dirname) = \
              file_selection(self.managed_node,
                             "Choose location for creating file for Dom Disk",
                             "select_folder", parentwin=self.dialog)
        
        if res and dirname:
            self.domDiskPath.set_text(dirname)

    def on_useLVM_toggled(self, widget, data=None):
        if widget.get_active():
            self.domDiskPath.set_sensitive(False)
            self.LVGList.set_sensitive(True)
            self.LVGLabel.set_sensitive(True)
            self.populateLVGList()
        else:
            self.LVGList.set_sensitive(False)
            self.LVGLabel.set_sensitive(False)
            self.domDiskPath.set_sensitive(True)

    def populateLVGList(self):
        lvglist_store = gtk.ListStore(gobject.TYPE_STRING)

        textrenderer = gtk.CellRendererText()
        self.LVGList.set_model(lvglist_store)
        self.LVGList.set_text_column(0)
                
        slistmodel = gtk.TreeModelSort(lvglist_store)
        slistmodel.set_sort_column_id(0, gtk.SORT_ASCENDING)
        lvgs = self.managed_node.lvm_proxy.listVolumeGroups()
        for lvg in lvgs:
            self.LVGList.append_text(lvg)
            
        if len(lvgs) >= 1:
            self.LVGList.set_active(0)
        

class InitParamsDialog:
    """ Class that handles events from collecting intial parameters dialog"""
    
    initialized = False
    
    def __init__(self, wtree, managed_node):
        """ Constructor"""
        self.managed_node = managed_node
        self.config = managed_node.config
        self.dialog = wtree.get_widget("initParamsDialog")
        self.defDiskLocation = \
                     wtree.get_widget("initParamsDialog_def_disk_location")
        self.defSnapshotLocation = \
                     wtree.get_widget("initParamsDialog_def_snapshot_location")

        if not InitParamsDialog.initialized :
            # setup handlers
            wtree.signal_connect("on_initParamsDialog_cancel_button_clicked",
                                 self.on_cancel_button_clicked)
            wtree.signal_connect("on_initParamsDialog_ok_button_clicked",
                                 self.on_ok_button_clicked)
            wtree.signal_connect("on_initParamsDialog_DiskLocationButton_clicked",
                                 self.on_disk_location_button_clicked)
            wtree.signal_connect("on_initParamsDialog_SnapshotLocationButton_clicked",
                                 self.on_snapshot_location_button_clicked)
            InitParamsDialog.initialized = True
            
        self.init()
        self.show()

    def init(self):
        disk_path = self.config.get(XMConfig.PATHS,constants.prop_disks_dir)
        snapshot_path = self.config.get(XMConfig.PATHS,
                                        constants.prop_snapshots_dir)
        if disk_path is None: disk_path = ""
        if snapshot_path is None: snapshot_path = ""
        
        self.defDiskLocation.set_text(disk_path)
        self.defSnapshotLocation.set_text(snapshot_path)
        
    def show(self):
        """ Displays initialization params dialog"""
        self.dialog.show() 

    def on_cancel_button_clicked(self, widget):
       """  Cancel on initialization params """
       self.dialog.hide()
       
    def on_ok_button_clicked(self, widget):
       """  OK creation of dom """
       self.config.set(XMConfig.PATHS,constants.prop_disks_dir,
                              self.defDiskLocation.get_text())
       self.config.set(XMConfig.PATHS,constants.prop_snapshots_dir,
                              self.defSnapshotLocation.get_text())
       self.dialog.hide()

    def on_disk_location_button_clicked(self, widget):
        (res, dirname) = file_selection(self.managed_node,
                                       "Select default location for VM disks",
                                        "select_folder", parentwin=self.dialog)
        
        if res and dirname:
            self.defDiskLocation.set_text(dirname)

    def on_snapshot_location_button_clicked(self, widget):
        (res, dirname) = file_selection(self.managed_node,
                                        "Select default location for Dom snapshots",
                                        "select_folder", parentwin=self.dialog)
        
        if res and dirname:
            self.defSnapshotLocation.set_text(dirname)



class AddServerGroupDialog:
    """ Class that handles adding new Server Pool"""
    
    initialized = False
    
    def __init__(self, wtree, left_nav):
        """ Constructor"""
        self.left_nav = left_nav
        
        self.dialog = wtree.get_widget("AddServerPool")
        self.group_name = wtree.get_widget("server_pool_name")
        
        self.group_name.set_activates_default(True)
        self.dialog.set_default_response(gtk.RESPONSE_OK)
        
        if not AddServerGroupDialog.initialized :
            # setup handlers
            wtree.signal_connect("on_add_server_pool_cancelbutton_clicked",
                                 self.on_cancel_button_clicked)
            wtree.signal_connect("on_add_server_pool_okbutton_clicked",
                                 self.on_ok_button_clicked)
            AddServerGroupDialog.initialized = True

        self.init()

    def init(self):
        self.group_name.set_text("")

        
    def show(self):
        """ Displays add server pool dialog"""

        ret = self.dialog.run()
        if ret == gtk.RESPONSE_DELETE_EVENT:
            self.dialog.hide()


    def on_cancel_button_clicked(self, widget):
       """  Cancel on add server pool dialog """
       self.init()
       self.dialog.hide()
       
    def on_ok_button_clicked(self, widget):
       """  Ok  button on add server pool """
       # validate parameters
       if self.group_name.get_text() == "":
           showmsg("Please enter valid group name")
           return

       try:
           group = ServerGroup(self.group_name.get_text())
           self.left_nav.add_group(group)
       except Exception , ex:
           showmsg(ex)
           return
       
       self.init()
       self.dialog.hide()


class CloneImageDialog:
    """ Class clone image """
    
    initialized = False
    
    def __init__(self, wtree, left_nav):
        """ Constructor"""
        self.left_nav = left_nav
        
        self.dialog = wtree.get_widget("CloneImage")
        self.img_name = wtree.get_widget("image_name")
        self.selected_image_label  = wtree.get_widget("selected_image")
        
        if not CloneImageDialog.initialized :
            # setup handlers
            wtree.signal_connect("on_clone_img_cancelbutton_clicked",
                                 self.on_cancel_button_clicked)
            wtree.signal_connect("on_clone_img_okbutton_clicked",
                                 self.on_ok_button_clicked)
            self.img_name.set_activates_default(True)
            self.dialog.set_default_response(gtk.RESPONSE_OK)
            CloneImageDialog.initialized = True

        self.init()

    def init(self):
        self.img_name.set_text("")

        
    def show(self, image_store, src_image_name):
        """ Displays add server pool dialog"""
        self.image_store = image_store
        self.src_image_name = src_image_name
        self.init()
        
        self.selected_image_label.set_text("Selected Image : " + src_image_name)
        ret = self.dialog.run()
        if ret == gtk.RESPONSE_DELETE_EVENT:
            self.dialog.hide()


    def on_cancel_button_clicked(self, widget):
       """  Cancel on add server pool dialog """
       self.init()
       self.dialog.hide()
       
    def on_ok_button_clicked(self, widget):
       """  Ok  button on add server pool """
       # validate parameters
       img_name = self.img_name.get_text()
       if img_name == "":
           showmsg("Please enter valid image name")
           return
       img_name = img_name.strip()
       # may be we should change spaces to underscore ourselves.
       if re.sub(ImageStore.INVALID_CHARS_EXP,"", img_name) != img_name:
           showmsg("Image name can not contain any special chars %s" % \
                              ImageStore.INVALID_CHARS)
           return
       if img_name in self.image_store.list():
           showmsg("Image already exists.")
           return
       
       self.image_store.clone_image(self.src_image_name, img_name)
       self.left_nav.refresh_nodes()
       self.left_nav.set_selection("Image Store",
                                   self.left_nav.IMAGE_STORE,
                                   op="row-activated")
       self.init()
       self.dialog.hide()



class AddNodeDialog:
    """ Class that handles adding new managed node to the host"""
    
    initialized = False
    # mapping for the combo box. Sequence related to entries in the
    # comobo box
    protocols = ["tcp", "ssl", "ssh_tunnel"]
    
    def __init__(self, wtree, client_config, left_nav):
        """ Constructor"""
        self.client_config = client_config
        self.left_nav = left_nav
        self.dialog = wtree.get_widget("AddNode")
        self.hostname = wtree.get_widget("add_node_hostname")
        self.creds_helper = CredentialsHelper(wtree)

        self.default_ssh_port = 22
        self.default_xen_protocol = "tcp"
        self.default_xen_port = "8006"
        self.default_use_keys = False
        self.default_migration_port = 8002
        
        self.protocol_combo = wtree.get_widget("add_node_xen_protocol")
        self.xen_port = wtree.get_widget("add_node_xen_port")
        self.ssh_port = wtree.get_widget("add_node_ssh_port")
        self.migration_port = wtree.get_widget("add_node_migration_port")
        self.username = wtree.get_widget("add_node_username")
        self.password = wtree.get_widget("add_node_password")
        self.use_keys_checkbox = wtree.get_widget("add_node_use_keys_checkbox")

        self.advanced = wtree.get_widget("advanced_expander")
        
        #self.protocol_combo.set_editable(False)
        # NOTE :set activate_default for widgets done in glade file.
        self.dialog.set_default_response(gtk.RESPONSE_OK)
                
        if not AddNodeDialog.initialized :
            # setup handlers
            wtree.signal_connect("on_add_node_cancelbutton_clicked",
                                 self.on_cancel_button_clicked)
            wtree.signal_connect("on_add_node_okbutton_clicked",
                                 self.on_ok_button_clicked)
            wtree.signal_connect("on_add_node_use_keys_checkbox_toggled",
                                 self.on_use_keys_toggled)
            
            AddNodeDialog.initialized = True

        default_use_keys = client_config.get(XMConfig.CLIENT_CONFIG,
                                             constants.prop_default_use_keys)
        if default_use_keys:
            self.default_use_keys = eval(default_use_keys)

        default_xen_port = client_config.get(XMConfig.CLIENT_CONFIG,
                                             constants.prop_default_xen_port)
        if default_xen_port:
            self.default_xen_port = int(default_xen_port)

        default_ssh_port = client_config.get(XMConfig.CLIENT_CONFIG,
                                             constants.prop_default_ssh_port)
        if default_ssh_port:
            self.default_ssh_port = int(default_ssh_port)

        default_migration_port = client_config.get(XMConfig.CLIENT_CONFIG,
                                             constants.prop_default_migration_port)
        if default_migration_port:
            self.default_migration_port = int(default_migration_port)

        default_xen_protocol = client_config.get(XMConfig.CLIENT_CONFIG,
                                                 constants.prop_default_xen_protocol)
        if default_xen_protocol :
            self.default_xen_protocol = default_xen_protocol
        #print "ssh_port is ", self.default_ssh_port, type(self.default_ssh_port)
        self.init()

    def on_use_keys_toggled(self, widget):
        use_keys = widget.get_active()
        #self.username.set_sensitive(not use_keys)
        self.password.set_sensitive(not use_keys)

    def init(self):
        self.protocol_combo.set_active(self.protocols.index(self.default_xen_protocol))
        self.ssh_port.set_text(str(self.default_ssh_port))
        self.migration_port.set_text(str(self.default_migration_port))
        self.xen_port.set_text(str(self.default_xen_port))
        self.username.set_text("root")
        self.password.set_text("")
        self.hostname.set_text("")
        self.hostname.set_sensitive(True)
        self.use_keys_checkbox.set_active(self.default_use_keys)
        self.advanced.set_expanded(self.default_use_keys)
        self.hostname.grab_focus()

    def show_in_edit(self, existing_node, group_name = None):
        """ Displays add node dialog in edit mode """
        self.existing_node = existing_node
        self.hostname.set_sensitive(False)
        self.mode = "edit"
        self.dialog.set_title("Edit Server")

        self.group_name = group_name


        self.protocol_combo.set_active(self.protocols.index(self.existing_node.protocol))
        self.ssh_port.set_text(str(self.existing_node.ssh_port))
        self.migration_port.set_text(str(self.existing_node.migration_port))
        self.xen_port.set_text(self.existing_node.tcp_port)
        self.username.set_text(self.existing_node.username)
        self.password.set_text("")
        self.hostname.set_text(self.existing_node.hostname)
        self.use_keys_checkbox.set_active(self.existing_node.use_keys)
        self.advanced.set_expanded(self.existing_node.use_keys)

        self.hostname.grab_focus()

        ret = self.dialog.run()
        if ret == gtk.RESPONSE_DELETE_EVENT:
            self.dialog.hide()


        
    def show(self, group_name = None):
        """ Displays add node dialog"""
        #self.dialog.show()
        self.init()
        self.mode = "add"
        self.dialog.set_title("Add Server")
        self.existing_node = None
        self.group_name = group_name
        ret = self.dialog.run()
        if ret == gtk.RESPONSE_DELETE_EVENT:
            self.dialog.hide()


    def on_cancel_button_clicked(self, widget):
       """  Cancel on add node dialog """
       self.init()
       self.dialog.hide()
       
    def on_ok_button_clicked(self, widget):
       """  Ok  button on add node """
       # validate parameters
       if self.hostname.get_text() == "":
           showmsg("Please enter valid host name")
           return

       if self.xen_port.get_text() == "":
           showmsg("Please enter valid xen port")
           return

       if self.ssh_port.get_text() == "":
           showmsg("Please enter valid ssh port")
           return

       if self.migration_port.get_text() == "":
           showmsg("Please enter valid migration port")
           return

       remote = is_host_remote(self.hostname.get_text())
       
       # allow for local host to be added.
       #if not remote:
       #    showmsg("Local host can not be added")
       #    return
       
       # create a new node
       ndx = self.protocol_combo.get_active()
       if ndx > -1:
           selected_protocol = self.protocols[ndx]
       else:
           showmsg("Invalid selection of protocol")

       try:
           node = XenNode(hostname = self.hostname.get_text(),
                          username = self.username.get_text(),
                          password = self.password.get_text(),
                          isRemote = remote, 
                          protocol = selected_protocol,
                          tcp_port = self.xen_port.get_text(),
                          #helper = self.creds_helper,
                          migration_port = int(self.migration_port.get_text()),
                          ssh_port = int(self.ssh_port.get_text()),
                          use_keys = self.use_keys_checkbox.get_active(),
                          address  = self.hostname.get_text()
                          )
           node.connect()
           if self.mode == "edit":
               self.left_nav.remove_node(self.existing_node, self.group_name)
               # do cleanup for the main thread.
               # invading privacy of node uggh
               if self.existing_node is not None and \
                      self.existing_node._node_proxy is not None: 
                   try:
                       Node.clean_locals(self.existing_node.node_proxy)
                   except Exception, ex:
                       print "Exception while cleaning locals ", ex
                       pass
           self.left_nav.add_node(node, self.group_name)
       except Exception , ex:
           traceback.print_exc()
           #try:
           #    if node is not None and node.node_proxy is not None:
           #        Node.clean_locals(node.node_proxy)
           #except Exception, e:
           #    pass
           showmsg(ex)
           return
       # username and password are not stored

       # clear the passwords from memory.
       self.init()
       self.dialog.hide()



class Cred:
    def __init__(self, username, password):
        self.username = username
        self.password = password

    def get_username(self):
        return self.username
    
    def get_password(self):
        return self.password
    

class CredentialsHelper:

    def __init__(self,wtree):
        self.wtree = wtree
    
    
    def get_credentials(self, hostname, username=None):
        dlg = CredentialsDialog(wtree)
        dlg.show(hostname, username)
        ret = dlg.dialog.run()
        cred = None
        if ret == gtk.RESPONSE_OK:
            cred = Cred(dlg.username.get_text(), dlg.password.get_text())
        dlg.init()
        return cred

class CredentialsDialog:
    """ Class that handles events from collecting credentials"""
    initialized = False
    def __init__(self, wtree):
        """ Constructor"""
        self.dialog = wtree.get_widget("Credentials")
        self.username = wtree.get_widget("cred_username")
        self.password = wtree.get_widget("cred_password")
        self.cred_label = wtree.get_widget("cred_label")

        self.password.set_activates_default(True)
        self.dialog.set_default_response(gtk.RESPONSE_OK)

        if not CredentialsDialog.initialized:
            # setup handlers
            wtree.signal_connect("on_cred_cancelbutton_clicked",
                                 self.on_cancel_button_clicked)
            wtree.signal_connect("on_cred_okbutton_clicked",
                                 self.on_ok_button_clicked)
            CredentialsDialog.initialized = True
        self.init()

    def init(self):
        self.username.set_text("")
        self.password.set_text("")
        
    def show(self, hostname, username = None):
        """ Displays initialization params dialog"""
        if username is not None:
            self.username.set_text(username)
            self.password.grab_focus()
        self.cred_label.set_text("Enter Credentials : " + hostname)
        #self.dialog.set_title("Credentials for " + hostname)
        self.dialog.show()

    def on_cancel_button_clicked(self, widget):
       """  Cancel on initialization params """
       self.dialog.hide()
       
    def on_ok_button_clicked(self, widget):
       """  Cancel creation of dom """
       self.dialog.hide()

    

class RemoteFileDialog:
    """ Simple dialog box to take file or path from the user """
    initialized = False
    def __init__(self, wtree):
        """ Constructor"""
        self.dialog = wtree.get_widget("RemoteFileDialog")
        self.filename = wtree.get_widget("remote_file_filename")
        self.file_selector = wtree.get_widget("remote_file_selector")
        self.file_label = wtree.get_widget("remote_file_label")
        self.hostname = wtree.get_widget("remote_file_hostname")
        self.frame_label = wtree.get_widget("remote_file_frame_title")
        if not RemoteFileDialog.initialized:
            # setup handlers
            wtree.signal_connect("on_remote_file_cancel_button_clicked",
                                 self.on_cancel_button_clicked)
            wtree.signal_connect("on_remote_file_okbutton_clicked",
                                 self.on_ok_button_clicked)
            wtree.signal_connect("on_remote_file_selectorButton_clicked",
                                 self.on_file_selector_clicked)
            RemoteFileDialog.initialized = True
            
        self.init()

    def init(self):
        self.filename.set_text("")

    def show(self, managed_node,
             title, fstype, init_folder,
             init_file, parentwin, multi_select=False):

        self.managed_node = managed_node
        self.title = title
        self.fstype = fstype
        self.init_folder = init_folder
        self.init_file = init_file
        self.parentwin = parentwin

        self.hostname.set_text(managed_node.hostname)

        if self.fstype == "select_folder":
            self.file_label.set_text("Folder name")
            self.frame_label.set_markup("<b>Specify Folder </b>")
        else:
            self.file_label.set_text("Filename")
            if multi_select:
                self.frame_label.set_markup("<b>Specify Filenames (comma separated) </b>")
            else:
                self.frame_label.set_markup("<b>Specify Filename </b>")
            
        self.dialog.set_title(title)

        # choose decent default
        if init_folder is not None and init_file is not None:
            # remove trailing / from path
            if init_folder.rfind("/") == len(init_folder) - 1:
                init_folder = init_folder[0:len(init_folder) -1]
            # remove leading / from filename
            if init_file.find("/") == 0: 
                init_file = init_file[1:]
            self.filename.set_text(init_folder + "/" + init_file)
            
        elif init_folder is not None :
            self.filename.set_text(init_folder)
        elif init_file is not None:
            self.filename.set_text(init_file)

    def on_cancel_button_clicked(self, widget):
       """  Cancel on RemoteFileDialog  """
       self.dialog.hide()
       
    def on_ok_button_clicked(self, widget):
       """  Ok on RemoteFileDialog """
       self.dialog.hide()

    def on_file_selector_clicked(self, widget):
       """  file selector clicked on RemoteFileDialog """
       # force the file selection to local/mounted files.
       (res, selection) = file_selection(self.managed_node,
                                         self.title,
                                         self.fstype,
                                         self.init_folder, self.init_file,
                                         self.parentwin,
                                         force_local = True,
                                         multi_select = True)
       
       if res and selection:
           text = ""
           for file in selection:
               text= text + file + ","
               
           if len(text) > 0 and text[-1] == ",":
               text = text[0:-1]
           
           self.filename.set_text(text)

    def run(self):
        return self.dialog.run()

    def get_filename(self):
        return self.filename.get_text()


class FileViewEditDialog:
    """ Allow a text file to be viewed/edited """
    initialized = False
    def __init__(self, wtree):
        """ Constructor"""
        self.dialog = wtree.get_widget("FileViewEditDialog")
        self.save_button = wtree.get_widget("file_view_edit_save")
        self.cancel_button = wtree.get_widget("file_view_edit_cancel")
        self.file_content = wtree.get_widget("file_content")
        if not FileViewEditDialog.initialized:
            # setup handlers
            wtree.signal_connect("on_file_view_edit_save_clicked",
                                 self.on_save_button_clicked)
            wtree.signal_connect("on_file_view_edit_cancel_clicked",
                                 self.on_cancel_button_clicked)
            FileViewEditDialog.initialized = True

        self.managed_node = None
        self.filename = None
        self.parentwin = None
        self.editable = True
        
    #    self.init()

    #def init(self):
    #    self.file_content.set_text("")

    def show(self, widget,  managed_node, filename, content = None,editable=True,
             parentwin = None):

        self.managed_node = managed_node
        self.filename = filename
        self.parentwin = parentwin
        self.editable = editable
        if content is not None:
            self.text = content
            self.editable = False
        else:
            if managed_node is not None and filename is not None:
                self.text = self.get_text_from_file(self.managed_node,
                                                    self.filename)
            else:
                self.text = ""
            
        self.text_buffer = gtk.TextBuffer()
        self.text_buffer.set_text(self.text)
        self.file_content.set_buffer(self.text_buffer)
        if self.managed_node is not None and self.filename is not None:
            self.dialog.set_title("File : "+  \
                                  self.managed_node.hostname + ":" + self.filename)
        else:
            self.dialog.set_title("Content")
        
        # adjusts the buttons
        if not self.editable:
            self.save_button.set_label("Ok")
        else:
            self.save_button.set_label("Save")

        self.cancel_button.set_property("visible", self.editable)

        if parentwin:
            self.dialog.set_transient_for(parentwin)
            self.dialog.set_position(gtk.WIN_POS_CENTER_ON_PARENT)

                    
        #self.dialog.show()
        ret = self.dialog.run()
        if ret == gtk.RESPONSE_DELETE_EVENT:
            self.dialog.hide()


    def on_save_button_clicked(self, widget):
        """  Save on FileViewEditDialog  """
        if self.editable:
            buf = self.file_content.get_buffer()
            self.text = buf.get_text(buf.get_start_iter(),
                                     buf.get_end_iter())
            print "save to text  " , self.filename
            self.save_text_to_file(self.managed_node,self.filename, self.text)
            
        self.dialog.hide()
        
    def on_cancel_button_clicked(self, widget):
        """  Cancel on FileViewEditDialog """
        self.dialog.hide()

    def get_text_from_file(self, managed_node,filename):
        text=""
        file = managed_node.node_proxy.open(filename)
        lines = file.readlines()
        text = "".join(lines)
        file.close()
        return text

    def save_text_to_file(self, managed_node,filename, text):
        file = managed_node.node_proxy.open(filename, "w")
        file.write(text)
        file.close()
        

class NodeSelection:
    """ Show dialog with list of nodes and return the selection """
    initialized = False
    def __init__(self, wtree):
        """ Constructor"""
        self.dialog = wtree.get_widget("NodeSelection")
        self.node_list_view = wtree.get_widget("select_node_list")
        self.migrate_live_check_button = wtree.get_widget("migrate_live")
        self.select_node_label = wtree.get_widget("select_node_label")
        #self.node_list_view.set_activates_default(True)
        self.dialog.set_default_response(gtk.RESPONSE_OK)
        self.list_model = None
        self.selected_node_name = None
        self.selected_node = None
        self.selected_node_type = None
        self.allow_groups = False
        self.migration_mode = False
        self.migrate_live_selection = False

        pbrenderer = gtk.CellRendererPixbuf()
        column = gtk.TreeViewColumn("       ", pbrenderer)
        column.set_cell_data_func(pbrenderer, get_state_pixbuf)
        self.node_list_view.append_column(column)
        
        textrenderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn("Name", textrenderer, text=0)
        
        self.node_list_view.append_column(column)
        
        self.node_list_model = gtk.TreeStore(gobject.TYPE_STRING,
                                             gobject.TYPE_STRING,
                                             gobject.TYPE_PYOBJECT,
                                             gobject.TYPE_STRING) # group name
        
        self.node_list_view.set_model(self.node_list_model)
        
        if not NodeSelection.initialized:
                        # setup handlers
            wtree.signal_connect("on_node_selection_ok_clicked",
                                 self.on_ok_button_clicked)
            wtree.signal_connect("on_node_selection_cancel_clicked",
                                 self.on_cancel_button_clicked)
            NodeSelection.initialized = True

    def get_selection(self):
        return (self.selected_node_name,
                self.selected_node_type,
                self.selected_node,
                self.migrate_live_selection)

    
    def show(self, widget, manager, migration = False, allow_groups =False,
             parentwin = None):

        self.parentwin = parentwin
        self.selected_node_name = None
        self.selected_node = None
        self.selected_node_type = None
        self.migrate_live_selection = False
        self.migration_mode = migration
        self.allow_groups = allow_groups

        if allow_groups:
            self.select_node_label.set_text("Select a Managed Server or a Server Pool")
        else:
            self.select_node_label.set_text("Select a Managed Server")
        
        populate_nodes(manager, self.node_list_view, self.node_list_model,
                       False)

        self.migrate_live_check_button.set_property("visible",
                                                    self.migration_mode)
        self.migrate_live_check_button.set_active(self.migration_mode)

        self.dialog.show()
        ret = self.dialog.run()
        if ret == gtk.RESPONSE_DELETE_EVENT:
            self.dialog.hide()


    def on_ok_button_clicked(self, widget):
        """  Ok on Select Node  """
        widget = self.node_list_view.get_selection()
        treemodel, treeiter = widget.get_selected()
        if treeiter:
            (name,type,node,group_name)= \
                                         (treemodel.get_value(treeiter, 0),
                                          treemodel.get_value(treeiter, 1),
                                          treemodel.get_value(treeiter, 2),
                                          treemodel.get_value(treeiter, 3))
            

            if type is not None and type == MANAGED_NODE or \
               (type is not None and type == SERVER_POOL and
                self.allow_groups == True):
                self.selected_node_name = name
                self.selected_node = node
                self.selected_node_type = type
                if self.migration_mode :
                    self.migrate_live_selection = self.migrate_live_check_button.get_active()
            else:
                showmsg("Invalid Selection.")
                return 
        self.dialog.hide()
        
    def on_cancel_button_clicked(self, widget):
        """  Cancel on FileViewEditDialog """
        self.dialog.hide()


# Make this non-modal as now can be displayed from multiple
# provisioning dialogs.

class ProvStatusDlg:
    """Displays the status message and output and logfile"""
    def __init__(self):
        """ Constructor"""
        self.gladefile = gladefile
        wtree = gtk.glade.XML(self.gladefile, "ProvStatusDlg")
        self.wtree = wtree
        self.dialog = wtree.get_widget("ProvStatusDlg")
        self.msg_widget = wtree.get_widget("msg_text")
        self.output = ""
        self.log_filename = None
        
        # setup handlers
        wtree.signal_connect("on_msg_close_clicked",
                             self.on_close_button_clicked)
        wtree.signal_connect("on_msg_view_output_clicked",
                             self.on_output_button_clicked)
        wtree.signal_connect("on_msg_view_log_clicked",
                             self.on_log_button_clicked)
        
    def show(self, widget, msg, managed_node, log_filename, output,
             parentwin = None):

        self.managed_node = managed_node
        self.parentwin = parentwin
        self.msg = msg
        self.output = output
        self.log_filename = log_filename

        self.msg_widget.set_text(msg)

        if parentwin:
            self.dialog.set_transient_for(parentwin)
            self.dialog.set_position(gtk.WIN_POS_CENTER_ON_PARENT)

        
        self.dialog.show()
        # Non-modal op now.
        #ret = self.dialog.run()
        #if ret == gtk.RESPONSE_DELETE_EVENT:
        #    self.dialog.hide()


    def on_output_button_clicked(self, widget):
        """  output button on ProvStatusDlg   """
        dlg = file_editor.show(widget,
                               None,
                               None,
                               content = self.output,
                               editable = False,
                               parentwin = self.dialog)

    def on_log_button_clicked(self, widget):
        """  output button on ProvStatusDlg   """
        dlg = file_editor.show(widget,
                               self.managed_node,
                               self.log_filename,
                               editable=False,
                               parentwin = self.dialog)
        


    def on_close_button_clicked(self, widget):
        """  Cancel on ProvStatusDlg """
        self.dialog.destroy()


class GetLocalIPDlg:

    def __init__(self):

        """ Constructor"""
        self.gladefile = gladefile
        wtree = gtk.glade.XML(self.gladefile, "local_ip_dlg")
        self.dialog = wtree.get_widget("local_ip_dlg")

        self.local_ip_entry = wtree.get_widget("local_ip_entry")

        self.dialog.set_default_response(gtk.RESPONSE_OK)
        # connect the handlers
        wtree.signal_connect("on_local_ip_cancel_clicked",
                                 self.on_cancel_clicked)
        wtree.signal_connect("on_local_ip_ok_clicked",
                             self.on_ok_clicked)


    def show(self, manager, local_node, parentwin = None):

        self.manager = manager
        self.local_node = local_node

        self.parentwin = parentwin
        
        if parentwin:
            self.dialog.set_transient_for(parentwin)
            self.dialog.set_position(gtk.WIN_POS_CENTER_ON_PARENT)

        adr = local_node.get_address()
        if adr and adr.lower() != "localhost":
            self.local_ip_entry.set_text(adr)
        else:
            self.local_ip_entry.set_text("")
        
        self.dialog.show()
        ret = self.dialog.run()
        
        self.dialog.destroy()
        return ret

    def on_cancel_clicked(self, widget):
        return False
        


    def on_ok_clicked(self, widget):
        local_ip = self.local_ip_entry.get_text()
        local_ip = local_ip.strip()
        if not local_ip:
            showmsg("Please enter a hostname of IP address.")
            return
        self.local_node.set_address(local_ip)
        self.manager._save_node(self.local_node)
        return True


class MigrationChecksResultsDlg:

    (TYPE,CATEGORY,DETAILS) = range(3)

    def __init__(self):

        self.gladefile = gladefile
        wtree = gtk.glade.XML(self.gladefile, "MigrationResultDlg")
        self.dialog = wtree.get_widget("MigrationResultDlg")

        """ Constructor"""
        # the list view
        self.list_view = wtree.get_widget("m_chk_tree_view")

        # set default response
        self.dialog.set_default_response(gtk.RESPONSE_OK)

        # state variables
        self.list_model = None


        # lets populate the colums

        # ENABLE THIS FOR Warning/Error Icon
        pbrenderer = gtk.CellRendererPixbuf()
        column = gtk.TreeViewColumn("Type", pbrenderer)
        column.set_cell_data_func(pbrenderer, self.get_msg_type_pb)
        column.set_sort_column_id(self.TYPE)
        self.list_view.append_column(column)
        
        lt_textrenderer = gtk.CellRendererText()
        lt_textrenderer.set_property("xalign", 0.0)

        ## column = gtk.TreeViewColumn("Type", lt_textrenderer, text=0)
##         column.set_clickable(True)
##         column.set_sort_column_id(self.TYPE)
##         column.set_resizable(True)
##         self.list_view.append_column(column)

        
        textrenderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn("Category", textrenderer, text=1)
        column.set_clickable(True)
        column.set_sort_column_id(self.CATEGORY)
        column.set_resizable(True)
        self.list_view.append_column(column)
        
        textrenderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn("Message", textrenderer, text=2)
        column.set_clickable(True)
        column.set_sort_column_id(self.DETAILS)
        column.set_resizable(True)
        self.list_view.append_column(column)

        
        self.list_model = gtk.TreeStore(gobject.TYPE_STRING, # Type
                                        gobject.TYPE_STRING, # Category
                                        gobject.TYPE_STRING) # Message


        
        self.list_view.set_model(self.list_model)

        self.list_model.set_sort_column_id(self.TYPE, gtk.SORT_ASCENDING)


    def populate_list(self, err_list, warn_list):
        for err in err_list:
            (cat, msg) = err
            iter = self.list_model.insert_before(None, None)
            self.list_model.set(iter,
                                self.TYPE,"Error",
                                self.CATEGORY,cat,
                                self.DETAILS,msg)

        for warn in warn_list:
            (cat, msg) = warn
            iter = self.list_model.insert_before(None, None)
            self.list_model.set(iter,
                                self.TYPE,"Warning",
                                self.CATEGORY,cat,
                                self.DETAILS,msg)


    def get_msg_type_pb(self,column,cell,model,iter):
        type = model.get_value(iter, self.TYPE)
        id = None
        if type:
            if type == "Warning":
                id = gtk.STOCK_DIALOG_WARNING
            elif type == "Error":
                id = gtk.STOCK_DIALOG_ERROR

        cell.set_property('stock-id', id)
        
    def show(self, err_list, warn_list, parentwin=None):
             
        self.err_list = err_list
        self.warn_list = warn_list
        self.parentwin = parentwin
        
        if parentwin:
            self.dialog.set_transient_for(parentwin)
            self.dialog.set_position(gtk.WIN_POS_CENTER_ON_PARENT)

        self.populate_list(self.err_list, self.warn_list)

        self.dialog.show()
        ret = self.dialog.run()
        self.dialog.destroy()
            
        if ret == gtk.RESPONSE_OK:
            return True
        else:
            return False

class ApplianceDetailsDlg:

    def __init__(self):

        """ Constructor"""
        self.gladefile = gladefile
        wtree = gtk.glade.XML(self.gladefile, "appliance_details_dlg")
        self.dialog = wtree.get_widget("appliance_details_dlg")

        self.w_app_protocol = wtree.get_widget("app_protocol")
        self.w_app_hostname = wtree.get_widget("app_hostname")
        self.w_app_port = wtree.get_widget("app_port")
        self.w_app_path = wtree.get_widget("app_path")

        self.w_app_mgmt_protocol = wtree.get_widget("app_mgmt_protocol")
        self.w_app_mgmt_port = wtree.get_widget("app_mgmt_port")
        
        self.dialog.set_default_response(gtk.RESPONSE_OK)
        # connect the handlers
        wtree.signal_connect("on_app_details_cancel_clicked",
                                 self.on_cancel_clicked)
        wtree.signal_connect("on_app_details_ok_clicked",
                             self.on_ok_clicked)

        self.mapping = ((self.w_app_protocol, "app_protocol",
                         "default_app_protocol"),
                        (self.w_app_hostname, "host",
                         "default_app_hostname"),
                        (self.w_app_port, "app_port", "default_app_port"),
                        (self.w_app_path, "app_path", "default_app_path"),
                        (self.w_app_mgmt_protocol, "app_mgmt_protocol",
                         "default_app_mgmt_protocol"),
                        (self.w_app_mgmt_port, "app_mgmt_port", "default_app_mgmt_port"))

        self.protocols = ["http", "https"]        

    def show(self, vm_config, defaults={}, parentwin = None):

        self.vm_config = vm_config
        self.defaults = defaults

        self.parentwin = parentwin
        
        if parentwin:
            self.dialog.set_transient_for(parentwin)
            self.dialog.set_position(gtk.WIN_POS_CENTER_ON_PARENT)



        # populate
        for w, vkey, dkey in self.mapping:
            val = self.vm_config.get(vkey)
            if not val:
                val = self.defaults.get(dkey)

            if not val:
                val = ""
                
            if isinstance(w, gtk.Entry):
                w.set_text(str(val))
            elif isinstance(w, gtk.ComboBox):
                print val
                if val in self.protocols:
                    w.set_active(self.protocols.index(val))
                else:
                    w.set_active(-1)
                   
        self.dialog.show()
        ret = self.dialog.run()

        if ret == gtk.RESPONSE_OK:
            for w, vkey, dkey in self.mapping:
                if isinstance(w, gtk.Entry):
                    val = w.get_text()
                    if val and vkey in ("app_port", "app_mgmt_port"):
                        val = int(val)

                elif isinstance(w, gtk.ComboBox):
                    val = self.protocols[w.get_active()]
                self.vm_config[vkey] = val

            self.vm_config.write()
            self.dialog.destroy()
            return True
        else:
            self.dialog.destroy()
            return False

    def on_cancel_clicked(self, widget):
        return False
        

    def on_ok_clicked(self, widget):
        return True


# show the news / updates for the product
class UpdatesDlg:

    def __init__(self):

        """ Constructor"""
        self.gladefile = gladefile
        wtree = gtk.glade.XML(self.gladefile, "updates_dlg")
        self.dialog = wtree.get_widget("updates_dlg")
        self.view_wnd = wtree.get_widget("updates_dlg_view_wnd")
        
        self.view = wtree.get_widget("update_view")
        self.view_wnd.remove(self.view)
        self.view = HtmlTextView()

        self.view.set_property("visible",True)
        self.view.set_wrap_mode(gtk.WRAP_WORD)
        self.view.set_editable(False)

        self.view.connect("url-clicked", self.url_clicked_cb)
        self.view_wnd.add(self.view)

        self.title_span = '<span style="font-size: 125%; font-family: serif; color:#0000FF;text-align: left">'

        # connect the handlers
        wtree.signal_connect("on_updates_dlg_close_clicked",
                                 self.on_close_button_clicked)


    def show(self, updates, widget = None,parentwin = None):
        self.updates = updates
        self.parentwin = parentwin
            
        if parentwin:
            self.dialog.set_transient_for(parentwin)
            self.dialog.set_position(gtk.WIN_POS_CENTER_ON_PARENT)

        text = "<body>"

        for update in updates:
            text += self.title_span
            text += '<a href="' + str(update["link"]) + '">'
            text = text + str(update["title"])
            text += "</a>"
            text += "</span>"
            text += "<br/>"
            text += "<br/>"
            
            text = text + str(update["description"])
            text += "<br/>"
            text += "<br/>"


        text += "</body>"
        try:
            self.view.display_html(text)
        except Exception, ex:
            print "error showing ConVirt updates : HTML -> " + text + "\n" + \
                  "Exception " + str(ex)
            
        self.dialog.show()

    def on_close_button_clicked(self, widget):
        self.dialog.destroy()

    def url_clicked_cb(htmlview, self, url, type):
        show_url(url)
        


#############
#    Util functions.. may move to util OR Managed Node
#    Also needs to cut this over to remote api
#



def createVBD(managed_node, filename, size=2048, backup=False):
    """Creates a file Virtual Block Device called 'name'
    at 'location'. 'size' is specified in kB.
    RAISES: IOError, OSError """

    if managed_node.node_proxy.file_exists(filename):
        if backup:
            managed_node.node_proxy.rename(filename,filename+'.bak')
        else:
            managed_node.node_proxy.remove(filename)

    # create the new disk block
    fd = managed_node.node_proxy.open(filename,'w')
    off = size * 1024L**2
    fd.seek(off, 0)
    fd.write('\x00')
    fd.close()


def quickcreateDom(managed_node,
                   name, mem, vbdlocation, vbdsize=2048, uselvm=False,
                   image_store = None, image_name = None):
    """Creates a new dom configuration file using default
    kernel and ramdisk attributes and starts the corresponding
    DomU instance.
    RAISES: IOError, OSError"""


    # Instantiate domconfig from dom template
    domfile = XenConfig(managed_node, image_store.get_vm_template(image_name))

    # create a new domfile object
    conf_dir = managed_node.config.get(XMConfig.PATHS,
                                       constants.prop_xenconf_dir)
    dom_filename = os.path.join(conf_dir, name)
    
    # set domfile attributes
    domfile['name'] = name
    domfile['memory'] = int(mem)

    # generate the disk entry
    if not uselvm :
        # create VBD and set corresponding attribute in domfile
        filename = "%s/%s.disk.xm" % (vbdlocation, name)
        #if domfile['disk']:
        #    domfile['disk'].append('file:%s,hda,w' % filename)
        #else:
        domfile['disk'] = ['file:%s,hda,w' % filename]
    else:
        #if domfile['disk']:
        #    domfile['disk'].append('phy:/dev/%s/%s.disk.xm,hda,w'%(vbdlocation,name))
        #else:
        domfile['disk'] = ['phy:/dev/%s/%s.disk.xm,hda,w' %(vbdlocation,name)]

    # instantiate the config
    template_map = {}
    template_map['IMAGE_NAME'] = image_name
    template_map['VM_NAME'] = name
    template_map['AUTOGEN_MAC'] = randomMAC()
        
    domfile.instantiate_config(template_map)

    # save the boot loader if specified with kernel and ramdisk
    bldr = None
    if (domfile["kernel"] is not None and domfile['kernel'] is not '') and \
       (domfile['bootloader'] is not None and domfile['bootloader'] is not ''):
        bldr = domfile['bootloader']
        domfile['bootloader'] = ''
    
    domfile.save(dom_filename)

    # execute provisioning script
    image_store.execute_provisioning_script(managed_node,
                                            image_name,
                                            domfile,
                                            None)

                                
    # TODO : Read the log file and show it to user.
    # useful in debugging.
      
    # prompt user to change the file here? or after starting dump him
    # in edit file mode ?

    # start the dom
    newdom = managed_node.create_dom(domfile)

    # replace the bootloader for next reboot. Kludge
    if bldr != None:
        # bootloader specified        
        domfile['kernel'] = ''
        domfile['ramdisk']= ''
        domfile['bootloader'] = bldr 
        domfile.write()

    


def cleanupQCDomain(managed_node, name, vbdlocation=''):
    """ Delete the xen configuration file and associated
    disk devices created during a quickcreate"""
    #domfilename = os.path.join(managed_node.config.get(XMConfig.PATHS,constants.prop_xenconf_dir),name)
    #if managed_node.node_proxy.file_exists(domfilename):
    #    managed_node.node_proxy.remove(domfilename)

    if name in managed_node.get_dom_names():
        dom = managed_node.get_dom(name)
        dom_config = None
        if dom:
            dom_config = dom.get_config()

        if dom_config:
            domfilename = dom_config.filename
            if domfilename and managed_node.node_proxy.file_exists(domfilename):
                managed_node.node_proxy.remove(domfilename)
                
            
            for file in dom_config.getDisks():
                # skip read only volumes, they are likely to be shared
                # or cdroms.
                # THIS IS NOT COMPLETE SOLN, NEED to prompt user for each
                # disk is the right soln
                if file.mode.find("w") == -1:
                    continue
                if file.type is 'lvm':
                    managed_node.lvm_proxy.removeLogicalVolume(file.filename)
                    print 'deleting: ' + file.filename
                elif file.type in ['file','tap:aio','tap:qcow']:
                    if managed_node.node_proxy.file_exists(file.filename):
                        print 'deleting: ' + file.filename
                        managed_node.node_proxy.remove(file.filename)
    else:
        print "Couldn't find the domain %s. Skipping deletion" % name






########## UI Utils #################

class CBException:
    def __init__(self, context, ex):
        self.context = context
        self.ex = ex

def cb_showmsg(msg):
    try:
        gtk.gdk.threads_enter()
        showmsg(msg)
    finally:
        gtk.gdk.threads_leave()


def showmsg(msg):
    """Displays the status message passed as an argument in a popup window"""
    input = msg

    d_msg = ""
    context = ""
    
    if isinstance(input, CBException):
        (context, input) =  (input.context, input.ex)

    if input and isinstance(input, Exception) and input.__dict__.get("args"):
        args = input.args
        if len(args) == 0:
            d_msg = str(input)
        else:
            for mx in args:
                if isinstance(mx, list):
                    for m in mx:
                        d_msg += str(m) + "\n"
                else:
                    d_msg += str(mx)
    else:
        d_msg = str(input)

    if context is not None and context != "":
        d_msg = context +"\n" + d_msg
        
    wtree.get_widget('StatusMsg').set_text(d_msg)
    dialog = wtree.get_widget('StatusDialog')
    dialog.run()
    dialog.hide()


from threading import Thread
import threading
import NodeProxy
class UIWorker(Thread):
    """
     
    """
    def __init__(self, ui_callback,context=None,
                 success_cb = None,failure_cb = cb_showmsg,
                 lock_ui = False):
                 #ui_cb_args=(), ui_cb_kwargs={}) :
        
        Thread.__init__(self)
        self.setDaemon(True)
        self.ui_callback = ui_callback
        #self.ui_cb_args = ui_cb_args
        #self.ui_cb_kwargs = ui_cb_kwargs
        self.success_cb = success_cb
        self.failure_cb = failure_cb
        self.context = context
        self.result = None
        self.done = False
        self.lock_ui = lock_ui




    def run(self):
        try:
            try:
                #self.ui_callback(*self.ui_cb_args,**self.ui_cb_kwargs)
                print "calling cb", self.context

                if not self.lock_ui:
                    self.result = self.ui_callback()
                else:
                    # might have to give up livelyness here..
                    # as we throw up login dialog from backend op too :(
                    try:
                        gtk.gdk.threads_enter()
                        self.result = self.ui_callback()
                    finally:
                        gtk.gdk.threads_leave()

                # call the success_cb
                if self.success_cb:
                    print "posting success cb", self.context
                    gobject.idle_add(self.success_cb)
            except Exception ,ex:
                if self.failure_cb:
                    #if self.context is not None :
                    #    msg = self.context + ":" + str(ex)
                    #else:
                    #    msg = str(ex)
                    traceback.print_exc()
                    print "posting exception cb", self.context, ex
                    gobject.idle_add(self.failure_cb, CBException(self.context, ex))

            if self.context is None:
                self.context = ""
            print "thread done. Context ", self.context
            self.done = True
        finally:
            # clean up andy local storage with the thread
            NodeProxy.Node.clean_locals()

def show_wait_cursor(win = None):
    """ Wait cursor displayed / all input disabled """ 
    
    if win is None:
        win = main_context["main_window"]
    watch = gtk.gdk.Cursor(gtk.gdk.WATCH)
    gdkwin = win.window
    gdkwin.set_cursor(watch)
    gtk.gdk.flush()

def hide_wait_cursor(win = None):
    if win is None:
        win = main_context["main_window"]
    gdkwin = win.window
    gdkwin.set_cursor(None)




def confirmation(msg):
    """Displays a confirmation dialog message and returns a boolean
    representing the result"""

    wtree.get_widget('ConfirmationDialog').set_title("Confirm")
    wtree.get_widget('ConfirmationMsg').set_text(msg)
    dialog = wtree.get_widget('ConfirmationDialog')
    res = dialog.run() == gtk.RESPONSE_YES
    dialog.hide()
    return res

def file_selection(managed_node,
                   title, fstype,
                   init_folder=None, init_file=None,
                   parentwin=None, force_local = False,
                   multi_select = False):
    """This function will setup a file selection dialogue and return a tuple
    consisting of a boolean which is true if the user selected a filename,
    followed by the filename. fstype is either "open" or "save" or
    "select_folder" and title is whateveryou want displayed as the
    window's title bar. The init_folder and init_file params would be used
    as initial values for the dialog box"""

    gnome_vfs_enabled = True
    if main_context.has_key("client_config"):
        client_config = main_context["client_config"]
        gnome_vfs_prop_val = client_config.get(XMConfig.CLIENT_CONFIG,
                                              constants.prop_gnome_vfs_enabled)
        if gnome_vfs_prop_val is not None:
             gnome_vfs_enabled = eval(gnome_vfs_prop_val)
             
    if gnome_vfs_enabled:
        back_end = 'gnome-vfs'
    else:
        back_end = None

    if back_end is None and managed_node.is_remote() and not force_local:
        remote_dlg = RemoteFileDialog(wtree)
        remote_dlg.show(managed_node, title, fstype, init_folder, init_file,
                    parentwin, multi_select=multi_select)
        ret = remote_dlg.run()
        res = ret == gtk.RESPONSE_OK
        filename = remote_dlg.get_filename()
        if multi_select:
            file_list = []
            if filename and filename is not "":
                file_list = filename.split(",")
                return (res, file_list)
        else:
            return (res, filename)


    if fstype == "save":
        filesel = gtk.FileChooserDialog(title,
                                        action=gtk.FILE_CHOOSER_ACTION_SAVE,
                                        buttons=(gtk.STOCK_CANCEL,
                                                 gtk.RESPONSE_CANCEL,
                                                 gtk.STOCK_SAVE,
                                                 gtk.RESPONSE_OK),
                                        backend=back_end)
    elif fstype == "open":
        filesel = gtk.FileChooserDialog(title,
                                        action=gtk.FILE_CHOOSER_ACTION_OPEN,
                                        buttons=(gtk.STOCK_CANCEL,
                                                 gtk.RESPONSE_CANCEL,
                                                 gtk.STOCK_OPEN,
                                                 gtk.RESPONSE_OK),
                                        backend = back_end)
        if multi_select:
            filesel.set_select_multiple(True)
    elif fstype == "select_folder":
        filesel = gtk.FileChooserDialog(title,
                                        action=gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER,
                                        buttons=(gtk.STOCK_CANCEL,
                                                 gtk.RESPONSE_CANCEL,
                                                 gtk.STOCK_OPEN,
                                                 gtk.RESPONSE_OK),
                                        backend= back_end)

    # enable remote file browsing.    
    filesel.set_local_only(False)


    # if remote node, create a url and set it.
    if managed_node.is_remote() and not force_local:
        hostname = managed_node.hostname
        username = managed_node.username
        if username is None:
            username = os.getlogin() 
        url = "ssh://"+username+"@"+hostname+"/"
        title = filesel.get_title()
        filesel.set_title(title + ":" + username +"@"+ hostname)
        if init_folder is not None and init_folder != "":
            if init_folder[0] == '/':
                url = url + init_folder[1:]
                filesel.set_current_folder_uri(url)
        else:
            filesel.set_current_folder_uri(url)
    else: # local node
        if init_folder is not None and init_folder is not "":
            filesel.set_current_folder(init_folder)
            
    if init_file is not None:
        filesel.set_current_name(init_file)
    
    if parentwin:
        filesel.set_transient_for(parentwin)
        filesel.set_position(gtk.WIN_POS_CENTER_ON_PARENT)

    res = filesel.run() == gtk.RESPONSE_OK

    filelist = []
    if res:
        uris = filesel.get_uris()
        if managed_node.is_remote() and not force_local:
            selected_uri = uris[0]
            selected_uri = filesel.get_uri()
            selected_uri = urllib.unquote(selected_uri)
            if selected_uri and selected_uri.find(hostname) == -1:
                msg = "File not selected from the managed node.\n"
                msg += "Please use Open Location and use ssh://"+username+"@" +hostname +"/ to access remote files.\n"
                msg += "Open Location available through right click menu."
                showmsg(msg)
                filelist = []
                filesel.destroy()
                return (None, filelist)
            
        for u in uris:
            filelist.append(get_filename_from_uri(u))

        
    filesel.destroy()

    if multi_select:
        return (res, filelist)
    else:
        if len(filelist) > 0:
            return (res, filelist[0])
        else:
            return (res, None)

def get_filename_from_uri(uri):
    if uri == None:
        return None
    # Kludge : as urllib functions require http
    if uri.find("ssh") == 0:
        uri =  uri.replace("ssh", "http", 1)
    (prot, host, path,param,query,fragments ) = urlparse.urlparse(uri)
    if path is not None:
        return path


    
def checkAndSetDefaultPaths(wtree, managed_node):
    """ Check if the default path for snapshot and disk location are set."""
    disk_prop = managed_node.config.get(XMConfig.PATHS,
                                        constants.prop_disks_dir)
    snapshot_prop = managed_node.config.get(XMConfig.PATHS,
                                            constants.prop_snapshots_dir)
    if disk_prop is None or snapshot_prop is None:
        # give user opportunity to fix the properties.
        InitParamsDialog(wtree,managed_node).dialog.run()


# Cell renderers for the colums in the tree view.
def get_state_pixbuf(column, renderer, model, iter, appliance_store=None):
    """ Call back to display the state column image"""
    dom_name = model.get_value(iter,0)
    node_type = model.get_value(iter,1)
    managed_node =  model.get_value(iter,2)
    g_name = model.get_value(iter, 3)
    
    if node_type is None :
        return

    pb = unknown_pb

    if node_type == MANAGED_NODE:
        pb = node_pb
    elif node_type == IMAGE_STORE:
        pb = image_store_pb
    elif node_type == IMAGE:
        pb = image_pb
        if g_name:
            provider_id = g_name
            ppb = ApplianceLogos.get_provider_logo(appliance_store,
                                                   provider_id)
            if ppb:
                pb = ppb

    elif managed_node is not None and node_type == DOMAIN: # dom
        dom = managed_node.get_dom(dom_name)

        if dom and dom.is_resident:
            try:
                state = dom["state"]
            except xmlrpclib.Fault :
                state = None

            if state is not None and state[2] == 'p':
                pb = paused_pb
            else:
                pb = resident_pb
        else:
            pb = not_resident_pb
    elif node_type == SERVER_POOL:
        pb = pool_pb
    elif node_type == DATA_CENTER:
        pb = dc_pb


    renderer.set_property('pixbuf', pb)

def get_node_type_pb(type):
    if type == SERVER_POOL:
        return pool_pb
    elif type == MANAGED_NODE:
        return node_pb

### create a node list 

def populate_nodes(manager, view, model, dummy = True):
    """
    refresh the nodes.
    Not expected to be called other than constructor.

    """
    model.clear()

    # Append a Global Server Pool
    d_iter = model.append(None,
                          ["Data Center",
                           DATA_CENTER,
                           None,
                           None])

    # sort them
    names = manager.getNodeNames()
    name_list = []
    for name in names:
        name_list.append(name)
    name_list.sort()

    # Kludge, append a blank node so the expanders show up
    nodes = manager.getNodeList()
    for name in name_list:
        node = nodes[name]
        iter = \
             model.append(d_iter,
                          [node.hostname,
                           MANAGED_NODE,
                           node,
                           None])
        if dummy:
            append_dummy_row(model, iter)

    # add the server pools
    group_names = manager.getGroupNames()
    group_names.sort()

    groups = manager.getGroupList()
    for group_name in group_names:
        group = groups[group_name]
        g_iter = model.append(d_iter,
                              [group.name,
                               SERVER_POOL,
                               group,
                               None])
        
        # sort them
        names = group.getNodeNames()
        name_list = []
        for name in names:
            name_list.append(name)
        name_list.sort()

        # Kludge, append a blank node so the expanders show up
        nodes = group.getNodeList()
        for name in name_list:
            node = nodes[name]
            iter = \
                 model.append(g_iter,
                              [node.hostname,
                               MANAGED_NODE,
                               node,
                               group.name])
            if dummy:
                append_dummy_row(model, iter)


        # NOTE : if this is re-introduced, we need to change
        # UI code for adding and removing groups
        
        #if len(names) == 0:
        #    if dummy:
        #        append_dummy_row(model, g_iter, "Empty")
                
    view.expand_row((0,),False)
    view.set_cursor((0,))

def append_dummy_row( model, iter, msg = "VMs not found"):
    model.append(iter, [msg,
                        "DUMMY",None, None])


# moved to util to prevent cirular dependency.
class ApplianceLogos:
    logos = {}

    @classmethod
    def get_provider_logo(cls, appliance_store, provider_id):
        if not cls.logos.get(provider_id):
            cache_dir = os.path.join(appliance_store.get_cache_dir(),
                                     provider_id)
            if not os.path.exists(cache_dir):
                os.makedirs(cache_dir)
            for ext in (".gif", ".ico", ".png"):
                logo_path =  os.path.join(cache_dir,  provider_id + ext)
                try:
                    pb = gtk.gdk.pixbuf_new_from_file(logo_path)
                    if pb:
                        cls.logos[provider_id] = pb.scale_simple(16,16,
                                                                 gtk.gdk.INTERP_BILINEAR )
                        break
                except Exception, ex:
                    pass
                if not cls.logos.get(provider_id):
                    try:
                        # fetch the logo from the given url
                        logo_url = appliance_store.get_logo_url(provider_id)
                        p = urlparse.urlparse(logo_url)[2]
                        logo_name= os.path.basename(p)
                        frag = logo_name.split(".")
                        if len(frag) > 1:
                            ext = "." + frag[-1]
                        else:
                            ext = ".gif" # can do better with content-type :TBD
                        logo_file = os.path.join(cache_dir,provider_id + ext)
                        fetch_isp(logo_url,logo_file, "image")
                        pb = gtk.gdk.pixbuf_new_from_file(logo_file)
                        if pb:
                            cls.logos[provider_id] = pb.scale_simple(16,16,
                                                                     gtk.gdk.INTERP_BILINEAR )
                    except Exception, e:
                        print "Exception processing ", provider_id, e
                        traceback.print_exc()
                        pass
                    
                # set it to none so we do not keep doing scans
                if not cls.logos.get(provider_id):
                    cls.logos[provider_id] = appliance_pb
                    print "setting default for ", provider_id

        return cls.logos[provider_id]
                
                
    



### Module initialization
# Types of nodes in the tree
SERVER_POOL = "SERVER_POOL"
MANAGED_NODE = "MANAGED_NODE"
DOMAIN  = "DOMAIN"
DATA_CENTER= "DATA_CENTER"

IMAGE_STORE = "IMAGE_STORE"
IMAGE = "IMAGE"

for path in (os.path.dirname(sys.argv[0]),'.', '/usr/share/convirt', '.'):
    filename = path + '/convirt.glade'
    if os.path.exists(filename):
        gladefile = filename
        break
else:
    print "ERROR: Couldn't find glade interface definition file!"
    sys.exit(1)   # bad, but ok for now.



paused_pb   = gtk.gdk.pixbuf_new_from_file(path +\
                                           "/pixmaps/small_pause.png")
resident_pb = gtk.gdk.pixbuf_new_from_file(path + \
                                           "/pixmaps/small_started_state.png")
not_resident_pb = gtk.gdk.pixbuf_new_from_file(path + \
                                               "/pixmaps/small_shutdown.png")

node_pb = gtk.gdk.pixbuf_new_from_file(path + \
                                       "/pixmaps/small_node.png")

dc_pb = gtk.gdk.pixbuf_new_from_file(path + \
                                     "/pixmaps/small_pool.png")
pool_pb = gtk.gdk.pixbuf_new_from_file(path + \
                                       "/pixmaps/group.png")

unknown_pb = gtk.gdk.pixbuf_new_from_file(path + \
                                          "/pixmaps/small_unknown_state.png")

image_store_pb = gtk.gdk.pixbuf_new_from_file(path + \
                                          "/pixmaps/small_image_store.png")

image_pb = gtk.gdk.pixbuf_new_from_file(path + \
                                        "/pixmaps/small_image.png")

connected_pb = gtk.gdk.pixbuf_new_from_file(path + \
                                        "/pixmaps/small_connect_blue.png")
disconnected_pb = gtk.gdk.pixbuf_new_from_file(path + \
                                        "/pixmaps/small_disconnect_yellow.png")

appliance_pb = gtk.gdk.pixbuf_new_from_file(path + \
                                        "/pixmaps/small_appliance.png")

convirt_pb = gtk.gdk.pixbuf_new_from_file(path + \
                                         "/pixmaps/convirt.png")

## error_img = gtk.Image()
## error_img.set_from_stock(gtk.STOCK_DIALOG_ERROR,
##                          gtk.ICON_SIZE_MENU)
## error_pb = error_img.get_pixbuf()

## warn_img = gtk.Image()
## warn_img.set_from_stock(gtk.STOCK_DIALOG_WARNING,
##                         gtk.ICON_SIZE_MENU)

## warn_pb = warn_img.get_pixbuf()

warn_pb = connected_pb
error_pb = disconnected_pb

gtk.window_set_default_icon(convirt_pb)

wtree = gtk.glade.XML(gladefile)

# KLUDGE : way to share global context
main_context = {}

file_editor =  FileViewEditDialog(wtree)
####
if __name__ == '__main__':
##     helper = CredentialsHelperDialog(wtree)
##     cred = helper.get_credentials("foobar", "root")
##     if cred:
##         print cred.get_username(), cred.get_password()
    dlg = MigrationChecksResultsDlg()
    err_list = (("Mem", "Test"), ("CPU", "TEST"))
    warn_list = (("HVM", "Warn Test"), ("Platform", "Warn TEST 1 sdslfjsdlfjsdjfs sdsdfsd sdfvdsfsdfs \n Second Line"))
    ret = dlg.show(err_list, warn_list)
    print ret
