#!/usr/bin/env python
#
#   ConVirt   -  Copyright (c) 2008 Jd & Hap Hazard
#   ======
#
# ConVirt is a Xen management tool with a GTK based graphical interface
# that allows for performing the standard set of domain operations
# (start, stop, pause, kill, shutdown, reboot, snapshot, etc...). It
# also attempts to simplify certain aspects such as the creation of
# domains, as well as making the consoles available directly within the
# tool's user interface.
#
#
# This software is subject to the GNU Lesser General Public License (LGPL)
# and for details, please consult it at:
#
#    http://www.fsf.org/licensing/licenses/lgpl.txt
#
# author : Jd <jd_jedi@users.sourceforge.net>
#


# All classes in thse files are required Interfaces for managing
# machines/host/nodes with virtualization technology.

import sys,os,re,types, traceback

from ManagedNode import ManagedNode

from utils import XMConfig, constants, search_tree,Poller
from KVMDomain import *
from KVMProxy import KVMProxy

from phelper import PHelper
from phelper import HostValidationException,\
     AuthenticationException, CommunicationException

from VM import *



class KVMNode(ManagedNode):
    """
    Interface that represents a node being managed.It defines useful APIs
    for clients to be able to do Management for a virtualized Node
    """                

    def __init__(self,
                 hostname = None,
                 username= Node.DEFAULT_USER,
                 password=None,
                 isRemote=False,
                 protocol = "ssh",
                 ssh_port = 22,
                 helper = None):

        ManagedNode.__init__(self, hostname,
                             ssh_port,
                             username, password,
                             isRemote, 
                             helper)
        self._node_info = None
        self._vmm = None
        self.dom_list = DomListHelper(self)
        self._dom0     = None
        self._managed_domfiles = None
        self.auto = '/etc/kvm/auto'

        self.metrics_helper = MetricsHelper(self)
        self.metrics_poller = None
        self.POLLING_INTERVAL = 5.0 # default no.of seconds between each metric fetch
        self.MAX_POLLS = 4 # default number of times metrics poller iterates

        self.protocol = protocol
        self.ssh_port = ssh_port

    # Try to get more information from the KVM node if possible
    def _init_node_info(self):
        if self._node_info is None:
            self._node_info = {}
        return self._node_info
    
    def _init_managed_domfiles(self):
        if self._managed_domfiles is None:
            mdf = self.config.get(XMConfig.APP_DATA,
                                  constants.prop_domfiles,
                                  )            
            if mdf is None:
                self._managed_domfiles = []
            else:
                self._managed_domfiles = eval(mdf)        
        return self._managed_domfiles

    def _init_vmm(self):
        if self._vmm is not None:
            return self._vmm

        self._vmm = KVMProxy(self.node_proxy)
        return self._vmm
    
                
    # implement lazy initialization. 
    def __getattr__(self, name):
        if name == 'node_info':
            return self._init_node_info()
        if name == 'vmm':
            return self._init_vmm()
        if name == 'managed_domfiles':
            return self._init_managed_domfiles()
        else:
            return ManagedNode.__getattr__(self,name)


    # encapsulate the host and dom0 information here.        
    def __getitem__(self, param):
        val = self.node_info.get(param)
        if val is not None:
            return val
        else:
            return ManagedNode.__getitem__(self,param)


    # public methods
    def get_dom_names(self):
        """
        return lists containing names of vm running on this node.
        exceptions :
        """
        return self.dom_list.iterkeys()


    def get_doms(self):
        """
        returns list containing information about nodes running on
        this node.
        returns list of Dom objects
        exceptions :
        NOTE: in most cases, get_dom_names() is a better option.
        """
        return self.dom_list

    def get_dom(self, name):
        return self.dom_list[name]

    
    def isDom0(self, name):
        if self.get_dom(name):
            return self.get_dom(name).isDom0()
        else:
            return False
    
    def isDomU(self, name):
        if self.get_dom(name):
            return self.get_dom(name).isDomU()
        else:
            return False
    
    def get_state(self, name):
        if self.get_dom(name):
            return self.get_dom(name)._state()
        else:
            return None

    def isResident(self, name):
        if self.get_dom(name):
            return self.get_dom(name).is_resident
        else:
            return False
    

    def create_dom(self, config):
       """
       create a new dom given a particular config.
       exceptions:
       """
       if config.filename is None:
           raise Exception("filename must be set in the config.")
       config.write()
       dom_name = self.add_dom_config(config.filename)
       self.start_dom(dom_name)
       self.refresh()
       return dom_name

    def create_dom_from_file(self, filename):
        dom_name = self.add_dom_config(filename)
        self.start_dom(dom_name)
        self.refresh()
        return dom_name

    def refresh(self):
        self.dom_list.refresh()
    
    # Manage external files.
    def add_dom_config(self, filename):
        return self.dom_list.add_dom_config(filename)

    def remove_dom_config(self, filename):
        return self.dom_list.remove_dom_config(filename)

    # Metrics
    def get_metrics(self, refresh=False):
        if refresh:
            self.metrics = self.metrics_helper.getFrame()
        else:
            if self.metrics is None:
                # first time get_metrics has been called
                self.get_metrics(refresh=True)

            if self.metrics_poller is None or not self.metrics_poller.isAlive():
                # kick off ascynchronous metrics polling
                # ... MAX_POLLS polls at POLLING_INTERVAL second intervals
                self.metrics_poller = Poller(self.POLLING_INTERVAL,self.get_metrics,
                                             args=[True],max_polls=self.MAX_POLLS)
                self.metrics_poller.start()
                
        return self.metrics


    # dom operations
    
    def start_dom(self, name):
       """
       start the given dom 
       exceptions:
       """
       # deligate to the dom itself
       self.get_dom(name)._start()
       

    def pause_dom(self, name):
       """
       pause a running dom
       exceptions:
       """
       self.get_dom(name)._pause()
       

    def resume_dom(self, name):
        """
        pause a running dom
        exceptions:
        """
        self.get_dom(name)._resume()
       

    def shutdown_dom(self, name):
       """
       shutdown a running dom. 
       """
       self.get_dom(name)._shutdown()
       

    def destroy_dom(self, name):
       """
       destroy dom
       """
       self.get_dom(name)._destroy()
       self.dom_list.refresh()
       

    def reboot_dom(self, name):
       """
       reboot dom
       """
       self.get_dom(name)._reboot()
       
    def restore_dom(self, filename):
        """
        restore from snapshot file
        """
        self.vmm.restore_vm(filename)
        self.dom_list.refresh()

    def get_console(self, name):
       """
       get the console for the dom
       API migght need revision...
       """
       pass

    def get_terminal(self, dom, username, password):
       """
       return tty terminal (telnet, ssh session ?)
       """
       pass

    def get_vnc(self,dom):
       """
       get VNC session for this dom. VNC would popup username/password.
       """
       pass

    
       

class DomListHelper:
    """
    Class represent list of dom being tracked by this managed
    node. 
    """
    # for implementing domlist

    def __init__(self, node):
        self._dom_dict = None
        self.node = node

    def _init_dom_list(self):
        """ take the dominfo from the API and return list of doms
        """
        if self._dom_dict is None:
            self.refresh()
        return self._dom_dict

    def __getattr__(self, name):
        if name == 'dom_dict': 
            return self._init_dom_list()

    # TODO : add synchronization ?
    def refresh(self):
        current_dict = {}
        # get the current running doms
        dom_list_info = self.node.vmm.get_domains()

        for dom_info in dom_list_info.itervalues():
            dom = KVMDomain(self.node,dom_info)
            current_dict[dom.name] = dom

        # now lets get /etc/kvm/auto files and associate with
        # running doms, also add the doms for managed files.

        files = [] 
        if self.node.node_proxy.file_exists(self.node.auto):
            files = self.node.node_proxy.listdir(self.node.auto)

        print "FILES : ", files

        for filename in files + \
                self.node.managed_domfiles:

            try: 
                if filename.find('/') >= 0:
                    if self.node.node_proxy.file_exists(filename):
                        new_dom = KVMDomain(self.node, filename)                    
                    else:
                        continue
                else:
                    filename = "%s/%s" % (self.node.auto, filename)
                    if self.node.node_proxy.file_exists(filename):
                        new_dom = KVMDomain(self.node, filename)
                    else:
                        continue
            except (Exception, StandardError), e:
                traceback.print_exc()
                print "Domain File %s is invalid. Reason: %s. Removing from list"\
                      % (filename, str(e))
                self.remove_dom_config(filename)
                continue

            dom_name = new_dom.name
            if current_dict.has_key(dom_name):
                current_dict[dom_name].set_config(new_dom.get_config())
            else:
                current_dict[dom_name] = new_dom                   

        self._dom_dict = current_dict

    def __getitem__(self, item):
        if not item: return None
        if type(item) is int:
            for name, dom in self.dom_dict.iteritems():
                if dom.is_resident and dom.id == item:
                    return dom
        else:
            if self.dom_dict.has_key(item):
                return self.dom_dict[item]
            else:
                return None


    def __iter__(self):
        return self.dom_dict.itervalues()


    def iterkeys(self):
        return self.dom_dict.keys()


    # start tracking file
    def add_dom_config(self, filename):
        if filename in self.node.managed_domfiles:
            return KVMConfig(self.node,filename)['name']
        else:
            new_dom = KVMDomain(self.node, filename)
            self.node.managed_domfiles.append(filename)
            self.node.config.set(XMConfig.APP_DATA,constants.prop_domfiles,
                                          repr(self.node.managed_domfiles),
                                          )
            self.dom_dict[new_dom.name] = new_dom

            return new_dom.name

    def remove_dom_config(self, filename):

        if filename in self.node.managed_domfiles:
            self.node.managed_domfiles.remove(filename)
            self.node.config.set(XMConfig.APP_DATA,constants.prop_domfiles,
                                          repr(self.node.managed_domfiles),
                                          )
            # check if running, shutdown, remove, etc.
            for d in self.dom_dict.itervalues():
                if d.get_config() is not None and \
                       d.get_config().filename == filename:
                    del self.dom_dict[d.name]
                    return True

        return False




class MetricsHelper:
    """A Helper to fetch and format runtime metrics from a KVM Host"""

    FRAME_CMD = 'top -b -i 2 -d 1'
    
    def __init__(self, node):
        self.node = node

    def getFrame(self):
        """returns a dictionary containing metrics for all running domains
        in a frame"""

        # construct the frame: Map
        # Top level / node level metrics : frame[toplevel] = value
        # followed by each domain
        #   frame [domain_name] = metrics map

        #
        frame = {} # the output metric frame (dict of dict's)
        (retbuf, retcode) = self.node.node_proxy.exec_cmd(self.FRAME_CMD,
                                                          self.node.exec_path)
        if retcode: return None

        # process and fill frame buffer

        
        return frame
        
    
        

        
            



# Test code
if __name__ == "__main__":
    test_domu = "test"
    host = "localhost"
    dom_file = '/etc/kvm/test'
    dom_2b_started = 'test'
    dom_2b_shutdown = 'test'

    username = 'root'
    print "Enter password "
    passwd = sys.stdin.readline().strip()

    
    # basic connectivity
    remote = True
    if not remote:
        host = "localhost"
    else:
        host = '192.168.12.101'
        test_domu = "test"
        dom_file = '/etc/kvm/test'
        dom_2b_started = 'test'
        dom_2b_shutdown = 'test'
        
    managed_node = KVMNode(hostname=host,
                           username = username,
                           password = passwd,
                           isRemote=remote)

    ## list doms, knows and running
    names = managed_node.get_dom_names()

    
    for dom_name in names:
        print dom_name, managed_node.get_state(dom_name)

#    managed_node.start_dom("ubuntu_vm")

    for dom_name in names:
        print dom_name, managed_node.get_state(dom_name)

    ## create/destroy dom
    
    

    
    
