#!/usr/bin/env python
#
#   ConVirt   -  Copyright (c) 2008 Jd & Hap Hazard
#   ======
#
# ConVirt is a Xen management tool with a GTK based graphical interface
# that allows for performing the standard set of domain operations
# (start, stop, pause, kill, shutdown, reboot, snapshot, etc...). It
# also attempts to simplify certain aspects such as the creation of
# domains, as well as making the consoles available directly within the
# tool's user interface.
#
#
# This software is subject to the GNU Lesser General Public License (LGPL)
# and for details, please consult it at:
#
#    http://www.fsf.org/licensing/licenses/lgpl.txt
#
# author : Jd <jd_jedi@users.sourceforge.net>
#


# KVM Domain

import sys,os,re,types


from utils import search_tree, XMConfig, PyConfig
from NodeProxy import Node
from constants import *
from VM import *

class KVMDomain(VM):
    
    def __init__(self, node,context):
        """
        initialize using dom_info.
        a. can be used to create a new dom
        b. can be used to instantiate dom representation of a runnig dom.
        """
        self.id = -1
        self._config   = None
        self._dom_info = None
        self._stats    = None
        self.node = node
        
        if type(context) == types.StringType:
            filename = context
            self._config = KVMConfig(node,filename)
            self.name = self._config.name
            self.is_resident = False
        
        else:
            # assume dom_info
            dom_info = context
            self._dom_info = dom_info
            self.name = self._dom_info.get('name')
            self.id = self._dom_info.get('domid')
            self.is_resident = True
            

        self.state = self._state()

    @classmethod
    def get_state(cls,state_str):
        if state_str == None or len(state_str) < 5:
            return VM.UNKNOWN
        
        if state_str[0] == "r":
            return VM.RUNNING
        elif state_str[1] == "b":
                return VM.BLOCKED
        elif state_str[2] == 'p':
            return VM.PAUSED
        elif state_str[3] == "s":
            return VM.SHUTDOWN
        elif state_str[4] == "c":
            return VM.CRASHED
        return VM.UNKNOWN
        

    def _state(self):
        """
        Translate the string in to state enum
        returns one of the following
        VM.RUNNING, VM.BLOCKED, VM.PAUSED,
        VM.SHUTDOWN, VM.CRASHED, VM.UNKNOWN.
        VM.NOT_STARTED 
        """
        if self._dom_info == None:
            return VM.NOT_STARTED

        # later : when more states available. particularly stopped/paused
        # put request on the kvm list.
        return self.get_state('r')
    
    def isDom0(self):
        """Test wether the dom is dom0"""

        return false

    def isDomU(self):
        """Test wether the dom is a guest dom"""

        return True

    
    def __getitem__(self, param):
        if param == "name":
            return self.name
        else:
            return search_tree(self._dom_info, param)

    # refresh the dom
    def refresh(self):
        if self.name is None:
            if self._config is not None:
                name = self._conifig.name
            else:
                raise Exception("Invalid domain state, no Name found")

        # try to get the dom information from xen
        try:
            self._dom_info= self.node.vmm.get_dom_info(self.name)
            self.name = self._dom_info.get('name')
            self.id   = self._dom_info.get('id')
            self.is_resident = True
        except VMOperationException, ex:
            self._dom_info = None
            self.is_resident = False

        self.state = self._state()
            
    # operation on doms.
    def _save(self, filename):
        """Save a snapshot of the dom"""

        self.node.vmm.xend.domain.save(self.id, filename)

    def _execShutdownCmd(self, cmd):
        """Execute the requested shutdown command on the dom"""
        self.node.vmm.xend.domain.shutdown(self.id, cmd)
        self._dom_info = None
        self.is_resident = False

    def _reboot(self):
        """Reboot the dom"""

        self._execShutdownCmd("reboot")

    def _shutdown(self):
        """Shutdown the dom"""

        self._execShutdownCmd("halt")

    def _destroy(self):
        """Destroy the dom, hence removing it from resident running doms
        without any attempt at shutting anything down on the guest OS"""

        self.node.vmm.xend.domain.destroy(self.id)
        self._dom_info = None
        self.is_resident = False

    def _pause(self):
        """Pause the dom"""

        self.node.vmm.xend.domain.pause(self.id)
        
    def _resume(self):
        """Resume the paused dom"""

        self.node.vmm.xend.domain.unpause(self.id)

    ## start
    def _start(self):
        self._start_dom()


    ## Changing properties of running dom
    def setMem(self, value):
        """Change the running memory setting"""
        
        self.node.vmm.xend.domain.setMemoryTarget(self.id, value)
        
    def setVCPUs(self, value):
        """Change the running number of CPUs setting"""

        self.node.vmm.xend.domain.setVCpuCount(self.id, value)


    ## config related intefaces
    def get_config(self):
        return self._config

    def set_config(self, config):
        self._config = config
    
    def _start_dom(self):
        if self._config is None:
            raise Exception("Configuration not set. Can not start domain")
        (output, exit_code) = self.node.vmm.start(self._config)
        if not exit_code:
            self.refresh()            
        else:
            raise Exception("Domain could not be started: " + output)
        
    ## get stats
    def get_snapshot(self):
        if self._stats == None:
            self._stats = KVMStats(self)
        return self._stats.get_snapshot()
        


# For now, we can implement the Xen specific stuff. Later we should
# drive it through metadata. (what attributes supported in file and
# running doms,nested attribute support) [May be do it now?]
#
class KVMConfig(VMConfig):
    """
    represnts startup config object (information in the conf file)
    """


    # DomConfig follows
    def __init__(self, node, filename = None):
        """
        read stuff from file and populate the config
        when filename is None, creates an empty config
        """
        VMConfig.__init__(self, node,filename)
        

    # kvm specific validation.
    def validate(self):
        """Attempts to validate that the settings are not going to lead to
        any errors when the dom is started, and returns a list of the
        errors as strings"""

        result = []

        if not self["name"]:
            result.append("Missing domain name.")

        if not self["disk"]:
            result.append("Missing disk specification.")

        # TODO : Add some more validations.

        # compare mem,cpus set to available on the node

        #
        return result


class KVMStats(VMStats):
    """
    represents statatistics/measurements for a vm. (CPU, I/O etc)
    This is abstracted out so we can cut over to some other source for
    runtime statastics/measurements
    """

    def __init__(self, dom):
        """
        constructor, dom for which the stats are to be obtained.
        """
        self.dom = dom
        self.stat = {}

    def get_snapshot(self):
        # get it from /proc/pid
        #for stat in ("memory", "cpu_time"):
        #   self.stat[stat] = ...
        return self.stat
    
    
### module initialization

    







    
    

    
    
