/*
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * Give credit where credit is due, keep the authors message below.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Authors:
 *   - Diogo Ferreira (playerX) <defer@underdev.org>
 *
 * This wouldn't have been possible without:
 *   - Ideas from the compiz community (mainly throughnothing's)
 *   - David Reveman's work
 *
 * */

#include <stdlib.h>
#include <string.h>

#include <compiz.h>

#include <math.h>

#define WIN_X(w) ((w)->attrib.x - (w)->input.left)
#define WIN_Y(w) ((w)->attrib.y - (w)->input.top)
#define WIN_W(w) ((w)->width + (w)->input.left + (w)->input.right)
#define WIN_H(w) ((w)->height + (w)->input.top + (w)->input.bottom)

#define SD_INITIATE_KEY_DEFAULT         "F7"
#define SD_INITIATE_MODIFIERS_DEFAULT   0

#define SD_DISPLAY_OPTION_INITIATE  0
#define SD_DISPLAY_OPTION_NUM       1

#define SD_SCREEN_OPTION_SPEED              0
#define SD_SCREEN_OPTION_TIMESTEP           1
#define SD_SCREEN_OPTION_DIRECTION          2
#define SD_SCREEN_OPTION_WINDOW_TYPE        3
#define SD_SCREEN_OPTION_USE_SCALE_SETTINGS 4
#define SD_SCREEN_OPTION_WINDOW_OPACITY     5
#define SD_SCREEN_OPTION_NUM                6


#define SD_DIRECTION_UP         0
#define SD_DIRECTION_DOWN       1
#define SD_DIRECTION_LEFT       2
#define SD_DIRECTION_RIGHT      3
#define SD_DIRECTION_UP_DOWN    4
#define SD_DIRECTION_LEFT_RIGHT 5
#define SD_DIRECTION_CORNERS    6


#define SD_DIRECTION_DEFAULT    SD_DIRECTION_UP_DOWN
#define SD_DIRECTION_MIN        0
#define SD_DIRECTION_MAX        5

#define SD_SPEED_DEFAULT        1.2f
#define SD_SPEED_MIN            0.1f
#define SD_SPEED_MAX            50.0f
#define SD_SPEED_PRECISION      0.1f

#define SD_TIMESTEP_DEFAULT   0.1f
#define SD_TIMESTEP_MIN       0.1f
#define SD_TIMESTEP_MAX       50.0f
#define SD_TIMESTEP_PRECISION 0.1f

#define SD_USE_SCALE_SETTINGS_DEFAULT FALSE

#define SD_WINDOW_OPACITY_DEFAULT   0.3
#define SD_WINDOW_OPACITY_MIN       0.1
#define SD_WINDOW_OPACITY_MAX       1.0
#define SD_WINDOW_OPACITY_PRECISION 0.01

#define SD_STATE_OFF          0
#define SD_STATE_ACTIVATING   1
#define SD_STATE_ON           2
#define SD_STATE_DEACTIVATING 3

#define NUM_OPTIONS(s) (sizeof ((s)->opt) / sizeof (CompOption))

/* window types */
static char *winType[] = {
    "Toolbar",
    "Utility",
    "Dialog",
    "ModalDialog",
    "Fullscreen",
    "Normal"
};
#define N_WIN_TYPE (sizeof (winType) / sizeof (winType[0]))

/* necessary plugin structs */
typedef struct _ShowdesktopPlacer {
    int onScreenX, onScreenY;
    int offScreenX, offScreenY;
} ShowdesktopPlacer;
typedef struct _ShowdesktopDisplay {
    int             screenPrivateIndex;
    CompOption opt[SD_DISPLAY_OPTION_NUM];

    HandleEventProc handleEvent;
} ShowdesktopDisplay;
typedef struct _ShowdesktopScreen {
    int                 windowPrivateIndex;

    PreparePaintScreenProc       preparePaintScreen;
    DonePaintScreenProc          donePaintScreen;
    SetScreenOptionForPluginProc setScreenOptionForPlugin;

    CompOption opt[SD_SCREEN_OPTION_NUM];

    ShowdesktopPlacer *placers;
    int               placersSize;
    int               nPlacers;

    int state;
    int moreAdjust;

    float speed;
    float timestep;
    float windowOpacity;

    int   direction;
    int   wMask;

    Bool ignoreNextTerminateEvent;
} ShowdesktopScreen;
typedef struct _ShowdesktopWindow {
    int sid;
    int distance;

    ShowdesktopPlacer *placer;

    GLfloat xVelocity, yVelocity;
    GLfloat tx, ty;
    GLfloat oldOpacity;

    float delta;
    Bool adjust;
} ShowdesktopWindow;

/* shortcut macros, usually named X_DISPLAY, X_SCREEN and X_WINDOW
 * these might seem overly complicated but they are shortcuts so we don't have to access the privates arrays all the time
 * */
#define GET_SHOWDESKTOP_DISPLAY(d)				     \
    ((ShowdesktopDisplay *) (d)->privates[displayPrivateIndex].ptr)

#define SD_DISPLAY(d)			   \
    ShowdesktopDisplay *sd = GET_SHOWDESKTOP_DISPLAY (d)

#define GET_SHOWDESKTOP_SCREEN(s, fd)					 \
    ((ShowdesktopScreen *) (s)->privates[(fd)->screenPrivateIndex].ptr)

#define SD_SCREEN(s)							\
    ShowdesktopScreen *ss = GET_SHOWDESKTOP_SCREEN (s, GET_SHOWDESKTOP_DISPLAY (s->display))

#define GET_SHOWDESKTOP_WINDOW(w, ss)					  \
    ((ShowdesktopWindow *) (w)->privates[(ss)->windowPrivateIndex].ptr)

#define SD_WINDOW(w)					       \
    ShowdesktopWindow *sw = GET_SHOWDESKTOP_WINDOW  (w,		       \
		            GET_SHOWDESKTOP_SCREEN  (w->screen,	       \
		            GET_SHOWDESKTOP_DISPLAY (w->screen->display)))
/* plugin private index */
static int displayPrivateIndex;


/* non interfacing code, aka the logic of the plugin */
static Bool
isSDWin (CompWindow *w)
{
    SD_SCREEN (w->screen);

    if (!(*w->screen->focusWindow) (w))
        return FALSE;

    if (!(ss->wMask & w->type))
        return FALSE;

    if (w->state & CompWindowStateSkipPagerMask)
        return FALSE;

    return TRUE;
}

static Bool
showdesktopTerminate (CompDisplay      *d,
               CompAction       *action,
               CompActionState  state,
               CompOption       *option,
               int              nOption)
{
    CompScreen *s;
    Window xid;

    xid = getIntOptionNamed (option, nOption, "root", 0);
    s = findScreenAtDisplay (d, xid);

    if (s)
    {
        SD_SCREEN (s);

        if (ss->state == SD_STATE_ON || ss->state == SD_STATE_ACTIVATING)
        {
            CompWindow *w;

            for (w = s->windows; w; w = w->next)
            {
                SD_WINDOW (w);
                if (sw->placer) {
                    sw->adjust = TRUE;
                    sw->xVelocity = sw->yVelocity = 0.0f;
                    w->paint.opacity = sw->oldOpacity;
                }
            }
            ss->state = SD_STATE_DEACTIVATING;
        }
    }
    return FALSE;
}

#define SD_PART_SIZE 20

static void
repositionSDPlacer (ShowdesktopPlacer *sp, CompWindow *w, int oldState)
{
    SD_SCREEN (w->screen);

    if (oldState == SD_STATE_OFF) {
        sp->onScreenX  = w->attrib.x;
        sp->onScreenY  = w->attrib.y;
    }

    switch (ss->direction)
    {
    case SD_DIRECTION_UP:
        sp->offScreenX = w->attrib.x;
        sp->offScreenY = w->screen->workArea.y - w->height + SD_PART_SIZE;
        break;
    case SD_DIRECTION_DOWN:
        sp->offScreenX = w->attrib.x;
        sp->offScreenY = w->screen->workArea.height - SD_PART_SIZE;
        break;
    case SD_DIRECTION_LEFT:
        sp->offScreenX = w->screen->workArea.x - w->width + SD_PART_SIZE;
        sp->offScreenY = w->attrib.y;
        break;
    case SD_DIRECTION_RIGHT:
        sp->offScreenX = w->screen->workArea.width - SD_PART_SIZE;
        sp->offScreenY = w->attrib.y;
        break;
    case SD_DIRECTION_UP_DOWN:
        sp->offScreenX = w->attrib.x;
        sp->offScreenY = w->screen->workArea.y - w->height + SD_PART_SIZE;
        if (w->attrib.y + (WIN_H (w)/2) > (w->screen->height/2))
            sp->offScreenY = w->screen->workArea.height - SD_PART_SIZE;
        break;
    case SD_DIRECTION_LEFT_RIGHT:
        sp->offScreenY = w->attrib.y;
        sp->offScreenX = w->screen->workArea.width - SD_PART_SIZE;
        if (w->attrib.x + (WIN_W (w)/2) < (w->screen->width + w->screen->x)/2)
            sp->offScreenX = w->screen->workArea.x - w->width + SD_PART_SIZE;
        break;
/*    case SD_DIRECTION_CORNERS:

        if ( (w->attrib.x + WIN_W(w)/2) < (w->screen->width + w->screen->x)/2)
        {
            sp->offScreenX = (-1) * w->attrib.width;
            if ( (w->attrib.y + WIN_H(w)/2) > (w->screen->height/2) )
                sp->offScreenY = w->screen->height + w->height;
            else
                sp->offScreenY = -1 * w->screen->height - w->height;
        }
        else
        {
            sp->offScreenX = 2 * w->attrib.width;
            if ( (w->attrib.y + WIN_H(w)/2) > (w->screen->height/2) )
                sp->offScreenY = w->screen->height;
            else
                sp->offScreenY = -1 * w->screen->height;
        }
        break;*/
    }
}

static Bool
prepareSDWindows (CompScreen *s, int oldState)
{
    CompWindow *w;
    CompWindow *desktopWindow;

    SD_SCREEN (s);

    ss->nPlacers = 0;
    desktopWindow = 0;

    for (w = s->windows; w; w = w->next)
    {
        SD_WINDOW (w);

        if (getWindowType (s->display, w->id) == CompWindowTypeDesktopMask)
            desktopWindow = w;

        if (!isSDWin (w))
            continue;

        if (sw->placer)
        {
            sw->placer = 0;
            sw->tx = sw->ty = 0;
            syncWindowPosition (w);
            sw->placer = 0;
        }

        if (ss->placersSize <=  ss->nPlacers)
        {
            ss->placers = realloc (ss->placers,
                                   sizeof (CompWindow *) * (ss->nPlacers + 32));
            if (!ss->placers)
                return FALSE;

            ss->placersSize = ss->nPlacers + 32;
        }

        sw->placer = &ss->placers[ss->nPlacers];
        repositionSDPlacer (&ss->placers[ss->nPlacers++], w, oldState);
        sw->tx = sw->ty = sw->xVelocity = sw->yVelocity = 0.0f;
        sw->adjust = TRUE;

        sw->oldOpacity   = w->paint.opacity;
        w->paint.opacity = ss->windowOpacity * OPAQUE;
    }

    if (desktopWindow)
        activateWindow (desktopWindow);

    return TRUE;
}

static Bool
showdesktopInitiate (CompDisplay      *d,
               CompAction       *action,
               CompActionState  state,
               CompOption       *option,
               int              nOption)
{
    CompScreen *s;
    Window xid;

    xid = getIntOptionNamed (option, nOption, "root", 0);
    s = findScreenAtDisplay (d, xid);

    if (s && otherScreenGrabExist (s, 0) == FALSE)
    {
        /*SD_DISPLAY (s->display);*/
        SD_SCREEN (s);

        if (ss->state == SD_STATE_OFF || ss->state == SD_STATE_DEACTIVATING)
        {
            if (prepareSDWindows (s, ss->state)) {
                XSetInputFocus (d->display, d->screens->root, RevertToPointerRoot,
                        CurrentTime);
                ss->state = SD_STATE_ACTIVATING;
            }

            if (state & CompActionStateInitButton)
                action->state |= CompActionStateTermButton;

            if (state & CompActionStateInitKey)
                action->state |= CompActionStateTermKey;
        }
        else
        {
            return showdesktopTerminate (d, action, state, option, nOption);
        }
    }

    return FALSE;
}

/* gconf entries */
static void
showdesktopDisplayInitOptions (ShowdesktopDisplay *sd, Display *display)
{
    CompOption *o;

    o = &sd->opt[SD_DISPLAY_OPTION_INITIATE];
    o->name                             = "initiate";
    o->shortDesc                        = "Initiate showdesktop mode";
    o->longDesc                         = "Initiate showdesktop mode";
    o->type                             = CompOptionTypeAction;
    o->value.action.initiate            = showdesktopInitiate;
    o->value.action.terminate           = showdesktopInitiate;
    o->value.action.bell                = FALSE;
    o->value.action.edgeMask            = (1 << SCREEN_EDGE_BOTTOMRIGHT);
    o->value.action.state               = CompActionStateInitEdge;
    o->value.action.type                = CompBindingTypeKey;
    o->value.action.state              |= CompActionStateInitKey;
    o->value.action.key.modifiers       = SD_INITIATE_MODIFIERS_DEFAULT;
    o->value.action.key.keycode         = 
        XKeysymToKeycode (display,
                          XStringToKeysym (SD_INITIATE_KEY_DEFAULT));
}

static void
showdesktopScreenInitOptions (ShowdesktopScreen *ss)
{
    CompOption *o;
    int        i;

    o = &ss->opt[SD_SCREEN_OPTION_SPEED];
    o->name               = "speed";
    o->shortDesc          = "Speed";
    o->longDesc           = "Window speed";
    o->type               = CompOptionTypeFloat;
    o->value.f            = SD_SPEED_DEFAULT;
    o->rest.f.min         = SD_SPEED_MIN;
    o->rest.f.max         = SD_SPEED_MAX;
    o->rest.f.precision   = SD_SPEED_PRECISION;

    o = &ss->opt[SD_SCREEN_OPTION_TIMESTEP];
    o->name               = "timestep";
    o->shortDesc          = "Timestep";
    o->longDesc           = "Showdesktop timestep";
    o->type               = CompOptionTypeFloat;
    o->value.f            = SD_TIMESTEP_DEFAULT;
    o->rest.f.min         = SD_TIMESTEP_MIN;
    o->rest.f.max         = SD_TIMESTEP_MAX;
    o->rest.f.precision   = SD_TIMESTEP_PRECISION;

    o = &ss->opt[SD_SCREEN_OPTION_DIRECTION];
    o->name             = "direction";
    o->shortDesc        = "Window direction";
    o->longDesc         = "0 - Up, 1 - Down, 2 - Left, 3 - Right, 4 - Up/Down, 5 - Left/Right, 6 - Corners";
    o->type             = CompOptionTypeInt;
    o->value.i          = SD_DIRECTION_DEFAULT;
    o->rest.i.min       = SD_DIRECTION_MIN;
    o->rest.i.max       = SD_DIRECTION_MAX;

    o = &ss->opt[SD_SCREEN_OPTION_WINDOW_TYPE];
    o->name	         = "window_types";
    o->shortDesc         = "Window Types";
    o->longDesc	         = "Window types that should go away in showdesktop mode";
    o->type	         = CompOptionTypeList;
    o->value.list.type   = CompOptionTypeString;
    o->value.list.nValue = N_WIN_TYPE;
    o->value.list.value  = malloc (sizeof (CompOptionValue) * N_WIN_TYPE);
    for (i = 0; i < N_WIN_TYPE; i++)
	o->value.list.value[i].s = strdup (winType[i]);
    o->rest.s.string     = windowTypeString;
    o->rest.s.nString    = nWindowTypeString;
    ss->wMask = compWindowTypeMaskFromStringList (&o->value);

    o = &ss->opt[SD_SCREEN_OPTION_USE_SCALE_SETTINGS];
    o->name              = "use_scale_settings";
    o->shortDesc         = "Use scale settings";
    o->longDesc          = "Use scale speed/timestep settings instead of the ones specified here";
    o->type              = CompOptionTypeBool;
    o->value.b           = SD_USE_SCALE_SETTINGS_DEFAULT;

    o = &ss->opt[SD_SCREEN_OPTION_WINDOW_OPACITY];
    o->name             = "window_opacity";
    o->shortDesc        = "Window opacity when showdesktop'd";
    o->longDesc         = "Window opacity when showdesktop'd";
    o->type             = CompOptionTypeFloat;
    o->value.f          = SD_WINDOW_OPACITY_DEFAULT;
    o->rest.f.min       = SD_WINDOW_OPACITY_MIN;
    o->rest.f.max       = SD_WINDOW_OPACITY_MAX;
    o->rest.f.precision = SD_WINDOW_OPACITY_PRECISION;
}

/* plugin initialization */

static Bool
showdesktopInit (CompPlugin *p)
{
    displayPrivateIndex = allocateDisplayPrivateIndex ();

    if (displayPrivateIndex < 0)
        return FALSE;

    return TRUE;
}

/* plugin finalization */
static void
showdesktopFini (CompPlugin *p)
{

    if (displayPrivateIndex >= 0)
        freeDisplayPrivateIndex (displayPrivateIndex);
}

/* Handle event function
 * I have intentions of using this in the future
 **/
static void
showdesktopHandleEvent (CompDisplay *d, XEvent *event)
{
    SD_DISPLAY (d);
    CompWindow *w;

    switch (event->type)
    {
        case DestroyNotify:
            w = findWindowAtDisplay (d, event->xdestroywindow.window);
            if (w)
            {
                SD_SCREEN (w->screen);
                ss->ignoreNextTerminateEvent = TRUE;
            }
            break;
        case ReparentNotify:
        {
            w = findWindowAtDisplay (d, event->xreparent.window);
            if (w)
            {
                SD_SCREEN (w->screen);
                ss->ignoreNextTerminateEvent = TRUE;
            }
            break;
        }
        case FocusIn:
        {

            w = findWindowAtDisplay (d, event->xfocus.window);

            if (w && w->managed && w->id != d->activeWindow)
            {
                SD_SCREEN (w->screen);
                SD_WINDOW (w);

                if (ss->ignoreNextTerminateEvent)
                {
                    ss->ignoreNextTerminateEvent = FALSE;
                    break;
                }

                if (sw->placer && (ss->state == SD_STATE_ON || ss->state == SD_STATE_ACTIVATING))
                {
                    CompOption o[1];

                    o[0].type       = CompOptionTypeInt;
                    o[0].name       = "root";
                    o[0].value.i    = w->screen->root;

                    showdesktopTerminate (d, NULL, 0, o, 1);
                }
            }
            break;
        }
    }


    UNWRAP (sd, d, handleEvent);
    (*d->handleEvent) (d, event);
    WRAP (sd, d, handleEvent, showdesktopHandleEvent);
}

/* adjust velocity for each animation step (adapted from the scale plugin) */
static int
adjustSDVelocity (CompWindow *w)
{
    float dx, dy, adjust, amount;
    float x1, y1;

    SD_WINDOW (w);
    SD_SCREEN (w->screen);

    x1 = y1 = 0.0;

    if (!sw->placer)
        return 0;

    if (ss->state == SD_STATE_ACTIVATING)
    {
        x1 = sw->placer->offScreenX;
        y1 = sw->placer->offScreenY;
    }
    else if (ss->state == SD_STATE_DEACTIVATING)
    {
        x1 = sw->placer->onScreenX;
        y1 = sw->placer->onScreenY;
    }

    dx = x1 - (w->serverX + sw->tx);

    adjust = dx * 0.15f;
    amount = fabs (dx) * 1.5f;
    if (amount < 0.5f)
        amount = 0.5f;
    else if (amount > 5.0f)
        amount = 5.0f;

    sw->xVelocity = (amount * sw->xVelocity + adjust) / (amount + 1.0f);

    dy = y1 - (w->serverY + sw->ty);

    adjust = dy * 0.15f;
    amount = fabs (dy) * 1.5f;
    if (amount < 0.5f)
        amount = 0.5f;
    else if (amount > 5.0f)
        amount = 5.0f;

    sw->yVelocity = (amount * sw->yVelocity + adjust) / (amount + 1.0f);

    if (fabs (dx) < 0.1f && fabs (sw->xVelocity) < 0.2f &&
        fabs (dy) < 0.1f && fabs (sw->yVelocity) < 0.2f)
    {
        sw->xVelocity = sw->yVelocity = 0.0f;
        sw->tx = x1 - w->serverX;
        sw->ty = y1 - w->serverY;

        return 0;
    }
    return 1;
}

/* this function gets called periodically (about every 15ms on this machine),
 * animation takes place here */
static void
showdesktopPreparePaintScreen (CompScreen *s,
                               int        msSinceLastPaint)
{
    SD_SCREEN (s);

    if (ss->state == SD_STATE_ACTIVATING || ss->state == SD_STATE_DEACTIVATING)
    {
        CompWindow *w;
        int steps, dx, dy;
        float amount, chunk;

        amount = msSinceLastPaint * 0.05f * ss->speed;
        steps = amount / (0.5f * ss->timestep);
        if (!steps) steps = 1;
        chunk = amount / (float) steps;

        while (steps--)
        {
            ss->moreAdjust = 0;

            for (w = s->windows; w; w = w->next)
            {
                SD_WINDOW (w);

                if (sw->placer && sw->adjust)
                {
                    sw->adjust = adjustSDVelocity (w);

                    ss->moreAdjust |= sw->adjust;

                    sw->tx += sw->xVelocity * chunk;
                    sw->ty += sw->yVelocity * chunk;

                    dx = (w->serverX + sw->tx) - w->attrib.x;
                    dy = (w->serverY + sw->ty) - w->attrib.y;

                    moveWindow (w, dx, dy, FALSE, FALSE);
                }
            }
            if (!ss->moreAdjust)
                break;
        }

    }

    UNWRAP (ss, s, preparePaintScreen);
    (*s->preparePaintScreen) (s, msSinceLastPaint);
    WRAP (ss, s, preparePaintScreen, showdesktopPreparePaintScreen);
}

/* this one gets called after the one above and periodically,
 * here the plugin checks if windows reached the end */
static void
showdesktopDonePaintScreen (CompScreen *s)
{
    SD_SCREEN (s);


    if (ss->moreAdjust)
    {
        damageScreen (s);
    }
    else
    {
        if (ss->state == SD_STATE_ACTIVATING
                || ss->state == SD_STATE_DEACTIVATING)
        {
            CompWindow *w;
            for (w = s->windows; w; w = w->next)
            {
                SD_WINDOW (w);
                syncWindowPosition (w);
                if (ss->state == SD_STATE_DEACTIVATING)
                    sw->placer = 0;
            }
            if (ss->state == SD_STATE_ACTIVATING)
                ss->state = SD_STATE_ON;
        
            if (ss->state == SD_STATE_DEACTIVATING)
                ss->state = SD_STATE_OFF;
        }
   }

    UNWRAP (ss, s, donePaintScreen);
    (*s->donePaintScreen) (s);
    WRAP (ss, s, donePaintScreen, showdesktopDonePaintScreen);
}

/* display initialization */

static Bool
showdesktopInitDisplay (CompPlugin *p, CompDisplay *d)
{
   ShowdesktopDisplay *sd;

   sd = malloc (sizeof (ShowdesktopDisplay)); /* allocate the display */
   if (!sd)
       return FALSE;

   sd->screenPrivateIndex = allocateScreenPrivateIndex (d);
   if (sd->screenPrivateIndex < 0)
   {
       free (sd);
       return FALSE;
   }
   showdesktopDisplayInitOptions (sd, d->display);

   WRAP (sd, d, handleEvent, showdesktopHandleEvent);

   d->privates[displayPrivateIndex].ptr = sd;

   return TRUE;
}

/* display finalization
 * free resources
 * */
static void
showdesktopFiniDisplay (CompPlugin *p, CompDisplay *d)
{
    SD_DISPLAY (d);

    freeScreenPrivateIndex (d, sd->screenPrivateIndex);

    /*UNWRAP (sd, d, handleEvent);*/

    free (sd);
}

static void
showdesktopUpdateScaleOptions (CompScreen *s)
{
    CompPlugin *p;

    SD_SCREEN (s);

    if (!ss->opt[SD_SCREEN_OPTION_USE_SCALE_SETTINGS].value.b)
        return;

    p = findActivePlugin ("scale");
    if (p && p->vTable->getScreenOptions)
    {
        CompOption *options, *option;
        int        nOptions;

        options = (*p->vTable->getScreenOptions) (s, &nOptions);

        option = compFindOption (options, nOptions, "speed", 0);
        if (option)
            ss->speed = option->value.f;

        option = compFindOption (options, nOptions, "timestep", 0);
        if (option)
            ss->timestep = option->value.f;

        option = compFindOption (options, nOptions, "window_types", 0);
        if (option)
            ss->wMask = compWindowTypeMaskFromStringList (&option->value);

    }
}

static Bool
showdesktopSetScreenOptionForPlugin (CompScreen         *s,
                                     char               *plugin,
                                     char               *name,
                                     CompOptionValue    *value)
{
    Bool status;

    SD_SCREEN (s);

    UNWRAP (ss, s, setScreenOptionForPlugin);
    status = (*s->setScreenOptionForPlugin) (s, plugin, name, value);
    WRAP (ss, s, setScreenOptionForPlugin, showdesktopSetScreenOptionForPlugin);

    if (status && strcmp (plugin, "scale") == 0)
        if (strcmp (name, "speed") == 0 || strcmp (name, "timestep") == 0
            || strcmp (name, "window_types") == 0)
        showdesktopUpdateScaleOptions (s);

    return status;
}

/* screen initialization
 * */
static Bool
showdesktopInitScreen (CompPlugin *p, CompScreen *s)
{
    ShowdesktopScreen *ss;

    SD_DISPLAY (s->display);

    ss = malloc (sizeof (ShowdesktopScreen));
    if (!ss)
        return FALSE;

    ss->windowPrivateIndex = allocateWindowPrivateIndex (s);
    if (ss->windowPrivateIndex < 0)
    {
        free (ss);
        return FALSE;
    }


    showdesktopScreenInitOptions (ss);

    ss->state = SD_STATE_OFF;
    
    ss->placers = 0;
    ss->placersSize = 0;
    ss->moreAdjust = 0;
    
    ss->speed = SD_SPEED_DEFAULT;
    ss->timestep = SD_TIMESTEP_DEFAULT;
    ss->direction = SD_DIRECTION_DEFAULT;
    ss->windowOpacity = SD_WINDOW_OPACITY_DEFAULT;

    ss->ignoreNextTerminateEvent = FALSE;

    addScreenAction (s, &sd->opt[SD_DISPLAY_OPTION_INITIATE].value.action);

    WRAP (ss, s, preparePaintScreen, showdesktopPreparePaintScreen);
    WRAP (ss, s, donePaintScreen, showdesktopDonePaintScreen);
    WRAP (ss, s, setScreenOptionForPlugin, showdesktopSetScreenOptionForPlugin);

    s->privates[sd->screenPrivateIndex].ptr = ss;

    showdesktopUpdateScaleOptions (s);

    return TRUE;

}

/* Free screen resources */
static void
showdesktopFiniScreen (CompPlugin *p, CompScreen *s)
{
    SD_SCREEN (s);

    UNWRAP (ss, s, preparePaintScreen);
    UNWRAP (ss, s, donePaintScreen);

    if (ss->placersSize)
        free (ss->placers);

    freeWindowPrivateIndex (s, ss->windowPrivateIndex);

    free (ss);
}


/* window init */
static Bool
showdesktopInitWindow (CompPlugin *p,
                       CompWindow *w)
{
    ShowdesktopWindow *sw;

    SD_SCREEN (w->screen);

    sw = malloc (sizeof (ShowdesktopWindow));
    if (!sw)
        return FALSE;

    sw->placer = 0;
    sw->tx = sw->ty = 0.0f;
    sw->adjust = FALSE;
    sw->xVelocity = sw->yVelocity = 0.0f;
    sw->delta = 1.0f;

    w->privates[ss->windowPrivateIndex].ptr = sw;

    return TRUE;
}

/* Free window resources */
static void
showdesktopFiniWindow (CompPlugin *p,
                       CompWindow *w)
{
    SD_WINDOW (w);

    free (sw);
}

/* gconf interfacing */

static CompOption *
showdesktopGetDisplayOptions (CompDisplay *display,
                              int         *count)
{
    SD_DISPLAY (display);

    *count = NUM_OPTIONS (sd);
    return sd->opt;
}

static Bool
showdesktopSetDisplayOption (CompDisplay     *display,
                             char            *name,
                             CompOptionValue *value)
{
    CompOption *o;
    int index;

    SD_DISPLAY (display);

    o = compFindOption (sd->opt, NUM_OPTIONS (sd), name, &index);

    if (!o)
        return FALSE;

    switch (index)
    {
        case SD_DISPLAY_OPTION_INITIATE:
            if (setDisplayAction (display, o, value))
                return TRUE;
        default:
            break;
    }

    return FALSE;
}

static CompOption *
showdesktopGetScreenOptions (CompScreen *screen,
                             int        *count)
{
    SD_SCREEN (screen);

    *count = NUM_OPTIONS (ss);
    return ss->opt;
}

static Bool
showdesktopSetScreenOption (CompScreen      *screen,
                            char            *name,
                            CompOptionValue *value)
{
    CompOption *o;
    int        index;

    SD_SCREEN (screen);

    o = compFindOption (ss->opt, NUM_OPTIONS (ss), name, &index);

    if (!o)
        return FALSE;

    switch (index) {
        case SD_SCREEN_OPTION_SPEED:
            if (compSetFloatOption (o, value))
            {
                if (!ss->opt[SD_SCREEN_OPTION_USE_SCALE_SETTINGS].value.b)
                    ss->speed = o->value.f;
                return TRUE;
            }
            break;
        case SD_SCREEN_OPTION_TIMESTEP:
            if (compSetFloatOption (o, value))
            {
                if (!ss->opt[SD_SCREEN_OPTION_USE_SCALE_SETTINGS].value.b)
                    ss->timestep = o->value.f;
                return TRUE;
            }
            break;
        case SD_SCREEN_OPTION_DIRECTION:
            if (compSetIntOption (o, value))
            {
                ss->direction = o->value.i;
                return TRUE;
            }
            break;
        case SD_SCREEN_OPTION_WINDOW_TYPE:
            if (compSetOptionList (o, value))
            {
                if (!ss->opt[SD_SCREEN_OPTION_USE_SCALE_SETTINGS].value.b)
                    ss->wMask = compWindowTypeMaskFromStringList (&o->value);
                return TRUE;
            }
            break;
        case SD_SCREEN_OPTION_USE_SCALE_SETTINGS:
            if (compSetBoolOption (o, value)) {
                if (o->value.b)
                {
                    showdesktopUpdateScaleOptions (screen);
                }
                else
                {
                    ss->speed    = ss->opt[SD_SCREEN_OPTION_SPEED].value.f;
                    ss->timestep = ss->opt[SD_SCREEN_OPTION_TIMESTEP].value.f;
                    ss->wMask    = compWindowTypeMaskFromStringList (&(ss->opt[SD_SCREEN_OPTION_WINDOW_TYPE].value)); 
                }
                return TRUE;
            }
            break;
        case SD_SCREEN_OPTION_WINDOW_OPACITY:
            if (compSetFloatOption (o, value)) {
                ss->windowOpacity = o->value.f;
                return TRUE;
            }
            break;
    }
    return FALSE;
}

/* plugin vtable */
static CompPluginVTable showdesktopVTable = {
    "showdesktop",
    "Shows the desktop",
    "Shows the desktop",
    showdesktopInit,
    showdesktopFini,
    showdesktopInitDisplay,
    showdesktopFiniDisplay,
    showdesktopInitScreen,
    showdesktopFiniScreen,
    showdesktopInitWindow,
    showdesktopFiniWindow,
    showdesktopGetDisplayOptions,
    showdesktopSetDisplayOption,
    showdesktopGetScreenOptions,
    showdesktopSetScreenOption,
    0, /* deps fade */
    0, /* sizeof (fadeDeps) / sizeof (fadeDeps[0]) */
};

/* send plugin info */
CompPluginVTable *
getCompPluginInfo (void)
{
    return &showdesktopVTable;
}
