/*
*
* cobex_tools.c - Gemensamma rutiner fr COBEX-program.
*
* Copyright (c) 2004,2005 Fredrik Srensson
*
* History:
* v0.1 - fsn - 04-01-20 - First version
* v0.5 - fsn - 04-07-08 - Better dump_packet_headers
* v0.6 - fsn - 05-07-19 - Generic signal handlers
* v0.7 - fsn - 05-07-19 - Better path handling
* v0.8 - fsn - 05-07-21 - Port locking added in 25 chars
* Source:
*
*/


#include <stdio.h>
#include <ezV24/ezV24.h>
#include <signal.h>
#include <string.h>
#include <math.h>

#include "cobex_tools.h"
#include "cobex_serial.h"

state globalState = { CTOOLS_STATE_NOT_CONNECTED, NULL };

// The generic signal handler

void ctools_signalHandler ( int reason ) {

	char aBuffer[513];
	obex_packet aPacket;

	aPacket.max=512;
	aPacket.buffer=aBuffer;

	int rc;


	if (globalState.connState == CTOOLS_STATE_NOT_CONNECTED) {
		exit(99);
	} else if ( globalState.connState == CTOOLS_STATE_INOBEX ) {
		rc = ctools_abort( &aPacket, globalState.port );
		rc = ctools_disconnect( &aPacket, globalState.port );
	} // else if ( globalState.connState == CTOOLS_STATE_CONNECTED ) {	Not needed for now
	// }
	v24ClosePort(globalState.port);
	ctools_set_state_not_connected();
	exit(99);
}

// The generic signal handler installer

void ctools_installSignalhandler ( void ) {
	signal(SIGINT,ctools_signalHandler);
	signal(SIGTERM,ctools_signalHandler);
}

// Generic function to set up state.

void ctools_setup_state ( v24_port_t *port ) {
	globalState.port = port;
}

// Change global state

void ctools_set_state_not_connected( void ) {
	globalState.connState = CTOOLS_STATE_NOT_CONNECTED;
}
void ctools_set_state_connected( void ) {
	globalState.connState = CTOOLS_STATE_CONNECTED;
}
void ctools_set_state_inobex( void ) {
	globalState.connState = CTOOLS_STATE_INOBEX;
}
int ctools_get_state( void) {
	return globalState.connState;
}

// Go into OBEX mode.

int ctools_connect(v24_port_t *aPort) {
	int rc;

	v24Puts(aPort,"AT*EOBEX\r");
	rc=cobex_waitFor( aPort, "CONNECT", 7, "NO ANSWER",9);
	if (!rc) {
		ctools_set_state_inobex();
		return rc;
	}
	return rc;
}

// Leave OBEX mode

int ctools_disconnect( obex_packet *packet, v24_port_t *port ) {

	int rc;
		
	rc = obex_opcode_disconnect ( packet );
	if (rc != COBEX_OK) return rc;	
	rc = cobex_packlgt( packet );
	if (rc != COBEX_OK) return rc;

	rc = cobex_packet_send( packet, port );
	if (rc != COBEX_OK) return rc;
	rc = cobex_packet_recieve_t ( packet, port,100 );
	if (rc != COBEX_OK) return rc;
	rc = cobex_waitFor( port,"OK",2,NULL,0 );
	if (rc != COBEX_OK) return rc;

	ctools_set_state_connected();
	
	return COBEX_OK;
}

// ctools_recursePath - go to a path in a nice manner

int ctools_recursePath ( obex_packet *packet, char *path, v24_port_t *port ) {

	char *nextLevel;
	char *pTo; 
	int nFrom = 0;
	int nTo = 0;
	int rc;

	nextLevel = calloc( strlen(path)+1, sizeof(char) );

	do {
		pTo=index(path+nFrom,'/');
		if (pTo == NULL) {nTo = strlen(path); }
		else {nTo = (int)(pTo-(path+nFrom)); }
		strncpy(nextLevel,(path+nFrom),nTo);
		nextLevel[nTo]=0x00;
		if (!strlen(nextLevel)) {return COBEX_OK;}	//  This one to handle trailing slashes
		
		rc=ctools_setPath(packet,nextLevel,port);
		if (rc) {return rc;}				// Propagate errors back up the hierarchy (No such dir)
		nFrom=nFrom+nTo+1;

	} while ( pTo != NULL );
	
	return COBEX_OK;
}

// setPath

int ctools_setPath( obex_packet *packet, char *path, v24_port_t *port ) {

	// SetPath

	int rc;
			
	rc = obex_opcode_setpath( packet, OBEX_SETPATH_DONTCREATE )  ;
	if (rc != COBEX_OK) return rc;
	rc = obex_hi_name ( packet, path, strlen(path)+1 );
	if (rc != COBEX_OK) return rc;
	rc = cobex_set_final_bit ( packet );
	if (rc != COBEX_OK) return rc;	
	rc = cobex_packlgt ( packet );
	if (rc != COBEX_OK) return rc;	

	rc = cobex_packet_send( packet, port);
	if (rc != COBEX_OK) return rc;	
	cobex_packet_recieve_t( packet, port ,200);
	if (rc != COBEX_OK) return rc;	

	return COBEX_OK;
}

// Abort

int ctools_abort( obex_packet *packet, v24_port_t *port ) {

	// SetPath

	int rc;
	
	rc = obex_opcode_abort( packet )  ;
	if (rc) return rc;
	rc = cobex_set_final_bit ( packet );
	if (rc) return rc;	
	rc = cobex_packlgt ( packet );
	if (rc) return rc;	

	rc = cobex_packet_send( packet, port);
	if (rc) return rc;	
	cobex_packet_recieve_t( packet, port ,200);
	if (rc) return rc;	

	return COBEX_OK;
}

// Delete

int ctools_deleteFile( obex_packet *packet, char *name, v24_port_t *port ) {
	int rc;

	obex_opcode_put ( packet );
	obex_hi_length ( packet, 0 );
	obex_hi_name ( packet, name, strlen(name)+1 );
	cobex_packlgt ( packet );
	cobex_set_final_bit( packet );

	rc=cobex_packet_send(packet, port);
	if (rc) {return rc;}
	return cobex_packet_recieve(packet, port);
	
}

// Internal: recieve body
	
int ctools_recBody( obex_packet *packet, v24_port_t *port ) {
	
	int rc;

	do {
		rc = obex_opcode_get ( packet );
		if (rc != COBEX_OK) return rc;
		rc = cobex_set_final_bit ( packet );
		if (rc != COBEX_OK) return rc;
		rc = cobex_packlgt ( packet );
		if (rc != COBEX_OK) return rc;
		rc = cobex_packet_send( packet, port);
		if (rc != COBEX_OK) return rc;
		rc = cobex_packet_recieve_t( packet, port,200);
		if (rc != COBEX_OK) return rc;

		rc = printBody(packet); 

	} while (cobex_response_code(packet) == (OBEX_RESPONSE_CONTINUE|OBEX_FINAL_BIT) );

	return COBEX_OK;
}

// Get a file by name

int ctools_getFileByName( obex_packet *packet, char *name, v24_port_t *port ) {

	int rc;

	// Get

	rc = obex_opcode_get ( packet );
	if (rc != COBEX_OK) return rc;
	rc = obex_hi_name ( packet, name, strlen(name)+1 );
	if (rc != COBEX_OK) return rc;
	rc = cobex_set_final_bit ( packet );
	if (rc != COBEX_OK) return rc;
	rc = cobex_packlgt ( packet );
	if (rc != COBEX_OK) return rc;
	rc = cobex_packet_send( packet, port );
	if (rc != COBEX_OK) return rc;
	rc = cobex_packet_recieve_t( packet, port,200);
	if (rc != COBEX_OK) return rc;
	//dump_packet_headers(packet);
	rc = printBody(packet);
	if (rc != COBEX_OK) return rc;
	
	// Fetch the body
	
	if (cobex_response_code(packet) == (OBEX_RESPONSE_CONTINUE|OBEX_FINAL_BIT) ) {
		rc = ctools_recBody( packet, port );
		if (rc != COBEX_OK) return rc;
	}
	
	return COBEX_OK;
}

// Get a file by type

int ctools_getFileByType( obex_packet *packet, char *type, v24_port_t *port ) {

	int rc;

	// Get

	rc = obex_opcode_get ( packet );
	if (rc != COBEX_OK) return rc;
	rc = obex_hi_type ( packet, type, strlen(type)+1 );
	if (rc != COBEX_OK) return rc;
	rc = cobex_set_final_bit ( packet );
	if (rc != COBEX_OK) return rc;
	rc = cobex_packlgt ( packet );
	if (rc != COBEX_OK) return rc;
	rc = cobex_packet_send( packet, port );
	if (rc != COBEX_OK) return rc;
	rc = cobex_packet_recieve_t( packet, port,200);
	if (rc != COBEX_OK) return rc;
	rc = printBody(packet);
	if (rc != COBEX_OK) return rc;

	// Fetch the body
	
	if (cobex_response_code(packet) == (OBEX_RESPONSE_CONTINUE|OBEX_FINAL_BIT) ) {
		rc = ctools_recBody( packet, port );
		if (rc != COBEX_OK) return rc;
	}
		
	return COBEX_OK;
}


// Internal: Print body, using the parser

int printBody(obex_packet *inPacket) {
	u_int p;
	parsedData pD;
	u_int hiPos;
		
	p=cobex_parseStart(inPacket);

	// Hooray! This proved MUCH cleaner with the new, simple parser.

	while ( p < inPacket->l ) {
		pD = cobex_parseNext( inPacket,p ); 
		if ( (pD.HI == OBEX_HI_BODY) || (pD.HI == OBEX_HI_END_OF_BODY) ) {
			
			hiPos=0;
			while (hiPos<pD.HIlen) {
				printf ("%c",*(pD.HIdata+hiPos) );
				hiPos++;
				if ((p+3+hiPos)>inPacket->l) { return COBEX_ERR; }
			}
		}
		p = pD.nextHIp;
	}
	return COBEX_OK;
	
}

void dump_packet_headers (obex_packet *inPacket) {

	static const char *HIText[] = {
		"OBEX_HI_COUNT",
		"OBEX_HI_NAME",
		"OBEX_HI_TYPE",
		"OBEX_HI_LENGTH",
//		"OBEX_HI_TIME_ISO8601",
//		"OBEX_HI_TIME_COMP",
		"OBEX_HI_TIME",
		"OBEX_HI_DESCRIPTION",
		"OBEX_HI_TARGET",
		"OBEX_HI_HTTP",
		"OBEX_HI_BODY",
		"OBEX_HI_END_OF_BODY",
		"OBEX_HI_WHO",
		"OBEX_HI_CONNECTION_ID",
		"OBEX_HI_APP_PARAMETERS",
		"OBEX_HI_AUTH_CHALLANGE",
		"OBEX_HI_AUTH_RESPONSE",
		"OBEX_HI_CREATOR_ID",
		"OBEX_HI_WAN_UUID",
		"OBEX_HI_OBJECT_CLASS",
		"OBEX_HI_SESSION_PARAMS",
		"OBEX_HI_SESSION_SEQ_NUM" 
	};
		
	u_int p;
	u_int i;
	int packLen = 0;
	parsedData pD;
		
	p=cobex_parseStart(inPacket);
	
	printf ("\nPacket dump:\n");
	printf ("  Result code/Opcode: %2X ",cobex_response_code(inPacket) );
	packLen = ((inPacket->buffer[1]<<8) + inPacket->buffer[2]);
	printf ("  Packet Length: %d, %2X %2X",packLen,inPacket->buffer[1],inPacket->buffer[2]);
	printf ("  Header Length: %d\n",p);
	if ( (p == 7) && (cobex_response_code(inPacket) == 0xA0) ) {
		printf ("  Obex version 1.0\n");
	}
	
	while ( p < inPacket->l ) {
		pD = cobex_parseNext( inPacket,p );
		printf ("  Header Identifier: %2X %s \n",pD.HI,HIText[pD.HI&0x3f]);
		switch (pD.HI&0xC0) {
			case 0x00:
				printf ("   Text, length %d :",pD.HIlen);
				for (i=0; i<pD.HIlen; i++) {
					if (*(pD.HIdata)>31) {
						printf ("%c",*(pD.HIdata+i) );
					} else {
						printf (".");
					}
				}
				printf ("\n");
				break;
			case 0x40:
				printf ("   Data, length (inc 3 bytes header) %d \n",pD.HIlen);
/*				for (i=0; i<pD.HIlen; i++) {
					if (*(pD.HIdata)>31) {
						printf ("%c",*(pD.HIdata+i) );
					} else {
						printf (".");
					}
				}
				printf ("\n");
*/
				break;
			case 0x80:
				printf ("   Byte : %2X \n",*pD.HIdata);
				break;
			case 0xC0:
				printf ("   Quad :");
				for (i=0; i<4; i++) {
					printf ("%2X ", *(pD.HIdata+i) );
				}
				printf("\n");
				break;
		}
		p = pD.nextHIp;
	}
	return;
}

// Better.

int ctools_buildPath ( char *inFullPath, char *outPath, char *outName ) {
	char *lastSep;
	
	if (	(strncmp(inFullPath, "telecom/", imin(strlen(inFullPath),8))==0)) {
		strcpy(outName, inFullPath);
		outPath[0] = 0x00;
	} else {

		lastSep = rindex ( inFullPath, '/');
		if (lastSep == NULL) {
			strcpy(outName, inFullPath);
			outPath[0] = 0x00;
			return COBEX_OK;
		} else {
			strncpy(outPath, inFullPath, (lastSep-inFullPath) );
			strncpy(outName, lastSep+1, (inFullPath + strlen(inFullPath) - lastSep));
			return COBEX_OK;
		}
	}

	return COBEX_OK;
}
