###########################################################################
# clive, video extraction utility
# Copyright (C) 2007 Toni Gundogdu
#
# clive is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 0.1.2-1307 USA
###########################################################################

import os
import ConfigParser

from optparse import OptionParser, OptionGroup

from clive.version import *


__all__ = ['Options']


class Options:
  """Command line and runtime option parser"""
  CLIVE_CONF_SECTION  = 'clive'
  CLIVE_CONF_FILE     = '.cliverc'
  CLIVE_COOKIE_FILE   = '.clivecookies'

  CLIVE_DEFAULT_FILENAME_CHARS  = 'A-Za-z0-9'
  CLIVE_DEFAULT_FILENAME_FORMAT = '%t.%e'

  def __init__(self):
    path = os.path.expanduser('~')
    self.cfile = os.path.join(path, self.CLIVE_CONF_FILE)
    self.cookiefile = os.path.join(path, self.CLIVE_COOKIE_FILE)

  def parse(self):
    self._parseconf()
    self._parseopts()
    return (self.opts, self.args, self.conf)

  """
  def saveconf(self):
    self._setconf('agent',      self.opts.agent)    
    self._setconf('proxy',      self.opts.proxy)
    self._setconf('encoder',    self.opts.encoder)
    self._setconf('player',     self.opts.player)
    self.conf.write(open(self.cfile,'w'))
  """

  # Non-public

  def _parseconf(self):
    self.conf = ConfigParser.SafeConfigParser()
    self.conf.read(self.cfile)

  def _parseopts(self):
    p = OptionParser(
      usage = PACKAGE_USAGE,
      description = PACKAGE_DESC,
      version = '%prog '+PACKAGE_VERSION+'\n'+PACKAGE_NOTICE
    )

    p.add_option('--overwrite','-o',
      dest='overwrite',
      action='store_true',
      default=False,
      help='overwrite existing files')

    p.add_option('--accept','-y',
      dest='accept',
      action='store_true',
      default=False,
      help='extract video(s) without prompting')

    p.add_option('--skip','-s',
      dest='skip_existing',
      action='store_true',
      default=False,
      help='skip existing videos without prompting')

    p.add_option('--emit','-e',
      dest='emit',
      action='store_true',
      default=False,
      help='emit a listing of video urls to stdout')

    p.add_option('--quiet','-q',
      dest='verbose',
      action='store_false',
      default=True,
      help='turn off verboseness')

    p.add_option('--no-accept','-A',
      dest='noaccept',
      action='store_true',
      default=False,
      help='explicitly turn off extraction without prompting')

    p.add_option('--no-skip','-S',
      dest='noskip',
      action='store_true',
      default=False,
      help='explicitly turn off skipping existing videos')

    p.add_option('--no-newt','-N',
      dest='nonewt',
      action='store_true',
      default=False,
      help='disable newt user interface')

    p.add_option('--no-strip','-T',
      dest='strip',
      action='store_false',
      default=True,
      help='do not strip stdout status messages')

    p.add_option('--agent',
      dest='agent',
      default=None,
      metavar='AGENT',
      help='identify as AGENT instead of urllib2/VERSION')

    p.add_option('--prefix',
      dest='prefix',
      default=None,
      metavar='PREFIX',
      help='save extracted videos to PREFIX/')

    p.add_option('--bugs',
      dest='bugs',
      action='store_true',
      default=False,
      help='how to report bugs')


    fgroup = OptionGroup(
      p,
      'Filename options',
    )

    fgroup.add_option('--filename-chars','-C',
      dest='filename_chars',
      default='A-Za-z0-9',
      metavar='STRING',
      help='set STRING (regexp) as accepted filename (title) characters, ' + 
            'e.g. --filename-chars="A-Za-z0-9"')

    fgroup.add_option('--filename-format','-F',
      dest='filename_format',
      default='%t.%e',
      metavar='STRING',
      help='use STRING for formatting video filenames, ' +
            'e.g. --filename-format="%t.%e" ' +
            'accepted identifiers: %t=title, %i=id, %h=host, %e=extension')

    fgroup.add_option('--output-file',
      dest='output_file',
      default=None,
      metavar='FILE',
      help='write video file(s) to FILE (overrides the above two options)')


    egroup = OptionGroup(
      p,
      'Re-encode options',
      'Use the -y switch with ffmpeg to overwrite existing output files'
    )

    egroup.add_option('--encoder',
      dest='encoder',
      default=None,
      metavar='STRING',
      help='e.g. --encoder="/usr/bin/ffmpeg -y -i %i %o"')

    egroup.add_option('--mpeg','-m',
      dest='mpeg',
      action='store_true',
      default=False,
      help='re-encode extracted video to mpeg')

    egroup.add_option('--avi','-a',
      dest='avi',
      action='store_true',
      default=False,
      help='re-encode extracted video to avi')

    egroup.add_option('--flv','-f',
      dest='flv',
      action='store_true',
      default=False,
      help='re-encode extracted video to flv')

    egroup.add_option('--no-encode','-E',
      dest='noencode',
      action='store_true',
      default=False,
      help='explicitly turn off re-encoding')


    pgroup = OptionGroup(p, 'Player options')

    pgroup.add_option('--player',
      dest='player',
      default=None,
      metavar='STRING',
      help='e.g. --player="/usr/bin/vlc %i"')

    pgroup.add_option('--play','-p',
      dest='play',
      action='store_true',
      default=False,
      help='play video after extraction')

    pgroup.add_option('--play-format','-P',
      dest='play_format',
      default=None,
      metavar='FORMAT',
      help='play re-encoded video')

    pgroup.add_option('--no-play','-L',
      dest='noplay',
      action='store_true',
      default=False,
      help='explicitly turn off play after extraction')


    pxgroup = OptionGroup(
      p, 
      'Proxy options',
      'Proxy is on by default if the appropriate env. variable is defined'
    )

    pxgroup.add_option('--proxy',
      dest='proxy',
      default=None,
      metavar='PROXY',
      help='set PROXY as proxy')

    pxgroup.add_option('--proxy-user',
      dest='proxy_user',
      default=None,
      metavar='USER',
      help='set USER as proxy username')

    pxgroup.add_option('--proxy-pass',
      dest='proxy_pass',
      default=None,
      metavar='PASS',
      help='set PASS as proxy password')

    pxgroup.add_option('--no-proxy','-R',
      dest='use_proxy',
      action='store_false',
      default=True,
      help='explicitly turn off proxy')


    ygroup = OptionGroup(p, 'Youtube options')

    ygroup.add_option('--youtube-user',
      dest='youtube_user',
      default=None,
      metavar='USER',
      help='set USER as youtube username')

    ygroup.add_option('--youtube-pass',
      dest='youtube_pass',
      default=None,
      metavar='PASS',
      help='set PASS as youtube password')

    ygroup.add_option('--youtube-status',
      dest='youtube_status',
      action='store_true',
      default=False,
      help='display youtube.com log-in info')

    ygroup.add_option('--youtube-logout',
      dest='youtube_logout',
      action='store_true',
      default=False,
      help='logout from youtube.com')

    ygroup.add_option('--no-youtube-login','-Y',
      dest='use_youtube_login',
      action='store_false',
      default=True,
      help='explicitly turn off login to youtube.com')


    p.add_option_group(fgroup)
    p.add_option_group(egroup)
    p.add_option_group(pgroup)
    p.add_option_group(pxgroup)
    p.add_option_group(ygroup)

    (self.opts, self.args) = p.parse_args()

    # Add these without exposing them through the CLI
    self.opts.newt = 1 # --no-newt, use_newt (cliverc), _newt_installed override this

    # NOTE: Some of the cliverc options override the cmdline options
    self._loadconf()

  def _loadconf(self):
    # The following are overridden by the cmdline opts

    if not self.opts.agent:
      self.opts.agent = self._getconf('user_agent')
    if not self.opts.prefix:
      self.opts.prefix = self._getconf('prefix')

    if not self.opts.proxy:
      self.opts.proxy = self._getconf('http_proxy')
    if not self.opts.proxy_user:
      self.opts.proxy_user = self._getconf('proxy_user')
    if not self.opts.proxy_pass:
      self.opts.proxy_pass = self._getconf('proxy_pass')

    if not self.opts.youtube_user:
      self.opts.youtube_user = self._getconf('youtube_user')
    if not self.opts.youtube_pass:
      self.opts.youtube_pass = self._getconf('youtube_pass')

    if not self.opts.player:
      self.opts.player = self._getconf('player')
    if not self.opts.encoder:
      self.opts.encoder = self._getconf('encoder')

    if not self.opts.play_format:
      # This sets `--play' and `--FORMAT' as well
      self.opts.play_format = self._getconf('play_format')

    # The following override the default values only if defined in cliverc

    value = self._getconf('filename_chars')
    if value and \
      self.opts.filename_chars == self.CLIVE_DEFAULT_FILENAME_CHARS:
      self.opts.filename_chars = value

    value = self._getconf('filename_format')
    if value and \
      self.opts.filename_format == self.CLIVE_DEFAULT_FILENAME_FORMAT:
      self.opts.filename_format = value

    # The following will ALWAYS override the cmdline opts.

    value = self._getconf('encode_format')
    if value:
      value = [v.strip() for v in value.lower().split(',')]
      for v in value:
        setattr(self.opts, v, True)

    value = self._getconf('always_accept')
    if value:
      self.opts.accept = self._istrue(value)

    value = self._getconf('always_play')
    if value:
      self.opts.play = self._istrue(value)

    value = self._getconf('always_skip')
    if value:
      self.opts.skip_existing = self._istrue(value)

    value = self._getconf('always_strip')
    if value:
      self.opts.strip = self._istrue(value)

    # NOTE: see clive/cli.py; disabled ALWAYS if newt is not installed
    value = self._getconf('use_newt')
    if value:
      self.opts.newt = self._istrue(value)

  def _getconf(self, name):
    try:
      value = self.conf.get(self.CLIVE_CONF_SECTION, name)
      return value.replace('%%','%')
    except:
      return None

  """
  def _setconf(self, name, value):
    if not value or len(value) == 0 or value == 'None':
      try:
        self.conf.remove_option(self.CLIVE_CONF_SECTION,name)
      except:
        pass
      return
    try:
      value = value.replace('%','%%')
      self.conf.set(self.CLIVE_CONF_SECTION, name, value)
    except ConfigParser.NoSectionError:
      self.conf.add_section(self.CLIVE_CONF_SECTION)
      self._setconf(name,value)
    except:
      pass
  """

  def _istrue(self, value):
    value = value.lower()
    return (value == 'on' or \
            value == 'yes' or \
            value == '1' or \
            value == 'true')
