###########################################################################
# clive, video extraction utility
# Copyright (C) 2007 Toni Gundogdu
#
# clive is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 0.1.2-1307 USA
###########################################################################

import os.path
try:
  from snack import *
  _newt_installed = True
except:
  _newt_installed = False

from clive.util import *


__all__ = ['newtUI','_newt_installed']


class newtUI:
  """newt portion of clive interface"""
  def main(self, videos, ignored):
    if not _newt_installed:
      raise CliveError('error : newt (snack module) not installed')
    scr = SnackScreen()
    try:
      self.cursel = []
      self.result = ''
      while self.result != 'ESC' and self.result != 'extract':
        self.result = self._main(scr,len(videos),ignored)
        if self.result == 'select':
          self._select(scr,videos)
        elif self.result == 'ignored':
          self._ignored(scr,ignored)
    finally:
      scr.finish()
    self.result = self.result != 'ESC'

  def checkoverwrite(self, videos, prefix):
    exist = []
    if not prefix: prefix=''
    for video in videos:
      path = pathjoin(prefix,video['file'])
      if os.path.exists(path):
        exist.append(video['file'])

    if len(exist) == 0:
      return True

    scr = SnackScreen()
    result = False

    try:
      ct = CheckboxTree(height=10, scroll=1)
      for item in exist:
        ct.append(item)

      btns = [('Cancel',False),('OK',True)]
      bb = ButtonBar(scr, btns)

      g = GridForm(scr, 'Overwrite existing files:', 1,2) # cols,rows
      g.add(ct,0,0)
      g.add(bb,0,1)
    
      result = bb.buttonPressed(g.runOnce())

      if result: # OK
        for video in videos:
          for item in exist:
            if video['file'] == item:
              # Exists already, user left unchecked -> Don't overwrite
              video['owrite'] = False
          for sel in ct.getSelection():
            if video['file'] == sel:
              # Exists, user checked -> Overwrite
              video['owrite'] = True
    finally:
      scr.finish()

    return result

  # Non-public

  def _main(self, scr, numvideos, ignored):
    """Main menu"""
    options = []
    if len(ignored) > 0:
      options.insert(0,('Show ignored (%d)' % len(ignored),'ignored'))
    if numvideos > 0:
      options.insert(0,('Select videos (%d)' % numvideos,'select'))
    options.insert(len(options),('--','-'))
    if numvideos > 0:
      options.insert(len(options),('OK','extract'))
    options.insert(len(options),('Cancel','ESC'))

    lb = Listbox(height=5, width=20, returnExit=1)
    for text,cmd in options:
      lb.append(text,cmd)

    g = GridForm(scr, 'Main', 1,1) # columns,rows
    g.add(lb,0,0)

    result = g.runOnce()
    if result != 'ESC' and result != 'F12':
      result = result.current()

    return result

  def _select(self, scr, videos):
    """Select extracted videos"""
    ct = CheckboxTree(height=10, scroll=1)

    for video in videos:
      file = video['file'][:50]
      if len(file) == 50: file += '...'
      size = byteshuman(video['size'])
      issel = 0
      if len(self.cursel) > 0:
        for sel in self.cursel:
          if video['file'] == sel['file']:
            issel = 1
            break
      ct.append('%s (%s)' % (file,size), video['file'], selected=issel)

    btns = [('Cancel',False),('OK',True)]
    bb = ButtonBar(scr, btns)

    g = GridForm(scr, 'Select videos:', 1,2) # columns,rows
    g.add(ct,0,0) # column,row
    g.add(bb,0,1)

    result = bb.buttonPressed(g.runOnce())

    if result: # OK
      self.cursel = []
      for item in ct.getSelection():
        for video in videos:
          if video['file'] == item:
            self.cursel.append(video)

  def _ignored(self, scr, ignored):
    """List ignored URLs with reasons"""
    ct = CheckboxTree(height=10, scroll=1)

    i = 0
    for item in ignored:
      url = item['url'][:50]
      if len(url) == 50: url + '...'
      ct.append(url)
      reason = item['reason'][:50]
      if len(reason) == 50: reason + '...'
      ct.addItem(reason, (i,snackArgs['append']))
      i += 1

    btn = Button('OK')

    g = GridForm(scr, 'Ignored URLs', 1,2) # columns,rows
    g.add(ct,0,0)
    g.add(btn,0,1)
    g.runOnce()


if __name__ == '__main__':
  videos = []
  for i in range(8):
    file = 'file%0d.avi' % (i+1)
    size = (i+1)*1024*1024
    videos.append({'url':None,'file':file,'size':size})
  ignored = []
  for i in range(10):
    url = 'http://domain.tld/%i' % i
    ignored.append({'url':url,'reason':'random reason'})
  ui = newtUI(videos,ignored)
  print ui.result
