/*
 *   This file is part of Clinica.
 *
 *   Clinica is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   Clinica is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Clinica.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Authors: Leonardo Robol <leo@robol.it>
 *            Gianmarco Brocchi <brocchi@poisson.phc.unipi.it>
 */
 
 using Gtk;

namespace Clinica {

    private bool has_gsettings = false;
    

    public class Settings : GLib.Settings {
    
        private ResourceManager resource_manager { get; set; }
        
        /**
         * @brief A pointer to the selected search engine for the medicines,
         * or null if not search engine is available nor selected.
         */
        internal unowned MedicineSearchEngine selected_search_engine;
        
        public Settings (ResourceManager resources) {
            string? my_schema = null;
            
            /* Check if clinica's schema is insatlled system-wide and, 
             * if it's found, use it. */
            if ("org.phcteam.clinica" in list_schemas ()) {
                has_gsettings = true;
                my_schema = "org.phcteam.clinica";
            }
            
            GLib.Object (
                schema: my_schema
            );
            resource_manager = resources;
            
            /* Select the right search engines according to user preferences */
            var se_name = get_string ("medicine-search-engine");
            if (se_name == "") {
                /* No preference, select the first that is found */
                if (resource_manager.medicine_search_engines.length () > 0) {
                    selected_search_engine = resource_manager.medicine_search_engines.nth_data (0);
                    set_string ("medicine-search-engine", selected_search_engine.get_name ());
                }
            }
            else {
                /* Select the one that was preferred, or nothing if that is not found */
                foreach (var engine in resource_manager.medicine_search_engines) {
                    if (engine.get_name () == se_name) {
                        selected_search_engine = engine;
                        break;
                    }
                }
            }
        }
        
        public unowned MedicineSearchEngine? get_medicine_search_engine () {
            return selected_search_engine;
        }
        
        /**
         * @brief Return a list of string with the name of the active
         * plugins as modules.
         */
        public string [] get_active_plugins () {
            if (has_gsettings)
                return get_strv ("active-plugins");
            else {
                return {};
            }
        }
        
        /**
         * @brief Check if the plugin with the given name has been activated.
         */
        public bool is_plugin_active (string plugin_name) {
            if (schema != null)
            {
                string [] active_plugins = get_strv ("active_plugins");
                return (plugin_name in active_plugins);
            }
            else
                return false;
        }
        
        public void activate_plugin (string module_name) {
            if (has_gsettings)
                if (! (module_name in get_strv ("active-plugins"))) {
                    string [] active_plugins = get_strv ("active-plugins");
                    active_plugins += module_name;
                    set_strv ("active-plugins", active_plugins);
                }
        }
        
        
        public void deactivate_plugin (string module_name) {
            if (has_gsettings && module_name in get_strv ("active-plugins")) 
            {
                string [] active_plugins = get_strv ("active-plugins");
                string [] new_active_plugins = {};
                foreach (string pl in active_plugins) {
                    if (pl != module_name)
                        new_active_plugins += pl;
                }
                set_strv ("active-plugins", new_active_plugins);
            }
        }
    }
}
