#include "cp_types.h"
#include "cp_proto.h"

/* =============== Debug routines: called from debugger ============= 

showhandle(struct RedList *handle): send doubly linked of faces to stderr
showlist(struct Vertlist *vertlist): send general vertex list to stderr
pfacered(struct p_data *p): send red face order to stderr
print_redfaces(struct p_data *p,struct RedList *redfaces): send list of
   redfaces to stderr
print_flower(struct K_data *K,int v): send flower to stderr
pack_to_stderr(struct p_data *p): send combinatorics of p to stderr
pack_face_draw_order(struct p_data *p): send draw order to stderr
face_debug(struct p_data *p): save face info to "/tmp/fbug.txt"
list_non_plots(struct p_data *p): send count/list of non-plotted circles
   to stderr
red_bug(struct RedList *redfaces): send redface data to "/tmp/redface_log"
print_one_redface(FILE *fd,struct RedList *redface): (does the work)
lightverts_q(struct p_light *pl): send vlist (original indices) for 
   light pack to "/tmp/lightverts_log"
vertlist_q(struct p_data *p,struct Vertlist *vertlist): send vertlist
   to "/tmp/vertlist_log"
list_q(struct p_data *p,int *list): put pos/neg info (??) in "/tmp/list_log"
util_q(struct p_data *p): put pos/neg info (??) on util_flag 
   in "/tmp/util_log"
colorlist_q(struct p_data *p,int *list): sent colors to "/tmp/colorlist_log"
redlist_q(struct p_data *p,struct RedList *redlist): send redlist to
   "/tmp/redlist_log"
record_redlist(struct RedList *redlist,int lim): adjoin redlist to 
   "/tmp/diagnostic_log"
record_bdrydata(struct p_data *p,struct BdryData *bdrydata,int **face_org):
   adjoin "BdryData" info to "/tmp/diagnostic_log"
pford(struct p_data *p): adjoin face order (as in f_data) 
   to "/tmp/diagnostic_log"
ck_list(struct RedList *redlist): check pointers of "RedList".
record_edge_pair(struct p_data *p): adjoin edge_pairing info to
   "/tmp/redface_log"
record_face_verts(struct p_data *p): adjoint face vert triples to
   "/tmp/face_verts".
print_pathlist(struct Pathlist *pathlist): save x,y's to "/tmp/pathlist".

*/

int showhandle(struct RedList *handle)
     /* show and check doubly linked list of faces */
{
  int count=0,total=1,first,last=0;
  struct RedList *trace;

  if ((trace=handle->next)==NULL) return 0;
  first=trace->face;
  while (trace!=handle && total < 2000)
    {
      total++;
      last=trace->face;
      trace=trace->next;
    }
  fprintf(stderr,
	  "\nCheck double linked face list: count=%d,first=%d,last=%d.\n",
	  total,first,last);
  while (trace!=handle && count <= total+1)
    {
      count++;
      fprintf(stderr,"  %d",trace->face);
      if (trace->prev->next!=trace)
	fprintf(stderr," prev->next wrong: %d %d, skips %d",
		handle->prev->face,handle->prev->next->face,
		handle->face);
      if (trace->next->prev!=trace)
	fprintf(stderr," next->prev wrong: %d %d, skips %d",
		handle->next->face,handle->next->prev->face,
		handle->face);
    }
  return count;
} /* showhandle */

int showlist(struct Vertlist *vertlist)
     /* show a Vertlist */
{
  int count=0,first,last;
  struct Vertlist *trace;

  if ((trace=vertlist)==NULL) return 0;
  first=vertlist->v;
  while (trace)
    {
      count++;
      last=trace->v;
      trace=trace->next;
    }
  fprintf(stderr,"\nShow a 'Vertlist': count=%d, first=%d, last=%d.\n",
	  count,first,last);
  trace=vertlist;
  while (trace)
    {
      fprintf(stderr," %d",trace->v);
      trace=trace->next;
    }
  fprintf(stderr,"  end\n");
  return 1;
} /* showlist */

int pfacered(struct p_data *p)
     /* print red face order according to pack */
{
  int f,count=1,stop;

  fprintf(stderr,"Pack red face order: %d \n",
	  (stop=p->first_red_face));
  f=stop;
  while ( p->faces[f].next_red!=stop 
	  && count < 2*p->facecount
	  && p->faces[f].next_red >0
	  && p->faces[f].next_red <=p->facecount && count++)
    fprintf(stderr,"%d ",(f=p->faces[f].next_red)); 
  return count;
} /* pfacered */

int print_redfaces(struct p_data *p,struct RedList *redfaces)
     /* print list of redfaces */
{
  int count=0;
  struct RedList *hold;

  if (!redfaces) 
    {
      fprintf(stderr,"Redfaces pointer not set.\n");
      return 0;
    }
  hold=redfaces;
  fprintf(stderr,"Red: %d  ",hold->face);
  hold=hold->next;
  while (hold!=redfaces && count<=2*p->facecount)
    {
      fprintf(stderr,"%d  ",hold->face);
      hold=hold->next;
      count++;
    }
  fprintf(stderr,"%d \n",hold->face);	
  return 1;
} /* print_redfaces */

int print_flower(struct K_data *K,int v)
     /* print out flower */
{
  int j;

  fprintf(stderr,"Flower v%d, num= %d ",
	  v,K[v].num);
  for (j=0;j<=K[v].num && j<100;j++)
    fprintf(stderr," %d, ",K[v].flower[j]);
  fprintf(stderr,"\n");
  return 1;
} /* print_flower */

int pack_to_stderr(struct p_data *p)
/* debugging help: write comb of p to stderr */
{
	writepack(stderr,p,0003,0);
  return 1;
} /* pack_to_strderr */

int pack_face_draw_order(struct p_data *p)
     /* show order of placing faces */
{
  int nf;

  nf=p->first_face;
  fprintf(stderr,"\nCircle-drawing order: %d ",nf);
  while ( (nf=p->faces[nf].next_face)!=p->first_face)
    fprintf(stderr,"%d ",nf);
  fprintf(stderr,"\n");
  return 1;
} /* pack_face_draw_order */

int face_debug(struct p_data *p)
     /* save face info for debugging */
{
  FILE *fp;
  int nf;

  fp=fopen("/tmp/fbug.txt","w");
  nf=p->first_face;
  fprintf(fp,"\n Face data in drawing order:\n");
  fprintf(fp,"face %d:\t (%d, %d, %d)\n",nf,
	  p->faces[nf].vert[p->faces[nf].index_flag],
	  p->faces[nf].vert[(p->faces[nf].index_flag+1)%3],
	  p->faces[nf].vert[(p->faces[nf].index_flag+2)%3]);
  while ( (nf=p->faces[nf].next_face)!=p->first_face)
    fprintf(fp,"face %d:\t (%d, %d, %d)\n",nf,
	    p->faces[nf].vert[p->faces[nf].index_flag],
	    p->faces[nf].vert[(p->faces[nf].index_flag+1)%3],
	    p->faces[nf].vert[(p->faces[nf].index_flag+2)%3]);
  fclose(fp);
  return 1;
} /* face_debug */

int list_non_plots(struct p_data *p)
     /* count and list non-plotted circles (ones whose placements 
are in question) */
{
  int i,count=0;

  for (i=1;i<=p->nodecount;i++)
    if (!p->packK_ptr[i].plot_flag)
      {
	fprintf(stderr," %d ",i);
	count++;
      }
  return count;
} /* list_non_plots */

int red_bug(struct RedList *redfaces)
     /* look through redface structure */
{
  int count=0;
  struct RedList *rtrace;
  FILE *diagfp;


  if (!(diagfp=fopen("/tmp/redface_log","a"))) return 1;
  fprintf(diagfp,"Data on red face list: ----------------------------- \n\n");
  if (!redfaces)
    {
      fprintf(stderr,"   No pointer to red chain ??\n");
      return 0;
    }
  print_one_redface(diagfp,redfaces);
  rtrace=redfaces->next;
  while (rtrace!= redfaces && count<1000)
    {
      print_one_redface(diagfp,rtrace);
      count++;
      rtrace=rtrace->next;
    }
  fclose(diagfp);
  return 1;
} /* red_bug */
      
int print_one_redface(FILE *fd,struct RedList *redface)
{
  int i;

  fprintf(fd," Face %d.     (prev,face,next)=(%d,%d,%d); \n",
	  redface->face,redface->prev->face,redface->face,
	  redface->next->face);
  fprintf(fd,"   v_flag = %d;      corner_flag: %d, %d, %d",
	  redface->v_flag,
	  redface->corner_flag[0],redface->corner_flag[1],
	  redface->corner_flag[2]);
  if (redface->next->face==redface->prev->face)
    fprintf(fd,"     BLUE face\n");
  else fprintf(fd,"\n");
  if (redface->next_edge) 
    fprintf(fd,"   next_edge face %d.\n",redface->next_edge->face);
  for (i=0;i<3;i++)
    {
      if (redface->cross[i])
	fprintf(fd,"      cross_edge index %d, face is %d;\n",
		i,redface->cross[i]->face);
    }
  fprintf(fd,"              rad = %f, center = (%f,%f).\n",
	  redface->rad,redface->center.re,redface->center.im);
  fprintf(fd,"\n");
  return 1;
} /* print_one_redface */

int lightverts_q(struct p_light *pl)
     /* create vlist (original indices) for light pack for CirclePack */
{
  int i;
  FILE *diagfp;

  if (!pl) return 0;
  if (!(diagfp=fopen("/tmp/lightverts_log","w"))) return 1;
  fprintf(diagfp,"\n\nCHECKCOUNT: %d \n\nVERT_LIST:\n\n",pl->counts[4]);
  for (i=1;i<=pl->counts[0];i++)
    fprintf(diagfp,"%d\n",pl->orig_indices[i]);
  fprintf(diagfp,"(done)\nEND\n");
  fclose(diagfp);
  return 1;
} /* lightverts_q */ 

int vertlist_q(struct p_data *p,struct Vertlist *vertlist)
     /* put vertlist in file for CirclePack */
{
  struct Vertlist *vlist;
  FILE *diagfp;

  if (!vertlist) return 0;
  vlist=vertlist;
  if (!(diagfp=fopen("/tmp/vertlist_log","w"))) return 1;
  fprintf(diagfp,"\n\nCHECKCOUNT: %d \n\nVERT_LIST:\n\n",p->nodecount);
  while (vlist)
    {
      fprintf(diagfp,"%d\n",vlist->v);
      vlist=vlist->next;
    }
  fprintf(diagfp,"(done)\nEND\n");
  fclose(diagfp);
  return 1;
} /* vertlist_q */ 

int list_q(struct p_data *p,int *list)
     /* put positives/negatives in file for CirclePack */
{
  int i,v;
  FILE *diagfp;

  if (!(diagfp=fopen("/tmp/list_log","w"))) return 1;
  fprintf(diagfp,"\n\nCHECKCOUNT: %d \n\nCIRCLE_COLORS:\n\n",p->nodecount);
  for (i=1;i<=p->nodecount;i++)
    {
      v=list[i];
      if (v==0) 
	fprintf(diagfp,"%d  %d\n",i,100);
      else if (v<0)
	fprintf(diagfp,"%d  %d\n",i,10);
      else if (v>0)
	fprintf(diagfp,"%d  %d\n",i,190);
    }
  fprintf(diagfp,"(done)\nEND\n");
  fclose(diagfp);
  return 1;
} /* list_q */ 

int util_q(struct p_data *p)
     /* put positives/negatives in file based on util_flag */
{
  int i,v;
  FILE *diagfp;

  if (!(diagfp=fopen("/tmp/util_log","w"))) return 1;
  fprintf(diagfp,"\n\nCHECKCOUNT: %d \n\nCIRCLE_COLORS:\n\n",p->nodecount);
  for (i=1;i<=p->nodecount;i++)
    {
      v=p->packK_ptr[i].util_flag;
      if (v==0) 
	fprintf(diagfp,"%d  %d\n",i,100);
      else if (v<0)
	fprintf(diagfp,"%d  %d\n",i,10);
      else if (v>0)
	fprintf(diagfp,"%d  %d\n",i,190);
    }
  fprintf(diagfp,"(done)\nEND\n");
  fclose(diagfp);
  return 1;
} /* util_q */ 

int colorlist_q(struct p_data *p,int *list)
     /* put colors in file for CirclePack */
{
  int i=1,v,maxn,minn,maxp,minp;
  double rn,rp;
  FILE *diagfp;

  if (!(diagfp=fopen("/tmp/colorlist_log","w"))) return 1;
  fprintf(diagfp,"\n\nCHECKCOUNT: %d \n\nCIRCLE_COLORS:\n\n",p->nodecount);
  maxp=0;minp=10000;
  maxn=-10000;minn=0;
  for (i=1;i<=p->nodecount;i++)
    {
      if ((v=list[i])<0)
	{
	  maxn = (v>maxn) ? v : maxn;
	  minn = (v<minn) ? v : minn;
	}
      else if (v>0)
	{
	  maxp = (v>maxp) ? v : maxp;
	  minp = (v<minp) ? v : minp;
	}	  
    }
  rn = ((maxn-minn)==0) ? 1 : (double)(maxn-minn);
  rp = ((maxp-minp)==0) ? 1 : (double)(maxp-minp);
  for (i=1;i<=p->nodecount;i++)
    {
      v=list[i];
      if (v==0) 
	fprintf(diagfp,"%d  %d\n",i,100);
      else if (v<0)
	fprintf(diagfp,"%d  %d\n",i,(int)(1+99*((v-minn)/rn)));
      else if (v>0)
	fprintf(diagfp,"%d  %d\n",i,(int)(199*99*((v-maxp)/rp)));
    }
  fprintf(diagfp,"(done)\nEND\n");
  fclose(diagfp);
  return 1;
} /* colorlist_q */ 

int redlist_q(struct p_data *p,struct RedList *redlist)
     /* put redlist in file for CirclePack. Return 0 on
error, -1 if redlist is not a closed list. */
{
  int wflag=0,count=0;
  struct RedList *trace;
  FILE *diagfp;

  if (!(diagfp=fopen("/tmp/redlist_log","w"))) return 0;
  fprintf(diagfp,"\n\nCHECKCOUNT: %d \n\nFACE_LIST:\n\n",p->nodecount);
  trace=redlist;
  while (count<p->facecount 
	 && (trace!=redlist->next || !(wflag++)))
    {
      fprintf(diagfp,"%d\n",trace->face);
      trace=trace->next;
      count++;
    }
  fprintf(diagfp,"(done)\nEND\n");
  if (count>=p->facecount)
    {
      fprintf(diagfp,"ERROR: redlist is not closed\n");
      fclose(diagfp);
      return -1;
    }
  fclose(diagfp);
  return 1;
} /* redlist_q */

int record_redlist(struct RedList *redlist,int lim)
     /* print redlist; safty limit in case of error. */
{
  int n=0;
  struct RedList *trace;
  FILE *diagfp;

  if (!(diagfp=fopen("/tmp/diagnostic_log","a"))) return 1;
  fprintf(diagfp,"\n\nRedlist: %d -> ",redlist->face);
  trace=redlist->next;
  while (trace!=redlist && n<lim)
    {
      fprintf(diagfp,"%d -> ",trace->face);
      trace=trace->next;
      n++;
    }
  fprintf(diagfp,"%d \n",trace->face);
  if (n>=lim) fprintf(diagfp,"\n reached safty limit of %d.\n",lim);
  fclose(diagfp);
  return 1;
} /* record_redlist */

int record_bdrydata(struct p_data *p,struct BdryData *bdrydata,int **face_org)
{
  int v,f1,f2,*face_ptr;
  struct BdryData *trace;
  FILE *diagfp;

  if (!(diagfp=fopen("/tmp/diagnostic_log","a"))) return 1;
  v=bdrydata->v;
  face_ptr=face_org[v]+1;
  f1=face_ptr[bdrydata->indx1];
  f2=face_ptr[bdrydata->indx2];
  fprintf(diagfp,"\n\nBdryData, vertex (first face, last face):\n %d (%d,%d) -> ",
	  v,f1,f2);

  trace=bdrydata->next;
  while (trace!=bdrydata)
    {
      v=trace->v;
      face_ptr=face_org[v]+1;
      f1=face_ptr[trace->indx1];
      f2=face_ptr[trace->indx2];
      fprintf(diagfp,"%d (%d,%d) -> ",v,f1,f2);
      trace=trace->next;
    }
  fclose(diagfp);
  return 1;
} /* record_bdrydata */

int pford(struct p_data *p)
     /* print the face order as recorded f_data */
{
  int f,count=1,stop;
  FILE *diagfp;

  if (!(diagfp=fopen("/tmp/diagnostic_log","a"))) return 1;
  fprintf(diagfp,"\n\nPack face order: %d \n", (stop=p->first_face));
  f=stop;
  while ( p->faces[f].next_face!=stop 
	  && count < 2*p->facecount
	  && p->faces[f].next_face >0
	  && p->faces[f].next_face <=p->facecount && count++)
    fprintf(diagfp,"%d ",(f=p->faces[f].next_face)); 
  fclose(diagfp);
  return count;
} /* pford */

int ck_list(struct RedList *redlist)
     /* see if pointers are screwed up */
{
  struct RedList *trace;

  trace=redlist;
  if (!redlist || redlist->next==redlist || redlist->prev==redlist) 
    fprintf(stderr,"Problem with redlist, face %d\n",redlist->face);
  while ((trace=trace->next)!=redlist)
    if (!trace || trace->next==trace || trace->prev==trace) 
      fprintf(stderr,"Problem with redlist, face %d\n",trace->face);
  return 1;
} /* ck_list */

int record_edge_pair(struct p_data *p)
     /* record edge_pairing info */
{
  int i;
  FILE *diagfp;

  if (!(diagfp=fopen("/tmp/redface_log","a"))) return 1;
  fprintf(diagfp,"\n\nEdge pairing info -----------------\n");
  for (i=1;p->edge_pair[i].edge;i++)
    {
      fprintf(diagfp,"  edge->face = %d, edge_indx = %d, (vert = %d)\n",
	      p->edge_pair[i].edge->face,p->edge_pair[i].edge_indx,
	      p->faces[p->edge_pair[i].edge->face].
	      vert[p->edge_pair[i].edge_indx]);
      if (p->edge_pair[i].mate)
	fprintf(diagfp,"  mate->face = %d, mate_indx = %d, (vert = %d)\n",
		p->edge_pair[i].mate->face,p->edge_pair[i].mate_indx,
		p->faces[p->edge_pair[i].mate->face].
		vert[p->edge_pair[i].mate_indx]);
      fprintf(diagfp,"      color = %d\n\n",p->edge_pair[i].color);
    }
  fclose(diagfp);
  return 1;
} /* record_edge_pair */
            
int record_face_verts(struct p_data *p)
     /* adjoin list of the order list of vertices for the faces of p */
{
  int i;
  FILE *diagfp;

  if (!(diagfp=fopen("/tmp/face_verts","a"))) return 1;
  fprintf(diagfp,"\n\nTriples of face verts -----------------\n");
  for (i=1;i<=p->facecount;i++)
    {
      fprintf(diagfp,"face %d: {%d, %d, %d}\n",i,
	      p->faces[i].vert[0],
	      p->faces[i].vert[1],
	      p->faces[i].vert[2]);
    }
  fprintf(diagfp,"\n---------------------------------------- \n\n");
  fclose(diagfp);
  return 1;
} /* record_face_verts */

int print_pathlist(struct Pathlist *pathlist)
     /* write pathlist to a file */
{
  struct Pathlist *plist;
  FILE *diagfp;

  if (!pathlist) return 0;
  if (!(diagfp=fopen("/tmp/pathlist","w"))) return 1;
  fprintf(diagfp,"\n\nPathlist: x y -----------------\n");
  plist=pathlist;
  while(plist)
    {
      fprintf(diagfp,"%f  %f\n",plist->x,plist->y);
      plist=plist->next;
    }
  fprintf(diagfp,"\n end");
  fclose(diagfp);
  return 1;
} /* print_pathlist */


