/*
 * Br_Image.cpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright 2005  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * Br_Image.cpp  -  mit name()-Funktion
 * 
 * Content:
 *   class Br_Image
 *   class Br_ImageVec
 */
#include <cstring>      // strdup()
#include <cstdlib>      // free()
#include <cstdio>       // printf()
#include <cmath>        // pow()

#include "Br_Image.hpp"

//#define YS_DEBUG
#undef YS_DEBUG
#include "YS/ys_dbg.hpp"


/**=====================================================================
 * Br_Image  -  class   (Bracketing-relevante Bild-Metadaten)
 * 
 * Klasse, die den C-String kopiert. Koennte man sich sparen durch Uebergang 
 * zu STL-Strings.
 *======================================================================*/
//--------------------
// Init Constructor...
//--------------------
Br_Image::Br_Image (const char* fn, 
                    int w, 
                    int h, 
                    Br_Precision prec, 
                    float t, 
                    bool act)
{ 
  SPUR(("Init-Constr (\"%s\")\n", fn));
  if (fn) name_ = strdup(fn); else name_ = NULL; 
  width_        = w; 
  height_       = h; 
  time          = t; 
  aperture      = 0.0; // not yet used
  ISO           = 0.0; // not yet used
  active        = act; 
  precision     = prec;
  brightness    = 0.0;
  n_low         = 0;
  n_high        = 0;
  r_wrange      = 0.0;
}
//----------------------------------------------
// copy_()  - used by copy constr. and =-op()...
//----------------------------------------------
void Br_Image::copy_(const Br_Image & img)
{ 
  SPUR(("copy_(\"%s\")\n", img.name()));
  if (img.name()) name_ = strdup(img.name()); else name_ = NULL; 
  width_     = img.width();
  height_    = img.height();
  time       = img.time;
  aperture   = img.aperture;
  ISO        = img.ISO;
  active     = img.active;
  precision  = img.precision;
  brightness = img.brightness;
  n_low      = img.n_low;
  n_high     = img.n_high;
  r_wrange   = img.r_wrange;
}
//----------------
// =-operator()...
//----------------
Br_Image & Br_Image::operator= (const Br_Image & img)
{ 
  SPUR(("=-Operator (\"%s\") = \"%s\"\n", name(), img.name()));
  if (&img != this) {
    free_();                // sich selbst freigeben
    copy_(img);
  }
  else {}                   // Selbstkopie -- mache nichts
  return *this;
}
//-----------------
// list contents...
//-----------------
void Br_Image::list() const
{ 
  printf("\"%s\" (%s):\n\t%d x %d, \"%s\", time=%f, brightn=%f\n",
         name(), 
         active ? "active" : "inactive",
         width(), 
         height(), 
         BR_PRECISION_SHORT_STR[precision],
         time, 
         brightness);
  printf("\tn_low=%d, n_high=%d, r_wrange=%f\n", n_low, n_high, r_wrange);
}
//------------------------------
// copy new name and free old...
//------------------------------
void Br_Image::name(const char* fn)
{ 
  if (name_) free(name_);
  if (fn) name_ = strdup(fn); else name_ = NULL;
}  



/**=====================================================================
 * Br_ImageVec  -  Container of Br_Image's
 *
 * NOTE: `vector' kopiert seine Objekte mittels deren Kopierkonstruktoren 
 *   auf den Heap. Wuerde bei C-Strings also nur fuer statische funktionieren,
 *   da nur die Adressen kopiert wuerden. Fuer nichtstatische ist Ueberladen
 *   des Kopiekonstruktors von Br_Image erforderlich -- oder mit STL-Strings
 *   arbeiten. -- Ueberladen inzwischen erfolgt.
 *======================================================================*/
//---------------------------
// number of active Images...
//---------------------------
int Br_ImageVec::size_active() const
{ 
  int s = 0;
  
  for (size_t i=0; i < size(); i++) 
    if ((*this)[i].active)  s++;
  
  return s;
}
//--------------------------
// list content of vector...
//--------------------------
void Br_ImageVec::list() const
{ 
  printf ("=== Br_ImageVec === [%d]\n", size());  
  for (size_t i=0; i < size(); i++)
  {
    printf("[%2d]: ", i);  (*this)[i].list();
  }
}
//-----------------------------------------------------------------
// Calc. times accordingly to 'stops' value, image[0].time as basis...
//-----------------------------------------------------------------
void Br_ImageVec::set_times_by_stops(double stops) 
{ 
  for (size_t i=1; i < size(); i++) 
  { 
    (*this)[i].time = (*this)[i-1].time * pow(2.0, stops);
  }
}


// END OF FILE
