#!/usr/bin/python
# -*- coding: UTF-8 -*- 

#~ JollyBOX chat+ server
#~ Copyright (C) 2006 Thomas Jollans
#~
#~ This program is free software; you can redistribute it and/or modify
#~ it under the terms of the GNU General Public License version 2 as 
#~ published by the Free Software Foundation
#~
#~ This program is distributed in the hope that it will be useful,
#~ but WITHOUT ANY WARRANTY; without even the implied warranty of
#~ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#~ GNU General Public License for more details.
#~
#~ You should have received a copy of the GNU General Public License
#~ along with this program; if not, write to the Free Software
#~ Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


def printwrapper ( str ):
	print str

import socket
import select
import thread
import traceback
import sys
import getopt
import time


class ChatServer:
    def __init__ ( self, roomname, port, bindaddr, printfn=printwrapper):
        self.serv_socket = socket.socket ( socket.AF_INET, socket.SOCK_STREAM )
        self.serv_socket.bind ( ( bindaddr, port ) )
        self.name = roomname
        self.clients = {}
        self.version = u"FZ's chat+ v0.1 //EXT:protocol"
        self.telluser = printfn
        self.protocols = {'chat+pr0': ChatPlusProtocol0} #no support
                                #for late tjchat proto switching'
    
    def main_loop ( self ):
        self.serv_socket.listen ( 2 )
        
        while True:
            sock, addr = self.serv_socket.accept ()
            cl = ChatClient (sock, self)
            cl.start()
    
    def add_client ( self, name, handler ):
        self.clients [ name ] = handler
        for to, client in self.clients.items():
            if not to == name:
                client.proto.send_special ( "new", name )
        self.telluser ( "NEW USER: " + name )
        
    def del_client ( self, name ):
        if name not in self.clients: return
        del self.clients [ name ]
        for to, client in self.clients.items():
            if not to == name:
                client.proto.send_special ( "hasleft", name )
        self.telluser ( "USER HAS LEFT: " + name )
    
    def private ( self, author, recpt, msg ):
        try:
            self.clients [recpt].proto.send_private(author, msg)
        except:
            pass
    
    def message ( self, author, msg ):
        for name, client in self.clients.items ():
            if not name == author:
                client.proto.send_msg ( author, msg )
    
    def special ( self, name, code, content='' ):
        if code == "#server-version":
            self.clients[name].proto.send_special ( "version", self.version )
        elif code == "#client-list":
            self.clients[name].proto.send_special ( "names", self.clients.keys())
        elif code == ".protocol-list":
            self.clients[name].proto.send_special ( "protocol-list", self.protocols.keys() )
        elif code == ".protocol-set":
            if content in self.protocols:
                self.clients[name].proto.send_special ( "protocol-ok", content )
                self.clients[name].proto = self.protocols[content](self.clients[name])
            else:
                self.clients[name].proto.send_special ( "error", "no such protocol: '%s'" % content )

class ChatClient:
    def __init__ ( self, sock, chatserv ):
        self.sock = sock
        self.server = chatserv
        self.buf = []
        self.flush = False
        self.proto = TjChatProtocol(self)
    
    def start ( self ):
        self.threadid = thread.start_new_thread ( self.mainloop, () )
    
    def mainloop ( self ):
        #actual loop
        try:
            while True:
                r, w, x  = None, None, None
                if self.flush:
                    if len ( self.buf ) == 0:
                        self.sock.close()
                        return
                    r, w, x = select.select ( [], [self.sock], [] )
                else:
                    r, w, x = select.select ( [self.sock], [self.sock], [] )
                    
                if r:
                    stringe = r[0].recv(512)
                    if len(stringe) == 0:
                        # EOF
                        self.close_down()
                        continue
                        
                    self.proto.parse ( stringe )
                if w and self.buf:
                    try:
                        sendit = self.buf.pop()
                        w[0].send ( sendit )
                    except:
                        # damn it. I think we'll stop this nonsense now.
                        self.close_down()
                else:
                    time.sleep(.1)

        except:
            traceback.print_exc()
    
    def buffer ( self, str ):
        self.buf.insert ( 0, str )
        
    def close_down ( self ):
        self.server.del_client ( self.name ) 
        self.buf = [] # that would result in unwanted exceptions
        self.flush = True

class TjChatProtocol:
    """TjChat compatible protocol-engine"""

    def __init__ ( self, client ):
        self.client = client
        self.first = True

    def buffer ( self, msg ):
        self.client.buffer( msg.encode("utf-16-le") )

    def send_msg ( self, author, msg ):
        self.buffer ( "$" + author + ";" + msg + "$"  )
        
    def send_private ( self, author, msg ):
        self.buffer ( "$~" + author + ";" + msg + "$"  )
        
    def send_special ( self, code, data ):
        if isinstance( data, list ):
            data = '~'.join(data)
        self.buffer ( "$@" + code + ";" + data + "$"  )
        self.buffer('')
    
    def parse ( self, string ):
        time.sleep(.1)
        str = string.decode("utf-16-le").strip()

        if self.first:
            if not str[0] == "$" and str[-1] == "$" :
                self.buffer ( "$@error;INVALID [first] QUERY: NO leading AND terminating $: '" 
                    + str + "' !$" )
                self.buffer ( "$@close;$" )
                self.client.flush = True
            elif  ( str[1:][:-1].find ( "@" ) > -1 or str[1:][:-1].find ( "~" ) > -1
                    or str[1:][:-1].find ( "$" ) > -1 ):
                self.buffer ( "$@error;INVALID NAME '"  + str[1:][:-1] + "' !$" )
                self.buffer ( "$@close;$" )
                self.client.flush = True
            elif str[1:][:-1] in self.client.server.clients.keys():
                self.buffer ( "$@error;NAME '"  + str + "' ALREADY IN USE !$" )
                self.buffer ( "$@close;$" )
                self.client.flush = True
            else:
                # name good !
                self.client.name = str[1:][:-1]
                self.client.server.add_client ( self.client.name, self.client )
                self.buffer ( "$@welcome_to_room;" + self.client.server.name + "$" )

            self.first = False
            return 

        if str[0] == '$' and str[-1] == '$' : #check if valid
            p1, p2 = str[1:][:-1].split(';', 1)
            if p1[0] == "@":
                #special
                if not p1[1:] == self.client.name:
                    self.buffer ( "$@error;'"  + me + "' IS NOT YOUR NAME !$" )
                else:
                    p2s = p2.split(';')
                    if len(p2s) > 1:
                        if '~' in p2s[1]: p2s[1] = p2s[1].split('~')
                    else: p2s.append('')
                    self.client.server.special ( self.client.name, p2s[0], p2s[1] )
            elif p1.find ( "~" ) > -1:
                #private
                me, thee = p1.split("~")
                if not me == self.client.name:
                    self.buffer ( "$@error;'"  + me + "' IS NOT YOUR NAME !$" )
                else:
                    self.client.server.private ( me, thee, p2 )
            else:
                if not p1 == self.client.name:
                    self.buffer ( "$@error;'"  + me + "' IS NOT YOUR NAME !$" )
                else:
                    self.client.server.message ( p1, p2 )
        
class ChatPlusProtocol0:
    """overworked protocol using UTF8
    messages are sepetated by a Record Seperator (\\x1e)"""

    def __init__ ( self, client ):
        self.client = client
        #self.first = True   # as the original protocol is always the first, we don't bother with names and all that
        self.old = u''

    def buffer ( self, msg ):
        self.client.buffer( msg.encode("utf-8") )

    def send_msg ( self, author, msg ):
        self.buffer ( "MESSAGE;%s;%s\x1e" % (author, msg)  )
        
    def send_private ( self, author, msg ):
        self.buffer ( "PRIVATE;%s;%s\x1e" % (author, msg)  )
        
    def send_special ( self, code, data ):
        #list sep is Unit Seperator, \x1f
        if isinstance( data, list ):
            data = '\x1f'.join(data)
        self.buffer ( "SPECIAL;%s;%s\x1e" % (code, data)  )
    
    def parse ( self, string ):
        time.sleep(.2)
        strs = string.decode("utf-8").split('\x1e')
        strs[0] = self.old + strs[0] # get possible left-over chars from last time
        self.old = strs.pop()        # the last is empty if there was a terminating RS, otherwise it's incomplete.

        for s in strs:
            prts = s.split(';')
            if prts[0] == 'MESSAGE':
                self.client.server.message(self.client.name, ';'.join(prts[1:]))
            elif prts[0] == 'PRIVATE':
                self.client.server.private(self.client.name, prts[1], ';'.join(prts[2:]))
            elif prts[0] == 'SPECIAL':
                data = ';'.join(prts[2:])
                if '\x1f' in data: data = data.split('\x1f')
                self.client.server.special(self.client.name, prts[1], data)
            else:
                self.send_special('error', 'illegal protocol action "%s"' % prts[0])


def main ( argv ):
    listenon = "0.0.0.0"
    port = 4056
    try:
        opts, args = getopt.getopt ( argv, "l:p:h", [ "listenon=", "port=", "help" ] )
    except getopt.GetoptError:
        usage()
        sys.exit( 2 )
    for opt,arg in opts:
        if opt in ("-h", "--help" ):
            usage ()
            sys.exit ()
        elif opt in ("-l", "--listenon" ):
            listenon = arg
            print "only listening on ", listenon
        elif opt in ("-p", "--port" ):
            port = int ( arg )
            print "listening on port ", port
    if len( args ) != 1:
        print "error: wrong number of arguments"
        usage()
        sys.exit ( 2 )
    else:
        roomname = args [0]
        c = ChatServer ( roomname, port, listenon )
        c.main_loop()
    
def usage ( ):
    print ( "usage: " + sys.argv [0]
        + ' [ --help | -h ] [ --listenon | -l IP-ADDRESS ] [ --port | -p PORT ] "ROOM NAME"' )



if __name__ == "__main__":
    try:
        main ( sys.argv[1:] )
    except KeyboardInterrupt:
        pass

#c = ChatServer ( "aha", 4056 )
#c.main_loop ()
