/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * File contains functions related to configuration of cdw:
 * - setting up options module afer launching cdw
 * - cleaning options module before closing cdw
 * - reading and writing configuration file
 * - displaying main options window visible to user
 *
 * Options are categorized and displayed on multiple tabs.
 *
 * 'struct conf config' is declared in this file.
 *
 * This file contains many wrapper functions - they are at the end.
 */


#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>

#include <ncursesw/ncurses.h>
#include <ncursesw/form.h>

#include <nl_types.h>
#include <libintl.h>
#include <errno.h>
#include "options.h"
#include "config.h"
#include "color.h"
#include "gettext.h"
#include "main.h"
#include "utils.h"
#include "cdw_widgets.h"


/* **************************** */
/* wrapper functions prototypes */
/* **************************** */
void init_default_config(void);
void write_config_file(FILE *conf_file, struct conf configuration);
void read_config_file(FILE *conf_file);

void put_ckbox(WINDOW *win, int y, int x);
void clear_main_win(void);

void first_page_build_fields(FIELD **fields, struct conf *tmp_config);
int first_page_driver(FORM *form_page, FIELD **fields);
void first_page_draw(void);

void second_page_build_fields(FIELD **fields, struct conf *tmp_config);
int second_page_driver(FORM *form_page, FIELD **fields);
void second_page_draw(void);

void first_page_tmp_save_changes(FIELD **fields, struct conf *tmp_config);
void second_page_tmp_save_changes(FIELD **fields, struct conf *tmp_config);

void copy_configuration(struct conf *dest, struct conf *src);
int options_set_paths(bool use_home_dir);


struct optwins {
	WINDOW *main;
} optwin;



/* Main application window */
extern WINDOW *mainwin;


/* project-wide variable holding current configuration of cdw */
struct conf config;

/* variable holding default, hardwired configuration of cdw */
static struct conf default_config;


/* set to true if something went wrong during module setup -
   cdw will work without reading/writing configuration to disk file */
static bool failsafe_mode = false;

/* homedir path is base path needed by some options values. It is also place
   where cdw config file is stored. If there are no errors this path is
   set to user home directory */
static char *homedir_path = NULL; /* full path to user home directory (without ending slash) */
static char *conf_file_full_path = NULL; /* full path to cdw configuration file */
static char *conf_file_name = "/.cdw.conf";


/* ****************************** */
/* some size and layout constants */
/* ****************************** */

/* width and height of main options window */
#define OPTIONS_WIDTH 70
#define OPTIONS_HEIGHT 20

/* number of form fields on first/second page
 * of options; ending field == NULL included */
#define OPTION_FIELDS_P1 39
#define OPTION_FIELDS_P2 9

/* constants defining layout of fields on form page;
 * they help to avoid magic numbers */
#define FIRST_COL_W 20        /* width of first column of options on options page */
#define FIRST_COL 1           /* starting column of first options column */
#define SECOND_COL_W 25                           /* width of second options column */
#define SECOND_COL ((OPTIONS_WIDTH / 2) + 2)      /* starting column of second options column */



/* names of tabs that group options - with shortcut keys in <> */
static char tab_one_name[30];
static char tab_two_name[30];
int tab_one_name_len = 0;
int tab_two_name_len = 0;




/**
 * Initialize options module
 *
 * Initialize options module - prepare some default option values and set paths
 * to base directory (should be user home directory) and to cdw configuration
 * file (will be in base directory).
 *
 * \returns 0
 */
int options_init(void)
{
	tab_one_name_len = strlen(_("<F1>Session"));
	strncpy(tab_one_name, _("<F1>Session"), tab_one_name_len);
	tab_two_name_len = strlen(_("<F2>Disc&Hardware"));
	strncpy(tab_two_name, _("<F2>Disc&Hardware"), tab_two_name_len);

	int rv = options_set_paths(true); /* true - use home dir */

	if (rv > 0) { /* paths set OK */
		/* initialize default config variable with default options values */
		init_default_config();
		/* Read cdw configuration file - it should be in default localization: user home dir;
		   overwrite default option values with values found in config file,
		   crate config file if it doesn't exist already */
		read_conf();
	} else { /* error situation; we can check what happened, but not in this release */
		dialogbox(_("Config file error"), _("Cannot find home directory and settings file. Will use temporary configuration."), DIALOG_OK);

		rv = options_set_paths(false); /* false - search for tmp dir */

		if (rv == -1) {
			/* user pressed escape, there will be no correct base
			   dir nor config file */
			options_failsafe_mode();
			/* initialize default config variable with default
			   options values */
			init_default_config();
			/* we do this instead of read_config() - default
			   configuration becomes current configuration */
			copy_configuration(&config, &default_config);
		} else {
			/* user entered some temp dir, use it as a base dir */
			/* initialize default config variable with default options values */
			init_default_config();
			/* Read cdw configuration file - it should be in
			   temporary localization; overwrite default option
			   values with values found in config file. If config
			   file does not exist in tmp location, it will be
			   created and filled with default values  */
			read_conf();
		}
	}

	return 0;
}




/**
 * Write current configuration to disk file
 *
 * Write content of config variable to disk file. This function only
 * opens (for writing) config file, (in future: does basic error checking)
 * calls function doing actual writing, and then closes config file.
 * If options module works in failsafe mode, this function silently skips
 * writing to file. Caller is responsible for informing user about
 * failsafe_mode being set to true (and about consequences of this fact).
 *
 * \returns CDW_FAILSAFE_MODE if failsafe mode is in effect and writing was skipped
 *          0 otherwise
 */
int write_conf(void)
{
	FILE *conf_file = NULL;

	if (failsafe_mode) { /* this might be redundant if caller checks
			      * for failsafe_mode, but just to be sure:
			      * silently skip writing */
		return CDW_FAILSAFE_MODE; /* emergency mode, don't work with filesystem */
	}

	conf_file = fopen(conf_file_full_path, "w");
	if ( conf_file == NULL ) {
		if (errno == EACCES) { /* conf file has no write permissions */
			dialogbox(_("Config file error"), _("An error occured when saving options. Please check config file permissions. Any changes will be lost after closing cdw."), DIALOG_OK);
			wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
			failsafe_mode = true;
		} else {
			dialogbox(_("Config file error"), _("Unknown error occured when saving options. Any changes will be lost after closing cdw."), DIALOG_OK);
			wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
			failsafe_mode = true;
		}
	} else {
		write_config_file(conf_file, config);
		fclose(conf_file);
	}

	return 0;
}




/**
 * Read config file, put config values into config variable
 *
 * Open config txt file, call function reading config file, close config
 * file. If config file does not exists, it is created and filled with
 * default values of options.
 * If options module works in failsafe mode, this function silently skips
 * reading from file. Caller is responsible for informing user about
 * failsafe_mode being set to true (and about consequences of this fact).
 *
 * \returns 0 if configuration was read from file without problems,
 *          1 if there was no configuration file, one has been created
 *                          and now we are using default configuration
 *          CDW_FAILSAFE_MODE if failsafe mode is in effect and reading was
 *                                skipped (using default configuration now)
 */
int read_conf(void)
{
	FILE *conf_file = NULL;

	if (failsafe_mode) {
		return CDW_FAILSAFE_MODE; /* emergency mode, don't work with filesystem */
	}

	/* fill fields of configuration variable with default values -
	 * just in case if conf file is missing or broken or incomplete */
	copy_configuration(&config, &default_config);

	conf_file = fopen(conf_file_full_path, "r");
	if (conf_file == NULL) {
		if (errno == ENOENT) { /* no such file, create one; */
			conf_file = fopen(conf_file_full_path, "w");
			if (conf_file == NULL) {
				dialogbox(_("Config file error"), _("Cannot create config file. Please check your home directory permissions."), DIALOG_OK);
				failsafe_mode = true;
				wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
				return CDW_FAILSAFE_MODE; /* no config file, using default configuration */
			} else { /* write default configuration to newly created config file */
				write_config_file(conf_file, default_config);
				fclose(conf_file);
				conf_file = NULL;
				/* dialogbox("Debug message", "Config file created from scratch.", DIALOG_OK);
				wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0)); */
				return 1; /* config file was created from scratch; using default configuration */
			}
		} else if (errno == EACCES) { /* file permission problems */
			dialogbox(_("Config file error"), _("Cannot open config file. Please check file permissions."), DIALOG_OK);
			failsafe_mode = true;
			wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
			return CDW_FAILSAFE_MODE; /* cannot read config file, using default configuration */
		}

	} else { /* config file already exists and it is now open to read */
		read_config_file(conf_file);
		fclose(conf_file);
		conf_file = NULL;
		return 0; /* configuration read from existing configuration file */
	}
}




/**
 * Draw first page of options
 *
 * Draw first page of options. This page contains basic and most often
 * changed options related to current session with cdw: writing speed,
 * mode, disc title, etc.
 * Any changed values will be saved in tmp_config.
 *
 * \param struct conf *tmp_config - temporary config variable, storing
 *               changes made on this page
 * \returns CDW_ESCAPE if user pressed ESCAPE
 *          F_(2) - if user pressed F_(2)
 *          F_(10) - if user pressed F_(10)
 */
static int page_one(struct conf *tmp_config)
{
	int ch, i;

	FIELD *fields[OPTION_FIELDS_P1]; /* last field == NULL is included */
	FORM *form_page[1];

	clear_main_win();

	/* build form fields represeting options and get their
	 * values from tmp_config */
	first_page_build_fields(fields, tmp_config);

	form_page[0] = new_form(fields);
	set_form_win(form_page[0], optwin.main);
	set_form_sub(form_page[0], derwin(optwin.main, OPTIONS_HEIGHT - 2, OPTIONS_WIDTH - 2, 1, 1));
	post_form(form_page[0]);

	/* draw additional stuff (checkboxes, tabs) in options window */
	first_page_draw();

	form_driver(form_page[0], REQ_END_LINE);
	wrefresh(optwin.main);

	/* main loop handling keys - moving cursor between
	 * fields and entering values */
	ch = first_page_driver(form_page[0], fields);

	/* save option values (possibly changed) to temporary variable;
	 * they will be kept there until user comes back to this page
	 * (then they will be displayed again, with changes
	 * made by user), or, after 'save options' request - key F_(10) -
	 * values from tmp_config will be copied to basic config variable */
	first_page_tmp_save_changes(fields, tmp_config);

	wrefresh(optwin.main);

	/* clean up - this page of options window will be erased */
	unpost_form(form_page[0]);
	free_form(form_page[0]);
	for (i = 0; i < OPTION_FIELDS_P1 - 1; i++) { /* -1: don't free last field that is NULL */
		free_field(fields[i]);
		fields[i] = NULL;
	}

	return ch; /* last key pressed */
}




/**
 * Draw second page of options
 *
 * Draw second page of options. This page contains options related to hardware
 * and will contain some options related to currently used disc.
 * Any changed values will be saved in tmp_config.
 *
 * \param struct conf *tmp_config - temporary config variable, storing
 *               changes made on this page
 * \returns CDW_ESCAPE if user pressed ESCAPE
 *          F_(1) - if user pressed F_(1)
 *          F_(10) - if user pressed F_(10)
 */
static int page_two(struct conf *tmp_config)
{
	int ch, i;

	FIELD *fields[OPTION_FIELDS_P2]; /* last field == NULL included */
	FORM *form_page[1];

	clear_main_win();

	/* build form fields represeting options and get their
	 * values from tmp_config */
	second_page_build_fields(fields, tmp_config);

	form_page[0] = new_form(fields);
	set_form_win(form_page[0], optwin.main);
	set_form_sub(form_page[0], derwin(optwin.main, OPTIONS_HEIGHT - 2, OPTIONS_WIDTH - 2, 1, 1));
	post_form(form_page[0]);

	/* draw additional stuff (checkboxes, tabs) in options window */
	second_page_draw();

	form_driver(form_page[0], REQ_END_LINE);
	wrefresh(optwin.main);

	/* main loop handling keys - moving cursor between
	 * fields and entering values */
	ch = second_page_driver(form_page[0], fields);

	/* save option values (possibly changed) to temporary variable;
	* they will be kept there until user comes back to this page
	* (then they will be displayed again, with changes
	* made by user), or, after 'save options' request - key F_(10) -
	* values from tmp_config will be copied to basic config variable */
	second_page_tmp_save_changes(fields, tmp_config);

	wrefresh(optwin.main);

	/* clean up - this page of options window will be erased */
	unpost_form(form_page[0]);
	free_form(form_page[0]);
	for (i = 0; i < OPTION_FIELDS_P2 - 1; i++) { /* -1: don't free last field that is NULL */
		free_field(fields[i]);
		fields[i] = NULL;
	}

	return ch; /* last key pressed */
}




/**
 * Show cdw options window
 *
 * Show options window with multiple tabs that group cdw options.
 * User can switch between them using F_(X) keys and save changed options
 * to config file or leave options window without saving changes. Current
 * options are read form disk file before being shown to the user,
 * and then are saved (if user demands it by pressing F_(10)) to disk file and
 * to global variable 'struct conf config' after closing options window.
 * Any changes made in options window will be discarded if user presses ESCAPE.
 *
 * Options module should be initilaized (using int options_init(void)) before
 * calling optins_window(), because it depends on defaul values being set.
 */
void options_window(void)
{
	int ch = 0, page = 0;

	/* this variable will store changes made _before_ pressing F_(10);
	 * these changes will be discarded if user pressed ESCAPE,
	 * or will be saved if user pressed F_(10) */
	struct conf tmp_config;

	/* create options window */
	optwin.main = newwin(OPTIONS_HEIGHT, OPTIONS_WIDTH,
			     (LINES - OPTIONS_HEIGHT) / 2, (COLS - OPTIONS_WIDTH) / 2);
	wbkgd(optwin.main, COLOR_PAIR(2));
	werase(optwin.main);
	keypad(optwin.main, TRUE);
	nice_box(optwin.main, _("CDW Options"), _("Press F10 to save config or ESC to cancel"));

	/* read configuration from disk file to config variable */
	read_conf();

	/* copy existing option values to keep consistency and to not
	 * overwrite valid options with some trash when saving options
	 * after key F_(10) was hit;
	 * now we will be using only tmp_config until user demands saving
	 * modified options */
	copy_configuration(&tmp_config, &config);

	/* initial state - show first page of options */
	ch = page_one(&tmp_config);
	page = 1;

	/* switch between options pages or close options window
	 * (with or without saving changed options) */
	while (ch != CDW_ESCAPE) {
		switch (ch) {
			case KEY_F(1):
				if (page != 1) {
					page = 1;
					ch = page_one(&tmp_config);
				}
			break;
			case KEY_F(2):
				if (page != 2) {
					page = 2;
					ch = page_two(&tmp_config);
				}
			break;
			case KEY_F(10): /* save modified configuration and exit */
				/* copy options values saved in tmp_config
				 * during work with options tabs to primary
				 * configuration variable - config */
				copy_configuration(&config, &tmp_config);
				if (failsafe_mode) { /* cannot work with file system because of previous errors */
					dialogbox(_("Config file error"), _("Cannot write to config file. Configuration will not be saved to file."), DIALOG_OK);
					/* cannot save changed values to file,
					 * but they are saved to config
					 * variable and will be used only
					 * in this session. */
				} else {
					/* write current configuration to disk file */
					write_conf();
				}
				delwin(optwin.main);
				optwin.main = NULL;
				return;
		}
	}

	if (optwin.main != NULL) { /* very unlikey */
		delwin(optwin.main);
		optwin.main = NULL;
	}

	return;
}




/**
 * Set path to cdw config file and to some start dir
 *
 * Set (default) full paths to some base directory (e.g. /home/username) and
 * to the config file (i.e. /base_dir/.cdw.conf). When called with
 * use_home_dir=true the function tries to set base directory to user home
 * directory. If called with use_home_dir=false it asks user for other dir.
 * The second situation may occur when first call to the function (with
 * use_home_dir=true) failed for some reason, but we need to get base
 * directory and some location for configuration file.
 * User home dir path does not contain ending slash. If this function cannot
 * get correct and valid base directory, it does not try to initialize
 * config file path and both paths are set to NULL.
 *
 * Base directory variable is char *homedir_path
 * Full config file path is char *conf_file_full_path
 * Both variables are global and static in this file.
 *
 * \param bool use_home_dir - if=true this function sets base path to user home dir
 *                            if=false this function asks user for other dir path
 *
 * \return length of full config file path if success, value < 0 if failure
 */
int options_set_paths(bool use_home_dir)
{
	int len1 = 0, len2 = 0;

	/* len1 - length of base dir path, ending '\0' not included */
	if (use_home_dir) { /* we assume that HOME exists */
		len1 = get_home_dir(&homedir_path);
	} else { /* calling this function with use_home_dir=true failed
		  * for some reason, thy to get other base dir */
		do {
			/* user can press ESC to skip this step  - then len1 = -1 */
			len1 = get_tmp_dir(&homedir_path);
		} while (len1 == CDW_FILE_INVALID);
	}

	if (len1 < 0) { /* Getting dir failed - check len1 for more details.
			 * Getting dir failed, so there is no way
			 * to get correct full config file path. */
		homedir_path = NULL;
		conf_file_full_path = NULL;
		return len1;
	}

	len2 = strlen(conf_file_name);
	conf_file_full_path = (char *) malloc(len1 + len2 + 1);
	if (conf_file_full_path == NULL) {
		free(homedir_path);
		homedir_path = NULL;
		return CDW_MEM_ERROR;
	}

	/* FIXME - to be replaced with concat() */
	strncpy(conf_file_full_path, homedir_path, len1); // without ending '\0' at len1 char
	strncpy(conf_file_full_path + len1, conf_file_name, len2);
	strncpy(conf_file_full_path + len1 + len2, "\0", 1);

	return strlen(conf_file_full_path);
}




/**
 * Deallocate 'options' module resources
 *
 * Deallocate all 'options' module resources that were allocated during
 * program run and were not freed before.
 */
void options_clean(void)
{
	/* some char * strings allocated in options_set_paths() */

	if (conf_file_full_path != NULL) {
		free(conf_file_full_path);
		conf_file_full_path = NULL;
	}

	if (homedir_path != NULL) {
		free(homedir_path);
		homedir_path = NULL;
	}

	return;
}




/* ***************************** */
/* ****** simple wrappers ****** */
/* ***************************** */


/**
 * Set default values in data structure holding current cdw configuration
 *
 * Set default values of cdw configuration variable. This function can be
 * called to make sure that some default values in default configuration
 * variable  exist - just in case if config file does not exists or is
 * broken or incomplete.
 *
 * This function depends on homedir_path to be set. If it is not set to some
 * valid directory (which is indicated by failsafe_mode set to true), last
 * attempt is made to set base dir to sane value: is is set to "/tmp".
 *
 * Note that this function does not write default configuration to disc file.
 */
void init_default_config(void)
{
	strcpy(default_config.scsi,"0,0,0");
	strcpy(default_config.cdrwdevice,"/dev/scd0");
	strcpy(default_config.tempdir,"/tmp/image.iso");
	strcpy(default_config.speed,"52");
	strcpy(default_config.blank,"fast");
	strcpy(default_config.logfile,"/tmp/cdw.log");
	strcpy(default_config.eject,"0");
	strcpy(default_config.dao,"0");
	strcpy(default_config.joliet,"1");
	strcpy(default_config.rockridge,"1");
	strcpy(default_config.dummy,"0");
	strcpy(default_config.volumeid,"cdrom");
	strcpy(default_config.cdrom,"/dev/cdrom");
	strcpy(default_config.usefulRR,"1");
	strcpy(default_config.showlog,"1");
	strcpy(default_config.showvol,"0");
	strcpy(default_config.pad,"1");
	strcpy(default_config.fromimage,"1");
	strcpy(default_config.multi,"0");
	strcpy(default_config.burnproof,"1");
	strcpy(default_config.mountpoint,"/cdrom");
	sprintf(default_config.audiodir,"%s/audio", failsafe_mode ? "/tmp" : homedir_path);
	strcpy(default_config.stereo,"1");
	strcpy(default_config.bitsperchn,"16");
	strcpy(default_config.echosound,"0");
	strcpy(default_config.encode,"0");
	strcpy(default_config.lame,"0");
	strcpy(default_config.highq,"1");
	strcpy(default_config.bitrate,"192");
	strcpy(default_config.cdbhost,"localhost");
	strcpy(default_config.cdbuser,"root");
	sprintf(default_config.sqlite_file,"%s/cdw.db", failsafe_mode ? "/tmp" : homedir_path);
	sprintf(default_config.sqlite_file,"");
	default_config.cdsize=657;
	default_config.user_cdsize=657;
	memset(default_config.other, '\0', 255);

	return;
}




/**
 * Write values from given configuration variable to cdw config file
 *
 * Write values from given configuration variable _config to disk
 * configuration file conf_file (should be opened for writing before passing
 * to function). This is low level function that does actual writing to file.
 * The file should be opened for writing before calling this function. This
 * function will not check for validity of file nor configuration variable.
 *
 * This is the only function that writes to configuration file. If you
 * would like to change layout of conifg file - you can do it here.
 *
 * \param FILE *conf_file - config file opened for writing
 * \param struct conf _config - configuration variable with option values to be written to file
 */
void write_config_file(FILE *conf_file, struct conf _config)
{
	fprintf(conf_file, "################################\n");
	fprintf(conf_file, "#    cdw configuration file    #\n");
	fprintf(conf_file, "################################\n");


	fprintf(conf_file, "\n\n\n");
	fprintf(conf_file, "########## session tab #########\n");
	fprintf(conf_file, "speed=%s\n", _config.speed);
	fprintf(conf_file, "eject=%s\n", _config.eject);
	fprintf(conf_file, "volumeid=%s\n", _config.volumeid);
	fprintf(conf_file, "showvol=%s\n", _config.showvol);
	fprintf(conf_file, "multi=%s\n", _config.multi);
	fprintf(conf_file, "WriteFromImage=%s\n", _config.fromimage); /* FIXME - looks like inconsistency in variable name */
	fprintf(conf_file, "fromimage=%s\n", _config.fromimage);


	fprintf(conf_file, "dao=%s\n", _config.dao);
	fprintf(conf_file, "burnproof=%s\n", _config.burnproof);
	fprintf(conf_file, "rockridge=%s\n", _config.rockridge);
	fprintf(conf_file, "joliet=%s\n", _config.joliet);
	fprintf(conf_file, "usefulRR=%s\n", _config.usefulRR);
	fprintf(conf_file, "pad=%s\n", _config.pad);
	fprintf(conf_file, "dummy=%s\n", _config.dummy);
	fprintf(conf_file, "blank=%s\n", _config.blank);


	fprintf(conf_file, "showlog=%s\n", _config.showlog);
	fprintf(conf_file, "logfile=%s\n", _config.logfile);
	fprintf(conf_file, "tempdir=%s\n", _config.tempdir);
	fprintf(conf_file, "bootimg=%s\n", _config.bootimg);
	fprintf(conf_file, "other=%s\n", _config.other);


	fprintf(conf_file, "\n\n\n");
	fprintf(conf_file, "###### disc/hardware tab #######\n");
	fprintf(conf_file, "scsi-dev=%s\n", _config.scsi); /* FIXME - looks like inconsistency in variable name */
	fprintf(conf_file, "scsi=%s\n", _config.scsi);
	fprintf(conf_file, "cdrwdevice=%s\n", _config.cdrwdevice);
	fprintf(conf_file, "mountpoint=%s\n", _config.mountpoint);
	fprintf(conf_file, "cdrom=%s\n", _config.cdrom);
	fprintf(conf_file, "cdsize=%d\n", _config.cdsize);
	fprintf(conf_file, "user_cdsize=%d\n", _config.user_cdsize);


	fprintf(conf_file, "\n\n\n");
	fprintf(conf_file, "########## audio tab ###########\n");
	fprintf(conf_file, "autodic=%s\n", _config.autodic);
	fprintf(conf_file, "audiodir=%s\n", _config.audiodir);
	fprintf(conf_file, "stereo=%s\n", _config.stereo);
	fprintf(conf_file, "bitsperchn=%s\n", _config.bitsperchn);
	fprintf(conf_file, "echosound=%s\n", _config.echosound);
	fprintf(conf_file, "encode=%s\n", _config.encode);
	fprintf(conf_file, "lame=%s\n", _config.lame);
	fprintf(conf_file, "highq=%s\n", _config.highq);
	fprintf(conf_file, "bitrate=%s\n", _config.bitrate);


	fprintf(conf_file, "\n\n\n");
	fprintf(conf_file, "########### CDDB tab ###########\n");
	fprintf(conf_file, "cdbhost=%s\n", _config.cdbhost);
	fprintf(conf_file, "cdbuser=%s\n", _config.cdbuser);
	fprintf(conf_file, "sqlite_file=%s\n", _config.sqlite_file);

	return;
}




/**
 * Read config file, store configuration in data structure
 *
 * Read config file line by line and store values found in that file into data
 * structure. This is low level function that deals with disk file. It should
 * be called by other function that will open file with correct permissions
 * before passing it to this function.
 * Config data structure is global.
 *
 * \param FILE *conf_file - config file opened for reading
 */
void read_config_file(FILE *conf_file)
{
	char option_buffer[255];
	char opt[41], prop[256];

	while (NULL != fgets(option_buffer, 255, conf_file)) { // read config file line by line
		if ( sscanf(option_buffer, "%30[^=]=%[^\n]", opt, prop) == 2 ) {
			if ( (strcmp(opt,"scsi-dev")) == 0 )
				strcpy(config.scsi, prop);
			if ( (strcmp(opt,"scsi")) == 0 )
				strcpy(config.scsi, prop); /* repeated because of "option name - option variable name" inconsistency */
			if ( (strcmp(opt,"cdrwdevice")) == 0 )
				strcpy(config.cdrwdevice, prop);
			if ( (strcmp(opt,"tempdir")) == 0 )
				strcpy(config.tempdir, prop);
			if ( (strcmp(opt,"speed")) == 0 )
				strcpy(config.speed, prop);
			if ( (strcmp(opt,"other")) == 0 )
				strcpy(config.other, prop);
			if ( (strcmp(opt,"blank")) == 0 )
				strcpy(config.blank, prop);
			if ( (strcmp(opt,"logfile")) == 0 )
				strcpy(config.logfile, prop);
			if ( (strcmp(opt,"eject")) == 0 )
				strcpy(config.eject, prop);
			if ( (strcmp(opt,"dao")) == 0 )
				strcpy(config.dao, prop);
			if ( (strcmp(opt,"joliet")) == 0 )
				strcpy(config.joliet, prop);
			if ( (strcmp(opt,"rockridge")) == 0 )
				strcpy(config.rockridge, prop);
			if ( (strcmp(opt,"dummy")) == 0 )
				strcpy(config.dummy, prop);
			if ( (strcmp(opt,"volumeid")) == 0 )
				strcpy(config.volumeid, prop);
			if ( (strcmp(opt,"cdrom")) == 0 )
				strcpy(config.cdrom, prop);
			if ( (strcmp(opt,"usefulRR")) == 0 )
				strcpy(config.usefulRR, prop);
			if ( (strcmp(opt,"showlog")) == 0 )
				strcpy(config.showlog, prop);
			if ( (strcmp(opt,"showvol")) == 0 )
				strcpy(config.showvol, prop);
			if ( (strcmp(opt,"cdsize")) == 0 )
				config.cdsize=atoi(prop);
			if ( (strcmp(opt,"pad")) == 0 )
				strcpy(config.pad, prop);
			if ( (strcmp(opt,"WriteFromImage")) == 0 )
				strcpy(config.fromimage, prop);
			if ( (strcmp(opt,"fromimage")) == 0 ) /* repeated because of "option name - option variable name" inconsistency */
				strcpy(config.fromimage, prop);
			if ( (strcmp(opt,"multi")) == 0 )
				strcpy(config.multi, prop);
			if ( (strcmp(opt,"burnproof")) == 0 )
				strcpy(config.burnproof, prop);
			if ( (strcmp(opt,"mountpoint")) == 0 )
				strcpy(config.mountpoint, prop);
			if ( (strcmp(opt,"autodic")) == 0 )
				strcpy(config.autodic, prop);
			if ( (strcmp(opt,"audiodir")) == 0 )
				strcpy(config.audiodir, prop);
			if ( (strcmp(opt,"stereo")) == 0 )
				strcpy(config.stereo, prop);
			if ( (strcmp(opt,"bitsperchn")) == 0 )
				strcpy(config.bitsperchn, prop);
			if ( (strcmp(opt,"echosound")) == 0 )
				strcpy(config.echosound, prop);
			if ( (strcmp(opt,"encode")) == 0 )
				strcpy(config.encode, prop);
			if ( (strcmp(opt,"lame")) == 0 )
				strcpy(config.lame, prop);
			if ( (strcmp(opt,"highq")) == 0 )
				strcpy(config.highq, prop);
			if ( (strcmp(opt,"bitrate")) == 0 )
				strcpy(config.bitrate, prop);
			if ( (strcmp(opt,"cdbhost")) == 0 )
				strcpy(config.cdbhost, prop);
			if ( (strcmp(opt,"cdbuser")) == 0 )
				strcpy(config.cdbuser, prop);
			if ( (strcmp(opt,"sqlite_file")) == 0 )
				strcpy(config.sqlite_file, prop);
			if ( (strcmp(opt,"bootimg")) == 0 )
				strcpy(config.bootimg, prop);
			if ( (strcmp(opt,"user_cdsize")) == 0 )
				config.user_cdsize=atoi(prop);
		} /* if */
	} /* while */
	return;
}




/**
 * Draw two square brackets around 1x1 input field
 *
 * Draw two square brackets around 1x1 input field. The field will behave
 * as checkbox, and drawing brackets around will make it look like this:
 *              option_name [x]
 * Second and third arguments are coordinates of input field, so brackets
 * should be put before and after it - make sure that there is enough space
 * for them on your window.
 *
 * \param WINDOW *win - window to draw on
 * \param int y - row in which input field is located
 * \param int x - column in which input field is located
 */
void put_ckbox(WINDOW *win, int y, int x)
{
	mvwprintw(win, y, x - 1, "[");
	mvwprintw(win, y, x + 1, "]");

	return;
}




#if 0 /* obsolete, replaced by void write_config_file(FILE *conf_file, struct conf _config) */
/*
 * Write content of config variable to text file
 *
 * Write content of config variable to text file. This is low level
 * function that does actual writing to file. The file should be opened for
 * writing before calling this function. This function will not check for
 * validity of file nor configuration variable.
 *
 * \param FILE *conf_file - opened configuration file
 */
void write_current_configuration(FILE *conf_file)
{
	fprintf(conf_file,"################################\n");
	fprintf(conf_file,"#    cdw configuration file    #\n");
	fprintf(conf_file,"################################\n");
	fprintf(conf_file, "\n");


	// Settings visible to the user - page one
	fprintf(conf_file, "\n#   \"Current session\" settings\n");
	// most basic options
	fprintf(conf_file,"volumeid=%s\n",config.volumeid);
	fprintf(conf_file,"speed=%s\n",config.speed);
	fprintf(conf_file,"multi=%s\n",config.multi);
	fprintf(conf_file,"WriteFromImage=%s\n",config.fromimage);
	fprintf(conf_file,"eject=%s\n",config.eject);
	fprintf(conf_file,"burnproof=%s\n",config.burnproof);
	// most basic options - 2
	fprintf(conf_file,"dao=%s\n",config.dao);
	fprintf(conf_file,"joliet=%s\n",config.joliet);
	fprintf(conf_file,"rockridge=%s\n",config.rockridge);
	fprintf(conf_file,"usefulRR=%s\n",config.usefulRR);
	// basic options - other
	fprintf(conf_file,"showvol=%s\n",config.showvol);
	fprintf(conf_file,"dummy=%s\n",config.dummy);
	fprintf(conf_file,"tempdir=%s\n",config.tempdir);
	fprintf(conf_file,"other=%s\n",config.other);
	fprintf(conf_file,"blank=%s\n",config.blank);
	fprintf(conf_file,"showlog=%s\n",config.showlog);
	fprintf(conf_file,"bootimg=%s\n",config.bootimg);
	fprintf(conf_file,"pad=%s\n",config.pad);

	fprintf(conf_file, "\n#   \"Hardware/CD\" settings\n");
	// cd drive setting
	fprintf(conf_file,"scsi-dev=%s\n",config.scsi);
	fprintf(conf_file,"cdrwdevice=%s\n",config.cdrwdevice);
	fprintf(conf_file,"cdrom=%s\n",config.cdrom);
	fprintf(conf_file,"mountpoint=%s\n",config.mountpoint);
	// disc settings
	fprintf(conf_file,"user_cdsize=%d\n",config.user_cdsize);
	fprintf(conf_file,"cdsize=%d\n",config.cdsize);

	fprintf(conf_file, "\n#   \"CDDB\" settings\n");
	fprintf(conf_file,"cdbhost=%s\n",config.cdbhost);
	fprintf(conf_file,"cdbuser=%s\n",config.cdbuser);
	fprintf(conf_file,"sqlite_file=%s\n",config.sqlite_file);

	fprintf(conf_file, "\n#   \"Audio\" settings\n");
	fprintf(conf_file,"audiodir=%s\n",config.audiodir);
	fprintf(conf_file,"stereo=%s\n",config.stereo);
	fprintf(conf_file,"bitsperchn=%s\n",config.bitsperchn);
	fprintf(conf_file,"echosound=%s\n",config.echosound);
	fprintf(conf_file,"encode=%s\n",config.encode);
	fprintf(conf_file,"lame=%s\n",config.lame);
	fprintf(conf_file,"highq=%s\n",config.highq);
	fprintf(conf_file,"bitrate=%s\n",config.bitrate);


	// haven't decided yet :)
	fprintf(conf_file,"logfile=%s\n",config.logfile);
	fprintf(conf_file,"autodic=%s\n",config.autodic);

	return;
}

#endif




/**
 * Redraw options window
 */
void clear_main_win(void)
{
	WINDOW *subwin = derwin(optwin.main, OPTIONS_HEIGHT - 2, OPTIONS_WIDTH - 2, 1, 1);
	werase(subwin);
	wrefresh(subwin);
}




/**
 * Create fields that are shown on first page of options
 *
 * Create fields that are shown on first page of options, including
 * initialization of size and position (all OPTION_FIELDS_P1 -1 of them, not
 * including last one which is of course NULL), set their types, options and
 * appearance.
 *
 * \param FIELD **fields - pointer to empty FILEDs table, to be filled with
 *                         alocated fields
 * \param struct conf *tmp_config - variable holding current configuration, and
 *                         temporarily storing modified values of options
 */
void first_page_build_fields(FIELD **fields, struct conf *tmp_config)
{
	int i = 0;

	/* put fields in  pairs: label field :: entry field */

	/* 1st row */
	fields[0] = new_field(1, FIRST_COL_W, 0, FIRST_COL, 0, 0);
	set_field_buffer(fields[0], 0, _("Writing speed"));

	fields[1] = new_field(1, 4, 0, FIRST_COL_W + 3, 0, 0);
	set_field_buffer(fields[1], 0, tmp_config->speed);
	set_field_type(fields[1], TYPE_REGEXP, "[0-9]{1,2}");
	field_opts_off(fields[1], O_AUTOSKIP);


	fields[2]=new_field(1, SECOND_COL_W, 0, SECOND_COL, 0, 0);
	set_field_buffer(fields[2], 0, _("Eject when done"));

	fields[3]=new_field(1, 1, 0, SECOND_COL + SECOND_COL_W + 2, 0, 0);
	set_field_buffer(fields[3], 0, (!strncmp(tmp_config->eject, "1", 1)?"X":"_"));
	set_field_type(fields[3], TYPE_REGEXP, "^(X|_)$");
	field_opts_off(fields[3], O_AUTOSKIP);



	/* 2nd row */
	fields[4]=new_field(1, FIRST_COL_W, 1, FIRST_COL, 0, 0);
	set_field_buffer(fields[4], 0, _("Default volume ID"));

	fields[5]=new_field(1, 18, 1, FIRST_COL_W + 3, 0, 0);
	set_field_buffer(fields[5], 0, tmp_config->volumeid);


	fields[6]=new_field(1, SECOND_COL_W, 1, SECOND_COL, 0, 0);
	set_field_buffer(fields[6], 0, _("Ask for volume ID"));

	fields[7]=new_field(1, 1, 1, SECOND_COL + SECOND_COL_W + 2, 0, 0);
	set_field_buffer(fields[7], 0, (!strncmp(tmp_config->showvol, "1", 1)?"X":"_"));
	set_field_type(fields[7], TYPE_REGEXP, "^(X|_)$");
	field_opts_off(fields[7], O_AUTOSKIP);



	/* 3rd row */
	fields[8]=new_field(1, FIRST_COL_W, 2, FIRST_COL, 0, 0);
	set_field_buffer(fields[8], 0, _("Write CD from image"));

	fields[9]=new_field(1, 1, 2, FIRST_COL_W + 3, 0, 0);
	set_field_buffer(fields[9], 0, (!strncmp(tmp_config->fromimage, "1", 1)?"X":"_"));
	set_field_type(fields[9], TYPE_REGEXP, "^(X|_)$");
	field_opts_off(fields[9], O_AUTOSKIP);


	fields[10]=new_field(1, SECOND_COL_W, 2, SECOND_COL, 0, 0);
	set_field_buffer(fields[10], 0, _("Multi-session"));

	fields[11]=new_field(1, 1, 2, SECOND_COL + SECOND_COL_W + 2, 0, 0);
	set_field_buffer(fields[11], 0, (!strncmp(tmp_config->multi, "1", 1)?"X":"_"));
	set_field_type(fields[11], TYPE_REGEXP, "^(X|_)$");
	field_opts_off(fields[11], O_AUTOSKIP);


	/* 4th row */
	fields[12]=new_field(1, FIRST_COL_W, 3, FIRST_COL, 0, 0);
	set_field_buffer(fields[12], 0, _("Default image file"));

	fields[13]=new_field(1, 30, 3, FIRST_COL_W + 3, 0, 0);
	set_field_buffer(fields[13], 0, tmp_config->tempdir);
	field_opts_off(fields[1], O_AUTOSKIP);


	/* 5th row */
	fields[14]=new_field(1, FIRST_COL_W, 5, FIRST_COL, 0, 0);
	set_field_buffer(fields[14], 0, _("Disk at once"));

	fields[15]=new_field(1, 1, 5, FIRST_COL_W + 3, 0, 0);
	set_field_buffer(fields[15], 0, (!strncmp(tmp_config->dao, "1", 1)?"X":"_"));
	set_field_type(fields[15], TYPE_REGEXP, "^(X|_)$");
	field_opts_off(fields[15], O_AUTOSKIP);


	fields[16]=new_field(1, SECOND_COL_W, 5, SECOND_COL, 0, 0);
	set_field_buffer(fields[16], 0, _("Burnproof"));

	fields[17]=new_field(1, 1, 5, SECOND_COL + SECOND_COL_W + 2, 0, 0);
	set_field_buffer(fields[17], 0, (!strncmp(tmp_config->burnproof, "1", 1)?"X":"_"));
	set_field_type(fields[17], TYPE_REGEXP, "^(X|_)$");
	field_opts_off(fields[17], O_AUTOSKIP);


	/* 6th row */
	fields[18]=new_field(1, FIRST_COL_W, 6, FIRST_COL, 0, 0);
	set_field_buffer(fields[18], 0, _("Rock Ridge"));

	fields[19]=new_field(1, 1, 6, FIRST_COL_W + 3, 0, 0);
	set_field_buffer(fields[19], 0, (!strncmp(tmp_config->rockridge, "1", 1)?"X":"_"));
	set_field_type(fields[19], TYPE_REGEXP, "^(X|_)$");
	field_opts_off(fields[19], O_AUTOSKIP);


	fields[20]=new_field(1, SECOND_COL_W, 6, SECOND_COL, 0, 0);
	set_field_buffer(fields[20], 0, _("Joliet information"));

	fields[21]=new_field(1, 1, 6, SECOND_COL + SECOND_COL_W + 2, 0, 0);
	set_field_buffer(fields[21], 0, (!strncmp(tmp_config->joliet, "1", 1)?"X":"_"));
	set_field_type(fields[21], TYPE_REGEXP, "^(X|_)$");
	field_opts_off(fields[21], O_AUTOSKIP);


	/* 7th row */
	fields[22]=new_field(1, FIRST_COL_W, 7, FIRST_COL, 0, 0);
	set_field_buffer(fields[22], 0, _("Useful RR attributes"));

	fields[23]=new_field(1, 1, 7, FIRST_COL_W + 3, 0, 0);
	set_field_buffer(fields[23], 0, (!strncmp(tmp_config->usefulRR, "1", 1)?"X":"_"));
	set_field_type(fields[23], TYPE_REGEXP, "^(X|_)$");
	field_opts_off(fields[23], O_AUTOSKIP);


	fields[24]=new_field(1, SECOND_COL_W, 7, SECOND_COL, 0, 0);
	set_field_buffer(fields[24], 0, _("Pad"));

	fields[25]=new_field(1, 1, 7, SECOND_COL + SECOND_COL_W + 2, 0, 0);
	set_field_buffer(fields[25], 0, (!strncmp(tmp_config->pad, "1", 1)?"X":"_"));
	set_field_type(fields[25], TYPE_REGEXP, "^(X|_)$");
	field_opts_off(fields[25], O_AUTOSKIP);


	/* 8th row */
	fields[26]=new_field(1, FIRST_COL_W, 8, FIRST_COL, 0, 0);
	set_field_buffer(fields[26], 0, _("Dummy write"));

	fields[27]=new_field(1, 1, 8, FIRST_COL_W + 3, 0, 0);
	set_field_buffer(fields[27], 0, (!strncmp(tmp_config->dummy, "1", 1)?"X":"_"));
	set_field_type(fields[27], TYPE_REGEXP, "^(X|_)$");
	field_opts_off(fields[27], O_AUTOSKIP);


	fields[28]=new_field(1, SECOND_COL_W, 8, SECOND_COL, 0, 0);
	set_field_buffer(fields[28], 0, _("Blank fast"));

	fields[29]=new_field(1, 1, 8, SECOND_COL + SECOND_COL_W + 2, 0, 0);
	set_field_buffer(fields[29], 0, (!strncmp(tmp_config->blank, "fast", 4)?"X":"_"));
	set_field_type(fields[29], TYPE_REGEXP, "^(X|_)$");
	field_opts_off(fields[29], O_AUTOSKIP);



	/* 9th row */
	fields[30]=new_field(1, FIRST_COL_W, 10, FIRST_COL, 0, 0);
	set_field_buffer(fields[30], 0, _("Show log after write"));

	fields[31]=new_field(1, 1, 10, FIRST_COL_W + 3, 0, 0);
	set_field_buffer(fields[31], 0, (!strncmp(tmp_config->showlog, "1", 1)?"X":"_"));
	set_field_type(fields[31], TYPE_REGEXP, "^(X|_)$");
	field_opts_off(fields[31], O_AUTOSKIP);



	/* 10th row */
	fields[32]=new_field(1, FIRST_COL_W, 11, FIRST_COL, 0, 0);
	set_field_buffer(fields[32], 0, _("Log file"));

	fields[33]=new_field(1, 30, 11, FIRST_COL_W + 3, 0, 0);
	set_field_buffer(fields[33], 0, tmp_config->logfile);
	field_opts_off(fields[33], O_AUTOSKIP);



	/* 11th row */
	fields[34]=new_field(1, FIRST_COL_W, 12, FIRST_COL, 0, 0);
	set_field_buffer(fields[34], 0, _("Boot image"));

	fields[35]=new_field(1, 30, 12, FIRST_COL_W + 3, 0, 0);
	set_field_buffer(fields[35], 0, tmp_config->bootimg);



	/* 12th row */
	fields[36]=new_field(1, FIRST_COL_W, 13, FIRST_COL, 0, 0);
	set_field_buffer(fields[36], 0, _("Other options"));

	fields[37]=new_field(1, 30, 13, FIRST_COL_W + 3, 0, 0);
	set_field_buffer(fields[37], 0, tmp_config->other);


	fields[OPTION_FIELDS_P1 - 1] = NULL;


	/* labels appearance */
	for (i = 0; i <= OPTION_FIELDS_P1 - 2; i += 2) {
		set_field_just(fields[i], JUSTIFY_RIGHT);
		field_opts_off(fields[i], O_ACTIVE);
		set_field_back(fields[i], COLOR_PAIR(2));
		set_field_fore(fields[i], COLOR_PAIR(2));
		set_field_back(fields[i], COLOR_PAIR(2));
		set_field_fore(fields[i], COLOR_PAIR(2));
	}

	/* input areas appearance */
	for (i = 1; i < OPTION_FIELDS_P1 - 1; i += 2) {
		set_field_just(fields[i], JUSTIFY_LEFT);
		set_field_back(fields[i], COLOR_PAIR(6));
		set_field_fore(fields[i], COLOR_PAIR(6));
		set_field_back(fields[i], COLOR_PAIR(6));
		set_field_fore(fields[i], COLOR_PAIR(6));
	}

	return;
}




/**
 * Handle keys while first page of options window is displayed
 *
 * Handle keys while first page of options window is displayed. Function
 * redraws options window to show updated values of options.
 * Function returns after user has hit F_2, F_10 or ESCAPE.
 *
 * \param FORM **form_page - current page of multi-page form
 * \param FIELD **fields - fields available on current page
 *
 * \return CDW_ESCAPE - exit otions window, no changes should be saved,
 *         KEY_F(10) - exit options window, changes should be permanently saved
 *         KEY_F(2) - change options tab, changes should be temporarly saved
 */
int first_page_driver(FORM *form_page, FIELD **fields)
{
	int ch;

	/* process keys in loop */
	while ( (ch = wgetch(optwin.main)) != CDW_ESCAPE ) {
		int chkbx = 0; /* flag - are we in checkbox field? */
		switch (ch) {
			case KEY_HOME:
				form_driver(form_page, REQ_BEG_LINE);
				break;
			case KEY_END:
				form_driver(form_page, REQ_END_LINE);
				break;
			case KEY_LEFT:
				form_driver(form_page, REQ_PREV_CHAR);
				break;
			case KEY_RIGHT:
				form_driver(form_page, REQ_NEXT_CHAR);
				break;
			case KEY_DOWN:
			case '\t':
				form_driver(form_page, REQ_NEXT_FIELD);
				form_driver(form_page, REQ_END_LINE);
				break;
			case KEY_UP:
			case KEY_BTAB:
				form_driver(form_page, REQ_PREV_FIELD);
				form_driver(form_page, REQ_END_LINE);
				break;
			case KEY_BACKSPACE:
				form_driver(form_page, REQ_DEL_PREV);
				break;
			case KEY_DC:
				form_driver(form_page, REQ_DEL_CHAR);
				break;
			case KEY_F(2):
			case KEY_F(10):
				/* leave keys handling loop */
				form_driver(form_page, REQ_VALIDATION); /* we DO check return value :) */
				return ch;
			default:
				if (ch == ' ') { /* check if user wants to toggle checkbox */
					FIELD *cur;
					int fld_index;
					cur = current_field(form_page);
					fld_index = field_index(cur);
					switch (fld_index) { /* if we are in checkbox then toggle checkbox */
						/* checkboxes */
						case 3:
						case 7:
						case 9:
						case 11:
						// case 13:

						case 15:
						case 17:
						case 19:
						case 21:
						case 23:
						case 25:
						case 27:
						case 29:
						case 31:
							/* toggle checkbox state */
							ch = (!strncmp(field_buffer(cur, 0), "X", 1) ? '_' : 'X');

							/* "Multisession" and "write from image" cannot be checked at the same time */
							if ((fld_index == 11) && (!strcmp(field_buffer(*(fields+11), 0), "_"))) {
								set_field_buffer(*(fields+9), 0, "_");
							}
							if ((fld_index == 9) && (!strcmp(field_buffer(*(fields+9), 0), "_"))) {
								set_field_buffer(*(fields+11), 0, "_");
							}
							wrefresh(optwin.main);
							chkbx = 1;
							break;
					} /* switch */
				} /* if */
				form_driver(form_page, ch);

				/* this is to fix following wrong behaviour:
				* go to checkbox, press space, checkbox is
				* checked, press space again, checkbox remains
				* checked */
				if (chkbx) {
					form_driver(form_page, REQ_NEXT_FIELD);
					form_driver(form_page, REQ_PREV_FIELD);
					form_driver(form_page, REQ_END_LINE);
					chkbx = 0;
				}
				break;
			wrefresh(optwin.main);
		} /* switch */
	}

	return ch; /* CDW_ESCAPE */
}




/**
 * Handle keys while second page of options window is displayed
 *
 * Handle keys while second page of options window is displayed. Function
 * redraws options window to show updated values of options.
 * Function returns after user hits F_1, F_10 or ESCAPE.
 *
 * \param FORM **form_page - current page of multi-page form
 * \param FIELD **fields - fields available on current page
 *
 * \return CDW_ESCAPE - exit options window, no changes should be saved,
 *         KEY_F(10) - exit options window, changes should be permanently saved
 *         KEY_F(1) - change options tab, changes should be temporarly saved
 */
int second_page_driver(FORM *form_page, FIELD **fields)
{
	int ch=0;
	while ( (ch = wgetch(optwin.main)) != CDW_ESCAPE ){
		switch(ch){
			case KEY_HOME:
				form_driver(form_page, REQ_BEG_LINE);
				break;
			case KEY_END:
				form_driver(form_page, REQ_END_LINE);
				break;
			case KEY_LEFT:
				form_driver(form_page, REQ_PREV_CHAR);
				break;
			case KEY_RIGHT:
				form_driver(form_page, REQ_NEXT_CHAR);
				break;
			case KEY_DOWN:
			case '\t':
				form_driver(form_page, REQ_NEXT_FIELD);
				form_driver(form_page, REQ_END_LINE);
				break;
			case KEY_UP:
			case KEY_BTAB:
				form_driver(form_page, REQ_PREV_FIELD);
				form_driver(form_page, REQ_END_LINE);
				break;
			case KEY_BACKSPACE:
				form_driver(form_page, REQ_DEL_PREV);
				break;
			case KEY_DC:
				form_driver(form_page, REQ_DEL_CHAR);
				break;
			case KEY_F(1):
			case KEY_F(10):
				form_driver(form_page, REQ_VALIDATION);
				return ch;
			default:
				form_driver(form_page, ch);
				break;

		} /* switch */
		wrefresh(optwin.main);
	}

	return ch;
}




/**
 * Draw checkboxes and tabs on first page of options
 *
 * Draw square brackets around some input fields that work
 * as checkboxes. Draw tabs with option categories names at
 * the bottom of first page of options.
 */
void first_page_draw(void)
{

	/* draw nice checkboxes */
	put_ckbox(optwin.main, 1, SECOND_COL + SECOND_COL_W + 3);
	put_ckbox(optwin.main, 2, SECOND_COL + SECOND_COL_W + 3);
	put_ckbox(optwin.main, 3, FIRST_COL_W + 4);
	put_ckbox(optwin.main, 3, SECOND_COL + SECOND_COL_W + 3);

	put_ckbox(optwin.main, 6, FIRST_COL_W + 4);
	put_ckbox(optwin.main, 6, SECOND_COL + SECOND_COL_W + 3);
	put_ckbox(optwin.main, 7, FIRST_COL_W + 4);
	put_ckbox(optwin.main, 7, SECOND_COL + SECOND_COL_W + 3);
	put_ckbox(optwin.main, 8, FIRST_COL_W + 4);
	put_ckbox(optwin.main, 8, SECOND_COL + SECOND_COL_W + 3);
	put_ckbox(optwin.main, 9, FIRST_COL_W + 4);
	put_ckbox(optwin.main, 9, SECOND_COL + SECOND_COL_W + 3);

	put_ckbox(optwin.main, 11, FIRST_COL_W + 4);


	/* tabs at the bottom of options window */

	/* tab one: label, right border and long line above */
	/* label */
	mvwprintw(optwin.main, OPTIONS_HEIGHT - 2, 1, _(tab_one_name));
	/* right border */
	mvwaddch(optwin.main, OPTIONS_HEIGHT - 2, 1 + tab_one_name_len + 1, ACS_VLINE);
	/* and long line above, starting from fixing fragment at the left window border */
	mvwaddch(optwin.main, OPTIONS_HEIGHT - 3, 0, ACS_VLINE);
	mvwaddch(optwin.main, OPTIONS_HEIGHT - 3, 1 + tab_one_name_len + 1, ACS_ULCORNER);
	mvwhline(optwin.main, OPTIONS_HEIGHT - 3, 1 + tab_one_name_len + 2, ACS_HLINE, OPTIONS_WIDTH - 1 - tab_one_name_len - 1 - 1);
	mvwaddch(optwin.main, OPTIONS_HEIGHT - 3, OPTIONS_WIDTH - 1, ACS_RTEE);

	/* tab two: just label and right border */
	mvwprintw(optwin.main, OPTIONS_HEIGHT - 2, 2 + tab_one_name_len + 1, _(tab_two_name));
	mvwaddch(optwin.main, OPTIONS_HEIGHT - 2, 2 + tab_one_name_len + 1 + tab_two_name_len + 1, ACS_VLINE);

	/* tab three: just label and right border */
	/*
	mvwprintw(optwin.main, OPTIONS_HEIGHT - 2, 2 + tab_one_name_len + 1 + 1 + tab_two_name_len + 1, _(tab_three_name));
	mvwaddch(optwin.main, OPTIONS_HEIGHT - 2, 2 + tab_one_name_len + 1 + tab_two_name_len + 2 + tab_three_len + 1, ACS_VLINE);
	*/

	return;
}




/**
 * Draw checkboxes and tabs on second page of options
 *
 * Draw square brackets around some input fields that work
 * as checkboxes. Draw tabs with option categories names at
 * the bottom of second page of options.
 */
void second_page_draw(void)
{
	/* draw nice checkboxes */
	/* NONE */

	/* tabs at the bottom of options window */

	/* tab one: label and right border */
	mvwprintw(optwin.main, OPTIONS_HEIGHT - 2, 1, _(tab_one_name));
	mvwaddch(optwin.main, OPTIONS_HEIGHT - 2, 1 + tab_one_name_len + 1, ACS_VLINE);

	/* tab two: label, right border and long line above */
	/* label */
	mvwprintw(optwin.main, OPTIONS_HEIGHT - 2, 2 + tab_one_name_len + 1, _(tab_two_name));
	/* right border */
	mvwaddch(optwin.main, OPTIONS_HEIGHT - 2, 2 + tab_one_name_len + 1 + tab_two_name_len + 1, ACS_VLINE);
	/* and long line above */
	mvwaddch(optwin.main, OPTIONS_HEIGHT - 3, 0, ACS_LTEE);
	mvwhline(optwin.main, OPTIONS_HEIGHT - 3, 1, ACS_HLINE, tab_one_name_len + 1);
	mvwaddch(optwin.main, OPTIONS_HEIGHT - 3, 1 + tab_one_name_len + 1, ACS_URCORNER);
	mvwaddch(optwin.main, OPTIONS_HEIGHT - 3, 1 + tab_one_name_len + 2 + tab_two_name_len + 1, ACS_ULCORNER);
	mvwhline(optwin.main, OPTIONS_HEIGHT - 3, 1 + tab_one_name_len + 2 + tab_two_name_len + 2, ACS_HLINE, OPTIONS_WIDTH - 2 - tab_one_name_len - 2 - tab_two_name_len - 2);
	mvwaddch(optwin.main, OPTIONS_HEIGHT - 3, OPTIONS_WIDTH - 1, ACS_RTEE);

	/* tab three: just label and right border */
	/*
	mvwprintw(optwin.main, OPTIONS_HEIGHT - 2, 2 + tab_one_name_len + 1 + 1 + tab_two_name_len + 1, _(tab_three_name));
	mvwaddch(optwin.main, OPTIONS_HEIGHT - 2, 2 + tab_one_name_len + 1 + tab_two_name_len + 2 + tab_three_len + 1, ACS_VLINE);
	*/

	return;
}




/**
 * Create fields that are shown on second page of options
 *
 * Create fields that are shown on second page of options, including
 * intialization of size and position (all OPTION_FIELDS_P2 - 1 of them,
 * last one is of course NULL), set their types, options and appearance.
 *
 * \param FIELD **fields - pointer to empty FILEDs table, to be filled with
 *                         alocated fields
 * \param struct conf *tmp_config - variable holding current configuation, and
 *                         temporarily storing modified values of options
 */
void second_page_build_fields(FIELD **fields, struct conf *tmp_config)
{
	int i = 0;

	/* 1st row */
	fields[0] = new_field(1, FIRST_COL_W, 0, FIRST_COL, 0, 0);
	set_field_buffer(fields[0], 0, _("SCSI device"));

	fields[1]=new_field(1, 35, 0, FIRST_COL_W + 2, 0, 0);
	set_field_buffer(fields[1], 0, tmp_config->scsi);
	//set_field_type(fields[1], TYPE_REGEXP, "^(ATAPI:|REMOTE:){0,1}[0-9],[0-9],[0-9]");



	/* 2nd row */
	fields[2]=new_field(1, FIRST_COL_W, 1, FIRST_COL, 0, 0);
	set_field_buffer(fields[2], 0, _("CD Reader/Wirter"));

	fields[3]=new_field(1, 35, 1, FIRST_COL_W + 2, 0, 0);
	set_field_buffer(fields[3], 0, tmp_config->cdrwdevice);



	/* 3rd row */
	fields[4]=new_field(1, FIRST_COL_W, 2, FIRST_COL, 0, 0);
	set_field_buffer(fields[4], 0, _("CD mountpoint"));

	fields[5]=new_field(1, 35, 2, FIRST_COL_W + 2, 0, 0);
	set_field_buffer(fields[5], 0, tmp_config->mountpoint);



	/* 4th row */
	fields[6]=new_field(1, FIRST_COL_W, 3, FIRST_COL, 0, 0);
	set_field_buffer(fields[6], 0, _("CD Reader"));

	fields[7]=new_field(1, 35, 3, FIRST_COL_W + 2, 0, 0);
	set_field_buffer(fields[7], 0, tmp_config->cdrom);


	fields[OPTION_FIELDS_P2 - 1] = NULL;

	/* labels - appearance */
	for (i = 0; i <= OPTION_FIELDS_P2 - 2; i += 2) {
		set_field_just(fields[i], JUSTIFY_RIGHT);
		field_opts_off(fields[i], O_ACTIVE);
		set_field_back(fields[i], COLOR_PAIR(2));
		set_field_fore(fields[i], COLOR_PAIR(2));
		set_field_back(fields[i], COLOR_PAIR(2));
		set_field_fore(fields[i], COLOR_PAIR(2));
	}

	/* input areas - appearance */
	for (i = 1; i < OPTION_FIELDS_P2 - 1; i += 2) {
		set_field_just(fields[i], JUSTIFY_LEFT);
		set_field_back(fields[i], COLOR_PAIR(6));
		set_field_fore(fields[i], COLOR_PAIR(6));
		set_field_back(fields[i], COLOR_PAIR(6));
		set_field_fore(fields[i], COLOR_PAIR(6));
	}

	return;
}




/**
 * Save option values from page two
 *
 * Save option values from page two to temporary config variable. The values
 * will be permanently stored if user demands it or discarded otherwise.
 *
 * \param FIELD **fields - option fields available on second page
 * \param struct cong **tmp_config - temporary varialble holding option values
 *                    (probably with changes) appearing on option pages
 */
void second_page_tmp_save_changes(FIELD **fields, struct conf *tmp_config)
{
	sprintf(tmp_config->scsi, rtrim(field_buffer(fields[1], 0)));
	sprintf(tmp_config->cdrwdevice, rtrim(field_buffer(fields[3], 0)));
	sprintf(tmp_config->mountpoint, rtrim(field_buffer(fields[5], 0)));
	sprintf(tmp_config->cdrom, rtrim(field_buffer(fields[7], 0)));

	return;
}




/**
 * Save options from page one to temporary config variable
 *
 * \param FIELD **fields - option fields available on second page
 * \param struct cong **tmp_config - temporary varialble holding option values
 *                    (probably with changes) appearing on option pages
 */
void first_page_tmp_save_changes(FIELD **fields, struct conf *tmp_config)
{
	sprintf(tmp_config->speed, rtrim(field_buffer(*(fields + 1), 0)));
	sprintf(tmp_config->eject, (!strncmp(field_buffer(*(fields + 3), 0), "X", 1)?"1":"0"));
	sprintf(tmp_config->volumeid, rtrim(field_buffer(*(fields + 5), 0)));
	sprintf(tmp_config->showvol, (!strncmp(field_buffer(*(fields + 7), 0), "X", 1)?"1":"0"));
	sprintf(tmp_config->fromimage, (!strncmp(field_buffer(*(fields + 9), 0), "X", 1)?"1":"0"));
	sprintf(tmp_config->multi, (!strncmp(field_buffer(*(fields + 11), 0), "X", 1)?"1":"0"));
	sprintf(tmp_config->tempdir, rtrim(field_buffer(*(fields + 13), 0)));
	sprintf(tmp_config->dao, (!strncmp(field_buffer(*(fields + 15), 0), "X", 1)?"1":"0"));
	sprintf(tmp_config->burnproof, (!strncmp(field_buffer(*(fields + 17), 0), "X", 1)?"1":"0"));
	sprintf(tmp_config->rockridge, (!strncmp(field_buffer(*(fields + 19), 0), "X", 1)?"1":"0"));
	sprintf(tmp_config->joliet, (!strncmp(field_buffer(*(fields + 21), 0), "X", 1)?"1":"0"));
	sprintf(tmp_config->usefulRR, (!strncmp(field_buffer(*(fields + 23), 0), "X", 1)?"1":"0"));
	sprintf(tmp_config->pad, (!strncmp(field_buffer(*(fields + 25), 0), "X", 1)?"1":"0"));
	sprintf(tmp_config->dummy, (!strncmp(field_buffer(*(fields + 27), 0), "X", 1)?"1":"0"));
	sprintf(tmp_config->blank, (!strncmp(field_buffer(*(fields + 29), 0), "X", 1)?"fast":"all"));
	sprintf(tmp_config->showlog, (!strncmp(field_buffer(*(fields + 31), 0), "X", 1)?"1":"0"));
	sprintf(tmp_config->logfile, rtrim(field_buffer(*(fields + 33), 0)));
	sprintf(tmp_config->bootimg, rtrim(field_buffer(*(fields + 35), 0)));
	sprintf(tmp_config->other, rtrim(field_buffer(*(fields + 37), 0)));

	return;
}




/**
 * Copy values of options from one options variable to other
 *
 * \param struct conf *dest - variable to be used as destination of copied values
 * \param struct conf *src - variable to be used as source of copied values
 */
void copy_configuration(struct conf *dest, struct conf *src)
{
	/* session page options */
	strcpy(dest->speed, src->speed);
	strcpy(dest->eject, src->eject);
	strcpy(dest->volumeid, src->volumeid);
	strcpy(dest->showvol, src->showvol);
	strcpy(dest->multi, src->multi);
	strcpy(dest->fromimage, src->fromimage);
	strcpy(dest->tempdir, src->tempdir);
	strcpy(dest->dao, src->dao);
	strcpy(dest->burnproof, src->burnproof);
	strcpy(dest->rockridge, src->rockridge);
	strcpy(dest->joliet, src->joliet);
	strcpy(dest->usefulRR, src->usefulRR);
	strcpy(dest->pad, src->pad);
	strcpy(dest->dummy, src->dummy);
	strcpy(dest->blank, src->blank);
	strcpy(dest->showlog, src->showlog);
	strcpy(dest->logfile, src->logfile);
	strcpy(dest->bootimg, src->bootimg);
	strcpy(dest->other, src->other);

	/* disk/hardware page options */
	strcpy(dest->scsi, src->scsi);
	strcpy(dest->cdrwdevice, src->cdrwdevice);
	strcpy(dest->mountpoint, src->mountpoint);
	strcpy(dest->cdrom, src->cdrom);

	/*oter */
	strcpy(dest->autodic, src->autodic);
	strcpy(dest->audiodir, src->audiodir);
	strcpy(dest->bitsperchn, src->bitsperchn);
	strcpy(dest->stereo, src->stereo);
	strcpy(dest->echosound, src->echosound);
	strcpy(dest->encode, src->encode);
	strcpy(dest->lame, src->lame);
	strcpy(dest->highq, src->highq);
	strcpy(dest->bitrate, src->bitrate);
	dest->cdsize = src->cdsize;
	dest->user_cdsize = src->user_cdsize;
	strcpy(dest->cdbhost, src->cdbhost);
	strcpy(dest->cdbuser, src->cdbuser);
	strcpy(dest->sqlite_file, src->sqlite_file);

	return;
}




/**
 * Set error flag for options module
 *
 * Call this function when something went wrong during options setup.
 * It will set internal error flag on, which will cause options module to work
 * without reading or writing disc config file.
 *
 * \returns CDW_FAILSAFE_MODE
 */
int options_failsafe_mode(void)
{
	failsafe_mode = true;

	return CDW_FAILSAFE_MODE;
}







/* UNUSED CODE BELOW */


#if 0  /* old version */
static int page_two()
{
	int ch, i;
	FIELD *fields[OPTION_FIELDS_P2]; /* last field == NULL included */
	FORM *form_page[1];


	clear_main_win();

	/* 1st row */
	field[0] = new_field(1, FIRST_COL_W, 0, FIRST_COL, 0, 0);
	set_field_buffer(field[0], 0, _("SCSI device"));
	field[1]=new_field(1, 12, 0, FIRST_COL_W + 2, 0, 0);
	set_field_buffer(field[1], 0, config.scsi);


	/* 2nd row */
	field[2]=new_field(1, FIRST_COL_W, 1, FIRST_COL, 0, 0);
	set_field_buffer(field[2], 0, _("CD Reader/Wirter"));
	field[3]=new_field(1, 35, 1, FIRST_COL_W + 2, 0, 0);
	set_field_buffer(field[3], 0, config.cdrwdevice);


	/* 3rd row */
	field[4]=new_field(1, FIRST_COL_W, 2, FIRST_COL, 0, 0);
	set_field_buffer(field[4], 0, _("CD mountpoint"));
	field[5]=new_field(1, 35, 2, FIRST_COL_W + 2, 0, 0);
	set_field_buffer(field[5], 0, config.mountpoint);


	/* 4th row */
	field[6]=new_field(1, FIRST_COL_W, 3, FIRST_COL, 0, 0);
	set_field_buffer(field[6], 0, _("CD Reader"));
	field[7]=new_field(1, 35, 3, FIRST_COL_W + 2, 0, 0);
	set_field_buffer(field[7], 0, config.cdrom);

	field[8] = NULL;

	/* scsi device
	set_field_type(field[i], TYPE_REGEXP, "^(ATAPI:|REMOTE:){0,1}[0-9],[0-9],[0-9]");
	*/


	/* Input fields */
	for (i=1; i<=13; i++) {
	switch (i){
#ifndef HAVE_LIBMYSQLCLIENT
	case 2:
	case 3:
		field_opts_off(field[(i*2)-1], O_ACTIVE);
		set_field_back(field[(i*2)-1], COLOR_PAIR(9) | A_BOLD);
		set_field_fore(field[(i*2)-1], COLOR_PAIR(9) | A_BOLD);
		break;
#endif
#ifndef HAVE_LIBSQLITE
	case 5:
		field_opts_off(field[(i*2)-1], O_ACTIVE);
		set_field_back(field[(i*2)-1], COLOR_PAIR(9) | A_BOLD);
		set_field_fore(field[(i*2)-1], COLOR_PAIR(9) | A_BOLD);
		break;
#endif
#ifndef OGGENC
	case 10:
		field_opts_off(field[(i*2)-1], O_ACTIVE);
		set_field_back(field[(i*2)-1], COLOR_PAIR(9) | A_BOLD);
		set_field_fore(field[(i*2)-1], COLOR_PAIR(9) | A_BOLD);
		break;
#endif
#ifndef LAME
	case 12:
		field_opts_off(field[(i*2)-1], O_ACTIVE);
		set_field_back(field[(i*2)-1], COLOR_PAIR(9) | A_BOLD);
		set_field_fore(field[(i*2)-1], COLOR_PAIR(9) | A_BOLD);
		break;
#endif
	case 1:
#ifdef HAVE_LIBMYSQLCLIENT
	case 2:
	case 3:
#endif
#ifdef HAVE_LIBSQLITE
	case 5:
#endif
	case 6:
	case 8:
	case 13:
		switch(i) {
			case 8:
				set_field_type(field[(i*2)-1], TYPE_REGEXP, "(8|12|16)+");
				break;
			case 13:
				set_field_type(field[(i*2)-1], TYPE_REGEXP, "(32|56|64|96|128|192|320)+");
				break;
		}
		set_field_back(field[(i*2)-1], COLOR_PAIR(6));
		set_field_fore(field[(i*2)-1], COLOR_PAIR(6));
		break;
	default:
		set_field_type(field[(i*2)-1], TYPE_REGEXP, "^(X|_)$");
		set_field_back(field[(i*2)-1], COLOR_PAIR(2));
		set_field_fore(field[(i*2)-1], COLOR_PAIR(2));
		break;
	}
}
		/* Lables */
		for (i=1; i<=13; i++) {
		set_field_just(field[(i*2)-2], JUSTIFY_RIGHT);
		field_opts_off(field[(i*2)-2], O_ACTIVE);
		switch (i){
#ifndef HAVE_LIBMYSQLCLIENT
			case 2:
			case 3:
				set_field_back(field[(i*2)-2], COLOR_PAIR(9) | A_BOLD);
				set_field_fore(field[(i*2)-2], COLOR_PAIR(9) | A_BOLD);
				break;
#endif
#ifndef HAVE_LIBSQLITE
			case 5:
				set_field_back(field[(i*2)-2], COLOR_PAIR(9) | A_BOLD);
				set_field_fore(field[(i*2)-2], COLOR_PAIR(9) | A_BOLD);
				break;
#endif
#ifndef OGGENC
			case 10:
				set_field_back(field[(i*2)-2], COLOR_PAIR(9) | A_BOLD);
				set_field_fore(field[(i*2)-2], COLOR_PAIR(9) | A_BOLD);
				break;
#endif
#ifndef LAME
			case 12:
				set_field_back(field[(i*2)-2], COLOR_PAIR(9) | A_BOLD);
				set_field_fore(field[(i*2)-2], COLOR_PAIR(9) | A_BOLD);
				break;
#endif
			default:
				set_field_back(field[(i*2)-2], COLOR_PAIR(2));
				set_field_fore(field[(i*2)-2], COLOR_PAIR(2));
				break;
		}
	}

	form_page[0]=new_form(field);
	set_form_win(form_page[0], optwin.main);
	set_form_sub(form_page[0], derwin(optwin.main, 18, 58, 1, 1));

	post_form(form_page[0]);
	put_ckbox(optwin.main, 4, 55);
	put_ckbox(optwin.main, 7, 23);
	put_ckbox(optwin.main, 8, 23);
#ifdef OGGENC
	put_ckbox(optwin.main, 9, 23);
#endif
	put_ckbox(optwin.main, 9, 55);
#ifdef LAME
	put_ckbox(optwin.main, 10, 23);
#endif
	mvwaddch(optwin.main, 17, 0, ACS_LTEE);
	mvwhline(optwin.main, 17, 1, ACS_HLINE, 15);
	mvwaddch(optwin.main, 17, 15, ACS_URCORNER);
	mvwaddch(optwin.main, 18, 15, ACS_VLINE);
	mvwprintw(optwin.main, 18, 1, _("<F1> General"));
	mvwaddch(optwin.main, 17, 30, ACS_ULCORNER);
	mvwhline(optwin.main, 17, 31, ACS_HLINE, 28);
	mvwaddch(optwin.main, 17, 59, ACS_RTEE);
	mvwprintw(optwin.main, 18, 18, _("<F2> Other"));
	mvwaddch(optwin.main, 18, 30, ACS_VLINE);

	form_driver(form_page[0], REQ_END_LINE);
	wrefresh(optwin.main);
	ch=0;
	while ( (ch=wgetch(optwin.main)) != 27 ){
		switch(ch){
			case KEY_HOME:
				form_driver(form_page[0], REQ_BEG_LINE);
				break;
			case KEY_END:
				form_driver(form_page[0], REQ_END_LINE);
				break;
			case KEY_LEFT:
				form_driver(form_page[0], REQ_PREV_CHAR);
				break;
			case KEY_RIGHT:
				form_driver(form_page[0], REQ_NEXT_CHAR);
				break;
			case KEY_DOWN:
				form_driver(form_page[0], REQ_NEXT_FIELD);
				form_driver(form_page[0], REQ_END_LINE);
				break;
			case KEY_UP:
				form_driver(form_page[0], REQ_PREV_FIELD);
				form_driver(form_page[0], REQ_END_LINE);
				break;
			case KEY_BACKSPACE:
				form_driver(form_page[0], REQ_DEL_PREV);
				break;
			case KEY_DC:
				form_driver(form_page[0], REQ_DEL_CHAR);
				break;
			case KEY_F(1):
			case KEY_F(10):
				form_driver(form_page[0], REQ_VALIDATION);
				sprintf(config.mountpoint, rtrim(field_buffer(field[1], 0)));
				sprintf(config.cdbhost, rtrim(field_buffer(field[3], 0)));
				sprintf(config.cdbuser, rtrim(field_buffer(field[5], 0)));
				sprintf(config.autodic, (!strncmp(field_buffer(field[7], 0), "X", 1)?"1":"0"));
				sprintf(config.audiodir, rtrim(field_buffer(field[11], 0)));
				sprintf(config.stereo, (!strncmp(field_buffer(field[13], 0), "X", 1)?"1":"0"));
				sprintf(config.bitsperchn, rtrim(field_buffer(field[15], 0)));
				sprintf(config.echosound, (!strncmp(field_buffer(field[17], 0), "X", 1)?"1":"0"));
				sprintf(config.encode, (!strncmp(field_buffer(field[19], 0), "X", 1)?"1":"0"));
				sprintf(config.highq, (!strncmp(field_buffer(field[21], 0), "X", 1)?"1":"0"));
				sprintf(config.lame, (!strncmp(field_buffer(field[23], 0), "X", 1)?"1":"0"));
				sprintf(config.bitrate, rtrim(field_buffer(field[25], 0)));
				sprintf(config.sqlite_file, rtrim(field_buffer(field[9], 0)));
				return ch;
				break;
			default:
				if (ch==' ') {
					FIELD *cur;
					int fld_index;

				cur=current_field(form_page[0]);
				fld_index=field_index(cur);
				switch (fld_index) {
					case 7:
					case 13:
					case 17:
					case 19:
					case 21:
					case 23:
					ch=(!strncmp(field_buffer(cur, 0), "X", 1)?'_':'X');
					if ((fld_index==19) && (!strcmp(field_buffer(field[19], 0), "_"))){
						set_field_buffer(field[23], 0, "_");
					}
					if ((fld_index==23) && (!strcmp(field_buffer(field[23], 0), "_"))){
						set_field_buffer(field[19], 0, "_");
					}
					break;
				}
			}
			form_driver(form_page[0], ch);
			break;
			wrefresh(optwin.main);
		}
	}
	ch=KEY_F(10);
	unpost_form(form_page[0]);
	free_form(form_page[0]);
	for (i=0; i<14; i++) free_field(field[i]);
	return ch;
}

#endif




/*

	field[2]=new_field(1, 20, 2, 1, 0, 0);
	set_field_buffer(field[2], 0, _("MySQL host"));
	field[3]=new_field(1, 35, 2, 22, 0, 0);
	set_field_buffer(field[3], 0, config.cdbhost);

	field[4]=new_field(1, 20, 3, 1, 0, 0);
	set_field_buffer(field[4], 0, _("MySQL user"));
	field[5]=new_field(1, 6, 3, 22, 0, 0);
	set_field_buffer(field[5], 0, config.cdbuser);

	field[6]=new_field(1, 24, 3, 29, 0, 0);
	set_field_buffer(field[6], 0, _("Auto Disk Catalog"));
	field[7]=new_field(1, 1, 3, 55, 0, 0);
	set_field_buffer(field[7], 0, (!strncmp(config.autodic, "1", 1)?"X":"_"));

	field[8]=new_field(1, 20, 4, 1, 0, 0);
	set_field_buffer(field[8], 0, _("SQLite DB file"));
	field[9]=new_field(1, 35, 4, 22, 0, 0);
	set_field_buffer(field[9], 0, config.sqlite_file);

	field[10]=new_field(1, 20, 5, 1, 0, 0);
	set_field_buffer(field[10], 0, _("Audio directory"));
	field[11]=new_field(1, 35, 5, 22, 0, 0);
	set_field_buffer(field[11], 0, config.audiodir);

	field[12]=new_field(1, 20, 6, 1, 0, 0);
	set_field_buffer(field[12], 0, _("Grab stereo mode"));
	field[13]=new_field(1, 1, 6, 23, 0, 0);
	set_field_buffer(field[13], 0, (!strncmp(config.stereo, "1", 1)?"X":"_"));

	field[14]=new_field(1, 24, 6, 29, 0, 0);
	set_field_buffer(field[14], 0, _("Bits per channel"));
	field[15]=new_field(1, 3, 6, 54, 0, 0);
	set_field_buffer(field[15], 0, config.bitsperchn);

	field[16]=new_field(1, 20, 7, 1, 0, 0);
	set_field_buffer(field[16], 0, _("Echo to sound dev."));
	field[17]=new_field(1, 1, 7, 23, 0, 0);
	set_field_buffer(field[17], 0, (!strncmp(config.echosound, "1", 1)?"X":"_"));

	field[18]=new_field(1, 20, 8, 1, 0, 0);
	set_field_buffer(field[18], 0, _("Encode to Ogg Vorbis"));
	field[19]=new_field(1, 1, 8, 23, 0, 0);
	set_field_buffer(field[19], 0, (!strncmp(config.encode, "1", 1)?"X":"_"));

	field[20]=new_field(1, 24, 8, 29, 0, 0);
	set_field_buffer(field[20], 0, _("High Quality"));
	field[21]=new_field(1, 1, 8, 55, 0, 0);
	set_field_buffer(field[21], 0, (!strncmp(config.highq, "1", 1)?"X":"_"));

	field[22]=new_field(1, 20, 9, 1, 0, 0);
	set_field_buffer(field[22], 0, _("Encode to Mp3"));
	field[23]=new_field(1, 1, 9, 23, 0, 0);
	set_field_buffer(field[23], 0, (!strncmp(config.lame, "1", 1)?"X":"_"));

	field[24]=new_field(1, 24, 9, 29, 0, 0);
	set_field_buffer(field[24], 0, _("Bitrate"));
	field[25]=new_field(1, 4, 9, 54, 0, 0);
	set_field_buffer(field[25], 0, config.bitrate);

*/

