/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2007 Brockmann Consult
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Seascount   seascount         Seasonal counts
*/

#include <cdi.h>

#include "cdo_int.h"
#include "cdo_season.h"

void *
Seascount(void *process)
{
  int64_t vdate0 = 0;
  int vtime0 = 0;
  int nrecs;
  int varID, levelID;
  int year, month, day, seas0 = 0;
  int oldmon = 0;

  cdoInitialize(process);

  cdoOperatorAdd("seascount", 0, 0, nullptr);

  const int season_start = get_season_start();

  CdoStreamID streamID1 = cdoOpenRead(0);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = vlistDuplicate(vlistID1);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  CdoStreamID streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  const int maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  size_t gridsizemax = vlistGridsizeMax(vlistID1);
  if (vlistNumber(vlistID1) != CDI_REAL) gridsizemax *= 2;

  Field field;
  field.resize(gridsizemax);

  FieldVector2D vars1;
  fieldsFromVlist(vlistID1, vars1, FIELD_VEC);

  int tsID = 0;
  int otsID = 0;
  while (true)
    {
      int nsets = 0;
      bool newseas = false;
      while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
        {
          const int64_t vdate = taxisInqVdate(taxisID1);
          const int vtime = taxisInqVtime(taxisID1);
          cdiDecodeDate(vdate, &year, &month, &day);

          int newmon = month;
          if (season_start == START_DEC && newmon == 12) newmon = 0;

          const int seas = month_to_season(month);

          if (nsets == 0)
            {
              seas0 = seas;
              oldmon = newmon;
            }

          if (newmon < oldmon) newseas = true;

          if ((seas != seas0) || newseas) break;

          oldmon = newmon;

          for (int recID = 0; recID < nrecs; recID++)
            {
              cdoInqRecord(streamID1, &varID, &levelID);

              if (tsID == 0)
                {
                  recList[recID].varID = varID;
                  recList[recID].levelID = levelID;
                  recList[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;
                }

              const size_t fieldsize = vars1[varID][levelID].size;

              if (nsets == 0)
                {
                  for (size_t i = 0; i < fieldsize; i++) vars1[varID][levelID].vec[i] = vars1[varID][levelID].missval;
                  vars1[varID][levelID].nmiss = fieldsize;
                }

              cdoReadRecord(streamID1, field.vec.data(), &field.nmiss);
              field.grid = vars1[varID][levelID].grid;
              field.missval = vars1[varID][levelID].missval;

              vfarcount(vars1[varID][levelID], field);
            }

          vdate0 = vdate;
          vtime0 = vtime;
          nsets++;
          tsID++;
        }

      if (nrecs == 0 && nsets == 0) break;

      taxisDefVdate(taxisID2, vdate0);
      taxisDefVtime(taxisID2, vtime0);
      cdoDefTimestep(streamID2, otsID);

      for (int recID = 0; recID < maxrecs; recID++)
        {
          if (otsID && recList[recID].lconst) continue;

          const int varID = recList[recID].varID;
          const int levelID = recList[recID].levelID;
          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, vars1[varID][levelID].vec.data(), vars1[varID][levelID].nmiss);
        }

      if (nrecs == 0) break;
      otsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
