/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Detrend    detrend         Detrend
*/

#include <cdi.h>

#include "cdo_int.h"
#include "cdo_options.h"
#include "datetime.h"
#include "cimdOmp.h"


#define NALLOC_INC 1024

static void
detrend(const long nts, const double missval1, const std::vector<double> &array1, std::vector<double> &array2)
{
  const double missval2 = missval1;
  double sumj = 0, sumjj = 0;
  double sumx = 0, sumjx = 0;
  long n = 0;
  for (long j = 0; j < nts; j++)
    if (!DBL_IS_EQUAL(array1[j], missval1))
      {
        const double zj = j;
        sumj += zj;
        sumjj += zj * zj;
        sumjx += zj * array1[j];
        sumx += array1[j];
        n++;
      }

  const double work1 = DIVMN(SUBMN(sumjx, DIVMN(MULMN(sumj, sumx), n)), SUBMN(sumjj, DIVMN(MULMN(sumj, sumj), n)));
  const double work2 = SUBMN(DIVMN(sumx, n), MULMN(DIVMN(sumj, n), work1));

  for (long j = 0; j < nts; j++) array2[j] = SUBMN(array1[j], ADDMN(work2, MULMN(j, work1)));
}

void *
Detrend(void *process)
{
  int nrecs;
  int varID, levelID;
  size_t nmiss;
  DateTimeList dtlist;

  cdoInitialize(process);

  CdoStreamID streamID1 = cdoOpenRead(0);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = vlistDuplicate(vlistID1);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  CdoStreamID streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  VarList varList;
  varListInit(varList, vlistID1);

  const int nvars = vlistNvars(vlistID1);
  FieldVector3D vars;

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      if ((size_t)tsID >= vars.size()) vars.resize(vars.size() + NALLOC_INC);

      dtlist.taxisInqTimestep(taxisID1, tsID);

      fieldsFromVlist(vlistID1, vars[tsID], FIELD_NONE);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          vars[tsID][varID][levelID].resize(varList[varID].gridsize);
          cdoReadRecord(streamID1, vars[tsID][varID][levelID].vec.data(), &nmiss);
          vars[tsID][varID][levelID].nmiss = nmiss;
        }

      tsID++;
    }

  const int nts = tsID;
  VECTOR_2D(double, array1, Threading::ompNumThreads, nts);
  VECTOR_2D(double, array2, Threading::ompNumThreads, nts);

  for (varID = 0; varID < nvars; varID++)
    {
      int nsteps = (varList[varID].timetype == TIME_CONSTANT) ? 1 : nts;
      double missval = varList[varID].missval;
      size_t gridsize = varList[varID].gridsize;
      const int nlevels = varList[varID].nlevels;
      for (levelID = 0; levelID < nlevels; levelID++)
        {
#ifdef _OPENMP
#pragma omp parallel for default(none) shared(gridsize, nsteps, missval, array1, array2, vars, varID, levelID)
#endif
          for (size_t i = 0; i < gridsize; i++)
            {
              const int ompthID = cdo_omp_get_thread_num();

              for (int tsID = 0; tsID < nsteps; ++tsID) array1[ompthID][tsID] = vars[tsID][varID][levelID].vec[i];

              detrend(nsteps, missval, array1[ompthID], array2[ompthID]);

              for (int tsID = 0; tsID < nsteps; ++tsID) vars[tsID][varID][levelID].vec[i] = array2[ompthID][tsID];
            }
        }
    }

  for (int tsID = 0; tsID < nts; tsID++)
    {
      dtlist.taxisDefTimestep(taxisID2, tsID);
      cdoDefTimestep(streamID2, tsID);

      for (varID = 0; varID < nvars; varID++)
        {
          if (tsID && varList[varID].timetype == TIME_CONSTANT) continue;
          const int nlevels = varList[varID].nlevels;
          for (levelID = 0; levelID < nlevels; levelID++)
            {
              nmiss = vars[tsID][varID][levelID].nmiss;
              cdoDefRecord(streamID2, varID, levelID);
              cdoWriteRecord(streamID2, vars[tsID][varID][levelID].vec.data(), nmiss);
            }
        }
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
