/*
 * KON2 - Kanji ON Console -
 * Copyright (C) 1992, 1993 MAEDA Atusi (mad@math.keio.ac.jp)
 * Copyright (C) 1992-1996 Takashi MANABE (manabe@papilio.tutics.tut.ac.jp)
 *
 * CCE - Console Chinese Environment -
 * Copyright (C) 1998-1999 Rui He (herui@cs.duke.edu)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY MAEDA ATUSI AND TAKASHI MANABE ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 */

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<unistd.h>
#if defined(linux)
#include	<sys/vt.h>
#endif
#include	<fcntl.h>
#include	<signal.h>
#if defined(linux)
#include	<termio.h>
#elif defined(__FreeBSD__)
#include	<termios.h>
#include	<machine/console.h>
#endif
#include	<sys/ioctl.h>
#ifdef linux
#include	<sys/kd.h>
#endif

#include	<getcap.h>
#include	<defs.h>
#include	<errors.h>
#include	<font.h>
#include	<mouse.h>
#include	<vt.h>
#include	<vc.h>
#include	<term.h>

#include        <vga.h>

DispInfo	dispInfo;
CursorInfo	cursorInfo;
VideoInfo	*pVideoInfo = NULL;
CursorInfo	mouseCursor;

/*
static bool	useHardScroll;
*/

int text_mode = FALSE;
static int   	curActive = FALSE;

static volatile bool	busy;		 /* TRUE iff updating screen */
static volatile bool	release;	 /* delayed VC switch flag */

static void	ShowCursor(CursorInfo *, bool);
void	LeaveVC(int);
static void	EnterVC(int);
static void     TextClearBand(ConInfo *con, int top, int btm);

/*

  con->flagBuff:
  |      7|      6|      5|4||3|2|1|0|
  |CLEAN_S|LATCH_2|LATCH_1| ||<----->|
  |0=latch|  byte2|  byte1| ||   LANG|

  Usage: CLEAN_S         indicate if we need to redraw this char
         LATCH_1/LATCH_2 double byte char (Chinese, Japanese, Korean)
         LANG            language, we can support BIG5/GB same screen
  */

#define	CCE_TMP_FILE	"/tmp/.ccetmp"

/* blatch clear the bit 7 of array head  byte , force to redraw */
static inline	void	blatch(void *head, int n)
{

    __asm__("pushl %%ebx\n"
            "pushl %%ecx\n"
            "\t clc\n"
	    "1:\n"
	    "\t andb %%bl, (%%eax)\n"
	    "\t incl %%eax\n"
	    "\t loop 1b\n"
            "popl %%ecx\n"
            "popl %%ebx\n"
	    :
	    : "eax" ((long)head), "bl" (0x7F), "c" (n)
	    );
}

/* long latch */
static inline	void	llatch(void *head, int n)
{

    __asm__("pushl %%ecx\n"
            "pushl %%ebx\n"
            "\t clc\n"
	    "1:\n"
	    "\t andl %%ebx, (%%eax)\n"
	    "\t addl $4, %%eax\n"
	    "\t loop 1b\n"
            "popl %%ebx\n"
            "popl %%ecx\n"
	    :
	    : "eax" ((long)head), "ebx" (0x7F7F7F7F), "c" (n>>2)
	    );
}

static inline u_int	TextAddress(ConInfo *con,u_int x, u_int y)
{
    return (con->textHead + x + y * (dispInfo.txmax)) % con->textSize;
}

static inline bool	IsKanji(ConInfo *con,u_int x, u_int y)
{
    return(*(con->flagBuff + TextAddress(con,x, y)) & CODEIS_1);
}

static inline bool	IsKanji2(ConInfo *con,u_int x, u_int y)
{
    return(*(con->flagBuff + TextAddress(con,x, y)) & CODEIS_2);
}

/****************************************************************************
 *                     TextDeleteChar/TextInsertChar                        *
 ****************************************************************************/

void	TextDeleteChar(ConInfo *con, int n)
{
    u_int	addr, dx;
    
    addr = TextAddress(con, con->x, con->y);
    dx = dispInfo.txmax - con->x - n;
    
    bmove(con->textBuff + addr, con->textBuff + addr + n, dx);
    bmove(con->attrBuff + addr, con->attrBuff + addr + n, dx);
    blatch(con->flagBuff + addr, dx);
    
    addr = TextAddress(con, dispInfo.txmax - n, con->y);

    bzero2(con->textBuff + addr, n);
    bzero2(con->attrBuff + addr, n);
    bzero2(con->flagBuff + addr, n);
}

void	TextInsertChar(ConInfo *con,int n)
{
    u_int	addr, dx;
    
    addr = TextAddress(con,dispInfo.txmax, con->y);
    dx = dispInfo.txmax - con->x - n;
    
    brmove(con->textBuff + addr, con->textBuff + addr - n, dx);
    brmove(con->attrBuff + addr, con->attrBuff + addr - n, dx);
    
    addr = TextAddress(con,con->x, con->y);
    blatch(con->flagBuff + addr + n, dx);
    bzero2(con->textBuff + addr, n);
    bzero2(con->attrBuff + addr, n);
    bzero2(con->flagBuff + addr, n);
}

/****************************************************************************
 *                     TextScrollUp/TextScrollDown                          *
 *                     TextMoveUp/TextMoveDown                              *
 *                        ScrollUp/ScrollDown                               *
 ****************************************************************************/

/* no more hardware scroll support */

#if 0
static void TextScrollUp(ConInfo *con,int line)
{
    int oldhead, len;

    oldhead = con->textHead;
    con->textHead += line * (dispInfo.txmax);

    if (con->textHead > con->textSize)
    {
        con->textHead -= con->textSize;
        len = con->textSize - oldhead;
        if (con->textHead)
        {
            lzero(con->textBuff, con->textHead);
            lzero(con->attrBuff, con->textHead);
            lzero(con->flagBuff, con->textHead);
        }
    }
    else len = con->textHead - oldhead;

    lzero(con->textBuff + oldhead, len);
    lzero(con->attrBuff + oldhead, len);
    lzero(con->flagBuff + oldhead, len);
}

static void TextScrollDown(ConInfo *con,int line)
{
    int oldhead, len;

    oldhead = con->textHead;
    con->textHead -= line * (dispInfo.txmax);

    if (con->textHead < 0)
    {
        con->textHead += con->textSize;
        if (oldhead)
        {
            lzero(con->textBuff, oldhead);
            lzero(con->attrBuff, oldhead);
            lzero(con->flagBuff, oldhead);
        }
        len = con->textSize - con->textHead;
    }
    else len = oldhead - con->textHead;

    lzero(con->textBuff + con->textHead, len);
    lzero(con->attrBuff + con->textHead, len);
    lzero(con->flagBuff + con->textHead, len);
}

#endif

void    TextMoveUp(ConInfo *con,int top, int btm, int line)
{
    int       n, src, dst;

    if (btm - top - line + 1 <= 0)
    {
        TextClearBand(con,top, btm);
        return;
    }
    for (n = top; n <= btm - line; n ++)
    {
        dst = TextAddress(con,0, n);
        src = TextAddress(con,0, n + line);
        lmove(con->textBuff + dst, con->textBuff + src, dispInfo.txmax);
        lmove(con->attrBuff + dst, con->attrBuff + src, dispInfo.txmax);
        lmove(con->flagBuff + dst, con->flagBuff + src, dispInfo.txmax);
        llatch(con->flagBuff + dst, dispInfo.txmax);
    }
    TextClearBand(con,btm - line + 1, btm);
}

void    TextMoveDown(ConInfo *con,int top, int btm, int line)
{
    int       n, src, dst;

    if (btm - top - line + 1 <= 0)
    {
        TextClearBand(con,top, btm);
        return;
    }
    for (n = btm; n >= top + line; n --)
    {
        dst = TextAddress(con,0, n);
        src = TextAddress(con,0, n - line);
        lmove(con->textBuff + dst, con->textBuff + src, dispInfo.txmax);
        lmove(con->attrBuff + dst, con->attrBuff + src, dispInfo.txmax);
        lmove(con->flagBuff + dst, con->flagBuff + src, dispInfo.txmax);
        llatch(con->flagBuff + dst, dispInfo.txmax);
    }
    TextClearBand(con,top, top + line - 1);
}

void    ScrollUp(ConInfo *con,int line)
{
#if 0
    if (useHardScroll && !con->soft)
    {
        TextScrollUp(con,line);
        con->scrollLine += line;
    }
    else 
#endif

    {
        TextMoveUp(con, con->ymin, con->ymax, line);
    }
}

void    ScrollDown(ConInfo *con,int line)
{
#if 0
    if (useHardScroll && !con->soft)
    {
        TextScrollDown(con, line);
        con->scrollLine -= line;
    }
    else 
#endif

    {
        TextMoveDown(con,con->ymin, con->ymax, line);
    }
}

/****************************************************************************
 *                       TextClearAll/TextClearEol                          *
 *                       TextClearEos/TextClearBand                         *
 ****************************************************************************/

void TextClearAll(ConInfo *con)
{
    // lzero(con->textBuff, con->textSize);
    // lzero(con->attrBuff, con->textSize);
    memset(con->textBuff, 0x20, con->textSize);
    memset(con->attrBuff, ((con->bcol << 4) | con->fcol), con->textSize);
    lzero(con->flagBuff, con->textSize);
    mInfo.sw = 0;
    con->textClear = TRUE;
}

void    TextClearEol(ConInfo *con,u_char mode)
{
    u_int       addr;
    u_char      len, x=0;

    switch(mode) 
    {
    case 1:
        len = con->x;
        break;
    case 2:
        len = dispInfo.txmax;
        break;
    default:
        x = con->x;
        len = dispInfo.txmax - con->x;
        break;
    }
    addr = TextAddress(con,x, con->y);
    // bzero2(con->textBuff + addr, len);
    // bzero2(con->attrBuff + addr, len);
    memset(con->textBuff + addr, 0x20, len);
    memset(con->attrBuff + addr, (con->bcol << 4) | con->fcol, len);
    bzero2(con->flagBuff + addr, len);/* needless to latch */
}

void    TextClearEos(ConInfo *con,u_char mode)
{
    int       addr, len, y;

    if (mode == 2)
    {
        TextClearAll(con);
        return;
    }
    switch(mode)
    {
    case 1:
        for (y = 0; y < con->y; y ++)
        {
            addr = TextAddress(con,0, y);

            // lzero(con->textBuff + addr, dispInfo.txmax);
            // lzero(con->attrBuff + addr, dispInfo.txmax);
            memset(con->textBuff + addr, 0x20, dispInfo.txmax);
            memset(con->attrBuff + addr, (con->bcol << 4) | con->fcol, dispInfo.txmax);
            lzero(con->flagBuff + addr, dispInfo.txmax);
              /* needless to latch */        
        }
        addr = TextAddress(con,0, con->y);
        // bzero2(con->textBuff + addr, con->x);
        // bzero2(con->attrBuff + addr, con->x);
        memset(con->textBuff + addr, 0x20, con->x);
        memset(con->attrBuff + addr, (con->bcol << 4) | con->fcol, con->x);
        bzero2(con->flagBuff + addr, con->x);  /* needless to latch */
        break;
    default:
        for (y = con->y + 1; y < dispInfo.tymax; y ++)
        {
            addr = TextAddress(con,0, y);
            // lzero(con->textBuff + addr, dispInfo.txmax);
            // lzero(con->attrBuff + addr, dispInfo.txmax);
            memset(con->textBuff + addr, 0x20, dispInfo.txmax);
            memset(con->attrBuff + addr, (con->bcol << 4) | con->fcol, dispInfo.txmax);
            lzero(con->flagBuff + addr, dispInfo.txmax);
               /* needless to latch */
        }
        addr = TextAddress(con,con->x, con->y);
        len = dispInfo.txmax - con->x;
        // bzero2(con->textBuff + addr, len);
        // bzero2(con->attrBuff + addr, len);
        memset(con->textBuff + addr, 0x20, len);
        memset(con->attrBuff + addr, (con->bcol << 4) | con->fcol, len);
        bzero2(con->flagBuff + addr, len);  /* needless to latch */
        break;
    }
}

static void     TextClearBand(ConInfo *con, int top, int btm)
{
    int       y, addr;

    for (y = top; y <= btm; y ++)
    {
        addr = TextAddress(con,0, y);
        // lzero(con->textBuff + addr, dispInfo.txmax);
        // lzero(con->attrBuff + addr, dispInfo.txmax);
        memset(con->textBuff + addr, 0x20, dispInfo.txmax);
        memset(con->attrBuff + addr, (con->bcol << 4) | con->fcol, dispInfo.txmax);
        lzero(con->flagBuff + addr, dispInfo.txmax);/* needless to latch */
    }
}

// --------------------------------------------------------------
// TextClearChars - Added by Holly Lee
//    ConInfo * con
//    int row
//    int fromColumn
//    int columns
// --------------------------------------------------------------
void TextClearChars(ConInfo * con, int len)
{
     int addr;
     
     // Must be in one line
     if ( con->x + len > dispInfo.txmax )
        len = dispInfo.txmax - con->x;

     addr = TextAddress(con, con->x, con->y);
     memset(con->textBuff + addr, 0x20, len);
     memset(con->attrBuff + addr, (con->bcol << 4) | con->fcol, len);
     bzero2(con->flagBuff + addr, len);  /* needless to latch */
}

/****************************************************************************
 *                   TextCopy/TextPaste/TextReverse                         *
 ****************************************************************************/

static inline void KanjiAdjust(ConInfo *con,int *x, int *y)
{
    if (IsKanji2(con,*x, *y))
    {
        --*x;
    }
}

void    TextCopy(ConInfo *con,int fx, int fy, int tx, int ty)
{
    int fd;
    int       from, to, y, swp, xx, x;
    u_char      ch, ch2;

    unlink(CCE_TMP_FILE);
    if ((fd = open(CCE_TMP_FILE, O_WRONLY|O_CREAT, 0600)) < 0) return;

    KanjiAdjust(con,&fx, &fy);
    KanjiAdjust(con,&tx, &ty);
    if (fy > ty)
    {
        swp = fy; fy = ty; ty = swp;
        swp = fx; fx = tx; tx = swp;
    }
    else if (fy == ty && fx > tx)
    {
        swp = fx; fx = tx; tx = swp;
    }

    for (xx = dispInfo.txmax-1, y = fy; y <= ty; y ++)
    {
        if (y == ty) xx = tx;
        from = TextAddress(con,fx, y);
        if (con->flagBuff[from] & CODEIS_2)
            /* 2nd byte of kanji */
            from--;
        to = TextAddress(con,xx, y);
        for (x = to; x >= from; x --)
           if (con->textBuff[x] > ' ') break;
        to = x;

        for (x = from; x <= to; x ++)
        {
            ch = con->textBuff[x];
            if (!ch) ch = ' ';
            if (con->flagBuff[x] & CODEIS_1) 
            {
                x ++;
                ch2 = con->textBuff[x];
                switch(lInfo.sysCoding)
                {
                case CODE_EUC:
                    ch2 |= 0x80;
                    ch |= 0x80;
                    break;
                case CODE_SJIS:
                    jistosjis(ch2, ch);
                    break;
                }
                write(fd, &ch2, 1);
                write(fd, &ch, 1);
                /* write the 2nd byte first! */
            } 
            else write(fd, &ch, 1);
        }

        if (y < ty)
        {
            ch = '\n';
            write(fd, &ch, 1);
        }
        fx = 0;
    }
    close(fd);
}

void    TextPaste(ConInfo *con)
{
    u_char      ch;
    int fd;

    if ((fd = open(CCE_TMP_FILE, O_RDONLY)) < 0) return;
    while(read(fd, &ch, 1) == 1)
        write(con->masterPty, &ch, 1);
    close(fd);
}

void    TextReverse(ConInfo *con,int fx, int fy, int tx, int ty)
{
    int       from, to, y, swp, xx, x;
    u_char      fc, bc, fc2, bc2;

    KanjiAdjust(con,&fx, &fy);
    KanjiAdjust(con,&tx, &ty);
    if (fy > ty)
    {
        swp = fy; fy = ty; ty = swp;
        swp = fx; fx = tx; tx = swp;
    }
    else if (fy == ty && fx > tx)
    {
        swp = fx; fx = tx; tx = swp;
    }

    for (xx = dispInfo.txmax-1, y = fy; y <= ty; y ++)
    {
        if (y == ty) xx = tx;
        from = TextAddress(con,fx, y);
        to = TextAddress(con,xx, y);
        if (con->flagBuff[from] & CODEIS_2)
            /* 2nd byte of kanji */
            from--;

        for (x = from; x <= to; x ++)
        {
            if (!con->textBuff[x]) continue;
            fc = con->attrBuff[x];
            bc = fc >> 4;
            bc2 = (bc & 8) | (fc & 7);
            fc2 = (fc & 8) | (bc & 7);
            con->attrBuff[x] = fc2 | (bc2 << 4);
            con->flagBuff[x] &= ~CLEAN_S;
        }
        fx = 0;
    }
}

/****************************************************************************
 *                TextWput1/TextWput2/TextWput/TextSput                     *
 ****************************************************************************/

 void TextWput1(ConInfo *con,u_char ch)
{
    u_int addr;

    addr = TextAddress(con,con->x, con->y);
    con->attrBuff[addr] = con->fcol | (con->bcol << 4);
    con->textBuff[addr] = ch;
    con->flagBuff[addr] = con->db;
}

void TextWput2(ConInfo *con,u_char ch)
{
    u_int addr;

    addr = TextAddress(con,con->x, con->y);
    con->textBuff[addr] = ch;
    con->flagBuff[addr] = LATCH_2;
}

void TextWput(ConInfo *con,u_char ch1, u_char ch2)
{
    u_int addr;

    addr = TextAddress(con,con->x, con->y);
    con->attrBuff[addr] = con->fcol | (con->bcol << 4);
    con->textBuff[addr] = ch2;
    con->textBuff[addr+1] = ch1;
    con->flagBuff[addr] = con->db;
    con->flagBuff[addr+1] = LATCH_2;
}

void TextSput(ConInfo *con,u_char ch)
{
    u_int addr;

    addr = TextAddress(con,con->x, con->y);
    con->flagBuff[addr] = LATCH_S|con->sb;
    con->attrBuff[addr] = con->fcol | (con->bcol << 4);
    con->textBuff[addr] = ch;
}


/****************************************************************************
 *                         TextRefresh/HardScroll                           *
 *   (will call pVideoInfo->* functions, only refresh the current terminal)   *
 ****************************************************************************/

/* relative scroll, so when switch between different tty, there'll be
   some problem */

/* no more hardware scroll support, software scroll is not much slower and
   it's compatible with frame buffer */

#if 0
void HardScroll(ConInfo *con)
{
     int scroll,color;
     
     scroll = con->currentScroll + con->scrollLine;
     color = (con->attr & ATTR_REVERSE ? con->fcol:con->bcol)&7;

      /* con->currentScroll should be in 0-18  0-42 18+24=42 */
      if (con->scrollLine > 0) 
      {
           if (scroll < 19) 
	       pVideoInfo->hard_scroll_up(con->scrollLine,color);
           else
           {
	       pVideoInfo->hard_scroll_down(con->currentScroll,color);
               llatch(con->flagBuff, con->textSize);  /* force to redraw */
               scroll = 0;
           }
      }
      else if (con->scrollLine < 0) 
      {
           if (scroll > 0)
		pVideoInfo->hard_scroll_down(-con->scrollLine,color);
           else if (scroll < 0)
           {
                if (con->currentScroll > 0)
		    pVideoInfo->hard_scroll_down(con->currentScroll,color);
                llatch(con->flagBuff, con->textSize); /* force to redraw */
                scroll = 0;
           }
      }
      con->currentScroll = scroll;
      con->scrollLine = 0;
}

#endif

void	TextRefresh(ConInfo *con)
{
    int	i,j;
    u_int fnt;
    u_char	ch, ch2, fc, bc;
    int color;

    color = (con->attr & ATTR_REVERSE ? con->fcol:con->bcol)&7; 

    busy = TRUE;
    if (!curActive) 
    {
	busy = FALSE;
	return;
    }

    /* during refreshing, close the cursor and mouse cursor */
    ShowCursor(&cursorInfo, FALSE);
    ShowCursor(&mouseCursor, FALSE);

    if (con->textClear)
	pVideoInfo->clear_all(color);
    /*
    if (useHardScroll) 
        HardScroll(con);
    */

    con->textClear = FALSE;

    /* con->textSize = (txmax) * (tymax) = 80*25=2000 */
    for (j = 0; j < con->textSize; j ++) 
    {
        i = (con->textHead +j ) % con->textSize;
	if (con->flagBuff[i] & CLEAN_S) continue;
              /* already clean, don't need to redraw */
	pVideoInfo->set_address(j);
	fc = con->attrBuff[i];
	bc = con->attrBuff[i] >> 4;
	ch = con->textBuff[i];
	con->flagBuff[i] |= CLEAN_S;
	if (con->flagBuff[i] & CODEIS_1) 
        {
	    dbFReg = &fDRegs[con->flagBuff[i] & LANG_CODE];
	    i ++;
	    con->flagBuff[i] |= CLEAN_S;
	    ch2 = con->textBuff[i];
            fnt = dbFReg->addr(ch2, ch);
	    if (con->ins) TextInsertChar(con,2);
	    if (fnt < dbFReg->size)
		pVideoInfo->wput(dbFReg->bitmap + fnt, fc, bc);
	} 
        else 
        {
	    sbFReg = &fSRegs[con->flagBuff[i] & LANG_CODE];
	    if (con->ins) TextInsertChar(con,1);
		pVideoInfo->sput(/*ch ? */sbFReg->bitmap + ((int)ch << 4)/*:0*/, fc, bc);
	}
    }
    cursorInfo.kanji = IsKanji(con,con->x, con->y);
	pVideoInfo->set_cursor_address(&cursorInfo, con->x, con->y);

    ShowCursor(&cursorInfo, TRUE);
    busy = FALSE;
    
    if (release)
	LeaveVC(SIGUSR1);

}

void    TextRepaintAll(ConInfo *con)
{
    llatch(con->flagBuff, con->textSize);
    /* force to redraw all the chars */

    //pVideoInfo->reset_hard_scroll();   // reset the start address of buffer
    con->currentScroll = con->scrollLine = 0;
   // con->textClear = TRUE;
    TextRefresh(con);
    RefreshInputArea();  // window-0/1/2..
}

/*
static int      ConfigHardScroll(const char *confstr)
{
    bool value = BoolConf(confstr);
    useHardScroll = value;
    if (value) 
        message("Hardware scroll mode.\r\n");
    else message("Software scroll mode.\r\n");
    return SUCCESS;
}
*/

/***************************************************************************
 *                         GraphMode/TextMode/SetTextMode                  *
 ***************************************************************************/

static struct winsize text_win;

static void SetTextMode(void)
{
//    int  color = (curCon->attr & ATTR_REVERSE ? curCon->fcol:curCon->bcol)&7;
    ShowCursor(&cursorInfo, FALSE);
 //   pVideoInfo->clear_all(color);
   // do we need to clear all ?
    pVideoInfo->text_mode();
    ioctl(0, KDSETMODE, KD_TEXT);
    ioctl(0, TIOCCONS, NULL);
}

/*  FreeBSD comment:
    /usr/include/sys/signal.h
    NSIG 32
    SIGUSR1  30
    SIGUSR2  31

   /usr/src/sys/i386/isa/syscons.c  
   ISSIGVALID(sig)  ((sig) > 0 && (sig) < NSIG)
   So, VT_SETMODE's 
 */
  
/* No one will call this routine, in kon, the Cleanup routine
   will calll TextMode to return to normal Text Mode. But 
   I move the code to ConsoleCleanUp now */

/* 
  VT_RELDISP:  screen swicher ioctl 
   data:
  VT_FALSE:   user refuses to release screen abort
  VT_TRUE:    user has released screen, go on
  VT_ACKACQ:  acquire acknowledged, switch completed
*/

void TextMode(void)
{
    struct vt_mode vtm;

    signal(SIGUSR1, SIG_DFL);
    signal(SIGUSR2, SIG_DFL);

    vtm.mode = VT_AUTO;
    vtm.waitv = 0;
    vtm.relsig = 0;
    vtm.acqsig = 0;
    /* in FreeBSD, relsig, acqsig, frsig not in 1-31 range, VT_SETMODE will
       fail, if mode is VT_AUTO, relsig/acqsig is not used */

    ioctl(0, VT_SETMODE, &vtm);
    
#if defined(__FreeBSD__)
    ioctl(0, VT_RELDISP, 1);
#endif
    text_mode = TRUE;
 //   SetTextMode();
    ioctl(curCon->masterPty, TIOCSWINSZ, &text_win);
}

void GraphMode(void)
{
    struct winsize win;
    struct vt_mode vtm;

    text_mode = FALSE;
    ioctl(0, KDSETMODE, KD_GRAPHICS);

#if defined(__FreeBSD__)
    ioctl(0, VT_RELDISP, VT_ACKACQ);
#endif
    /* in FreeBSD, the new console need to acknowledge the acquirement,
       sometimes GraphMode() is not called by EnterVC, then this ioctl
       call will fail, but no problem */

    signal(SIGUSR1, LeaveVC);
    signal(SIGUSR2, EnterVC);
    vtm.mode = VT_PROCESS;

    /* FreeBSD: /usr/include/machine/console.h
        VT_PROCESS: switching controlled by prog
        VT_KERNEL:  switching controlled by kernel */

    vtm.waitv = 0;
    vtm.relsig = SIGUSR1;
    vtm.acqsig = SIGUSR2;
#if defined(__FreeBSD__)
    vtm.frsig = SIGUSR1; 
   /* for freebsd, added by raner or ioctl will fail */
#endif

   ioctl(0, VT_SETMODE, &vtm);

   pVideoInfo->graph_mode();
//   pVideoInfo->set_start_address();
    
   win.ws_row = dispInfo.tymax;	
        /* Note: con->ymax may be changed by application */
   win.ws_col = dispInfo.txmax;
   win.ws_xpixel = win.ws_ypixel = 0;
   ioctl(curCon->masterPty, TIOCSWINSZ, &win);
   ioctl(curCon->masterPty, TIOCCONS, NULL);
    
   llatch(curCon->flagBuff, curCon->textSize);
 //  curCon->textClear = TRUE;
   TextRefresh(curCon);
   RefreshInputArea();
}

/***************************************************************************
 *                            EnterVC/LeaveVC                              *
 ***************************************************************************/

void LeaveVC(int signum)
{
    signal(SIGUSR1, LeaveVC);  /* should use sigaction()? */
    if (busy) 
    {
        release = TRUE;
	return;
    }
    release = FALSE;
    curActive = FALSE;
    SetTextMode();
#ifdef	HAS_MOUSE
    if (mInfo.has_mouse) 
    {
	MouseResetRfd(mouseFd);
	MouseCleanup();
    }
#endif

#if defined(linux)
    ioctl(0, VT_RELDISP, 1);
#elif defined(__FreeBSD__)
    ioctl(0, VT_RELDISP, VT_TRUE);
#endif

    /* read the syscons.c source code, 
       if old console is in VT_PROCESS mode,switch_scr() will set the
       mark SWITCH_WAIT_REL and wait for the old console to
       release screen, old console call VT_RELDISP to clear the
       SWITCH_WAIT_REL and call exchange_scr() to continue switching,

       if the new console is in VT_PROCESS mode, the SWITCH_WAIT_ACQ
        mark will be set and signal will be issued, until new console
        to issue VT_RELDISP with VT_ACKACQ, then the whole process is
        completed */

    RestoreKeymap();
}

static void	EnterVC(int signum)
{
    signal(SIGUSR2, EnterVC);
    if (!curActive) 
    {
	curActive = TRUE;
	GraphMode();
    	signal(SIGUSR2, EnterVC);  // ?? 
#ifdef	HAS_MOUSE
	if (mInfo.has_mouse) 
        {
	    MouseStart();
	    MouseSetRfd(mouseFd);
	}
#endif
    }

    SetupKeymap();
}

/***************************************************************************
 *                         Screen Saver routines                           *
 ***************************************************************************/
 
static int      saveTime, saverCount;
static bool     saved;

static void     SaveScreen(bool save)
{
    if (saved != save)
    {
        saved = save;
        pVideoInfo->screen_saver(save);
    }
    saverCount = 0;
}

static int      ConfigSaver(const char *confstr)
{
    saveTime = atoi(confstr) * 600; /* convert unit from minitue to 1/10 sec */
    /* default 5 min. */
    return SUCCESS;
}


/****************************************************************************
 *                         Cursor related routines.                         *
 ****************************************************************************/

static void	ToggleCursor(CursorInfo *c)
{
    c->count = 0;
    if (text_mode)
	return;
    c->shown = ! c->shown;
    pVideoInfo->cursor(c);  /* draw the cursor using XOR operation */
}

static void	ShowCursor(CursorInfo *c, bool show)
{
    if (!curActive || !c->sw)
	return;
    if (c->shown != show)
	ToggleCursor(c);
}

static void PollMouseCursor(void)
{
    ShowCursor(&mouseCursor, FALSE);
    if (mInfo.sw > 0) 
    {
	--mInfo.sw;
	if (cursorInfo.shown) 
        {
	    int	x = mInfo.x, y = mInfo.y;
	    
	    KanjiAdjust(curCon,&x, &y);
	    mouseCursor.kanji = IsKanji(curCon,x, y);
       	    pVideoInfo->set_cursor_address(&mouseCursor, x, y);
	    ShowCursor(&mouseCursor, TRUE);
	}
    }
}

/* Called when some action was over, or every 1/10 sec when idle. */

void	PollCursor(bool wakeup)
{
    if (!curActive)
	return;

    if (wakeup) 
    {
	SaveScreen(FALSE);
	ShowCursor(&cursorInfo, TRUE);
	PollMouseCursor();
	return;
    }

    /* Idle, screen saver */
    if (saved)
	return;

    // screen saver
    if ((saveTime > 0) && (++saverCount == saveTime)) 
    {
	ShowCursor(&cursorInfo, FALSE);
	ShowCursor(&mouseCursor, FALSE);
	SaveScreen(TRUE);
	return;
    }

    // make it blinking
    if ((cursorInfo.interval > 0) && (++cursorInfo.count == cursorInfo.interval)) 
	ToggleCursor(&cursorInfo);

    if (mInfo.has_mouse) 
	PollMouseCursor();
}

static int      ConfigInterval(const char *confstr)
{
    cursorInfo.interval = mouseCursor.interval = atoi(confstr);
    return SUCCESS;
}

/****************************************************************************
 *                           Beep routines.                                 *
 ****************************************************************************/

#define	COUNTER_ADDR	0x61

static int	beepCount;

static int	ConfigBeep(const char *confstr)
{
    beepCount = atoi(confstr) * 10000;
#if defined(linux)
    if (beepCount > 0)
	ioperm(COUNTER_ADDR, 1, TRUE);
#endif
    return SUCCESS;
}

void	Beep(void)
{
    if (!curActive || beepCount <= 0) return;
#if defined(linux)
    PortOutb(PortInb(COUNTER_ADDR)|3, COUNTER_ADDR);
    usleep(beepCount);
    PortOutb(PortInb(COUNTER_ADDR)&0xFC, COUNTER_ADDR);
#endif
}

/***************************************************************************
 *                      Initialize & Cleanup routine.                      *
 ***************************************************************************/

/* After init, in ReadConfig, before Start */
static int ConfigDisplay(const char *confstr)
{
    int fd = -1;
    char name[10] = "VGA";  // VGA or S3 or... buffer overflow?

    if (strlen(confstr) < 10) 
    {
        sscanf(confstr, "%s", name);
    }

#if defined(linux)
    fd = open("/dev/fb",O_RDWR);
#endif

    if (fd == -1)  // error, maybe not in frame buffer mode
    {
        if (!strcasecmp(name,"S3"))   // no-case compare
           pVideoInfo = &VgaS3Info;
        else pVideoInfo = &VgaInfo;  
    }

#if defined(linux)
    else pVideoInfo = &VgaFBInfo;
#endif

    close(fd);

    return SUCCESS;
}

void	ConsoleInit()
{
    curActive = TRUE;

    DefineCap("Display", ConfigDisplay, "VGA");

//    DefineCap("HardScroll", ConfigHardScroll, "On");
    DefineCap("BeepCounter", ConfigBeep, "5");
    DefineCap("CursorInterval", ConfigInterval, "4");
    DefineCap("SaveTime", ConfigSaver, "4");

    DisplayDefaultCaps();
}

void	ConsoleStart(void)
{
    curActive = TRUE;

    ioctl(0, KDSETMODE, KD_GRAPHICS);
    ioctl(0, TIOCGWINSZ, &text_win);

    pVideoInfo->start();

    cursorInfo.shown = mouseCursor.shown = FALSE;
    mouseCursor.sw = TRUE;
    saved = FALSE;
    GraphMode();
}

void	ConsoleCleanup()
{
    struct vt_mode vtm;

#ifdef linux
    ioperm(COUNTER_ADDR, 1, FALSE);
#endif
    
  //  pVideoInfo->detach();

    /* TextMode(), but curCon == NULL now */
    signal(SIGUSR1, SIG_DFL);
    signal(SIGUSR2, SIG_DFL);

    vtm.mode = VT_AUTO;
    vtm.waitv = 0;
    vtm.relsig = 0;
    vtm.acqsig = 0;
    ioctl(0, VT_SETMODE, &vtm);
#if defined(__FreeBSD__)
    ioctl(0, VT_RELDISP, 1);
#endif

    /* SetTextMode(), curCon == NULL now */
    if (curActive)
    {
        pVideoInfo->text_mode();
        ioctl(0, KDSETMODE, KD_TEXT);
        ioctl(0, TIOCCONS, NULL);
    }

    pVideoInfo->detach();
}
