/*
 * CCE - Console Chinese Environment -
 * Copyright (C) 1998-1999 Rui He (herui@cs.duke.edu)
 *
 * Arthur Que <arthur@turbolinux.com.cn>
 * FreeBSD keymap patch
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY TAKASHI MANABE ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 */

#include	<stdio.h>
#include	<stdlib.h>
#include	<unistd.h>
#include	<string.h>
#include	<signal.h>
#include 	<sys/file.h>

#if defined(linux)
#include	<termio.h>
#elif defined(__FreeBSD__)
#include	<termios.h>
#define termio termios
#endif

#if defined(linux)
#include        <sys/vt.h>
#include        <sys/kd.h>
#include 	<linux/keyboard.h>
#include 	<vgakeyboard.h>
#elif defined(__FreeBSD__)
#include        <machine/console.h>
#endif

#include	<term.h>
#include        <hzinput.h>
#include	<vt.h>
#include	<key.h>

#if defined(linux)

#define CTRL_ALT   ((1 << KG_CTRL) + (1 << KG_ALT))

struct _KeymapData
{  
    int table, index, oldValue, newValue;
}KMData[NR_SPEC_KEY] =  // 18
{
  { CTRL_ALT,  SCANCODE_0,  0, CTRL_ALT_0},
  { CTRL_ALT,  SCANCODE_1,  0, CTRL_ALT_1},
  { CTRL_ALT,  SCANCODE_2,  0, CTRL_ALT_2},
  { CTRL_ALT,  SCANCODE_3,  0, CTRL_ALT_3},
  { CTRL_ALT,  SCANCODE_4,  0, CTRL_ALT_4},
  { CTRL_ALT,  SCANCODE_5,  0, CTRL_ALT_5},
  { CTRL_ALT,  SCANCODE_6,  0, CTRL_ALT_6},
  { CTRL_ALT,  SCANCODE_7,  0, CTRL_ALT_7},
  { CTRL_ALT,  SCANCODE_8,  0, CTRL_ALT_8},
  { CTRL_ALT,  SCANCODE_9,  0, CTRL_ALT_9},

  { CTRL_ALT,  SCANCODE_A,  0, CTRL_ALT_A},
  { CTRL_ALT,  SCANCODE_X,  0, CTRL_ALT_X},
  { CTRL_ALT,  SCANCODE_P,  0, CTRL_ALT_P},
  { CTRL_ALT,  SCANCODE_N,  0, CTRL_ALT_N},
  { CTRL_ALT,  SCANCODE_R,  0, CTRL_ALT_R},

  { 1 << KG_CTRL,  SCANCODE_SPACE,       0, CTRL_SPACE},
  { 1 << KG_SHIFT, SCANCODE_SPACE,       0, SHIFT_SPACE},
  { 0,             SCANCODE_SCROLLLOCK,  0, SCROLL_LOCK}
};

#elif defined(__FreeBSD__)

keymap_t old_keymap;

#endif

static void GetOldKeymap(void)
{
#if defined(linux)

   struct kbentry spec;
   int i,cfd;

   cfd = open("/dev/console", O_WRONLY);
   if (cfd < 0 && (cfd = open("/dev/console", O_RDONLY)) < 0)
   {
        PerrorExit("/dev/console");
   }

   for (i = 0; i < NR_SPEC_KEY; i++)
   {
       spec.kb_table = KMData[i].table;
       spec.kb_index = KMData[i].index ;
       ioctl(cfd, KDGKBENT, &spec);
       KMData[i].oldValue =  spec.kb_value;
    }

#elif defined(__FreeBSD__)
 
   if (ioctl(0, GIO_KEYMAP, &old_keymap) < 0)
   {
     PerrorExit("getting keymap");
   }

#endif

}

void SetupKeymap(void)
{
#if defined(linux)

   int i,cfd;
   struct kbentry spec;

   cfd = open("/dev/console", O_WRONLY);
   if (cfd < 0 && (cfd = open("/dev/console", O_RDONLY)) < 0) 
   {
        PerrorExit("/dev/console");
   }

   for(i = 0; i < NR_SPEC_KEY; i++)
   {
      spec.kb_table = KMData[i].table;
      spec.kb_index = KMData[i].index ;
      spec.kb_value = K(KT_LATIN, KMData[i].newValue);
      ioctl(cfd, KDSKBENT, &spec);
  }

#elif defined(__FreeBSD__)
  
  keymap_t new_keymap;

  memcpy(&new_keymap, &old_keymap, sizeof(keymap_t));
  
  new_keymap.key[57].map[2] = CTRL_SPACE; // 210
  new_keymap.key[57].map[1] = SHIFT_SPACE; // 211

  new_keymap.key[59].map[4] = ALT_F1;      
  new_keymap.key[59].map[6] = CTRL_ALT_F1;  
  new_keymap.key[60].map[4] = ALT_F2;     
  new_keymap.key[60].map[6] = CTRL_ALT_F2;
  new_keymap.key[61].map[4] = ALT_F3;     
  new_keymap.key[61].map[6] = CTRL_ALT_F3;
  new_keymap.key[62].map[4] = ALT_F4;
  new_keymap.key[62].map[6] = CTRL_ALT_F4;
  new_keymap.key[63].map[4] = ALT_F5;
  new_keymap.key[63].map[6] = CTRL_ALT_F5;
  new_keymap.key[64].map[4] = ALT_F6;
  new_keymap.key[64].map[6] = CTRL_ALT_F6;
  new_keymap.key[65].map[4] = ALT_F7;
  new_keymap.key[65].map[6] = CTRL_ALT_F7;

  new_keymap.key[2].map[6] = CTRL_ALT_1; // 201
  new_keymap.key[3].map[6] = CTRL_ALT_2; // 202
  new_keymap.key[4].map[6] = CTRL_ALT_3; // 203
  new_keymap.key[5].map[6] = CTRL_ALT_4; // 204
  new_keymap.key[6].map[6] = CTRL_ALT_5; // 205
  new_keymap.key[7].map[6] = CTRL_ALT_6; // 206
  new_keymap.key[8].map[6] = CTRL_ALT_7; // 207
  new_keymap.key[9].map[6] = CTRL_ALT_8; // 208
  new_keymap.key[10].map[6] = CTRL_ALT_9; // 209
  new_keymap.key[11].map[6] = CTRL_ALT_0; // 200

  new_keymap.key[30].map[6] = CTRL_ALT_A; // 212
  new_keymap.key[45].map[6] = CTRL_ALT_X; // 213
  new_keymap.key[49].map[6] = CTRL_ALT_N; // 214
  new_keymap.key[25].map[6] = CTRL_ALT_P; // 215
  new_keymap.key[19].map[6] = CTRL_ALT_R; // 216
//  new_keymap.key[70].map[2] = CTRL_SCROLL; // 217

  // Write Keyboard Keymap now;

  if (ioctl(0, PIO_KEYMAP, &new_keymap) < 0) 
  {
    PerrorExit("setting keymap\n");
  }
#endif
   
}

void RestoreKeymap(void)
{
#if defined(linux)
  
  struct kbentry spec;
  int i,cfd;

  cfd = open("/dev/console", O_WRONLY);
  if (cfd < 0 && (cfd = open("/dev/console", O_RDONLY)) < 0) 
  {
       PerrorExit("/dev/console");
  }

  for (i = 0; i < NR_SPEC_KEY; i++)
  {
    spec.kb_table = KMData[i].table;
    spec.kb_index = KMData[i].index;
    spec.kb_value = KMData[i].oldValue;
    ioctl(cfd, KDSKBENT, &spec);
  }

#elif defined(__FreeBSD__)

  if (ioctl(0, PIO_KEYMAP, &old_keymap) < 0)
    PerrorExit("restoring keymap\n");

#endif  

}

void KeymapInit(void)
{
   GetOldKeymap();
   SetupKeymap();
}

void KeymapCleanup(void)
{
   RestoreKeymap();
}

void ProcessNormalModeKey(int tty_fd,unsigned char c)
{
    switch(c)
    {

#if defined(__FreeBSD__)

      case ALT_F1: // case CTRL_ALT_F1:
      case ALT_F2: // case CTRL_ALT_F2:
      case ALT_F3: // case CTRL_ALT_F3:
      case ALT_F4: // case CTRL_ALT_F4:
      case ALT_F5: // case CTRL_ALT_F5:
      case ALT_F6: // case CTRL_ALT_F6:
      case ALT_F7: // case CTRL_ALT_F7:
      /*
      case ALT_F8: // case CTRL_ALT_F8:
      case ALT_F9: // case CTRL_ALT_F9:
      case ALT_F10: // case CTRL_ALT_F10:
      */
      {
          int cfd = open("/dev/vga",O_WRONLY);
          if (cfd >= 0)
          {
              ioctl(0, VT_ACTIVATE, c-ALT_F1+1);
          /* if current vt is in graphics state and mode is VT_AUTO,
              VT_ACTIVATE will fail */
              close(cfd);
          }
          break;
      }
#endif

       case CTRL_ALT_0:
       case CTRL_ALT_1:
       case CTRL_ALT_2:
       case CTRL_ALT_3:
       case CTRL_ALT_4:
       case CTRL_ALT_5:
       case CTRL_ALT_6:
       case CTRL_ALT_7:
       case CTRL_ALT_8:
       case CTRL_ALT_9:
	  SetCurrentInputMethod( c - CTRL_ALT_0);
          break;
       case CTRL_ALT_A:
          TermOpen();
          TextRepaintAll(curCon);
          break;
       case CTRL_ALT_X:
          if ( curCon->terminate == 0 )
               kill( curCon->childPid, SIGHUP);
          break;
       case CTRL_ALT_N:
          ChangeTerm(1);
          break;
       case CTRL_ALT_P:
          ChangeTerm(0);
          break;
       case CTRL_ALT_R:
          TextRepaintAll(curCon);
          break;
       case CTRL_SPACE:
	  ToggleInputMethod();;
          break;
       case SHIFT_SPACE:
	  ToggleHalfFull();
          break;
       case SCROLL_LOCK:
          break;
       default:
          HZFilter(tty_fd,c);
          break;
    }
}

