#include "mutation.h"

#include "catacharset.h"
#include "debug.h"
#include "game.h"
#include "input.h"
#include "output.h"
#include "player.h"
#include "translations.h"
#include "string_formatter.h"

#include <algorithm> //std::min
#include <sstream>

// '!' and '=' are uses as default bindings in the menu
const invlet_wrapper
mutation_chars( "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ\"#&()*+./:;@[\\]^_{|}" );

void draw_exam_window( const catacurses::window &win, const int border_y )
{
    int width = getmaxx( win );
    mvwputch( win, border_y, 0, BORDER_COLOR, LINE_XXXO );
    mvwhline( win, border_y, 1, LINE_OXOX, width - 2 );
    mvwputch( win, border_y, width - 1, BORDER_COLOR, LINE_XOXX );
}

const auto shortcut_desc = []( const std::string &comment, const std::string &keys )
{
    return string_format( comment.c_str(),
                          string_format( "<color_yellow>%s</color>", keys.c_str() ).c_str() );
};

void show_mutations_titlebar( const catacurses::window &window, std::string &menu_mode,
                              input_context &ctxt )
{
    werase( window );
    std::ostringstream desc;
    if( menu_mode == "reassigning" ) {
        desc << _( "Reassigning." ) << "  " <<
             _( "Select a mutation to reassign or press <color_yellow>SPACE</color> to cancel. " );
    }
    if( menu_mode == "activating" ) {
        desc << "<color_green>" << _( "Activating" ) << "</color>  " <<
             shortcut_desc( _( "%s to examine mutation, " ), ctxt.get_desc( "TOGGLE_EXAMINE" ) );
    }
    if( menu_mode == "examining" ) {
        desc << "<color_light_blue>" << _( "Examining" ) << "</color>  " <<
             shortcut_desc( _( "%s to activate mutation, " ), ctxt.get_desc( "TOGGLE_EXAMINE" ) );
    }
    if( menu_mode != "reassigning" ) {
        desc << shortcut_desc( _( "%s to reassign invlet, " ), ctxt.get_desc( "REASSIGN" ) );
    }
    desc << shortcut_desc( _( "%s to assign the hotkeys." ), ctxt.get_desc( "HELP_KEYBINDINGS" ) );
    fold_and_print( window, 0, 1, getmaxx( window ) - 1, c_white, desc.str() );
    wrefresh( window );
}

void player::power_mutations()
{
    if( !is_player() ) {
        // TODO: Implement NPCs activating mutations
        return;
    }

    std::vector<trait_id> passive;
    std::vector<trait_id> active;
    for( auto &mut : my_mutations ) {
        if( !mut.first->activated ) {
            passive.push_back( mut.first );
        } else {
            active.push_back( mut.first );
        }
        // New mutations are initialized with no key at all, so we have to do this here.
        if( mut.second.key == ' ' ) {
            for( const auto &letter : mutation_chars ) {
                if( trait_by_invlet( letter ).is_null() ) {
                    mut.second.key = letter;
                    break;
                }
            }
        }
    }

    // maximal number of rows in both columns
    const int mutations_count = std::max( passive.size(), active.size() );

    int TITLE_HEIGHT = 2;
    int DESCRIPTION_HEIGHT = 5;

    // Main window
    /** Total required height is:
    * top frame line:                                         + 1
    * height of title window:                                 + TITLE_HEIGHT
    * line after the title:                                   + 1
    * line with active/passive mutation captions:               + 1
    * height of the biggest list of active/passive mutations:   + mutations_count
    * line before mutation description:                         + 1
    * height of description window:                           + DESCRIPTION_HEIGHT
    * bottom frame line:                                      + 1
    * TOTAL: TITLE_HEIGHT + mutations_count + DESCRIPTION_HEIGHT + 5
    */
    int HEIGHT = std::min( TERMY, std::max( FULL_SCREEN_HEIGHT,
                                            TITLE_HEIGHT + mutations_count + DESCRIPTION_HEIGHT + 5 ) );
    int WIDTH = FULL_SCREEN_WIDTH + ( TERMX - FULL_SCREEN_WIDTH ) / 2;
    int START_X = ( TERMX - WIDTH ) / 2;
    int START_Y = ( TERMY - HEIGHT ) / 2;
    catacurses::window wBio = catacurses::newwin( HEIGHT, WIDTH, START_Y, START_X );

    // Description window @ the bottom of the bionic window
    int DESCRIPTION_START_Y = START_Y + HEIGHT - DESCRIPTION_HEIGHT - 1;
    int DESCRIPTION_LINE_Y = DESCRIPTION_START_Y - START_Y - 1;
    catacurses::window w_description = catacurses::newwin( DESCRIPTION_HEIGHT, WIDTH - 2,
                                       DESCRIPTION_START_Y, START_X + 1 );

    // Title window
    int TITLE_START_Y = START_Y + 1;
    int HEADER_LINE_Y = TITLE_HEIGHT + 1; // + lines with text in titlebar, local
    catacurses::window w_title = catacurses::newwin( TITLE_HEIGHT, WIDTH - 2, TITLE_START_Y,
                                 START_X + 1 );

    int scroll_position = 0;
    int second_column = 32 + ( TERMX - FULL_SCREEN_WIDTH ) /
                        4; // X-coordinate of the list of active mutations

    input_context ctxt( "MUTATIONS" );
    ctxt.register_updown();
    ctxt.register_action( "ANY_INPUT" );
    ctxt.register_action( "TOGGLE_EXAMINE" );
    ctxt.register_action( "REASSIGN" );
    ctxt.register_action( "HELP_KEYBINDINGS" );
    bool redraw = true;
    std::string menu_mode = "activating";

    while( true ) {
        // offset for display: mutation with index i is drawn at y=list_start_y+i
        // drawing the mutation starts with mutation[scroll_position]
        const int list_start_y = HEADER_LINE_Y + 2 - scroll_position;
        int max_scroll_position = HEADER_LINE_Y + 2 + mutations_count -
                                  ( ( menu_mode == "examining" ) ? DESCRIPTION_LINE_Y : ( HEIGHT - 1 ) );
        if( redraw ) {
            redraw = false;

            werase( wBio );
            draw_border( wBio, BORDER_COLOR, _( " MUTATIONS " ) );
            // Draw line under title
            mvwhline( wBio, HEADER_LINE_Y, 1, LINE_OXOX, WIDTH - 2 );
            // Draw symbols to connect additional lines to border
            mvwputch( wBio, HEADER_LINE_Y, 0, BORDER_COLOR, LINE_XXXO ); // |-
            mvwputch( wBio, HEADER_LINE_Y, WIDTH - 1, BORDER_COLOR, LINE_XOXX ); // -|

            // Captions
            mvwprintz( wBio, HEADER_LINE_Y + 1, 2, c_light_blue, _( "Passive:" ) );
            mvwprintz( wBio, HEADER_LINE_Y + 1, second_column, c_light_blue, _( "Active:" ) );

            if( menu_mode == "examining" ) {
                draw_exam_window( wBio, DESCRIPTION_LINE_Y );
            }
            nc_color type;
            if( passive.empty() ) {
                mvwprintz( wBio, list_start_y, 2, c_light_gray, _( "None" ) );
            } else {
                for( size_t i = scroll_position; i < passive.size(); i++ ) {
                    const auto &md = passive[i].obj();
                    const auto &td = my_mutations[passive[i]];
                    if( list_start_y + static_cast<int>( i ) ==
                        ( menu_mode == "examining" ? DESCRIPTION_LINE_Y : HEIGHT - 1 ) ) {
                        break;
                    }
                    type = ( has_base_trait( passive[i] ) ? c_cyan : c_light_cyan );
                    mvwprintz( wBio, list_start_y + i, 2, type, "%c %s", td.key, md.name.c_str() );
                }
            }

            if( active.empty() ) {
                mvwprintz( wBio, list_start_y, second_column, c_light_gray, _( "None" ) );
            } else {
                for( size_t i = scroll_position; i < active.size(); i++ ) {
                    const auto &md = active[i].obj();
                    const auto &td = my_mutations[active[i]];
                    if( list_start_y + static_cast<int>( i ) ==
                        ( menu_mode == "examining" ? DESCRIPTION_LINE_Y : HEIGHT - 1 ) ) {
                        break;
                    }
                    if( td.powered ) {
                        type = ( has_base_trait( active[i] ) ? c_green : c_light_green );
                    } else {
                        type = ( has_base_trait( active[i] ) ? c_red : c_light_red );
                    }
                    // TODO: track resource(s) used and specify
                    mvwputch( wBio, list_start_y + i, second_column, type, td.key );
                    std::ostringstream mut_desc;
                    mut_desc << md.name;
                    if( md.cost > 0 && md.cooldown > 0 ) {
                        //~ RU means Resource Units
                        mut_desc << string_format( _( " - %d RU / %d turns" ),
                                                   md.cost, md.cooldown );
                    } else if( md.cost > 0 ) {
                        //~ RU means Resource Units
                        mut_desc << string_format( _( " - %d RU" ), md.cost );
                    } else if( md.cooldown > 0 ) {
                        mut_desc << string_format( _( " - %d turns" ), md.cooldown );
                    }
                    if( td.powered ) {
                        mut_desc << _( " - Active" );
                    }
                    mvwprintz( wBio, list_start_y + i, second_column + 2, type,
                               mut_desc.str().c_str() );
                }
            }

            // Scrollbar
            if( scroll_position > 0 ) {
                mvwputch( wBio, HEADER_LINE_Y + 2, 0, c_light_green, '^' );
            }
            if( scroll_position < max_scroll_position && max_scroll_position > 0 ) {
                mvwputch( wBio, ( menu_mode == "examining" ? DESCRIPTION_LINE_Y : HEIGHT - 1 ) - 1,
                          0, c_light_green, 'v' );
            }
        }
        wrefresh( wBio );
        show_mutations_titlebar( w_title, menu_mode, ctxt );
        const std::string action = ctxt.handle_input();
        const long ch = ctxt.get_raw_input().get_first_input();
        if( menu_mode == "reassigning" ) {
            menu_mode = "activating";
            const auto mut_id = trait_by_invlet( ch );
            if( mut_id.is_null() ) {
                // Selected an non-existing mutation (or escape, or ...)
                continue;
            }
            redraw = true;
            const long newch = popup_getkey( _( "%s; enter new letter." ),
                                             mutation_branch::get_name( mut_id ).c_str() );
            wrefresh( wBio );
            if( newch == ch || newch == ' ' || newch == KEY_ESCAPE ) {
                continue;
            }
            if( !mutation_chars.valid( newch ) ) {
                popup( _( "Invalid mutation letter. Only those characters are valid:\n\n%s" ),
                       mutation_chars.get_allowed_chars().c_str() );
                continue;
            }
            const auto other_mut_id = trait_by_invlet( newch );
            if( !other_mut_id.is_null() ) {
                std::swap( my_mutations[mut_id].key, my_mutations[other_mut_id].key );
            } else {
                my_mutations[mut_id].key = newch;
            }
            // TODO: show a message like when reassigning a key to an item?
        } else if( action == "DOWN" ) {
            if( scroll_position < max_scroll_position ) {
                scroll_position++;
                redraw = true;
            }
        } else if( action == "UP" ) {
            if( scroll_position > 0 ) {
                scroll_position--;
                redraw = true;
            }
        } else if( action == "REASSIGN" ) {
            menu_mode = "reassigning";
        } else if( action == "TOGGLE_EXAMINE" ) { // switches between activation and examination
            menu_mode = menu_mode == "activating" ? "examining" : "activating";
            werase( w_description );
            redraw = true;
        } else if( action == "HELP_KEYBINDINGS" ) {
            redraw = true;
        } else {
            const auto mut_id = trait_by_invlet( ch );
            if( mut_id.is_null() ) {
                // entered a key that is not mapped to any mutation,
                // -> leave screen
                break;
            }
            const auto &mut_data = mut_id.obj();
            if( menu_mode == "activating" ) {
                if( mut_data.activated ) {
                    if( my_mutations[mut_id].powered ) {
                        add_msg_if_player( m_neutral, _( "You stop using your %s." ), mut_data.name.c_str() );

                        deactivate_mutation( mut_id );
                        // Action done, leave screen
                        break;
                    } else if( ( !mut_data.hunger || get_hunger() <= 400 ) &&
                               ( !mut_data.thirst || get_thirst() <= 400 ) &&
                               ( !mut_data.fatigue || get_fatigue() <= 400 ) ) {

                        g->draw();
                        add_msg_if_player( m_neutral, _( "You activate your %s." ), mut_data.name.c_str() );
                        activate_mutation( mut_id );
                        // Action done, leave screen
                        break;
                    } else {
                        popup( _( "You don't have enough in you to activate your %s!" ), mut_data.name.c_str() );
                        redraw = true;
                        continue;
                    }
                } else {
                    popup( _( "\
You cannot activate %s!  To read a description of \
%s, press '!', then '%c'." ), mut_data.name.c_str(), mut_data.name.c_str(),
                           my_mutations[mut_id].key );
                    redraw = true;
                }
            }
            if( menu_mode == "examining" ) { // Describing mutations, not activating them!
                draw_exam_window( wBio, DESCRIPTION_LINE_Y );
                // Clear the lines first
                werase( w_description );
                fold_and_print( w_description, 0, 0, WIDTH - 2, c_light_blue, mut_data.description );
                wrefresh( w_description );
            }
        }
    }
}
