/* The Cantus project.
 * (c)2002, 2003 by Samuel Abels (spam debain org)
 * This project's homepage is: http://www.debain.org/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "gui_gtkfilelist.h"


static const gchar *defaultpattern[] = { "*.*", NULL };


/******************************************************************************
 * Constructor/Destructor
 ******************************************************************************/
GtkFileList::GtkFileList(void)
{
  blocked     = FALSE;
  curdir      = getenv("HOME");
  showhidden  = FALSE;
  filepattern = defaultpattern;
  store       = Gtk::ListStore::create(columns);
  sortstore   = Gtk::TreeModelSort::create(store);
  sortstore->set_sort_column_id(columns.filename, Gtk::SORT_ASCENDING);
  
  // Attach the liststore to the treeview.
  set_model(sortstore);
  append_column(_("Filename"), columns.filename);
  get_column(0)->set_resizable(TRUE);
  get_column(0)->set_sort_column_id(0);
  append_column(_("Size"),     columns.size);
  get_column(1)->set_resizable(TRUE);
  get_column(1)->set_sort_column_id(1);
  append_column(_("Mode"),     columns.mode);
  get_column(2)->set_resizable(TRUE);
  get_column(2)->set_sort_column_id(2);
  append_column(_("Owner"),    columns.owner);
  get_column(3)->set_resizable(TRUE);
  get_column(3)->set_sort_column_id(3);
  
  set_headers_clickable();
  get_selection()->set_mode(Gtk::SELECTION_EXTENDED);
  get_selection()->signal_changed().connect(
          SigC::slot(*this, &GtkFileList::on_selection_changed), TRUE);
}


GtkFileList::~GtkFileList(void)
{
}


/******************************************************************************
 * Public
 ******************************************************************************/
/* Jump to the given directory. Returns TRUE on success, otherwise FALSE.
 */
bool GtkFileList::set_directory(std::string dir)
{
  struct stat filestat;
  if (dir.substr(dir.length() - 1, 1) != "/")  // Append a trailing slash.
    dir = dir + "/";
  stat(dir.c_str(), &filestat);                // Make sure it is a directory.
  if (!filestat.st_mode & S_IFDIR)
    return FALSE;
  curdir = dir;
  update();
  return TRUE;
}


/* Specify a list of patterns that represent filenames to be shown in the
 * filelist.
 */
void GtkFileList::set_filepattern(const gchar **pfilepattern)
{
  filepattern = pfilepattern;
  update();
}


/* Specify a comma-sperated list of patterns that represent filenames to be
 * shown in the filelist.
 */
void GtkFileList::set_filepattern(std::string str)
{
  if (str == "")
    return;
  char *part = NULL;
  int i = 0;
  filepattern_str = str;
  for (part = strtok((char*)filepattern_str.c_str(), ",");
       part != NULL;
       i++, part = strtok(NULL, ","))
    filepattern[i] = part;
  filepattern[i] = NULL;
  set_filepattern(filepattern);
}


/* Specify whether or not hidden files and directories should be visible.
 */
void GtkFileList::set_showhidden(gboolean show_hidden)
{
  if (showhidden == show_hidden)
    return;
  showhidden = show_hidden;
  update();
}


/* Triggers a refresh of the current directory.
 * If "hard" is TRUE, the filelist is cleared and re-read (the selection is
 * lost).
 */
void GtkFileList::update(bool hard)
{
  if (hard) {
    store->clear();
    in_list.clear();
  }
  else
    remove_dead();
  load();
}


/* Returns a list of all selected filenames as a C-compatible GList.
 */
GList *GtkFileList::get_selection_glist(void)
{
  GList *filelist = NULL;
  get_selection()->selected_foreach(
        SigC::bind(
            SigC::slot(*this, &GtkFileList::glist_filename_append), &filelist));
  return filelist;
}


/******************************************************************************
 * Protected
 ******************************************************************************/
void GtkFileList::on_selection_changed(void)
{
  if (blocked)
    get_selection()->signal_changed().emission_stop();
}


/* Helper function to create a GList of all selected filenames.
 */
void GtkFileList::glist_filename_append(const Gtk::TreeModel::iterator& iter,
                                        GList **filelist)
{
  Gtk::TreeRow row     = *iter;
  std::string filename = row.get_value(columns.fullfilename);
  *filelist = g_list_append(*filelist,
                            (void*)in_list.find(filename)->c_str());
}


/* Helper function for iterating through the tree model. This function
 * checks if the given file does still exist and removes dead items from the
 * "in_list" list. Also, it appends the path to a list.
 */
bool GtkFileList::remove_if_dead(const Gtk::TreeIter& iter,
                                 std::list<Gtk::TreeIter> *dead)
{
  std::string filename = iter->get_value(columns.fullfilename);
  struct stat filestatus;
  if (stat(filename.c_str(), &filestatus) == -1) {
    dead->push_back(iter);
    in_list.erase(filename);
  }
  return FALSE;
}


/* Re-reads all files and updates differences only.
 */
void GtkFileList::remove_dead(void)
{
  std::list<Gtk::TreeIter>           deadfiles;
  std::list<Gtk::TreeIter>::iterator iter;
  store->foreach(
       SigC::bind(SigC::slot(*this, &GtkFileList::remove_if_dead), &deadfiles));
  iter = deadfiles.begin();
  blocked = TRUE;
  while (iter != deadfiles.end()) {
    store->erase(*iter);
    iter++;
  }
  blocked = FALSE;
}


/* Clears the filelist and re-reads all files.
 */
void GtkFileList::load(void)
{
  DIR                *stream       = NULL;
  struct dirent      *filestruct   = NULL;
  struct stat         filestatus;
  struct passwd      *pwd;
  struct group       *grp;
  std::string         fullfilename;
  char                size[20];
  char                mode[30];
  Gtk::TreeModel::Row row;
  
  if (!(stream = opendir(curdir.c_str()))) {        // Open the directory.
    printf("GtkFileList::update_hard(): Path open failed. %s\n",
           curdir.c_str());
    return;
  }
  
  while ((filestruct = readdir(stream)) != NULL) {  // Walk through all files.
    // Skip hidden files?
    if (!showhidden && strncmp(filestruct->d_name, ".", 1) == 0)
      continue;
    
    // Create a full path.
    fullfilename = curdir + filestruct->d_name;
    
    // Already in the list?
    if (in_list.find(fullfilename) != in_list.end())
      continue;
    
    // Does the glob pattern match?
    if (filepattern
      && !shellpattern_match_any(filestruct->d_name, filepattern, FALSE))
      continue;
    
    // Stat the file.
    if (stat(fullfilename.c_str(), &filestatus) == -1)
      continue;
    
    // Skip directories.
    if (S_ISDIR(filestatus.st_mode) != 0)
      continue;
    
    // Create a human readable size and mode.
    snprintf(size, 19, "%i kb",  (gint)(filestatus.st_size / 1024));
    snprintf(mode, 29, "%i%i%i", (gint)(filestatus.st_mode & 0700) >> 6,
                                 (gint)(filestatus.st_mode & 0070) >> 3,
                                 (gint)(filestatus.st_mode & 0007));
    
    // Create a human readable owner.
    pwd = getpwuid(filestatus.st_uid);
    grp = getgrgid(filestatus.st_gid);
    std::string owner;
    owner.append((pwd && pwd->pw_name) ? pwd->pw_name : "?");
    owner.append(":");
    owner.append((grp && grp->gr_name) ? grp->gr_name : "?");
    
    // Add the new row/file.
    row = *store->append();
    in_list.insert(fullfilename);
    
#ifdef _DEBUG_
      printf("GtkFileList::update_hard(): %s, %s\n", filestruct->d_name, mode);
#endif
    
    row[columns.filename]     = filestruct->d_name;
    row[columns.size]         = size;
    row[columns.mode]         = mode;
    row[columns.owner]        = owner;
    row[columns.fullfilename] = fullfilename;
  }
  
  closedir(stream);
}
