/* The Cantus project.
 * (c)2002, 2003, 2004 by Samuel Abels (spam debain org)
 * This project's homepage is: http://www.debain.org/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "tageditor.h"

//#define _DEBUG_


/******************************************************************************
 * Constructor/Destructor
 ******************************************************************************/
TagEditor::TagEditor(CantusHash *pplugindata)
{
  plugindata = pplugindata;
  selected   = NULL;
  
  // Initialize the mapping between hash keys and tag struct.
  hashmap["OGG:Artist"]   = &tag.artist;
  hashmap["OGG:Song"]     = &tag.title;
  hashmap["OGG:Album"]    = &tag.album;
  hashmap["OGG:Track"]    = &tag.track;
  hashmap["OGG:Year"]     = &tag.year;
  hashmap["OGG:Genre"]    = &tag.genre;
  hashmap["OGG:Comment"]  = &tag.comment;
  
  // Connect all signals with Cantus.
  CantusAddListenerSigCFunc addlistener = (CantusAddListenerSigCFunc)
                  cantushash_get_pointer(plugindata, "Cantus:AddListenerSigC");
  g_return_if_fail(addlistener != NULL);
  listenerids.push_back(
    addlistener("Filelist:Read:Start",
              SigC::slot(*this, &TagEditor::on_selection_changed_event)));
  listenerids.push_back(
    addlistener("File:Read:Finished",
              SigC::slot(*this, &TagEditor::on_file_read_finished_event)));
  listenerids.push_back(
    addlistener("Filelist:Read:Finished",
              SigC::slot(*this, &TagEditor::on_filelist_read_finished_event)));
  listenerids.push_back(
    addlistener("GUI:PluginWidget:Destroyed",
              SigC::slot(*this, &TagEditor::on_uiwidget_destroyed_event)));
}


TagEditor::~TagEditor(void)
{
#ifdef _DEBUG_
  printf("vorbis(): TagEditor::~TagEditor(): Called.\n");
#endif
  // Disconnect all signals from Cantus.
  CantusRemoveListenerFunc removelistener = (CantusRemoveListenerFunc)
                  cantushash_get_pointer(plugindata, "Cantus:RemoveListener");
  g_return_if_fail(removelistener != NULL);
  std::list<long>::iterator iter = listenerids.begin();
  while (iter != listenerids.end()) {
    removelistener(*iter);
    iter++;
  }
#ifdef _DEBUG_
  printf("vorbis(): TagEditor::~TagEditor(): Finished.\n");
#endif
}


/******************************************************************************
 * Public
 ******************************************************************************/
GtkWidget *TagEditor::editarea_build(gboolean vertical)
{
  GtkWidget *editareawidget = editarea.build(vertical);
  editarea.signal_button_save_clicked.connect(
                SigC::slot(*this, &TagEditor::on_editarea_button_save_clicked));
  return editareawidget;
}


void TagEditor::gui_to_hash(CantusHash *hash)
{
#ifdef _DEBUG_
  printf("vorbis(): TagEditor::gui_to_hash(): Called.\n");
#endif
  std::map<const gchar*, void*>::iterator cur = hashmap.begin();
  const gchar *field = NULL;
  while (cur != hashmap.end()) {
#ifdef _DEBUG_
  printf("vorbis(): TagEditor::gui_to_hash(): Handling %s.\n", (*cur).first);
#endif
    gchar *fieldname = strchr((*cur).first, ':') + 1;
    gchar *checkname = g_strconcat(fieldname, ":Check", NULL);
    if (editarea.get_check_active(checkname)) {
      const gchar *value = NULL;
      if (strcmp(fieldname, "Comment") == 0)
        value = editarea.get_textview_text(fieldname).data();
      else
        value = editarea.get_entry_text(fieldname).data();
#ifdef _DEBUG_
  printf("vorbis(): TagEditor::gui_to_hash(): Setting %s: %s.\n", (*cur).first,
                                                                 value);
#endif
      cantushash_set_char(hash, (*cur).first, value);
    }
    g_free(checkname);
    cur++;
  }
}


void TagEditor::set_from_hash(CantusHash *hash)
{
#ifdef _DEBUG_
  printf("vorbis(): TagEditor::set_from_hash(): Called.\n");
#endif
  std::map<const gchar*, void*>::iterator cur = hashmap.begin();
  const gchar *field = NULL;
  while (cur != hashmap.end()) {
    field = cantushash_get_char(hash, (*cur).first);
#ifdef _DEBUG_
    std::cout << "OGG(): TagEditor::set_from_hash(): "
              << "Handling " << (*cur).first << ": " << field << "\n";
#endif
    if (field != NULL)
      strcpy((gchar*)(*cur).second, field ? field : "");
    cur++;
  }
}


void TagEditor::delete_if_not_in_hash(CantusHash *hash)
{
  std::map<const gchar*, void*>::iterator cur = hashmap.begin();
  const gchar *field = NULL;
  while (cur != hashmap.end()) {
#ifdef _DEBUG_
  printf("vorbis(): TagEditor::delete_if_not_in_hash(): "
         "Handling %s.\n", (*cur).first);
#endif
    field = cantushash_get_char(hash, (*cur).first);
    if (!field || (field && strcmp((gchar*)(*cur).second, field) != 0))
      *(gchar*)(*cur).second = '\0';
    cur++;
  }
}


void TagEditor::show_tag(void)
{
  std::map<const gchar*, void*>::iterator cur = hashmap.begin();
  while (cur != hashmap.end()) {
    const gchar *fieldname = strchr((*cur).first, ':') + 1;
    if (!strcmp(fieldname, "Comment"))
      editarea.set_textview_text(fieldname, (gchar*)(*cur).second);
    else
      editarea.set_entry_text(fieldname, (gchar*)(*cur).second);
    cur++;
  }
}


/******************************************************************************
 * Private
 ******************************************************************************/
/* Called when the selection changed, but the selected files have not yet been
 * read.
 */
void TagEditor::on_selection_changed_event(void *pselected)
{
#ifdef _DEBUG_
  printf("vorbis(): TagEditor::on_selection_changed_event(): Called.\n");
#endif
  if (pselected && ((GList*)pselected)->next) // Make the buttons inactive if at
    editarea.set_active(FALSE);               // least two files are selected.
  isfirst = TRUE;
  if (selected)
    g_list_free(selected);
  selected = g_list_copy((GList*)pselected);
  memset(&tag, 0, sizeof(Tag));
}


/* The selection has now been read. */
void TagEditor::on_file_read_finished_event(void *pinfo)
{
#ifdef _DEBUG_
  printf("vorbis(): TagEditor::on_file_read_finished_event(): Called.\n");
#endif
  g_return_if_fail(pinfo != NULL);
  CantusHash *info = (CantusHash*)pinfo;
  
  // Copy the data from the hash into the tag struct.
  if (isfirst) {
    set_from_hash(info);
    isfirst = FALSE;
    return;
  }
  
  // Or remove all fields that do not equal the data.
  delete_if_not_in_hash(info);
}


/* The files that have been selected have now all been read. */
void TagEditor::on_filelist_read_finished_event(void *trash)
{
#ifdef _DEBUG_
  printf("vorbis(): TagEditor::on_filelist_read_finished_event(): Called.\n");
#endif
  editarea.set_active(TRUE);
  this->show_tag();
}


void TagEditor::on_editarea_button_save_clicked(void)
{
  // Get a pointer to all required functions.
  CantusEmitFunc emit = (CantusEmitFunc)
                    cantushash_get_pointer(plugindata, "Cantus:Emit");
  CantusFileInfoGetFunc get_info = (CantusFileInfoGetFunc)
                    cantushash_get_pointer(plugindata, "Cantus:FileInfoGet");
  CantusFileInfoUnlockFunc unlock_info = (CantusFileInfoUnlockFunc)
                    cantushash_get_pointer(plugindata, "Cantus:FileInfoUnlock");
  
  g_return_if_fail(emit != NULL || get_info != NULL || unlock_info != NULL);
  GList *item = selected;
  while (item) {
    const gchar *filename = (const gchar*)item->data;
    g_assert(filename != NULL);
    CantusHash *info = get_info(filename);
    gui_to_hash(info);
    cantushash_set_bool(info, "OGG:Changed", TRUE);
    unlock_info(filename);
    item = item->next;
  }
  
  // Emit the write request in the eventbus.
  GValue arg = {0, };
  g_value_init(&arg, G_TYPE_POINTER);
  g_value_set_pointer(&arg, (gpointer)selected);
  emit("Filelist:Save:Request", &arg);
  g_value_unset(&arg);
}


void TagEditor::on_uiwidget_destroyed_event(void *name)
{
#ifdef _DEBUG_
  printf("vorbis(): TagEditor::on_uiwidget_destroyed_event(): Called.\n");
#endif
  if (strcmp((gchar*)name, cantushash_get_char(plugindata, "Plugin:Name")) == 0)
    delete this;
}
