/* The Cantus project.
 * (c)2002, 2003, 2004 by Samuel Abels (spam debain org)
 * This project's homepage is: http://www.debain.org/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#ifndef HAVE_PLUGIN_H
#define HAVE_PLUGIN_H

class Plugin;

#include <iostream>
#include <dlfcn.h>
#include <sigc++/object.h>
#include <sigc++/signal.h>
#include "libs/lib_shellpattern.h"
#include "plugins/cantusplugin.h"
#include "pluginwrappers.h"

#define PLUGIN_COMPAT_LEVEL 1

typedef struct {
  const gchar *name;      // A symbol name.
  const gchar *keyname;   // The plugindata hash key name.
  gboolean     required;  // Whether or not the symbol is mandatory.
} Symbol;


class Plugin : public SigC::Object {
public:
  Plugin();
  ~Plugin();
  
  /* Triggered directly before the plugin will be deleted (= when the */
  /* refcounter has reached a value <= 0). */
  SigC::Signal1<void, Plugin*> signal_plugin_deleted;
  
  /* Loads a plugin and all its symbols into the object.
   */
  gint load(const gchar *filename);
  
  /* Increase a plugin's refcounter. When the refcounter is <= 0 the plugin will
   * be unloaded. On object creation, the refcounter is 1.
   */
  void ref(void);
  
  /* Decrease a plugin's refcounter.
   */
  void unref(void);
  
  /* Define the plugin's priority.
   */
  void set_priority(gint priority);
  
  /* Returns the plugin's priority.
   */
  gint get_priority(void);
  
  /* Returns the plugin name.
   */
  const gchar *get_name(void);
  
  /* Returns the name of the tags handled by this plugin.
   */
  const gchar *get_tagname(void);
  
  /* Returns the title for the plugin's notebook tab label.
   */
  const gchar *get_label(void);
  
  /* Returns the the plugin description (should be 200 chars max.).
   */
  const gchar *get_description(void);
  
  /* Returns the plugin's major version number.
   */
  gint get_majorversion(void);
  
  /* Returns the plugin's minor version number.
   */
  gint get_minorversion(void);
  
  /* Returns a function pointer to the plugin's read() function.
   */
  const ReadFunc get_readfunc(void);
  
  /* Returns a function pointer to the plugin's write() function.
   */
  const WriteFunc get_writefunc(void);
  
  /* Calls the plugin's uiwidget() function.
   */
  void *get_uiwidget(gboolean vertical);
  
  /* Given a file name, this function returns TRUE if the plugin is responsible
   * for handling this filetype, otherwise FALSE.
   */
  gboolean handles(const gchar *filename);
  
private:
  /* Checks, whether the plugin meets all mandatory requirements. Returns TRUE
   * if the plugin is valid, otherwise FALSE.
   */
  gboolean check_plugin(void);
  
  void       *dlhandle;    // The handle for dl_open.
  CantusHash *plugindata;  // A pointer to the C-compatible plugin data.
  gint        refct;       // A refcounter indicating whether the plugin is
                           // still needed.
  gint        priority;    // A priority, to be used by the backend only.
};

#endif
