/* The Cantus project.
 * (c)2002, 2003, 2004 by Samuel Abels (spam debain org)
 * This project's homepage is: http://www.debain.org/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

class FileInfoManager;

#ifndef HAVE_FILEINFOMANAGER_H
#define HAVE_FILEINFOMANAGER_H

#include <glibmm.h>
#include <glibmm/thread.h>
#include <glibmm/timer.h>
#include <sigc++/class_slot.h>
#include <deque>
#include <string>
#include "libs/gnuhashmap.h"
#include "fileinfo.h"
#include "fileinfomanagerevent.h"
#include "plugin.h"


class FileInfoManager : public SigC::Object {
protected:
  // Prevent copies.
  FileInfoManager(FileInfoManager &f) { g_assert_not_reached(); }
  // Prevent assignments.
  FileInfoManager& operator=(FileInfoManager &f) { g_assert_not_reached(); }
  
public:
  FileInfoManager();
  ~FileInfoManager();
  
  /* Triggered whenever a file has been read. */
  SigC::Signal1<void, FileInfo*>  signal_file_read_finished;
  /* Triggered whenever a file could not be read (e.g. wrong permissions). */
  SigC::Signal1<void, FileInfo*>  signal_file_read_failed;
  /* Triggered whenever a file has been written. */
  SigC::Signal1<void, FileInfo*>  signal_file_write_finished;
  /* Triggered whenever a file could not be written (e.g. wrong permissions). */
  SigC::Signal1<void, FileInfo*>  signal_file_write_failed;
  /* Triggered whenever a file has been renamed. */
  SigC::Signal1<void, FileInfo*>  signal_file_rename_finished;
  /* Triggered whenever a file could not be renamed. (e.g. file exists). */
  SigC::Signal1<void, FileInfo*>  signal_file_rename_failed;
  /* Triggered whenever the read queue has been finished. */
  SigC::Signal0<void>             signal_queue_read_finished;
  /* Triggered whenever the write queue has been finished. */
  SigC::Signal0<void>             signal_queue_write_finished;
  
  /* Stores a copy of the pluginlist in the object.
   * This is required because the object invokes the FileInfo read function,
   * which needs a plugin as an argument.
   */
  void register_plugins(std::list<Plugin*> *pluginlist);
  
  /* Replaces the list of files that still need to be read. (read queue)
   * (filenames == NULL is allowed).
   */
  void readqueue_set(GList *filenames);
  
  /* Clear the read queue. (=Abort read).
   */
  void readqueue_clear(void);
  
  /* Appends stuff to the list of files that still need to be
   * written. (write queue)
   */
  void writequeue_append(GList *filenames);
  
  /* Clear the write queue. (=Abort write)
   */
  void writequeue_clear(void);
  
  /* This function locks the fileinfo object with the given name and returns
   * the hash containing all file data from the fileinfo object.
   * The FileInfo object will remain locked, so make sure to unlock!!
   */
  CantusHash *get_info_locked(const gchar *filename);
  
  /* Unlocks the FileInfo object with the given name.
   */
  void unlock(const gchar *filename);
  
  /* Returns the progress in percent as a float.
   */
  float get_progress(void);
  
  /* Returns a human readable status (e.g. "Reading /my/file.ogg", or "Ready.").
   */
  std::string get_status(void);
  
private:
  /* Unregisters all plugins.
   */
  void unregister_plugins(void);
  
  /* A worker thread doing all read operations.
   */
  void reader_func(void);         
  
  /* A worker thread doing all write operations.
   */
  void writer_func(void);
  
  /* Receives one fileinfo object from the cache. If the item is not in the
   * cache yet, it will try to read it using all apropriate plugins.
   */
  FileInfo *read(const gchar *filename);
  
  /* Receives one fileinfo object from the cache. If the item is not in the
   * cache yet, read it. Then try to write it using all apropriate plugins.
   */
  void write(gint inode);
  
  /* Checks whether the filename has been edited and renames the file.
   * Returns TRUE on success, FALSE otherwise.
   */
  gboolean try_rename(FileInfo *info);
  
  /* This dispatcher is called via an asynchronous pipe.
   * Once called, it emits all queued events via a synchronous SigC signal.
   */
  void signal_dispatcher(void);
  
  Glib::Mutex   mutex;        // One mutex for parent, reader and writer.
  Glib::Cond    readcond;     // Pushed whenever there is something to read.
  Glib::Cond    writecond;    // Pushed whenever there is something to write.
  gboolean      stopthreads;  // When true, the threads will terminate themself.
  Glib::Thread *reader;       // The reader_func() thread.
  Glib::Thread *writer;       // The writer_func() thread.
  
  std::deque<FileInfoManagerEvent*> eventqueue; // The event-dispatcher's queue.
  Glib::Dispatcher signal_dispatch;             // Triggered asynchronously to
                                                // call the dispatcher.
  
  std::list<Plugin*>       *plugins;     // A list of all available plugins.
  std::deque<const gchar *> readqueue;   // The read queue (list of filenames).
  std::deque<glong>         writequeue;  // The write queue (list of inodes).
  
                          // A hash mapping filename to inode.
  __gnu_cxx::hash_map<std::string, glong> inodemap;
                          // A hash mapping inode to FileInfo object.
  __gnu_cxx::hash_map<glong, FileInfo*>   cache;
  
  int queued_read;        // The number of files currently queued for reading.
  int processed_read;     // The number of files read.
  int queued_write;       // The number of files currently queued for writing.
  int processed_write;    // The number of files written.
};

#endif
