/* The Cantus project.
 * (c)2002, 2003, 2004 by Samuel Abels (spam debain org)
 * This project's homepage is: http://www.debain.org/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#ifndef HAVE_FILEINFO_H
#define HAVE_FILEINFO_H

class FileInfo;

#include <iostream>
#include <glibmm.h>
#include <sys/stat.h>
#include "plugins/cantushash.h"

#define FILEINFO_HISTORY_LEN 10


typedef gint (*ReadFunc) (const gchar *filename, CantusHash *data);
typedef gint (*WriteFunc)(const gchar *filename, CantusHash *data);


class FileInfo {
protected:
  // Prevent copies.
  FileInfo(FileInfo &f) { g_assert_not_reached(); }
  // Prevent assignments.
  FileInfo& operator=(FileInfo &s) { g_assert_not_reached(); }
  
public:
  /* Triggered whenever a file has been read. */
  SigC::Signal1<void, FileInfo*>  signal_read_finished;
  /* Triggered whenever a file has been written. */
  SigC::Signal1<void, FileInfo*>  signal_write_finished;
  
  /* The filename is being stored in the real hash and the edited hash as
   * well.
   */
  FileInfo(const gchar *filename);
  ~FileInfo();
  
  /* Returns the inode number of the file, or < 0 on an error.
   */
  glong get_inode(void);
  
  /* Read all file informations into the real hash using the given Input
   * Plugin.
   * Returns an errorcode or 0.
   */
  gint read(const ReadFunc func);
  
  /* Write all file informations from the real hash to the filesystem, using
   * the given output plugin.
   * Returns an errorcode or 0.
   */
  gint write(const WriteFunc func);
  
  /* Returns a pointer to the file's "edited_hash". Make sure to "lock()" when
   * you do this!
   */
  CantusHash *get_edited_hash(void);
  
  /* Lock the object from being accessed.
   */
  void lock(void);
  
  /* Unlock the object so that it can be accessed.
   */
  void unlock(void);
  
  /* Returns the filename from the real hash. Make sure to lock() before you
   * use this!
   */
  const gchar *get_filename(void);
  
  /* Returns the filename from the edited hash. Make sure to lock() before you
   * use this!
   */
  const gchar *get_edited_filename(void);
  
  /* Move the "real hash" onto an "UNDO" stack. Then, copy the edited hash into
   * the new real hash. Also, clear the "REDO" stack.
   */
  void commit(void);
  
  /* Copy the real hash into the edited hash.
   */
  void revert(void);
  
  /* Push the edited hash onto the "REDO" stack. Then, pop the last item from
   * the "UNDO" stack into the edited hash.
   * Returns FALSE when the "UNDO" list was empty so nothing has been changed.
   */
  gboolean undo(void);
  
  /* Push the edited hash onto the "UNDO" stack. Then, pop the last item from
   * the "REDO" stack into the edited hash.
   * Returns FALSE when the "REDO" list was empty so nothing has been changed.
   */
  gboolean redo(void);
  
private:
  /* Push something to the history (for UNDO).
   */
  void history_push(CantusHash *hash);
  
  /* Get something from the history (for UNDO).
   */
  CantusHash *history_pop(void);
  
  /* Get something to the future (for REDO).
   */
  void future_push(CantusHash *hash);
  
  /* Get something from the future (for REDO).
   */
  CantusHash *future_pop(void);
  
  /* Clear the future (for REDO).
   */
  void future_clear(void);
  
  CantusHash *real_hash;            // The file's data as on the filesystem.
  CantusHash *edited_hash;          // The file's data as edited.
  std::list<CantusHash *> history;  // The history of the data for UNDO.
  std::list<CantusHash *> future;   // The future of the data for REDO.
  
  Glib::Mutex mutex;                // One mutex to lock the whole object.
};

#endif
