/* the cantus project.
 * (c)2002 by Samuel Abels (sam@manicsadness.com)
 * This project's homepage is: http://software.manicsadness.com/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <libgnomeui/gnome-messagebox.h>
#include "freedb.h"
#include "lib_freedb.h"
#include "gui_freedb.h"
#include "support.h"
#include "interface.h"
#include "mp3info.h"
#include "gui_queue.h"
#include "lib_id3v1.h"
#include "lib_id3v2.3.h"
#include "lib_vorbis.h"
#include "configfile.h"
#include "plugs.h"
#include "shared.h"
#include "gui_shared.h"


/***************************************************************************************
 * BELOW FOLLOW THE STATICS
 ***************************************************************************************/
/*
 * Returns a pointer to the item containing the given fullfilename.
 */
static Mp3Info * g_list_find_matching_fullfilename (GList *haystack, gchar *fullfilename)
{
  GList * haystack_item = NULL;
  Mp3Info * haystack_mp3info = NULL;
  gchar haystack_fullfilename[2048];
  
  haystack_item = haystack;
  while ( haystack_item ) {
    haystack_mp3info = (Mp3Info *)haystack_item->data;
    
    snprintf (haystack_fullfilename, 2047, "%s%s", haystack_mp3info->directory, haystack_mp3info->filename);
    if ( strncmp (fullfilename, haystack_fullfilename, 2048) == 0 )
      return haystack_mp3info;
    
    haystack_item = haystack_item->next;
  }
  
  return NULL;
}


/*
 * Check if the user entered all data correctly.
 */
static gboolean check_userinput (void)
{
  extern GtkWidget * cantus;
  
  /* Fulltext search selected. */
  if ( gtk_toggle_button_get_active ((GtkToggleButton *)lookup_widget (cantus, "radio_cantus_freedb_fulltext")) ) {
    /* No fulltextstring entered. */
    if ( *gtk_entry_get_text ((GtkEntry *)lookup_widget (cantus, "entry_cantus_freedb_fulltext_string")) == '\0' ) {
      popup_info ("Please enter a search string!");
      return (FALSE);
    }
    
    return (TRUE);
  }
  
  /* Search by freedb id. */
  if ( gtk_toggle_button_get_active ((GtkToggleButton *)lookup_widget (cantus, "radio_cantus_freedb_lookup_id")) ) {
    /* Check if something has been entered for the category. */
    if ( *gtk_entry_get_text ((GtkEntry *)lookup_widget (cantus, "entry_cantus_freedb_id_category")) == '\0' ) {
      popup_info ("You must enter a FreeDB category!");
      return (FALSE);
    }
    
    /* Check if something has been entered for the id. */
    if( *gtk_entry_get_text ((GtkEntry *)lookup_widget (cantus, "entry_cantus_freedb_id_id")) == '\0' ) {
      popup_info ("You must enter a FreeDB ID!");
      return (FALSE);
    }
    
    return (TRUE);
  }
  
  return (TRUE);
}


/*
 * Copy the filenames from the queue to the current CD struct.
 */
static void get_filenames_from_queue (CDInfo * cd)
{
  extern GtkWidget * cantus;
  GtkCList * queue = NULL;
  CDTrack * curtrack = NULL;
  Mp3Info * mp3info = NULL;
  gint row = 0;
  
  if ( !cd )
    return;
  if ( cd->track[0] == NULL )
    return;
  
  queue = (GtkCList *)lookup_widget (cantus, "clist_cantus_queue");
  
  curtrack = cd->track[0];
  while ( (mp3info = gtk_clist_get_row_data (queue, row)) ) {
    /* Copy from the queueitem to the trackitem. */
    snprintf (curtrack->filename, 2047, "%s%s", mp3info->directory, mp3info->filename);
    
    curtrack = cd->track[++row];
    
    /* No more tracks? We're done, then. */
    if ( !curtrack )
      break;
    
    memset (curtrack->filename, '\0', sizeof (curtrack->filename));
  }
  
  return;
}
/***************************************************************************************
 * END OF STATICS
 ***************************************************************************************/


/*
 * Get list of available FreeDB servers from the configfile into a struct.
 */
static GList * read_serverlist (void)
{
  gchar config[10000];
  Host * curhost = NULL;
  GList * hostlist = NULL;
  gchar option[2048];
  gchar value[2048];
  int curserver = 0;
  
  /* Read configfile. */
  config_read (config, 10000);
  
  curserver = 0;
  while ( 1 ) {
    curhost = (Host *)malloc (sizeof (Host));
    
    /* Get the hostname. */
    snprintf (option, 2047, "server%i#1", curserver);
    configfile_option_get(config, "freedb servers", option, curhost->name);
    
    /* Get the port. */
    snprintf (option, 2047, "server%i#2", curserver);
    configfile_option_get (config, "freedb servers", option, value);
    curhost->port = (unsigned int)atoi (value);
    
    /* Protocol. */
    snprintf (option, 2047, "server%i#3", curserver);
    configfile_option_get (config, "freedb servers", option, value);
    if ( strcmp (value, "HTTP") == 0 )
      curhost->protocol = PROTO_HTTP;
    else
      curhost->protocol = PROTO_CDDB;
    
    /* Cgi-Path. */
    snprintf (option, 2047, "server%i#4", curserver);
    configfile_option_get (config, "freedb servers", option, curhost->cgipath);
    
    /* All values present? */
    if ( *curhost->name == '\0'
      || curhost->port == 0
      || curhost->protocol == PROTO_NONE
      || *curhost->cgipath == '\0' )
      break;
    
    hostlist = g_list_append (hostlist, curhost);
    curserver++;
  }
  
  free (curhost);
  
  return (hostlist);
}


static Host * read_proxy (void)
{
  gchar config[10000];
  Host * proxy = (Host *)malloc (sizeof (Host));
  gchar value[2048];

  /* Get the proxy server. */
  configfile_option_get(config, "freedb", "proxy address", proxy->name);
  
  /* Proxy port. */
  configfile_option_get(config, "freedb", "proxy port", value);
  proxy->port = (unsigned int)atoi (value);
  
  /* Enabled state. */
  configfile_option_get (config, "freedb", "enable proxy", value);
  if ( strcmp (value, "TRUE") != 0 )
    *proxy->name = '\0';
  
  /* All values present? */
  if ( *proxy->name == '\0'
    || proxy->port == 0 )
    *proxy->name = '\0';
  
  return (proxy);
}


void freedb_get (void)
{
  extern GtkWidget * cantus;
  extern GtkWidget * tracks_sort;
  GList * serverlist = NULL;
  GList * curserver = NULL;
  Host * curhost = NULL;
  Host * proxy = NULL;
  GList *cdlist = NULL;
  gchar * searchstring = NULL;
  
  /* Check if the user gave us all infos and display error messages. */
  if ( !check_userinput () )
    return;
  
  /* Read the configuration from the configfile. */
  serverlist = read_serverlist ();
  proxy = read_proxy ();
  
  /* No hosts? Then return. */
  if ( !serverlist ) {
    popup_info ("No FreeDB hosts configured! Please edit the preferences.");
    goto cleanups;
  }
  
  /* Get by a searchstring. */
  if ( gtk_toggle_button_get_active ((GtkToggleButton *)lookup_widget(cantus, "radio_cantus_freedb_fulltext")) ) {
    gint socket = 0;
    
    /* Connect to a freedb host. */
    if( (socket = freedb_connect_host ("www.freedb.org", 80, proxy->name, proxy->port)) < 0 ) {
      popup_info ("Connecting the FreeDB host failed!");
      return;
    }
    
    searchstring = gtk_entry_get_text ((GtkEntry *)lookup_widget (cantus, "entry_cantus_freedb_fulltext_string"));
    
    /* Get album list from freedb. */
    if ( freedb_get_albumlist_by_searchstring (socket, searchstring, &cdlist) < 0 ) {
      close (socket);
      popup_info ("Error: Connected successfully, but invalid answer while receiving the albumlist.");
      return;
    }
    
    /* Close connection. */
    close (socket);
    
    /* Show gui and insert the data. */
    tracks_sort = create_tracks_sort ();
    clist_album_choose_albums_add (cdlist);
    gtk_widget_show (tracks_sort);
    g_list_free (cdlist);
  }
  
  /* Get the album from the freedb id. */
  if ( gtk_toggle_button_get_active ((GtkToggleButton *)lookup_widget (cantus, "radio_cantus_freedb_lookup_id")) ) {
    CDInfo * cd = calloc (1, sizeof (CDInfo));
    gint socket = 0;
    
    memset (cd->year, 0, 20);
    
    /* Get genre and id from the gui. */
    strncpy (cd->genre, gtk_entry_get_text ((GtkEntry *)lookup_widget (cantus, "entry_cantus_freedb_id_category")), 512);
    cd->id = strtoul (gtk_entry_get_text ((GtkEntry *)lookup_widget (cantus, "entry_cantus_freedb_id_id")), 0, 16);
    
    /* Connect to any host. */
    curserver = serverlist;
    while ( curserver ) {
      curhost = (Host *)curserver->data;
      socket = freedb_connect_host (curhost->name, curhost->port, proxy->name, proxy->port);
      
      /* Successfully connected? */
      if ( socket > 0 )
        break;
      
      curserver = curserver->next;
    }
    
    /* Still not connected? Then return. */
    if ( !socket > 0 ) {
      popup_info ("Connection to every freedb host failed!");
      return;
    }
    
    /* Get album from FreeDB. */
    if ( freedb_get_album_by_id_and_category (socket, curhost->name, curhost->port, curhost->cgipath, "cantus", VERSION, &cd) != 0 ) {
      popup_info ("Connected successfully, but i received an error while receiving the album.");
      close (socket);
      return;
    }
    
    /* Close connection. */
    close (socket);
    
    /* Add the CURRENT filenames to the struct, also. */
    get_filenames_from_queue (cd);
    
    /* Show the track_sort widget and insert the albums data (=Tracks/Titles). */
    tracks_sort = create_tracks_sort ();
    clist_tracks_sort_tracks_add (cd);
    gtk_widget_show (tracks_sort);
  }
  
  /* Cleanups. */
cleanups:
  free (proxy);
  
  curserver = serverlist;
  while ( curserver ) {
    g_free (curserver->data);
    curserver = curserver->next;
  }
}



void
freedb_album_clicked(void)
{
  extern GtkWidget *tracks_sort;
  GtkCList * cdlist = NULL;
  GList * serverlist = NULL;
  GList * curserver = NULL;
  Host * curhost = NULL;
  Host * proxy = NULL;
  gint clickedrow = 0;
  gint curtrack = 0;
  CDInfo *cd = NULL;
  gint socket = 0;
  
  cdlist = (GtkCList *)lookup_widget (tracks_sort, "clist_tracks_sort_cdlist");
  
  /* If no cd is selected, return. */
  if( !cdlist->selection )
    return;
  
  /* Get the chosen albums data. */
  clickedrow = (gint)cdlist->selection->data;
  cd = (CDInfo *)gtk_clist_get_row_data (cdlist, clickedrow);
  
  if ( cd == NULL )
    return;
  
  /* Empty out the cd struct. */
  curtrack = -1;
  while( cd->track[++curtrack] != NULL ) {
    free (cd->track[curtrack]);
    cd->track[curtrack] = NULL;
  }
  
  /* Read the configuration from the configfile. */
  serverlist = read_serverlist ();
  proxy = read_proxy ();
  
  /* No hosts? Then return. */
  if ( !serverlist ) {
    popup_info ("No FreeDB hosts configured! Please edit the preferences.");
    goto cleanups;
  }
  
  /* Connect to any host. */
  curserver = serverlist;
  while ( curserver ) {
    curhost = (Host *)curserver->data;
    socket = freedb_connect_host (curhost->name, curhost->port, proxy->name, proxy->port);
    
    /* Successfully connected? */
    if ( socket > 0 )
      break;
    
    curserver = curserver->next;
  }
  
  /* Still not connected? Then return. */
  if ( !socket > 0 ) {
    popup_info ("Connection to every freedb host failed!");
    return;
  }
  
  /* Get album from FreeDB. */
  if ( freedb_get_album_by_id_and_category (socket, curhost->name, curhost->port, curhost->cgipath, "cantus", VERSION, &cd) != 0 ) {
    popup_info ("Connected successfully, but received an error while receiving the album.");
    close (socket);
    return;
  }
  
  /* Close connection. */
  close (socket);
  
  /* Add the CURRENT filenames to the struct, also. */
  get_filenames_from_queue (cd);
  
  /* Insert the album data to the GUI (=Tracks/Titles). */
  clist_tracks_sort_tracks_add (cd);
  
  /* Cleanups. */
cleanups:
  free (proxy);
  
  curserver = serverlist;
  while ( curserver ) {
    g_free (curserver->data);
    curserver = curserver->next;
  }
}



/*
 * This function renames the files in the queue according to the freedb list.
 */
/* FIXMEFIXME: Needs some split-down */
void
freedb_tracks_ok(void)
{
  extern GtkWidget *cantus;
  extern GtkWidget *tracks_sort;
  extern GList *queue;
  extern char *active_tag;
  
  char newfilename[2048];
  char newfilename_full[2048];
  char extension[2048];
  CDInfo *cd = NULL;
  CDTrack *curtrack = NULL;
  int row = -1, num = -1;
  int whichtag = 0;
  
  id3Tag *tag = malloc (sizeof (id3Tag));
  Mp3Info *mp3info = NULL;
  
  cd = (CDInfo *)gtk_clist_get_row_data (GTK_CLIST(lookup_widget (tracks_sort, "clist_tracks_sort_tracks")), 0);
  
  if ( !cd )
    goto cleanups;
  
  /* Parse through the track list. */
  row = -1;
  while ( cd->track[++row] ) {
    curtrack = (CDTrack *)cd->track[row];
    
    if( *(curtrack->filename) == '\0' )
      continue;
    
		mp3info = g_list_find_matching_fullfilename (queue, curtrack->filename);
    
    /* Check write permissions. */
    if ( access (curtrack->filename, W_OK) != 0 ) {
      strncpy (mp3info->status, "Locked!", 255);
      goto nexttrack;
    }
    
    memset(tag->artist,  '\0', 1024);
    memset(tag->title,   '\0', 1024);
    memset(tag->album,   '\0', 1024);
    memset(tag->year,    '\0', 5);
    memset(tag->track,   '\0', 3);
    memset(tag->comment, '\0', 1024);
    memset(tag->genre,   '\0', 512);
    
    /* Read the old tag. */
    if ( strtoucmp (curtrack->filename + strlen (curtrack->filename) - 4, ".mp3") == 0
        || strtoucmp (curtrack->filename + strlen (curtrack->filename) - 5, ".flac") == 0  ) {
      id3Tag *v2 = malloc (sizeof (id3Tag));
      
      /* Read ID3V1 tags here. */
      get_id3v1_tag (tag, curtrack->filename);

      /* V2 tags have higher prio, so it will overwrite the V1 tag. */
      if ( get_id3v2_tag (v2, curtrack->filename) == 0 ) {
        g_free (tag);
        tag = v2;
      }
      else
        g_free (v2);
      
      whichtag = 1;
    }
#ifdef HAVE_OGG_H
    else if ( strtoucmp (curtrack->filename + strlen (curtrack->filename) - 4, ".ogg") == 0 ) {
      get_vorbis_tag (tag, curtrack->filename);
      whichtag = 3;
    }
#endif
    else {
      goto nexttrack;
    }
    
    /* The CD-title can have two different formats in freedb: */
    /* Either "Artist / Album" or only "Sampler". */
    /* So we look wether an " / " occurs and split the string into artist and cdtitle. */
    /* Otherwise we set both artist and cdtitle the same. */
    /* Copy freedb album title to tag. */
    if( strstr (cd->cdtitle, " / ") != NULL ) {
      strncpy (tag->artist, cd->cdtitle, 1023);
      *(strstr (tag->artist, " / ")) = '\0';
      strncpy (tag->album, strstr (cd->cdtitle, " / ") + 3, 1023);
    }
    else {
      strncpy (tag->artist, cd->cdtitle, 1023);
      strncpy (tag->album, cd->cdtitle, 1023);
    }
    
    /* The TRACK-title has the same format as the CD-title above. */
    if ( strstr (curtrack->title, " / ") != NULL ) {
      strncpy (tag->artist, curtrack->title, 1023);
      *(strstr (tag->artist, " / ")) = '\0';
      strncpy (tag->title, strstr (curtrack->title, " / ") + 3, 1023);
    }
    else
      strncpy (tag->title, curtrack->title, 1023);
    
    /* Track number and year. */
    snprintf (tag->track, 3, "%s", curtrack->number);
    strncpy (tag->year, cd->year, 5);
    
    
    /* Now, save the tag. */
    switch ( whichtag ) {
      /* Handle ID3 tags here. */
      case 1:
        if ( set_id3v1_tag (tag, curtrack->filename) != 0 ) {
          strncpy (mp3info->status, "V1 tag write failure", 255);
          goto nexttrack;
        }
        
        if ( set_id3v2_tag (tag, curtrack->filename) != 0 ) {
          strncpy (mp3info->status, "V2 tag write failure!", 255);
          goto nexttrack;
        }
        break;
        
#ifdef HAVE_OGG_H
      /* Handle OGG tags here. */
      case 3:
        if ( set_vorbis_tag (tag, curtrack->filename) == 0 ) {
          strncpy (mp3info->status, "OGG tag write failure", 255);
          goto nexttrack;
        }
        break;
#endif
        
      default:
        goto nexttrack;
    }
    
    /* Ok, so the tag has been written successfully. */
    /* Re-read the tag. */
    mp3file_get_info_1 (mp3info->directory, mp3info->filename, mp3info);
    
    /* Create the new filename for renaming. */
    switch ( whichtag ) {
      /* Handle ID3 tags here. */
      case 1:
        // Remember the extension.
        strncpy(extension, strrchr(curtrack->filename, '.'), 2047);
        snprintf (newfilename, 2047, "%s - %s%s", curtrack->number, curtrack->title, extension);
        break;
      
#ifdef HAVE_OGG_H
      /* Handle OGG tags here. */
      case 3:
        snprintf (newfilename, 2047, "%s - %s.ogg", curtrack->number, curtrack->title);
        break;
#endif
    }
    
    apply_rule_replace_invalid_characters (newfilename);
    
    /* Create a new fullfilename. */
    strncpy (newfilename_full, curtrack->filename, 2047);
    *(strrchr (newfilename_full, '/') + 1) = '\0';
    strncpy (newfilename_full + strlen (newfilename_full), newfilename, 2047 - strlen (newfilename_full));
    
    /* Just to be sure we do nothing stupid we replace a " / " with a " - ". */
    if ( strstr (newfilename_full, " / ") != NULL )
      *(strstr (newfilename_full, " / ") + 1) = '-';
    
    /* Be sure not to overwrite a file. */
    if ( access (newfilename_full, F_OK) == 0 ) {
      strncpy (mp3info->status, "File exists!", 255);
      goto nexttrack;
    }
    
    /* Rename the file. */
    if ( rename (curtrack->filename, newfilename_full) == 0 ) {
      strncpy (mp3info->filename, newfilename, 2047);
      strncpy (mp3info->status, "FreeDB OK!", 255);
      if ( strcmp (active_tag, curtrack->filename) == 0)
        strncpy (active_tag, newfilename_full, 2047);
    }
    else
      strncpy (mp3info->status, "Tags set, rename failed!", 255);
    
nexttrack:
    /* Update the queue. */
    clist_file_update (GTK_CLIST(lookup_widget (cantus, "clist_cantus_queue")), mp3info);
    clist_file_update (GTK_CLIST(lookup_widget (cantus, "clist_cantus_queueonly_queue")), mp3info);
    
    g_free (curtrack);
    cd->track[row] = NULL;
	}

  /* Cleanups. */
cleanups:
  g_free (tag);
  
  /* Free every cd in the cd list. */
  row = -1;
  num = -1;
  while ( (cd = gtk_clist_get_row_data (GTK_CLIST(lookup_widget (tracks_sort, "clist_tracks_sort_cdlist")), ++row)) ) {
    num = -1;
    while ( cd->track[++num] )
      g_free (cd->track[num]);
    
    g_free (cd);
  }
}





void freedb_tracks_cancel(void)
{
  extern GtkWidget *tracks_sort;
  CDInfo *cd = NULL;
  int row = -1, num = -1;
  
  /* Free every cd in the cd list. */
  while ( (cd = gtk_clist_get_row_data (GTK_CLIST(lookup_widget(tracks_sort, "clist_tracks_sort_cdlist")), ++row)) ) {
    num = -1;
    while ( cd->track[++num] )
      g_free (cd->track[num]);
    
    g_free (cd);
  }
  
  gtk_widget_destroy (tracks_sort);
}
