/* the cantus project.
 * (c)2002 by Samuel Abels (sam@manicsadness.com)
 * This project's homepage is: http://software.manicsadness.com/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "memory.h"
#include "shared.h"

#include "freedb.h"
#include "lib_freedb.h"

#include "gui_dirtree.h"
#include "gui_filelist.h"
#include "gui_shared.h"
#include "gui_rules.h"
#include "gui_custom_view.h"
#include "gui_queue.h"
#include "gui_freedb.h"

#include "configfile.h"
#include "mp3info.h"
#include "queue.h"
#include "rules.h"
#include "rules_plugs.h"
#include "options.h"
#include "tageditor.h"
#include "file2tag.h"
#include "popups.h"
#include "custom_find.h"
#include "custom_view.h"
#include "tag2filename.h"


/***************************************************************************************
 * BELOW FOLLOW THE STATICS
 ***************************************************************************************/

// Returns the datapointer to the queueitems data with the given fullfilename
static Mp3Info *
g_list_find_matching_fullfilename(GList *haystack, gchar *fullfilename)
{
	GList *haystack_item = NULL;
	Mp3Info *haystack_mp3info = NULL;
	gchar haystack_fullfilename[2048];

	haystack_item = g_list_first(haystack);
	while(haystack_item)
	{
		haystack_mp3info = (Mp3Info *)haystack_item->data;
		haystack_item = g_list_next(haystack_item);
		
		snprintf(haystack_fullfilename, 2047, "%s%s", haystack_mp3info->directory, haystack_mp3info->filename);
		if(strncmp(fullfilename, haystack_fullfilename, 2047)==0)
			return haystack_mp3info;
	}
	return NULL;
}

/***************************************************************************************
 * END OF STATICS
 ***************************************************************************************/




void
on_button_rules_back_clicked           (GtkObject       *object,
                                        gpointer         user_data)
{
	extern GtkWidget *rules;
	rules_close();
	gtk_widget_destroy(rules);
}


void
on_button_rules_delete_clicked         (GtkButton       *button,
                                        gpointer         user_data)
{
	rules_remove_selected();
}


void
on_button_rules_up_clicked             (GtkButton       *button,
                                        gpointer         user_data)
{
	rules_rules_up();
}


void
on_button_rules_down_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
	rules_rules_down();
}


void
on_button_rules_add_clicked            (GtkButton       *button,
                                        gpointer         user_data)
{
	rules_add();
}


void
on_button_rules_clear_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
	rules_clear();
}


void
on_button_rules_save_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *rules;
	
	rules_dialog_save(
		gtk_entry_get_text(GTK_ENTRY(lookup_widget(rules, "entry_rules_presets"))) );
	rules_presets_update();
}


void
on_button_rules_load_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *rules;
	
	rules_dialog_load(
		gtk_entry_get_text(GTK_ENTRY(lookup_widget(rules, "entry_rules_presets"))) );
}


void
on_cantus_destroy                        (GtkObject       *object,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	extern GtkWidget *popup_cantus_queue;
	extern GtkWidget *popup_cantus_filelist;
	extern GtkWidget *popup_cantus_dirtree;
	extern GtkWidget *popup_cantus_log_log;
	extern GtkWidget *wait;
	
	queue_clear();
	
	ctree_destroy_dirtree( GTK_CTREE(lookup_widget(cantus, "ctree_cantus_directories")),
		gtk_ctree_node_nth(GTK_CTREE(lookup_widget(cantus, "ctree_cantus_directories")), 0) );

	gtk_widget_destroy(popup_cantus_queue);
	gtk_widget_destroy(popup_cantus_filelist);
	gtk_widget_destroy(popup_cantus_dirtree);
	gtk_widget_destroy(popup_cantus_log_log);
	gtk_widget_destroy(wait);
	
	gtk_main_quit();
}


void
on_button_cantus_clearqueue_clicked      (GtkButton       *button,
                                        gpointer         user_data)
{
	queue_clear();
}


void
on_button_cantus_selectdir_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	gtk_clist_select_all(GTK_CLIST(lookup_widget(cantus, "clist_cantus_filelist")));
}


void
on_button_cantus_addfiles_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
	queue_files_add_selected();
}


void
on_button_cantus_addrecursive_clicked    (GtkButton       *button,
                                        gpointer         user_data)
{
	extern gchar *current_dir;
	queue_files_add_recursive(current_dir);
}


void
on_button_cantus_findandlists_clicked    (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *custom_find;
	extern gchar *current_dir;
	custom_find = create_custom_find();
	gtk_entry_set_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_filename_directory")), current_dir);
	gtk_widget_show(custom_find);
}


void
on_button_cantus_options_clicked         (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *options;
	
	gtk_widget_show(options = create_options());
	options_dialog_load();
	options_listview_items_update( GTK_CLIST(lookup_widget(options, "clist_options_listview_items")) );
}


void
on_button_cantus_about_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *about;
//	gchar abouttext[4000];
	
/*	sprintf(
		abouttext,
		"Cantus - Version %s\n\n"
		"This program comes WITHOUT ANY WARRANTY. USE ON YOUR OWN RISC!\n"
		"PLEASE REPORT BUGS! You can do so in the support forum on http://software.manicsadness.com/cantus/"
		"or by email. Most time I am fast in fixing them.\n\n"
		"The cost of programming this tool was: 400 hours of my time, over 20 bottles of beer and several people loosing their mp3's...\n"
		"By now such errors should be merged out, so have fun using it!",
		VERSION);
*/
	about = create_about();
	gtk_widget_show(about);
//	gtk_label_set_text(GTK_LABEL(lookup_widget(about, "label_about")), abouttext);
}


void
on_button_cantus_help_clicked            (GtkButton       *button,
                                        gpointer         user_data)
{

}


void
on_button_cantus_exit_clicked            (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	
	gtk_widget_destroy(cantus);
}

void
on_ctree_cantus_directories_select_row   (GtkCList        *clist,
                                        gint             row,
                                        gint             column,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	extern gchar *current_dir;
	extern Cfg cfg;
	GtkCTree *ctree = GTK_CTREE(lookup_widget(cantus, "ctree_cantus_directories"));
	GtkCTreeNode *node = gtk_ctree_node_nth(ctree, row);
	
// update global "current_dir"
	strncpy( current_dir, gtk_ctree_node_get_row_data(ctree, node), 2047 );
	
// update the filelist
	clist_filelist_update( GTK_CLIST(lookup_widget(cantus, "clist_cantus_filelist")),
		current_dir,
		cfg.showhidden );
}


void
on_ctree_cantus_directories_tree_expand  (GtkCTree        *ctree,
                                        GList           *node,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	extern Cfg cfg;

// expand the directory tree
	ctree_expand_dirtree(
		GTK_CTREE(lookup_widget(cantus, "ctree_cantus_directories")),
		g_list_position(g_list_first(node), node),
		cfg.showhidden );
	
// and set status again.
	gnome_appbar_set_status(GNOME_APPBAR (lookup_widget (cantus, "appbar")), "Ready.");
	gnome_appbar_set_progress(GNOME_APPBAR (lookup_widget (cantus, "appbar")), 0);
}


void
on_ctree_cantus_directories_tree_collapse
                                        (GtkCTree        *ctree,
                                        GList           *node,
                                        gpointer         user_data)
{
	GtkCTreeNode *ctreenode = NULL;
	
	ctreenode = (GtkCTreeNode*)node;

	gtk_clist_freeze((GtkCList*)ctree);
	
// collapse the directory tree
	if ( GTK_CTREE_ROW(ctreenode)->children != NULL )
		ctree_collapse_dirtree (ctree, ctreenode);
	gtk_clist_thaw((GtkCList*)ctree);
}


gboolean
on_ctree_cantus_directories_button_press_event
                                        (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	extern gint lastclick_x;
	extern gint lastclick_y;
	extern GtkWidget *popup_cantus_dirtree;
	gint row, col;

	if( event->button == 3 )
	{
		lastclick_x = event->x;
		lastclick_y = event->y;
		
		gtk_clist_get_selection_info(GTK_CLIST (lookup_widget (cantus, "ctree_cantus_directories")), event->x, event->y, &row, &col);
		gtk_clist_select_row(GTK_CLIST (lookup_widget (cantus, "ctree_cantus_directories")), row, col);
		gtk_menu_popup( GTK_MENU( popup_cantus_dirtree ),
			NULL, NULL, NULL, NULL, event->button, event->time );
		return TRUE;
	}
	
	return FALSE;
}


gboolean
on_clist_cantus_queue_button_press_event (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
	extern GtkWidget *popup_cantus_queue;
	extern GtkWidget *lastclicked;
	
	lastclicked = widget;

	if( event->button == 3 )
	{
		clist_cantus_queue_rightclicked(GTK_CLIST(widget), event->x, event->y);
		gtk_menu_popup( GTK_MENU( popup_cantus_queue ),
			NULL, NULL, NULL, NULL, event->button, event->time );
		return TRUE;
	}
	
 	return FALSE;
}


void
on_clist_cantus_queue_select_row         (GtkCList        *clist,
                                        gint             row,
                                        gint             column,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	extern gchar *active_tag;
	extern Cfg cfg;
	
	Mp3Info *rowdata = NULL;
	gint whichtag = 0;
	
// update filename2tag info.
	file2tag_preview_selected(GTK_CLIST(lookup_widget (cantus, "clist_cantus_queue")));
	
// set new activetag
	rowdata = gtk_clist_get_row_data(clist, row);
	snprintf(active_tag, 2047, "%s%s", rowdata->directory, rowdata->filename);

#ifdef HAVE_OGG_H
	if( strtoucmp(active_tag + strlen(active_tag) - 4, ".ogg") == 0 )
	{
		gtk_widget_set_sensitive(GTK_WIDGET(lookup_widget(cantus, "radio_cantus_tagedit_ogg")), TRUE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_ogg")), TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(lookup_widget(cantus, "radio_cantus_tagedit_v1")), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(lookup_widget(cantus, "radio_cantus_tagedit_v2")), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(lookup_widget(cantus, "radio_cantus_tagedit_both")), FALSE);
	}
	else if( strtoucmp(active_tag + strlen(active_tag) - 4, ".mp3") == 0
          || strtoucmp(active_tag + strlen(active_tag) - 5, ".flac") == 0 )
	{
		if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_ogg"))))
		{
			gtk_widget_set_sensitive(GTK_WIDGET(lookup_widget(cantus, "radio_cantus_tagedit_v1")), TRUE);
			gtk_widget_set_sensitive(GTK_WIDGET(lookup_widget(cantus, "radio_cantus_tagedit_v2")), TRUE);
			gtk_widget_set_sensitive(GTK_WIDGET(lookup_widget(cantus, "radio_cantus_tagedit_both")), TRUE);
			
			if(cfg.defaulttag == 1)
				gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v1")), TRUE);
			if(cfg.defaulttag == 2)
				gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v2")), TRUE);
			if(cfg.defaulttag == 3)
				gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_both")), TRUE);
			
			gtk_widget_set_sensitive(GTK_WIDGET(lookup_widget(cantus, "radio_cantus_tagedit_ogg")), FALSE);
		}
	}
#endif
	
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v1"))))
		whichtag = 1;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v2"))))
		whichtag = 2;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_both"))))
		whichtag = 3;
#ifdef HAVE_OGG_H
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_ogg"))))
		whichtag = 4;
#endif

// if there is more than one row selected, we do not completely update the tag information,
// but want to show only that fields, that are IDENTICALLY in every file.
	if( (clist->selection)
		&& ((clist->selection)->next || (clist->selection)->prev) )
		tagedit_info_update(active_tag, whichtag, 1);
	else
		tagedit_info_update(active_tag, whichtag, 0);
}


// need that for the function below
static gint
sortbynumber(GtkCList *clist, gconstpointer *ptr1, gconstpointer *ptr2)
{
	gchar *text1 = NULL;
	gchar *text2 = NULL;

	GtkCListRow *row1 = (GtkCListRow *) ptr1;
	GtkCListRow *row2 = (GtkCListRow *) ptr2;

	switch(row1->cell[clist->sort_column].type)
	{
	case GTK_CELL_TEXT:
		text1 = GTK_CELL_TEXT (row1->cell[clist->sort_column])->text;
		break;
	case GTK_CELL_PIXTEXT:
		text1 = GTK_CELL_PIXTEXT (row1->cell[clist->sort_column])->text;
		break;
	default:
		break;
	}

	switch(row2->cell[clist->sort_column].type)
	{
	case GTK_CELL_TEXT:
		text2 = GTK_CELL_TEXT (row2->cell[clist->sort_column])->text;
		break;
	case GTK_CELL_PIXTEXT:
		text2 = GTK_CELL_PIXTEXT (row2->cell[clist->sort_column])->text;
		break;
	default:
		break;
	}

	if(!text2 || *text2 == '\0')
		return(text1 != NULL || *text1 == '\0');

	if(!text1 || *text1 == '\0')
		return -1;
	
	if(atoi((gchar *)text1) == atoi((gchar *)text2))
		return 0;
	
	if(atoi((gchar *)text1) < atoi((gchar *)text2))
		return -1;
	
	return 1;
}

void
on_clist_cantus_queue_click_column       (GtkCList        *clist,
                                        gint             column,
                                        gpointer         user_data)
{
	gtk_clist_set_sort_column(clist, column);
	if( strcmp(gtk_clist_get_column_title(clist, column), "V1-Track") == 0
		|| strcmp(gtk_clist_get_column_title(clist, column), "V2-Track") == 0
		|| strcmp(gtk_clist_get_column_title(clist, column), "Ogg-Track") == 0 )
	{
		gtk_clist_set_compare_func(clist, (GtkCListCompareFunc)sortbynumber);
	}
	else
	{
		gtk_clist_set_compare_func(clist, NULL);
	}
	
	gtk_clist_sort(clist);
	clist_queue_toggle_colors(clist);
}


gboolean
on_clist_cantus_queue_key_press_event    (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
// delete-key = remove selected items
	if(event->keyval==65535)
		queue_files_remove_selected(GTK_CLIST(widget));
// Ctrl/a = select all
	if(event->keyval==97)
		gtk_clist_select_all(GTK_CLIST(widget));
// Ctrl/i = invert selection
	if(event->keyval==105)
		popup_cantus_queue_select(GTK_CLIST(widget), "invert");
	return TRUE;
}


void
on_button_cantus_rename_editrules_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *rules;
	rules = create_rules();
	rules_open();
	dialog_rules_init();
	rules_presets_update();
}


void
on_button_cantus_rename_renamefiles_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	extern gchar *current_dir;
	extern GList *queue;
	extern Cfg cfg;
	
	rules_plugs_rename(queue);
	clist_filelist_update( GTK_CLIST(lookup_widget(cantus, "clist_cantus_filelist")),
		current_dir,
		cfg.showhidden );
	gtk_clist_columns_autosize(GTK_CLIST(lookup_widget(cantus, "clist_cantus_queue")));
	gtk_clist_columns_autosize(GTK_CLIST(lookup_widget(cantus, "clist_cantus_queueonly_queue")));
}


void
on_button_cantus_rename_preview_clicked  (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	extern GList *queue;
	
	rules_plugs_preview(queue);
	gtk_clist_columns_autosize(GTK_CLIST(lookup_widget(cantus, "clist_cantus_queue")));
	gtk_clist_columns_autosize(GTK_CLIST(lookup_widget(cantus, "clist_cantus_queueonly_queue")));
}


void
on_entry_cantus_file2tag_file_changed    (GtkEditable     *editable,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	file2tag_preview_selected(GTK_CLIST(lookup_widget (cantus, "clist_cantus_queue")));
}


void
on_entry_cantus_file2tag_directory_changed
                                        (GtkEditable     *editable,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	file2tag_preview_selected(GTK_CLIST(lookup_widget (cantus, "clist_cantus_queue")));
}


void
on_button_cantus_file2tag_settags_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	file2tag_proceed();
	gtk_clist_columns_autosize(GTK_CLIST(lookup_widget(cantus, "clist_cantus_queue")));
	gtk_clist_columns_autosize(GTK_CLIST(lookup_widget(cantus, "clist_cantus_queueonly_queue")));
}


void
on_button_cantus_tagedit_del_clicked     (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	extern GList *queue;
	extern gchar *active_tag;
	gint whichtag = 0;
	
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v1"))))
		whichtag = 1;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v2"))))
		whichtag = 2;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_both"))))
		whichtag = 3;
#ifdef HAVE_OGG_H
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_ogg"))))
		whichtag = 4;
#endif
	
// Set either one tag (the active tag) or all tags in queue
	if(gtk_toggle_button_get_active
		(GTK_TOGGLE_BUTTON(lookup_widget (cantus, "check_cantus_tagedit_single"))))
	{
		Mp3Info* mp3info = NULL;
		
// no file selected? return.
		if ( *active_tag == '\0' )
			return;
		
// Try to find the file in the queue
		mp3info = (Mp3Info*)g_list_find_matching_fullfilename(queue, active_tag);
		if ( mp3info == NULL )
		{
// it was not found, so we mus create an info struct.
			mp3info = calloc (1, sizeof(Mp3Info));
			
			strncpy (mp3info->filename, strrchr (active_tag, '/'), 2047);
			strncpy (mp3info->directory, active_tag, 2047);
			*(strrchr (mp3info->directory, '/') + 1) = '\0';
			
			tagedit_tag_delete (mp3info, whichtag);
			free (mp3info);
		}
		else
		{
			tagedit_tag_delete (mp3info, whichtag);
		}
	}
	else
		tagedit_tag_delete_queue(whichtag);

	tagedit_info_update(active_tag, whichtag, 0);
	
	gtk_clist_columns_autosize(GTK_CLIST(lookup_widget(cantus, "clist_cantus_queue")));
	gtk_clist_columns_autosize(GTK_CLIST(lookup_widget(cantus, "clist_cantus_queueonly_queue")));
	
// For security reasons i better set the togglebutton to "single tag only" again...
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "check_cantus_tagedit_single")), TRUE);
}


void
on_button_cantus_tagedit_change_clicked  (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	extern gchar *active_tag;
	gint whichtag = 0;
	
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v1"))))
		whichtag = 1;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v2"))))
		whichtag = 2;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_both"))))
		whichtag = 3;
#ifdef HAVE_OGG_H
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_ogg"))))
		whichtag = 4;
#endif
	
// Set either one tag (the active tag) or all tags in queue
	if(gtk_toggle_button_get_active
		(GTK_TOGGLE_BUTTON(lookup_widget (cantus, "check_cantus_tagedit_single"))))
		tagedit_tag_set(active_tag, whichtag);
	else
		tagedit_tag_set_queue(whichtag);
	
	gtk_clist_columns_autosize(GTK_CLIST(lookup_widget(cantus, "clist_cantus_queue")));
	gtk_clist_columns_autosize(GTK_CLIST(lookup_widget(cantus, "clist_cantus_queueonly_queue")));
	
// For security reasons i better set togglebutton to "single tag only" again...
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "check_cantus_tagedit_single")), TRUE);

/* !!!!!!!!!!!!!TODO!!!!!!!!!!!!!!!!!!!!
// activate the next item in the queue or in the filelist
	gtk_clist_select_row(GTK_CLIST(lookup_widget(cantus, "clist_cantus_queue"))
*/
}


void
on_popup_cantus_queue_remove_item_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	extern GtkWidget *lastclicked;
	
	popup_cantus_queue_remove(GTK_CLIST(lastclicked));
}


void
on_popup_cantus_queue_select_errors_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	extern GtkWidget *lastclicked;
	popup_cantus_queue_select(GTK_CLIST(lastclicked), "Rename failed!");
	popup_cantus_queue_select(GTK_CLIST(lastclicked), "Locked!");
}


void
on_popup_cantus_queue_select_todo_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	extern GtkWidget *lastclicked;
	popup_cantus_queue_select(GTK_CLIST(lastclicked), "Todo");
}


void
on_popup_cantus_queue_invert_selection_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	extern GtkWidget *lastclicked;
	popup_cantus_queue_select(GTK_CLIST(lastclicked), "invert");
}


void
on_popup_cantus_queue_select_anytag_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	extern GtkWidget *lastclicked;
	popup_cantus_queue_select(GTK_CLIST(lastclicked), "id3v1");
	popup_cantus_queue_select(GTK_CLIST(lastclicked), "id3v2");
}


void
on_popup_cantus_queue_select_notag_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	extern GtkWidget *lastclicked;
	popup_cantus_queue_select(GTK_CLIST(lastclicked), "notag");
}


void
on_popup_cantus_queue_select_id3v1_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	extern GtkWidget *lastclicked;
	popup_cantus_queue_select(GTK_CLIST(lastclicked), "id3v1");
}


void
on_popup_cantus_queue_select_id3v2_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	extern GtkWidget *lastclicked;
	popup_cantus_queue_select(GTK_CLIST(lastclicked), "id3v2");
}


void
on_popup_cantus_clear_activate           (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	queue_clear();
}


void
on_button_cantus_rename_file_cancel_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *popup_rename_file;
	gtk_widget_destroy(popup_rename_file);
}


void
on_button_cantus_rename_file_ok_clicked  (GtkButton       *button,
                                        gpointer         user_data)
{
	popup_cantus_filelist_rename_file_ok();
}


void
on_entry_cantus_rename_file_name_activate
                                        (GtkEditable     *editable,
                                        gpointer         user_data)
{
	popup_cantus_filelist_rename_file_ok();
}


void
on_button_options_freedb_add_clicked   (GtkButton       *button,
                                        gpointer         user_data)
{
	options_freedb_server_add();
}


void
on_button_options_freedb_delete_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	options_freedb_server_delete();
}


void
on_button_options_freedb_up_clicked    (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *options;
	clist_move_selected_up(GTK_CLIST (lookup_widget (options, "clist_options_freedb_server")));
}


void
on_button_options_freedb_down_clicked  (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *options;
	clist_move_selected_down(GTK_CLIST (lookup_widget (options, "clist_options_freedb_server")));
}


void
on_check_options_freedb_proxy_toggled  (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	extern GtkWidget *options;
	gtk_widget_set_sensitive(
		GTK_WIDGET(lookup_widget(options, "entry_options_freedb_proxy")),
		!GTK_WIDGET_IS_SENSITIVE(lookup_widget(options, "entry_options_freedb_proxy")));
	gtk_widget_set_sensitive(
		GTK_WIDGET(lookup_widget(options, "entry_options_freedb_proxy_port")),
		!GTK_WIDGET_IS_SENSITIVE(lookup_widget(options, "entry_options_freedb_proxy_port")));
}


gboolean
on_clist_options_freedb_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	if(event->keyval==65535)
		options_freedb_server_delete();
	return TRUE;
}


void
on_radio_options_freedb_mode_http_toggled
                                        (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	extern GtkWidget *options;
	gtk_widget_set_sensitive(
		GTK_WIDGET(lookup_widget(options, "entry_options_freedb_path")),
		!GTK_WIDGET_IS_SENSITIVE(lookup_widget(options, "entry_options_freedb_path")));
}


void
on_button_options_listview_add_clicked (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *options;
	options_listview_add( GTK_CLIST(lookup_widget(options, "clist_options_listview_items")),
		GTK_CLIST(lookup_widget(options, "clist_options_listview_columns")) );
}


void
on_button_options_listview_remove_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *options;
	options_listview_add( GTK_CLIST(lookup_widget(options, "clist_options_listview_columns")),
		GTK_CLIST(lookup_widget(options, "clist_options_listview_items")) );
	gtk_clist_sort(GTK_CLIST(lookup_widget(options, "clist_options_listview_items")));
}


void
on_button_options_listview_up_clicked  (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *options;
	clist_move_selected_up(GTK_CLIST (lookup_widget (options, "clist_options_listview_columns")));
}


void
on_button_options_listview_down_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *options;
	clist_move_selected_down(GTK_CLIST (lookup_widget (options, "clist_options_listview_columns")));
}


void
on_button_options_cancel_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *options;
	gtk_widget_destroy(options);
}


void
on_button_options_ok_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *options;
	extern Cfg cfg;
	
	options_dialog_save();
	
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(options, "radio_options_preferences_defaulttag_v1"))))
		cfg.defaulttag = 1;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(options, "radio_options_preferences_defaulttag_v2"))))
		cfg.defaulttag = 2;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(options, "radio_options_preferences_defaulttag_both"))))
		cfg.defaulttag = 3;
	
	gtk_widget_destroy (options);
	
// Re-read configuration.
	destroy_config (&cfg);
	get_config_to_struct (&cfg);
	
// we must update the queue list here, because the user may have changed the appearance of it.
	queue_update (TRUE);
}


void
on_popup_cantus_dirtree_rename_directory_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	popup_cantus_dirtree_rename_directory();
}


void
on_button_cantus_rename_directory_cancel_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *popup_rename_directory;
	gtk_widget_destroy(popup_rename_directory);
}


void
on_button_cantus_rename_directory_ok_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	popup_cantus_dirtree_rename_directory_ok();
}


void
on_entry_cantus_rename_directory_name_activate
                                        (GtkEditable     *editable,
                                        gpointer         user_data)
{
	popup_cantus_dirtree_rename_directory_ok();
}


void
on_button_custom_find_cancel_clicked   (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *custom_find;
	gtk_widget_destroy(custom_find);
}


void
on_button_custom_find_find_clicked     (GtkButton       *button,
                                        gpointer         user_data)
{
	custom_find_find();
}


void
on_button_custom_view_cancel_clicked   (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *custom_view;
	gpointer data = NULL;
	gint cur = -1;
		
	while( (data = gtk_clist_get_row_data(GTK_CLIST(lookup_widget(custom_view, "clist_custom_view_files")), ++cur)) )
		free(data);
	
	gtk_widget_destroy(custom_view);
}


void
on_button_custom_view_add_clicked      (GtkButton       *button,
                                        gpointer         user_data)
{
	custom_view_add();
}

void
on_button_about_ok_clicked             (GtkObject       *object,
                                        gpointer         user_data)
{
	extern GtkWidget *about;
	
	gtk_widget_hide(about);
	gtk_widget_destroy(about);
}


void
on_clist_cantus_filelist_click_column    (GtkCList        *clist,
                                        gint             column,
                                        gpointer         user_data)
{
	gtk_clist_set_sort_column(clist, column);
	gtk_clist_sort(clist);
	clist_queue_toggle_colors(clist);
}


void
on_clist_cantus_filelist_select_row      (GtkCList        *clist,
                                        gint             row,
                                        gint             column,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	extern gchar *active_tag;
	extern gchar *current_dir;
	extern Cfg cfg;
	
	gchar *filename = NULL;
	
	GdkPixmap *pixmap = NULL;
	GdkBitmap *mask = NULL;
	guint8 i = 0;
	gint whichtag = 0;
	
// set new activetag
	gtk_clist_get_pixtext(clist, row, 0, &filename, &i, &pixmap, &mask);
	snprintf(active_tag, 2047, "%s%s", current_dir, filename);

#ifdef HAVE_OGG_H
	if( strtoucmp(active_tag + strlen(active_tag) - 4, ".ogg") == 0 )
	{
		gtk_widget_set_sensitive(GTK_WIDGET(lookup_widget(cantus, "radio_cantus_tagedit_ogg")), TRUE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_ogg")), TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(lookup_widget(cantus, "radio_cantus_tagedit_v1")), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(lookup_widget(cantus, "radio_cantus_tagedit_v2")), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(lookup_widget(cantus, "radio_cantus_tagedit_both")), FALSE);
	}
	else if( strtoucmp(active_tag + strlen(active_tag) - 4, ".mp3") == 0
          || strtoucmp(active_tag + strlen(active_tag) - 5, ".flac") == 0 )
	{
		if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_ogg"))))
		{
			gtk_widget_set_sensitive(GTK_WIDGET(lookup_widget(cantus, "radio_cantus_tagedit_v1")), TRUE);
			gtk_widget_set_sensitive(GTK_WIDGET(lookup_widget(cantus, "radio_cantus_tagedit_v2")), TRUE);
			gtk_widget_set_sensitive(GTK_WIDGET(lookup_widget(cantus, "radio_cantus_tagedit_both")), TRUE);
			
			if(cfg.defaulttag == 1)
				gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v1")), TRUE);
			if(cfg.defaulttag == 2)
				gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v2")), TRUE);
			if(cfg.defaulttag == 3)
				gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_both")), TRUE);
			
			gtk_widget_set_sensitive(GTK_WIDGET(lookup_widget(cantus, "radio_cantus_tagedit_ogg")), FALSE);
		}
	}
#endif
	
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v1"))))
		whichtag = 1;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v2"))))
		whichtag = 2;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_both"))))
		whichtag = 3;
#ifdef HAVE_OGG_H
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_ogg"))))
		whichtag = 4;
#endif

// If something is already selected, just show, which fields are the same.
	if( (clist->selection)
		&& ((clist->selection)->next || (clist->selection)->prev) )
		tagedit_info_update(active_tag, whichtag, 1);
	else
// otherwise simply show the tag
		tagedit_info_update(active_tag, whichtag, 0);
}


void
on_popup_cantus_filelist_rename_activate (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	popup_cantus_filelist_rename_file();
}


void
on_popup_cantus_filelist_refresh_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	popup_cantus_filelist_refresh();
}


gboolean
on_clist_cantus_filelist_button_press_event
                                        (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
	extern GtkWidget *popup_cantus_filelist;

	if( event->button == 3 )
	{
		clist_cantus_filelist_rightclicked(event->x, event->y);
		gtk_menu_popup( GTK_MENU( popup_cantus_filelist ),
			NULL, NULL, NULL, NULL, event->button, event->time );
		return TRUE;
	}
	
	return FALSE;
}



void
on_button_cantus_freedb_get_clicked      (GtkButton       *button,
                                        gpointer         user_data)
{
	freedb_get();
}


void
on_radio_cantus_freedb_fulltext_toggled  (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	gtk_widget_set_sensitive(
		GTK_WIDGET(lookup_widget(cantus, "entry_cantus_freedb_id_category")),
		!GTK_WIDGET_IS_SENSITIVE(lookup_widget(cantus, "entry_cantus_freedb_id_category")));
	gtk_widget_set_sensitive(
		GTK_WIDGET(lookup_widget(cantus, "entry_cantus_freedb_id_id")),
		!GTK_WIDGET_IS_SENSITIVE(lookup_widget(cantus, "entry_cantus_freedb_id_id")));
	gtk_widget_set_sensitive(
		GTK_WIDGET(lookup_widget(cantus, "entry_cantus_freedb_fulltext_string")),
		!GTK_WIDGET_IS_SENSITIVE(lookup_widget(cantus, "entry_cantus_freedb_fulltext_string")));
}


void
on_button_tracks_sort_up_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
	clist_tracks_move_track_up();
}


void
on_button_tracks_sort_down_clicked     (GtkButton       *button,
                                        gpointer         user_data)
{
	clist_tracks_move_track_down();
}


void
on_button_tracks_sort_cancel_clicked   (GtkButton       *button,
                                        gpointer         user_data)
{
	freedb_tracks_cancel();
}


void
on_button_tracks_sort_ok_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	extern GtkWidget *tracks_sort;
	extern Cfg cfg;
	extern gchar *current_dir;
	
	freedb_tracks_ok();
	gtk_widget_destroy(tracks_sort);
	
	clist_filelist_update( GTK_CLIST(lookup_widget(cantus, "clist_cantus_filelist")),
		current_dir,
		cfg.showhidden );
}


void
on_button_tracks_sort_previous_clicked (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *tracks_sort;
	gtk_widget_destroy(tracks_sort);
	
	freedb_get();
}


gint
on_splash_timeout                      (gpointer         user_data)
{
	extern GtkWidget *splash;
	extern GtkWidget *cantus;
	extern gchar *current_dir;
	
	gtk_widget_hide(splash);
	gtk_widget_destroy(splash);
	
// Show mainwindow
	gtk_widget_show(cantus);

// Update dirtree.
	ctree_dirtree_open_directory(
		GTK_CTREE(lookup_widget(cantus, "ctree_cantus_directories")),
		current_dir);

	return(0);
}


gboolean
on_clist_cantus_filelist_key_press_event (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
// Ctrl/a = select all
	if( event->keyval == 97 )
		gtk_clist_select_all(GTK_CLIST(widget));
	return TRUE;
}


void
on_entry_cantus_freedb_fulltext_string_activate
                                        (GtkEditable     *editable,
                                        gpointer         user_data)
{
	freedb_get();
}


void
on_entry_cantus_freedb_id_category_activate
                                        (GtkEditable     *editable,
                                        gpointer         user_data)
{
	freedb_get();
}


void
on_entry_cantus_freedb_id_id_activate    (GtkEditable     *editable,
                                        gpointer         user_data)
{
	freedb_get();
}


void
on_button_wait_cancel_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *wait;
	gtk_widget_hide(wait);
}


void
on_button_cantus_tag2filename_selected_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	extern gchar *current_dir;
	extern Cfg cfg;

	tag2filename_selection();
	clist_filelist_update( GTK_CLIST(lookup_widget(cantus, "clist_cantus_filelist")),
		current_dir,
		cfg.showhidden );
}


void
on_button_cantus_tag2filename_queue_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	extern gchar *current_dir;
	extern Cfg cfg;
	
	tag2filename_queue();
	clist_filelist_update( GTK_CLIST(lookup_widget(cantus, "clist_cantus_filelist")),
		current_dir,
		cfg.showhidden );
}

void
on_clist_tracks_sort_cdlist_select_row (GtkCList        *clist,
                                        gint             row,
                                        gint             column,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	freedb_album_clicked();
}

void
on_radio_cantus_tagedit_v2_toggled       (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	extern gchar *active_tag;
	gint whichtag = 0;
	
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v1"))))
		whichtag = 1;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v2"))))
		whichtag = 2;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_both"))))
		whichtag = 3;
	
	tagedit_info_update(active_tag, whichtag, 0);
}


void
on_radio_cantus_tagedit_v1_toggled       (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	extern gchar *active_tag;
	gint whichtag = 0;
	
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v1"))))
		whichtag = 1;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v2"))))
		whichtag = 2;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_both"))))
		whichtag = 3;
	
	tagedit_info_update(active_tag, whichtag, 0);
}


void
on_button_rules_del_clicked            (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *rules;
	
	rules_preset_delete(
		gtk_entry_get_text(GTK_ENTRY(lookup_widget(rules, "entry_rules_presets"))) );
	rules_presets_update();
}


void
on_popup_cantus_log_log_clear_log_activate       (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	gtk_clist_clear(GTK_CLIST(lookup_widget(cantus, "ctree_cantus_log_log")));
}


gboolean
on_ctree_cantus_log_log_button_press_event
                                        (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
	extern GtkWidget *popup_cantus_log_log;

	if( event->button == 3 )
	{
		clist_cantus_filelist_rightclicked(event->x, event->y);
		gtk_menu_popup( GTK_MENU( popup_cantus_log_log ),
			NULL, NULL, NULL, NULL, event->button, event->time );
		return TRUE;
	}
	
	return FALSE;
}


void
on_entry_options_characters_destination_activate
                                        (GtkEditable     *editable,
                                        gpointer         user_data)
{
	options_specialchar_add();
}


void
on_button_options_characters_ok_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	options_specialchar_add();
}


void
on_button_options_characters_del_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	options_specialchar_del();
}

void
on_clist_options_characters_list_select_row
                                        (GtkCList        *clist,
                                        gint             row,
                                        gint             column,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	options_specialchar_update();
}



void
on_button_cantus_defaultdir_clicked      (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	extern gchar *current_dir;
	extern Cfg cfg;
	gchar *config = malloc(10000);
	gchar value[4096];
	
// read configfile
	config_read(config, 10000);
	configfile_option_get(config, "program general", "default directory", value);

// and open the default directory
	if(value && *value != '\0')
	{
		ctree_dirtree_open_directory(
			GTK_CTREE(lookup_widget(cantus, "ctree_cantus_directories")),
			value);
		strncpy(current_dir, value, 2048);
		clist_filelist_update(GTK_CLIST(lookup_widget(cantus, "clist_cantus_filelist")),
			current_dir,
			cfg.showhidden );
	}
	free(config);
}


void
on_button_cantus_homedir_clicked         (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	extern gchar *current_dir;
	extern Cfg cfg;
	gchar homedir[4096];
	
	strncpy(homedir, getenv("HOME"), 4096);

// and open the home directory
	if(homedir && *homedir != '\0')
	{
		ctree_dirtree_open_directory(
			GTK_CTREE(lookup_widget(cantus, "ctree_cantus_directories")),
			homedir);
		strncpy(current_dir, homedir, 2048);
		clist_filelist_update(GTK_CLIST(lookup_widget(cantus, "clist_cantus_filelist")),
			current_dir,
			cfg.showhidden );
	}
}




gboolean
on_entry_cantus_rename_file_name_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_filechar_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_cantus_file2tag_file_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_filechar_pattern_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_cantus_file2tag_directory_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_dirchar_pattern_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_cantus_tag2filename_format_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_filechar_pattern_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_cantus_tagedit_album_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;

// Tabindex
	if ( event->keyval == 65056
		|| (event->keyval == GDK_Tab && (event->state & GDK_SHIFT_MASK)) )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_prev(widget);
		return TRUE;
	}
	if ( event->keyval == GDK_Tab )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_next(widget);
		return TRUE;
	}
	
// maxlen for v1 tag
	if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v1")))
		&& !is_cursorkey(event->keyval)
		&& strlen(gtk_entry_get_text(GTK_ENTRY(widget))) > 30 )
	{
		gtk_signal_emit_stop_by_name (GTK_OBJECT(widget), "key_press_event");
		return FALSE;
	}

	return entry_allow_all_but_lf(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_cantus_tagedit_artist_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;

// Tabindex
	if ( event->keyval == 65056
		|| (event->keyval == GDK_Tab && (event->state & GDK_SHIFT_MASK)) )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_prev(widget);
		return TRUE;
	}
	if ( event->keyval == GDK_Tab )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_next(widget);
		return TRUE;
	}

// maxlen for v1 tag
	if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v1")))
		&& !is_cursorkey(event->keyval)
		&& strlen(gtk_entry_get_text(GTK_ENTRY(widget))) > 30 )
	{
		gtk_signal_emit_stop_by_name (GTK_OBJECT(widget), "key_press_event");
		return FALSE;
	}
	
	return entry_allow_all_but_lf(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_cantus_tagedit_song_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;

// Tabindex
	if ( event->keyval == 65056
		|| (event->keyval == GDK_Tab && (event->state & GDK_SHIFT_MASK)) )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_prev(widget);
		return TRUE;
	}
	if ( event->keyval == GDK_Tab )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_next(widget);
		return TRUE;
	}
	
// maxlen for v1 tag
	if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v1")))
		&& !is_cursorkey(event->keyval)
		&& strlen(gtk_entry_get_text(GTK_ENTRY(widget))) > 30 )
	{
		gtk_signal_emit_stop_by_name (GTK_OBJECT(widget), "key_press_event");
		return FALSE;
	}
	
	return entry_allow_all_but_lf(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_cantus_tagedit_year_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
// Tabindex
	if ( event->keyval == 65056
		|| (event->keyval == GDK_Tab && (event->state & GDK_SHIFT_MASK)) )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_prev(widget);
		return TRUE;
	}
	if ( event->keyval == GDK_Tab )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_next(widget);
		return TRUE;
	}
	
// maxlen
	if( !is_cursorkey(event->keyval)
		&& strlen(gtk_entry_get_text(GTK_ENTRY(widget))) > 4 )
	{
		gtk_signal_emit_stop_by_name (GTK_OBJECT(widget), "key_press_event");
		return FALSE;
	}
	
	return entry_allow_digit_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_cantus_tagedit_track_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
// Tabindex
	if ( event->keyval == 65056
		|| (event->keyval == GDK_Tab && (event->state & GDK_SHIFT_MASK)) )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_prev(widget);
		return TRUE;
	}
	if ( event->keyval == GDK_Tab )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_next(widget);
		return TRUE;
	}
	
// maxlen for v1 tag
	if( !is_cursorkey(event->keyval)
		&& strlen(gtk_entry_get_text(GTK_ENTRY(widget))) > 2 )
	{
		gtk_signal_emit_stop_by_name (GTK_OBJECT(widget), "key_press_event");
		return FALSE;
	}
	
	return entry_allow_digit_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_cantus_tagedit_genre_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
// Tabindex
	if ( event->keyval == 65056
		|| (event->keyval == GDK_Tab && (event->state & GDK_SHIFT_MASK)) )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_prev(widget);
		return TRUE;
	}
	if ( event->keyval == GDK_Tab )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_next(widget);
		return TRUE;
	}
	
	return entry_allow_all_but_lf(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_cantus_freedb_fulltext_string_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_all_but_lf(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_cantus_freedb_id_category_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_all_but_lf(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_cantus_freedb_id_id_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_all_but_lf(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_options_preferences_defaultdir_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_dirchar_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_options_freedb_server_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_all_but_lf(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_options_freedb_port_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_digit_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_options_freedb_path_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_dirchar_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_options_freedb_proxy_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_all_but_lf(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_options_freedb_proxy_port_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_digit_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_options_freedb_path_local_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_dirchar_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_options_characters_source_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_all_but_lf(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_options_characters_destination_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_filechar_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_cantus_rename_directory_name_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_filechar_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_custom_find_filename_must_match_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_all_but_lf(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_custom_find_filename_directory_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_dirchar_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_custom_find_filename_directory_not_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_dirchar_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_custom_find_filename_must_not_match_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_all_but_lf(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_custom_find_other_filesize_kb_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_digit_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_rules_replace_from_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_filechar_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_rules_replace_to_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_filechar_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_rules_fromformat_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_filechar_pattern_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_rules_toformat_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_filechar_pattern_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_rules_v2tag_get_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_filechar_pattern_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_rules_v1tag_get_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_filechar_pattern_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_rules_presets_key_press_event (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_all_but_lf(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_rules_delete_occurence_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_filechar_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_rules_move_file_to_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_dirchar_only(GTK_OBJECT(widget), event->keyval);
}


void
on_popup_cantus_filelist_delete_file_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	popup_cantus_filelist_delete_file();
}


void
on_popup_cantus_dirtree_create_directory_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	extern GtkWidget *popup_create_directory;
	
	popup_create_directory = create_create_directory();
	gtk_widget_show(popup_create_directory);
}


void
on_popup_cantus_dirtree_delete_directory_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	popup_cantus_dirtree_delete_directory();
}


void
on_button_cantus_really_do_cancel_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *popup_really_do;
	
	gtk_widget_destroy(popup_really_do);
}


void
on_button_cantus_really_do_ok_clicked    (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;
	extern GtkWidget *popup_really_do;
	extern Cfg cfg;
	extern gchar *current_dir;
	
	popup_cantus_dirtree_really_do_ok();
	
	gtk_widget_destroy(popup_really_do);
	
	clist_filelist_update(GTK_CLIST(lookup_widget(cantus, "clist_cantus_filelist")),
		current_dir,
		cfg.showhidden );
}


void
on_button_create_directory_cancel_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *popup_create_directory;
	
	gtk_widget_destroy(popup_create_directory);
}


void
on_button_create_directory_ok_clicked  (GtkButton       *button,
                                        gpointer         user_data)
{
	extern GtkWidget *popup_create_directory;
	
	popup_cantus_createdir_ok();
	
	gtk_widget_destroy(popup_create_directory);
}


void
on_entry_create_directory_activate     (GtkEditable     *editable,
                                        gpointer         user_data)
{
	extern GtkWidget *popup_create_directory;
	
	popup_cantus_createdir_ok();
	
	gtk_widget_destroy(popup_create_directory);
}


gboolean
on_entry_create_directory_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_filechar_only(GTK_OBJECT(widget), event->keyval);
}


gboolean
on_entry_rules_vorbistag_get_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	return entry_allow_filechar_pattern_only(GTK_OBJECT(widget), event->keyval);
}


void
on_popup_cantus_queue_select_all_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	extern GtkWidget *lastclicked;
	gtk_clist_select_all(GTK_CLIST(lastclicked));
}


gboolean
on_text_cantus_tagedit_comment_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
	extern GtkWidget *cantus;

// Tabindex
	if ( event->keyval == 65056
		|| (event->keyval == GDK_Tab && (event->state & GDK_SHIFT_MASK)) )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_prev(widget);
		return TRUE;
	}
	if ( event->keyval == GDK_Tab )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_next(widget);
		return TRUE;
	}

// maxlen for v1 tag
	if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(cantus, "radio_cantus_tagedit_v1")))
		&& !is_cursorkey(event->keyval)
		&& gtk_text_get_length(GTK_TEXT(widget)) > 28 )
	{
		gtk_signal_emit_stop_by_name (GTK_OBJECT(widget), "key_press_event");
		return FALSE;
	}

	return TRUE;
}


void
on_popup_cantus_filelist_addselected_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	queue_files_add_selected();
}


gboolean
on_check_cantus_tagedit_artist_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
// Tabindex
	if ( event->keyval == 65056
		|| (event->keyval == GDK_Tab && (event->state & GDK_SHIFT_MASK)) )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_prev(widget);
		return TRUE;
	}
	if ( event->keyval == GDK_Tab )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_next(widget);
		return TRUE;
	}
	return FALSE;
}


gboolean
on_check_cantus_tagedit_song_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
// Tabindex
	if ( event->keyval == 65056
		|| (event->keyval == GDK_Tab && (event->state & GDK_SHIFT_MASK)) )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_prev(widget);
		return TRUE;
	}
	if ( event->keyval == GDK_Tab )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_next(widget);
		return TRUE;
	}
	return FALSE;
}


gboolean
on_check_cantus_tagedit_album_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
// Tabindex
	if ( event->keyval == 65056
		|| (event->keyval == GDK_Tab && (event->state & GDK_SHIFT_MASK)) )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_prev(widget);
		return TRUE;
	}
	if ( event->keyval == GDK_Tab )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_next(widget);
		return TRUE;
	}
	return FALSE;
}


gboolean
on_check_cantus_tagedit_year_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
// Tabindex
	if ( event->keyval == 65056
		|| (event->keyval == GDK_Tab && (event->state & GDK_SHIFT_MASK)) )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_prev(widget);
		return TRUE;
	}
	if ( event->keyval == GDK_Tab )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_next(widget);
		return TRUE;
	}
	return FALSE;
}


gboolean
on_check_cantus_tagedit_track_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
// Tabindex
	if ( event->keyval == 65056
		|| (event->keyval == GDK_Tab && (event->state & GDK_SHIFT_MASK)) )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_prev(widget);
		return TRUE;
	}
	if ( event->keyval == GDK_Tab )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_next(widget);
		return TRUE;
	}
	return FALSE;
}


gboolean
on_check_cantus_tagedit_comment_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
// Tabindex
	if ( event->keyval == 65056
		|| (event->keyval == GDK_Tab && (event->state & GDK_SHIFT_MASK)) )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_prev(widget);
		return TRUE;
	}
	if ( event->keyval == GDK_Tab )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_next(widget);
		return TRUE;
	}
	return FALSE;
}


gboolean
on_check_cantus_tagedit_genre_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
// Tabindex
	if ( event->keyval == 65056
		|| (event->keyval == GDK_Tab && (event->state & GDK_SHIFT_MASK)) )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_prev(widget);
		return TRUE;
	}
	if ( event->keyval == GDK_Tab )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_next(widget);
		return TRUE;
	}
	return FALSE;
}


gboolean
on_button_cantus_tagedit_del_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
// Tabindex
	if ( event->keyval == 65056
		|| (event->keyval == GDK_Tab && (event->state & GDK_SHIFT_MASK)) )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_prev(widget);
		return TRUE;
	}
	if ( event->keyval == GDK_Tab )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_next(widget);
		return TRUE;
	}
	return FALSE;
}


gboolean
on_check_cantus_tagedit_single_key_press_event
                                        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
// Tabindex
	if ( event->keyval == 65056
		|| (event->keyval == GDK_Tab && (event->state & GDK_SHIFT_MASK)) )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_prev(widget);
		return TRUE;
	}
	if ( event->keyval == GDK_Tab )
	{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tagedit_focus_next(widget);
		return TRUE;
	}
	return FALSE;
}
