/*  CVideoDevice: video capture base class
    part of CamStream: a collection of GUI webcam tools
    Copyright (C) 2002 Nemosoft Unv.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For questions, remarks, patches, etc. for this program, the author can be
    reached at camstream@smcc.demon.nl.
*/

#ifndef VIDEO_DEVICE_HPP
#define VIDEO_DEVICE_HPP

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <qglobal.h>

#if (QT_VERSION >= 0x030000)
#include <qmutex.h>
#endif

#include <qarray.h>
#include <qcolor.h>
#include <qdom.h>
#include <qimage.h>
#include <qlist.h>
#include <qobject.h>
#include <qsocketnotifier.h>
#include <qtimer.h>
#include <qthread.h>
#include <qvector.h>

#include "VideoFrame.h"

#include "TVChannel.h"

class CChannelEditorDlg;

class CVideoDevice: public QObject, protected QThread
{
friend class CVideoCollector;
   Q_OBJECT
public:
   enum Signals {
     frame_ready = 1,
   };

private:
   enum Palettes {
     RGB,
     YUV,
   };

   QList<CVideoFrame> m_VideoFrames;
   QList<CVideoFrame> m_EmptyFrames, m_FullFrames;
   CVideoFrame *m_pFillFrame;   ///< Pointer to frame that can be filled.
   QMutex m_BufMutex;           ///< Lock when you're about to manipulate m_CurBuffer or access an image
///   QWaitCondition m_WaitForEmpty;
   QWaitCondition m_WaitFromFrame;
   CChannelEditorDlg *m_pTunerDlg;

   int m_CaptureCount;

   void IncrementPalette(Palettes);
   void DecrementPalette(Palettes);

   void ParseTVChannels(const QDomNode &);
   void GetTVChannels(QDomNode &) const;

private slots:
   virtual void NewFrequency(float) = 0;
           void NewCurrentChannel();

protected:
   QList<TVChannel> m_TVChannels;

   bool m_Validated;
   QString m_NodeName, m_IntfName, m_SerialNumber;
   bool m_HasDisplayDialog, m_HasFormatDialog, m_HasSourceDialog, m_HasTunerDialog;
   bool m_Mutable;

   int m_OpenCount;

   int m_PalRGB;  ///< RGB images are requested
   int m_PalYUV;  ///< YUV images are requested
   uint m_RequestedBuffers, m_Buffers;
   int m_FrameCount;  ///< Reset at \ref Open()
   int m_FrameDropped;

   QSize m_ImageSize;
   QVector<QImage> m_RGB;
   QVector<QImage> m_Y, m_U, m_V;
   QImage *m_pNullImage;
   QRgb m_GrayScale[256];

   int GetCaptureCount() const;

   virtual bool Init() = 0;
   virtual void Exit() = 0;

   virtual void CreateImagesRGB() = 0;
   virtual void DeleteImagesRGB() = 0;
   virtual void CreateImagesYUV() = 0;
   virtual void DeleteImagesYUV() = 0;

   void CreateVideoFrames();
   void DeleteVideoFrames();
   CVideoFrame *GetFillFrame();
   void ReturnFillFrame(bool filled = true);

   virtual bool StartCapture() = 0;
   virtual void StopCapture() = 0;

   void SendSignal(Signals s);

public:
   enum ErrorCodes {
     NoError,
     SizeChangeFailed,
   };

   CVideoDevice();
   virtual ~CVideoDevice();

   virtual bool event(QEvent *);

   virtual void SetConfiguration(const QDomNode &);
   virtual void GetConfiguration(QDomNode &) const;

   bool IsValid() const;
   bool Open(uint buffers = 0);
   void Close();
   bool IsOpen() const;

   virtual int GetNumberOfInputs() const = 0;

   /**
     \brief Select an input
     \param input The input to use. Between 0 and GetNumberOfInputs()

     This will select an input on your TV card. Most cards have more
     than one input, for example S-VHS, composite and tuner. Even if you
     have only one input, you may want to select it so it gets initialized
     properly.
     \note \ref GetConfiguration() and \ref SetConfiguration() will remember
     the selected input.
   */
   virtual void SetInput(int input) = 0;
   /**
     \brief Get currently selected input
     \return A positive integer, or -1 in case input is unknown

     Returns the number of the currently selected input (if known). Otherwise
     returns -1.

     Usually this variable will be initialized during SetConfiguration().
    */
   virtual int GetInput() const = 0;
   /**
     \brief Get number of tuners attached to this video device
     \return A positive integer

     When you use a TV card, it has a tuner on board which you can use to
     select a frequency to watch. This function tells the number of tuners
     attached. Rarely this will be larger than 1.

     You may have to select a specific input to watch TV; some cards also
     have S-VHS or composite input. Unfortunately it is not possible to
     determine to which input a tuner is attached.
   */
   virtual int GetNumberOfTuners() const = 0;
   /**
     \brief Select a tuner
     \param tuner The tuner to use. between 0 and GetNumberOfTuners()

     This will select a tuner. Even if you have only one tuner, you may want
     to select it so it gets initialized properly.

     \note \ref GetConfiguration() and \ref SetConfiguration() will remember
     the selected tuner.
   */
   virtual void SetTuner(int tuner) = 0;
   virtual int GetTuner() const = 0;

   unsigned int GetNumberOfTVChannels() const;
   TVChannel QueryTVChannel(unsigned int preset);
   TVChannel GetCurrentTVChannel() const;
   TVChannel SelectTVChannel(unsigned int preset);
   TVChannel SelectNextTVChannel(bool wrap = true);
   TVChannel SelectPrevTVChannel(bool wrap = true);

   /** \brief System dependant descriptor to device.

     File descriptor under Unix, WHND under Windows; should be casted
     to apropriate type.

     Only valid when the device is open!
   */
   virtual long GetDescriptor() const = 0;

   QString GetNodeName() const;
   QString GetIntfName() const;
   QString GetSerialNumber() const;

   int GetBuffers() const;

   CVideoFrame *GetLatestVideoFrame(int backlog = 0);

   bool HasDisplayDialog() const;
   bool HasFormatDialog() const;
   bool HasSourceDialog() const;
   bool HasTunerDialog() const;

   virtual QSize GetSize() const;

   // Do we need these?
   bool IsMutable() const;
   virtual void Mute(bool on) const = 0;

public slots:
   virtual void ShowDisplayDialog() = 0;
   virtual void ShowFormatDialog() = 0;
   virtual void ShowSourceDialog() = 0;
   virtual void ShowTunerDialog();

   void EnableRGB();
   void DisableRGB();
   void EnableYUV();
   void DisableYUV();

signals:
   /** The device is being opened by the first user; the device is now ready and streaming */
   void Opened();
   /** The device is being closed by the last user; the device is not accessible */
   void Closed();
   /** New size */
   void SizeChanged(const QSize &);
   /** New framerate */
   void FramerateChanged(int rate);
   /** An error occured. err_num is one of the errno values. */
   void Error(int err_num);
   /** One ore more new image(s) is(are) available. */
   void FrameReady();
   /** A new TV channel (frequency) was selected */
   void TVChannelChanged();
};

#endif
