/**
  \class CFTPClient
  \brief A low-level, asynchronous FTP client based on RFC 959.

  This class implements an FTP client. It allows the usual things done
  through FTP (uploading, downloading, dir-listing, etc), but at a very
  basic level.

  It may be a bit difficult to use, because it is so low level.  It isn't as
  simple as "open ftp server, do dir-listing, upload file, close". All these
  functions are there, but they are asynchronuous. That is, they return
  immediately and don't wait for the end result. Nor can you queue commands:
  only one command can be executed at a time. This means you will have to do
  the high-level protocol stuff in your own class.  On the other hand, it
  allows you to accurately follow the progress of your commands. Think of it
  as a wrapper around the FTP protocol itself, with some helper functions
  for the data transfers.

  With this class you have to make a distinction between 'commands' and
  'state'. A command is just that: the user demands a certain action, and
  the class executes it. A string is sent to the FTP server to perform the
  command; if a data stream is needed that is opened as well.

  A command will cause various changes in the class that can be tracked
  through the ref StateChange signal. For example, uploading a file will
  cause the following states: SendingPort, WaitData, Transfer, ClosingData,
  Idle (or Failed). It reflects the various steps necessary by the FTP
  protocol to perform an actual filetransfer. It is up to the calling
  class to interpret these responses (most aren't really interesting to the
  user anyway).

  All commands will end in either "Idle" (== success) or "Failed" state.
  "Failed" indicates something went wrong; the \b result code from
  StateChange contains the numeric FTP response code that explains
  the error. "Idle", which also doubles as "Success", indicates completion
  of the command without error.

  Note that it is possible for a state to be set multiple times in a row.
  If, for example, Login was succesful and was followed by a "Change Dir"
  that also succeeds, StateChange will be emitted twice with state "Idle".
  This is okay; 2 commands were issued, so you get two StateChange(Idle)
  (possibly intermingled with other StateChange()s).

  Most commands listed below show a short state transition diagram, so
  you know what StateChange()s to expect.

*/

#include <qglobal.h>

#include <ctype.h>
#include <errno.h>
#include <sys/types.h>

#if defined(Q_OS_UNIX) || defined(_OS_UNIX_)
#include <fcntl.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <unistd.h>

#define OS_CLOSE  ::close
#define OS_ERRNO   errno
#endif

#if defined(Q_OS_WIN32) || defined(_OS_WIN32_)
#include <winsock2.h>
#define EWOULDBLOCK  WSAEWOULDBLOCK
#define EINPROGRESS  WSAEINPROGRESS
typedef int socklen_t;

#define OS_CLOSE  closesocket
#define OS_ERRNO  WSAGetLastError()
#endif

#include "FTPClient.h"
#include "tracer.h"

TR_MODULE("FTPClient.cc");

#undef DEBUG_TRANSFER
#define DEBUG_TRANSFER

const char *CFTPClient::CommandStr[] =
{
   "Nop",
   "Login",
   "Logout",
   "SetType",
   "ChangeDir",
   "ListDir",
   "Upload",
   "Download",
   "Rename",

   "Passive",
   "DeleteLocalFile",
   "Destroy",
};

const char *CFTPClient::StateStr[] =
{
   "NOC",
   "Resolving servername",
   "Connecting",
   "Connected",
   "Login",
   "Authenticate",

   "Idle",

   "SendingPort",
   "WaitData",
   "Transfer",
   "ClosingData",

   "Failed",
   "Unknown",
};


CFTPClient::CFTPClient(QObject *parent, const char *name)
    : QObject(parent, name)
{
   TR_ENTER();
   m_pCtrlFD = 0;
   m_DataFD = -1;
   m_pDataNotifier = 0;
   m_pDataConnectNotifier = 0;
   m_pListenFD = 0;

   inputbuffer = new char[1024];
   linebuffer = new char[1024];
   LineLen = 0;

   m_TransferBufferSize = 32 * 1024;
   m_TransferBufferHead = 0;
   m_TransferBufferUsed = 0;
   m_TransferBuffer = new char[m_TransferBufferSize];

   m_TransferMethod = tmUnknown;
   m_Passive = false;
   m_Log = false;
   CurrentCommand = cmdNop;
   CurrentState = stNOC;
   TR_LEAVE();
}

CFTPClient::~CFTPClient()
{
   TR_ENTER();
   // SendQuit();
   CloseAllFD();
   delete [] inputbuffer;
   delete [] linebuffer;
   delete [] m_TransferBuffer;
   TR_LEAVE();
}

// private

void CFTPClient::InitBuffers()
{
   LastChar = '\0';
   m_ResponseBuffer = "";
   Response = 0;
   m_TransferMethod = tmAscii; // ASCII is default
   m_TransferBufferUsed = 0;
   m_TransferBufferHead = 0;
}

/**
  \brief Closes all TCP connections immediately
*/

void CFTPClient::CloseAllFD()
{
   if (m_Log) qDebug(">> CFTPClient::CloseAllFD()");
   CloseData();
   CloseListen();
   CloseLocal();
   CloseControl();
   if (m_Log) qDebug("<< CFTPClient::CloseAllFD()");
}

/**
  \brief Try to connect to the server's FTP control port.
*/
void CFTPClient::SetupControl()
{
   if (m_Log) qDebug(">> CFTPClient::SetupControl()");
   if (m_pCtrlFD != 0) // already connected
     return;
   m_pCtrlFD = new QSocket(this, "FTP control socket");
   if (m_pCtrlFD == 0) {
     qWarning("Failed to create FTP control socket");
     SetState(stFailed);
   }
   else {
     connect(m_pCtrlFD, SIGNAL(hostFound()), SLOT(ControlResolved()));
     connect(m_pCtrlFD, SIGNAL(connected()), SLOT(ControlConnected()));
     connect(m_pCtrlFD, SIGNAL(readyRead()), SLOT(ControlRead()));
     connect(m_pCtrlFD, SIGNAL(error(int)),  SLOT(ControlError(int)));
     connect(m_pCtrlFD, SIGNAL(connectionClosed()),    SLOT(ControlClosed()));
     connect(m_pCtrlFD, SIGNAL(delayedCloseFinised()), SLOT(ControlClosed()));
     SetState(stDNSBusy);
     m_pCtrlFD->connectToHost(m_ServerName, m_ServerPort);
   }
   if (m_Log) qDebug("<< CFTPClient::SetupControl()");
}


/**
  \brief Shut down connection to FTP server control port.
*/
void CFTPClient::CloseControl()
{
   if (m_Log) qDebug(">> CFTPClient::CloseControl() Shutting down control socket [%d].", m_pCtrlFD ? m_pCtrlFD->socket() : -2);
   delete m_pCtrlFD;
   m_pCtrlFD = 0;
   emit ControlPortClosed();
   SetState(stNOC);
   if (m_Log) qDebug("<< CFTPClient::CloseControl() Shutting down control socket.");
}

/**
  \brief Sets up our incoming port where we will listen

  In normal FTP mode, the FTP server connects to us; therefor we must set up
  a listing port ourselves.
*/
bool CFTPClient::SetupListen()
{
   if (m_Log) qDebug(">> CFTPClient::SetupListen()");
   if (m_pListenFD == 0) {
     m_pListenFD = new CFTPListen(0, this); // pick a port
     if (!m_pListenFD->ok())
     {
       qWarning("CFTPClient::SetupListen() Failed to create CFTPListen instance.");
       delete m_pListenFD;
       m_pListenFD = 0;
       return false;
     }
     connect(m_pListenFD, SIGNAL(ConnectionEstablished()), SLOT(ListenConnect()));
     if (m_Log) qDebug("CFTPClient::SetupListen() Opened listen fd [%d] at port %d.", m_pListenFD->socket(), m_pListenFD->port());
   }
   if (m_Log) qDebug("<< CFTPClient::SetupListen()");
   return true;
}

/**
  \brief Close our listen port
*/
void CFTPClient::CloseListen()
{
   if (m_Log) qDebug(">> CFTPClient::CloseListen() Shutting down listen fd [%d].", m_pListenFD ? m_pListenFD->socket() : -2);
   delete m_pListenFD;
   m_pListenFD = 0;
   if (m_Log) qDebug("<< CFTPClient::CloseListen()");
}

/**
  \brief Set up the local file for transfer
  \param filename Local filename
  \param write Open file for writing
  \param truncate Truncate file when writing
*/
bool CFTPClient::SetupLocal(const QString &filename, bool write, bool truncate)
{
   bool ret;

   if (m_Log) qDebug(">> CFTPClient::SetupLocal(...)");
   if (m_LocalFile.handle() >= 0) {
     qWarning("CFTPClient::SetupLocal() Error: local file fd still open [%d]!\n", m_LocalFile.handle());
     m_LocalFile.close();
   }

   m_LocalFile.setName(filename);
   if (write)
     ret = m_LocalFile.open(IO_WriteOnly | (truncate ? IO_Truncate : 0));
   else
     ret = m_LocalFile.open(IO_ReadOnly);
   if (ret)
     if (m_Log) qDebug("Opened local file fd [%d].", m_LocalFile.handle());
   else
     qWarning("CFTPClient::SetupLocal() failed.");
   if (m_Log) qDebug("<< CFTPClient::SetupLocal(...)");
   return ret;
}

/**
  \brief Close fd of local file on disk
*/
void CFTPClient::CloseLocal()
{
   if (m_Log) qDebug(">> CFTPClient::CloseLocal() Shutting down local file fd [%d].", m_LocalFile.handle());
   m_LocalFile.close();
   if (m_Log) qDebug("<< CFTPClient::CloseLocal()");
}


/**
  \brief Set up a data pipe in active mode (that is, we listen)
*/
bool CFTPClient::SetupDataActive()
{
   if (m_Log) qDebug(">> CFTPClient::SetupDataActive()");
   if (m_pCtrlFD == 0)
     return false;
   if (!SetupListen()) {
     qWarning("CFTPCLient::SetupDataActive() SetupListen() failed.");
     return false;
   }

   /* Note that we use m_pCtrlFD for our address; m_pListen is 0.0.0.0 */
   if (m_Log) qDebug("CFTPClient::SetupDataActive() %s:%d", (const char *)m_pCtrlFD->address().toString(), m_pListenFD->port());

   /* Tell the other side we have a connection ready */
   TotalTransfered = 0;
   SetState(stSendingPort);
   SendPort(m_pCtrlFD->address(), m_pListenFD->port());
   if (m_Log) qDebug("<< CFTPClient::SetupDataActive()");
   return true;
}

bool CFTPClient::SetupDataPassive()
{
   bool Ret = true;

   if (m_Log) qDebug(">> CFTPClient::SetupDataPassive()");
   // In passive mode we connect to the other side; the address was given with the PASV response.
   if (m_DataFD >= 0) {
     qWarning("CFTPClient::SetupDataPassive() We are still connected?!");
     OS_CLOSE(m_DataFD);
     m_DataFD = -1;
   }

   TotalTransfered = 0;

   /* Old school socket stuff. Qt's socket sucks, in some ways */
   m_DataFD = ::socket(AF_INET, SOCK_STREAM, 0);
   if (m_DataFD < 0) {
     qWarning("CFTPClient::SetupDataPassive() Can't create data socket! errno = %d", OS_ERRNO);
   }
   else
   {
     struct sockaddr_in SocketAddress;
     int ConRet;
#ifdef Q_OS_UNIX
     int SocketFlags;

     // We connect non-blocking
     if (m_Log) qDebug("Setting data port non-blocking");
     SocketFlags = fcntl(m_DataFD, F_GETFL);
     if (SocketFlags < 0)
     {
       qWarning("fcntl(F_GETFL) failed. errno = %d", OS_ERRNO);
     }
     else
     {
       SocketFlags |= O_NONBLOCK;
       if (fcntl(m_DataFD, F_SETFL, SocketFlags) < 0)
       {
         qWarning("fcntl(F_SETFL) failed. errno = %d", OS_ERRNO);
       }
     }
#endif
     SocketAddress.sin_family = AF_INET;
     SocketAddress.sin_port = htons(m_PassivePort);
     SocketAddress.sin_addr.s_addr = htonl(m_PassiveAddress.ip4Addr());

     ConRet = ::connect(m_DataFD, (const struct sockaddr *)&SocketAddress, sizeof(SocketAddress));
     if (ConRet == 0 || (ConRet < 0 && OS_ERRNO == EINPROGRESS)) {
       if (ConRet < 0) {
         qDebug("Connecting using non-blocking method.");
         // non-blocking, set up notifier
         m_pDataConnectNotifier = new QSocketNotifier(m_DataFD, QSocketNotifier::Write, this, "data connect notifier");
         if (m_pDataConnectNotifier != 0) {
           connect(m_pDataConnectNotifier, SIGNAL(activated(int)), this, SLOT(DataConnect(int)));
           m_pDataConnectNotifier->setEnabled(true);
         }
       }
       else {
         // We were connected immediate
         qDebug("Immediately connected.");
         DataConnect(m_DataFD);
       }
     }
     else {
       qWarning("CFTPClient::SetupDataPassive() connect failed! errno = %d", OS_ERRNO);
       Ret = false;
     }
   }
   if (m_Log) qDebug("<< CFTPClient::SetupDataPassive()");
   return Ret;
}


/**
  \brief Close data socket
*/
void CFTPClient::CloseData()
{
   qDebug("CFTPClient::CloseData() Shutting down data fd [%d].", m_DataFD);
   // TODO: flush input buffer
//   if (Direction == dirDownload)
//     Write

   delete m_pDataNotifier;
   m_pDataNotifier = 0;
   delete m_pDataConnectNotifier;
   m_pDataConnectNotifier = 0;
   if (m_DataFD >= 0) {
     OS_CLOSE(m_DataFD);
     m_DataFD = -1;
   }
}

/**
   Called when the FTP data connection has been established. Sets up
   the signals for read and write events.
*/
void CFTPClient::HookData()
{
   if (m_Log) qDebug(">> CFTPClient::HookData()");

   if (m_DataFD < 0) {
     qWarning("CFTPClient::HookData() Trying to hook up to a empty data socket...");
     return;
   }

   // make non-blocking
#if 0
#ifdef Q_OS_UNIX
   int SocketFlags;
   int fd = m_pDataFD->socket();

   SocketFlags = fcntl(fd, F_GETFL);
   if (SocketFlags < 0)
   {
     qWarning("fcntl(F_GETFL) failed. errno = %d", OS_ERRNO);
   }
   else
   {
     SocketFlags |= O_NONBLOCK;
     if (fcntl(fd, F_SETFL, SocketFlags) < 0)
     {
       qWarning("fcntl(F_SETFL) failed. errno = %d", OS_ERRNO);
     }
   }
#endif
#endif

//   connect(m_pDataFD, SIGNAL(connectionClosed()),     SLOT(DataClose()));
//   connect(m_pDataFD, SIGNAL(delayedCloseFinished()), SLOT(DataClose()));

   if (CurrentCommand == cmdDownload || CurrentCommand == cmdListDir) {
     m_pDataNotifier = new QSocketNotifier(m_DataFD, QSocketNotifier::Read, this, "FTP Data read notifier");
     connect(m_pDataNotifier, SIGNAL(activated(int)), SLOT(DataRead()));
   }
   if (CurrentCommand == cmdUpload) {
     m_pDataNotifier = new QSocketNotifier(m_DataFD, QSocketNotifier::Write, this, "FTP Data write notifier");
     connect(m_pDataNotifier, SIGNAL(activated(int)), SLOT(DataWrite()));
   }
   if (CurrentState != stWaitData && m_pDataNotifier != 0) {
     m_pDataNotifier->setEnabled(false); // wait until 125/150 response
   }
   if (m_Log) qDebug("<< CFTPClient::HookData()");
}





/**
  \brief Set new state en emit signal
*/
void CFTPClient::SetState(States new_st, int response)
{
   CurrentState = new_st;
   emit StateChange(CurrentCommand, new_st, response == 0 ? Response : response, m_ResponseBuffer);
}

/**
  \brief Takes a line from the control port input buffer and unfolds it.
*/
void CFTPClient::InterpretLine()
{
   int l, resp;

//printf("InterpretLine(): %s\n", linebuffer);
   l = strlen(linebuffer);
   if (l < 3) {
     if (m_Log) qDebug("InterpretLine(): short line.");
     return;
   }

   // See if we have 3 digits at the beginning of the line
   resp = 0;
   if (isdigit(linebuffer[0]) && isdigit(linebuffer[1]) && isdigit(linebuffer[2]))
     resp = 100 * (linebuffer[0] - '0') + 10 * (linebuffer[1] - '0') + (linebuffer[2] - '0');
   if (resp == 0) {
     qWarning("CFTPClient::InterpretLine() could not decipher digits.");
     return;
   }

   if (l > 4) { // Append any text there may be present at the line. Append linefeeds.
     m_ResponseBuffer.append(&linebuffer[4]);
     m_ResponseBuffer.append('\n');
   }
   if (m_ResponseBuffer.length() > 20000) { // uh... this doesn't look right
     qWarning("CFTPClient::InterpretLine() ResponseBuffer overload. Bogus FTP server?");
     m_ResponseBuffer = ""; // clear
   }

   // Multiline responses have a '-' right after the digits. If not, this is the end of the response.
   // Single line responses simply don't have the '-'
   if (l == 3 || linebuffer[3] != '-') {
     Response = resp;
     InterpretResponse();
     m_ResponseBuffer = "";
     Response = 0;
   }
}


/**
  \brief Takes a complete, unfolded line from the control port and interprets the response
*/
void CFTPClient::InterpretResponse()
{
   if (m_Log) qDebug(">> CFTPClient::InterpretResponse(%d, \"%s\"), CurrentCmd = %d (\"%s\")", Response, m_ResponseBuffer.ascii(), CurrentCommand, CommandStr[CurrentCommand]);
   switch(Response) {
     case 120: // Service ready in nnn minutes
       SetState(stFailed);
       CloseControl();
       SetState(stNOC);
       break;

     case 125: // port is open
       //SetState(stTransfer);
       //break;

     case 150: // Transfer is about to start
       /* There is a continuous race condition between the 150 response
          and the actual opening of the data connection. It could be
          the 150 arrives after the datapipe has been opened (happens
          often on a local LAN).
        */
       if (CurrentCommand != cmdListDir && CurrentCommand != cmdUpload && CurrentCommand != cmdDownload)
         qWarning("Received 150 repose, but not expecting any data.");
       // Careful here; in passive mode, it is possible m_DataFD is already setup (in SetupDataPassive),
       // but the connection hasn't been established yet (non-blocking mode, remember). It is possible
       // the 150 response got here earlier than the connect. In that case m_DataFD !=0 but m_pDataNotifier
       // is still 0.
       // Solution is to set stWaitData and then wait for the actual connect; in HookData() the
       // m_pDataNotifier is left enabled in this specific case (normally disabled)
       // Note: I can't create and enable the DataNotifier immediately, since in case of a
       // semi-connected socket, we'll get spammed by the notifier.
       if (m_DataFD >= 0 && m_pDataNotifier != 0) {
         qDebug("CFTPClient::InterpretResponse() 150 Enabling data pipe.");
         m_pDataNotifier->setEnabled(true);
       }
       else {
         SetState(stWaitData);
         qDebug("CFTPClient::InterpretResponse() Got 150 response but no data pipe yet.");
       }
       break;

     case 200: // General 'okay' response
       switch(CurrentCommand) {
         case cmdSetType:
           // host accepted type
           CurrentCommand = cmdNop;
           SetState(stIdle);
           break;

         case cmdUpload:
           StartSending();
           break;

         case cmdDownload:
           StartReceiving();
           break;

         case cmdListDir:
           // Send the actual command
           SendList();
           break;

         case cmdChangeDir:
           qWarning("Received 200 response for CWD command.");
           CurrentCommand = cmdNop;
           SetState(stIdle);
           break;

         default:
           qWarning("Unexpected 200 response");
           CurrentCommand = cmdNop;
           SetState(stUnknown);
           break;
       }    // ..switch CurrentState
       break;

     case 220: // Ready for new user
       if (CurrentCommand == cmdLogin) {
         SetState(stLogin);
         SendUser();
       }
       break;

     case 221: // Logged out
       if (CurrentCommand == cmdLogout) {
         CloseAllFD();
         SetState(stNOC);
       }
       else
         qWarning("Got 221 response, but I didn't logout?");
       break;

     case 226: // Transfer completed
       if (CurrentState == stClosingData || CurrentState == stTransfer) {
         SetState(stIdle);
       }
       else
         qWarning("226 without transfer?");

       if (CurrentCommand == cmdDownload || CurrentCommand == cmdUpload || CurrentCommand == cmdListDir)
         CurrentCommand = cmdNop;
       else
         qWarning("226 Response, but not in data transfer mode? (upload/download/listdir)");
       break;

     case 227: // Transfering in passive mode
       {
         QString braces; // text between ()
         Q_UINT32 lAddress;
         int left, right, i;
         bool addrOk = true;

         braces = m_ResponseBuffer;
         left = braces.find('(');
         right = braces.findRev(')', -1);
         if (left >= 0 && right > left) {
           braces = braces.mid(left + 1, right - left - 1);
           /* Hmpf. Just think of all the FTP clients that have to switch to IPv6...  */
           lAddress = 0;
           for (i = 0; i < 4; i++) {
              left = braces.find(',');
              if (left > 0) {
                lAddress = (lAddress << 8) + braces.left(left).toInt();
                braces = braces.mid(left + 1);
              }
              else {
                addrOk = false;
                break;
              }
           }
           m_PassiveAddress.setAddress(lAddress);
           //qDebug("Passive addr = 0x%x (%ud)", m_PassiveAddress, m_PassiveAddress);

           m_PassivePort = 0;
           left = braces.find(',');
           if (left > 0) {
             m_PassivePort = braces.left(left).toInt();
             braces = braces.mid(left + 1);
           }
           else {
             addrOk = false;
           }
           m_PassivePort = (m_PassivePort << 8) + braces.toInt();

           //qDebug("Passive port = 0x%x (%d)", m_PassivePort, m_PassivePort);
           if (!addrOk) {
             SetState(stFailed);
           }
           else {
             switch(CurrentCommand) {
               case cmdDownload: StartReceiving(); break;
               case cmdUpload: StartSending(); break;
               case cmdListDir: SendList(); break;
               default: SetState(stFailed); break; // the command doesn't make sense
             }
             if (!SetupDataPassive()) {
               SetState(stFailed);
             }
           }
         }
         else {
           qWarning("Could not interpret 227 response properly");
           SetState(stFailed);
         }
       }
       break;

     case 230: // Login succeeded
       if (CurrentCommand == cmdLogin && (CurrentState == stAuthenticate || CurrentState == stLogin)) {
         CurrentCommand = cmdNop;
         SetState(stIdle);
         emit LoggedIn();
         // SendTimeOut ??
       }
       break;

     case 250: // CWD succeeded
       if (CurrentCommand == cmdChangeDir) {
         CurrentCommand = cmdNop;
         SetState(stIdle);
       }
       break;

     case 331: // Password required
       if (CurrentCommand == cmdLogin && CurrentState == stLogin) {
         SetState(stAuthenticate);
         SendPass();
       }
       break;

     case 421: // Service not available
       SetState(stFailed);
       CloseControl();
       SetState(stNOC);
       break;

     case 426: // Transfer aborted
       if (m_DataFD >= 0) {
         CloseData();
       }
       SetState(stFailed);
       break;

     case 500: // Syntax error
       SetState(stFailed);
       break;

     case 501: // Syntax error in parameters
       SetState(stFailed);
       break;

     case 504: // Command not implemented for that parameter (Duh? 'parameter not valid for that command' sounds a lot more logical...)
       SetState(stFailed);
       break;

     case 530: // Login failed/not logged in
       SetState(stFailed);
       emit LoginFailed();
       //SetState(stConnected); // hmm, retries?
       break;

     case 550:
       if (CurrentCommand == cmdChangeDir) {
         // ChDir failed
         SetState(stFailed);
       }
       else if (CurrentCommand == cmdUpload || CurrentCommand == cmdDownload) {
         // Permission deniend
         SetState(stFailed);
       }
       else {
         qWarning("Received unexpected 550 response");
         SetState(stUnknown);
       }
       break;

     case 553: // no rights
       if (CurrentCommand == cmdUpload || CurrentCommand == cmdDownload) {
         SetState(stFailed);
       }

     default:
       qWarning("Unknown response %d.", Response);
       SetState(stUnknown);
       break;
   }
   Response = 0;
   if (m_Log) qDebug("<< CFTPClient::InterpretResponse");
}


/**
  \brief Send content of outputbuffer
*/
void CFTPClient::Send()
{
  uint l;

//qDebug("CFTPClient::Send(): %s", (const char *)outputbuffer);
  l = outputbuffer.length();
  if (l > 0 && m_pCtrlFD != 0)
    m_pCtrlFD->writeBlock(outputbuffer.latin1(), l); // Should be non-blocking as well
}


void CFTPClient::SendUser()
{
   outputbuffer = "USER " + m_UserName + "\r\n";
   Send();
}

void CFTPClient::SendPass()
{
   outputbuffer = "PASS " + m_Password + "\r\n";
   Send();
}

void CFTPClient::SendList()
{
   outputbuffer = "LIST\r\n";
   Send();
}

void CFTPClient::SendPort(const QHostAddress &addr, Q_UINT16 port)
{
   Q_UINT32 l;

   if (addr.isIp4Addr()) {
     l = addr.ip4Addr();
     outputbuffer.sprintf("PORT %d,%d,%d,%d,%d,%d\r\n",
                  (l >> 24) & 0xff, (l >> 16) & 0xff, (l >> 8) & 0xff, l & 0xff,
                  (port >> 8) & 0xff, port & 0xff);
   }
   Send();
}

void CFTPClient::SendStore(const QString &filename)
{
   outputbuffer = "STOR " + filename + "\r\n";
   Send();
}



void CFTPClient::StartSending()
{
   /* We are ready to go.. we have our local filedescriptor, the remote
    * side has our port, now send a STOR and see what happens
    */
   SetState(stTransfer);
   SendStore(RemoteFileName);
}

void CFTPClient::StartReceiving()
{
   SetState(stTransfer);
   //SendRetrieve(RemoteFileName);
}


// private slots


void CFTPClient::ControlRead()
{
   int err, i;
   char c;

   /* Read data from control socket. Because FTP responses can consist of
      multiple lines we need a 3-stage process:
        inputbuffer contains the raw input data from the socket;
        linebuffer contains a whole line (possibly truncated);
        m_ResponseBuffer has the complete command.

      Linebuffer is delimited on the \r\n sequence; the \r\n sequence is
      removed and replaced by a '\0' to make it a proper string.  To prevent
      bad servers from trying to overflow our stack/memory, linebuffer will
      never contain more than 1023 characters; if we receive more than this
      before a \r\n sequence, the rest is ignored.
    */
   err = m_pCtrlFD->readBlock(inputbuffer, 1023);
   if (err < 0) {
     // Emit error?
     qDebug("CFTPClient::ControlRead() Error reading from control pipe: %d\n", err);
     return;
   }
   if (err == 0) { // EOF
     qDebug("CFTPClient::ControlRead() Control pipe closed by remote end.\n");
     //SetState(stClosing);
     CloseAllFD();
     SetState(stNOC);
     return;
   }

   inputbuffer[err] = '\0';
   //qDebug("CFTPClient::ControlRead() Read from control socket: `%s'", inputbuffer);

   for (i = 0; i < err; i++) {
      c = inputbuffer[i];
      if (c == '\n') { // End-Of-Line
        if (LastChar != '\r') {
          qWarning("Bogus line from FTP server (invalid CRLF sequence)");
        }
        else {
          linebuffer[LineLen] = '\0'; // Terminate; LineLen is never larger than 1023
          InterpretLine();
        }
        LineLen = 0;
      }
      else if (isprint(c)) { // Copy character
        if (LineLen < 1023)
          linebuffer[LineLen++] = c;
      }
      // Anything else is ignored

      LastChar = c;
   }

}

void CFTPClient::ControlError(int e)
{
   if (m_Log) qDebug(">> CFTPClient::ControlError(%d); CurrentState = %d", e, CurrentState);
   switch(CurrentState) {
     case stDNSBusy:
       SetState(stFailed, 810);
       break;
     case stConnecting:
       SetState(stFailed, 811);
       break;
     default:
       SetState(stFailed, 0);
       break;
   }
   if (m_Log) qDebug("<< CFTPClient::ControlError()");
}

void CFTPClient::ControlResolved()
{
   if (m_Log) qDebug(">> CFTPClient::ControlResolved()");
   SetState(stConnecting);
   if (m_Log) qDebug(">> CFTPClient::ControlResolved()");
}

void CFTPClient::ControlConnected()
{
   if (m_Log) qDebug(">> CFTPClient::ControlConnected()");
   m_MyAddress = m_pCtrlFD->address();
   SetState(stConnected);
   if (m_Log) qDebug("<< CFTPClient::ControlConnected()");
}


/**
  \brief Helper slot; called when the remote server closes the control connection
*/
void CFTPClient::ControlClosed()
{
  if (m_Log) qDebug(">> CFTPClient::ControlConnected()");
  emit ControlPortClosed();
  SetState(stNOC);
  if (m_Log) qDebug("<< CFTPClient::ControlConnected()");
}



/**
  \brief Activated when our listening data port is contacted.

  In active mode, we (the client) wait for the FTP server to contact us
  for the data-transfer; we have sent the address & port with a PORT
  command before. A connection on a listening socket is signalled by
  a 'read' on that socket.
*/
void CFTPClient::ListenConnect()
{
   int NewFD = 0;
   struct sockaddr_in SockAddr;
   QHostAddress HostAddr;
   socklen_t SockAddrLen;
   int HostPort;

   if (m_Log) qDebug(">> CFTPClient::ListenConnect()");
   NewFD = m_pListenFD->FetchSocket();
   if (NewFD < 0) {
     qWarning("Duh? Got new incoming connection but no socket?");
     return;
   }

   if (m_DataFD >= 0) {
     // Hey? That's odd...
     qWarning("Warning: got a second connection on data port; dropping it");// from %s:%d", (const char *)new_data->peerName(), new_data->peerPort());
     OS_CLOSE(NewFD); // drop it on the floor
   }
   else {
     SockAddrLen = sizeof(SockAddr);
     getpeername(NewFD, (struct sockaddr *)&SockAddr, &SockAddrLen);
     HostPort = ntohs(SockAddr.sin_port);
     if (HostPort != (m_ServerPort - 1)) {
       qWarning("Got an incoming connection from an unmatching port (%d)", HostPort);
     }
     HostAddr.setAddress(ntohl(SockAddr.sin_addr.s_addr));
     qDebug("NewFD: %s:%d", HostAddr.toString().ascii(), HostPort);

     m_DataFD = NewFD;

     // The listen port has fullfilled its purpose, close it now.
     CloseListen();

     ///if (m_Log) qDebug("CFTPClient::ListenConnect() Opened data fd [%d] = %s:%d.", m_pDataFD->socket(), (const char *)m_pDataFD->peerName(), m_pDataFD->peerPort());
     HookData();
     SetState(stTransfer);
   }
   if (m_Log) qDebug("<< CFTPClient::ListenConnect()");
}


/**
  \brief Asynchronuous data connect in passive mode

  Called when our outgoing data port connects. In Passive mode we
  setup the data connection to the server ourselves.
*/
void CFTPClient::DataConnect(int socket)
{
   if (m_Log) qDebug(">> CFTPClient::DataConnect(%d)", socket);
   if (m_DataFD < 0 || socket != m_DataFD) {
     qWarning("CFTPClient::DataConnect() without data socket/invalid socket.");
   }
   else
   {
     delete m_pDataConnectNotifier; // This has fulfilled its purpose
     m_pDataConnectNotifier = 0;

     // Get error status
     int ErrorStatus = 0;
     socklen_t ErrorStatusLen = sizeof(ErrorStatus);

     if (getsockopt(m_DataFD, SOL_SOCKET, SO_ERROR,
#if defined(Q_OS_WIN32) || defined (_OS_WIN32_)
                    (char *)&ErrorStatus,
#else
                    &ErrorStatus,
#endif
                    &ErrorStatusLen) < 0) {
       qWarning("CFTPClient::DataConnect() getsockopt failed! errno = %d", OS_ERRNO);
     }
     else {
       if (ErrorStatus != 0) {
         qWarning("CFTPClient::DataConnect() connection failed, reason = %d", ErrorStatus);
         SetState(stFailed);
       }
       else {
         HookData();
         SetState(stTransfer);
       }
     }
   }
   if (m_Log) qDebug("<< CFTPClient::DataConnect()");
}


/**
  \brief Activated when data is coming in on our data socket
*/
void CFTPClient::DataRead()
{
   char buf[10240];
   int rd;

   if (m_DataFD < 0)
     return;

#if defined(_OS_WIN32_) || defined(Q_OS_WIN32)
   rd = recv(m_DataFD, buf, 10000, 0);
#else
   rd = ::read(m_DataFD, buf, 10000);
#endif
   if (m_Log) qDebug("CFTPClient::DataRead() Read %d bytes.", rd);
   buf[rd] = '\0';

   if (rd == 0) { // End-Of-File
     SetState(stClosingData);
     CloseData();
   }
}

/**
  \brief Activated when there is output space available in the data socket

  We use this slot to write more data to the output socket.
*/
void CFTPClient::DataWrite()
{
   int InBuf, OutWrite;

   // Check if we're already nearly done
   if (CurrentState == stClosingData) {
     return;
   }

   if (m_DataFD < 0 || m_pDataNotifier == 0) {
     qWarning("CFTPClient::DataWrite() without data socket?");
     return;
   }

   // Disable now; we must enable it later so we will get triggered when there's
   // room available; otherwise nothing happens.
   m_pDataNotifier->setEnabled(false);

   if (m_TransferBufferUsed == 0) { // Our buffer is empty
     m_TransferBufferHead = 0;
     if (m_LocalFile.handle() >= 0) { // we still have data to read from the file
       InBuf = m_LocalFile.readBlock(m_TransferBuffer, m_TransferBufferSize);
       if (InBuf > 0) {
         m_TransferBufferUsed = InBuf;
       }
       else { // end-of-file, or error
         if (InBuf < 0) {
           qWarning("CFTPClient::DataWrite() Error reading local file; status = %d", m_LocalFile.status());
         }
         CloseLocal();
       }
     }
   }

   // If buffer is still empty then we've read all there is. Close data pipe and disable notifier.
   if (m_TransferBufferUsed == 0) {
     SetState(stClosingData);
#ifdef DEBUG_TRANSFER
     qDebug("Closing data socket...");
#endif
     CloseData();
   }

   if (m_TransferBufferUsed > 0) {
     // We have some data in our buffer. Stuff it in the ouput socket
#ifdef DEBUG_TRANSFER
     qDebug("writing data socket (%d bytes)...", m_TransferBufferUsed);
#endif
#if defined(_OS_WIN32_) || defined(Q_OS_WIN32)
     OutWrite = send(m_DataFD, &m_TransferBuffer[m_TransferBufferHead], m_TransferBufferUsed, 0);
#else
     OutWrite = ::write(m_DataFD, &m_TransferBuffer[m_TransferBufferHead], m_TransferBufferUsed);
#endif
#ifdef DEBUG_TRANSFER
     qDebug("written to data socket (%d bytes)...", OutWrite);
#endif
     if (OutWrite < 0) {
       if (OS_ERRNO != EWOULDBLOCK) {
         qWarning("CFTPClient::DataWrite() Error writing socket; errno = %d", OS_ERRNO);
       }
#ifdef DEBUG
       else {
         // EWOULDBLOCK is normal behaviour for this socket
         qDebug("EWOULDBLOCK");
       }
#endif
     }
     else {
       m_TransferBufferHead += OutWrite;
       m_TransferBufferUsed -= OutWrite;

       TotalTransfered += OutWrite;
     }
   }
   if (m_pDataNotifier != 0) {
     m_pDataNotifier->setEnabled(true);
   }
   emit Progress(TotalTransfered);
}

/* Helper slot */
void CFTPClient::DataClose()
{
  if (m_Log) qDebug(">> CFTPClient::DataClose()");
  CloseData();
  SetState(stIdle);
  if (m_Log) qDebug("<< CFTPClient::DataClose()");
}

// public

void CFTPClient::SetLogging(bool log)
{
   m_Log = log;
}


int CFTPClient::GetCommand() const
{
   return CurrentCommand;
}

int CFTPClient::GetState() const
{
   return (int)CurrentState;
}

/** \brief Set transfer mode to passive

  Usually FTP transfers (both up- and downloads) are initiated by
  the \i server, after the command has been sent by the client. However,
  this does not always work (firewalls, braindead FTP implementations, etc)
  and the client has to do the work; this is called 'passive mode', and
  this function turns this mode on.

  Note: this function does not trigger a state change.
*/
void CFTPClient::SetPassive()
{
   m_Passive = true;
}

/** \brief Turns off passive mode; see \ref SetPassive */
void CFTPClient::SetActive()
{
   m_Passive = false;
}

/**
  \brief Initiate connection to FTP server
  \param user The username
  \param pass The password
  \param server The server name
  \param port Port number on the server

  This function initiates a connection to the server. Since Qt provides
  asynchronuous DNS lookups, we don't even block here. However, that means
  we are not connected yet when this function returns.
  */
void CFTPClient::Connect(const QString &user, const QString &pass, const QString &server, int port)
{
   if (m_pCtrlFD != 0) {
     // SendAbort();
     CloseAllFD();
     SetState(stNOC);
   }

   m_UserName = user;
   m_Password = pass;
   m_ServerName = server;
   m_ServerPort = port;
   CurrentCommand = cmdLogin;
   InitBuffers();
   SetupControl();
}


/**
  \brief Upload file to remote site
  \param local_file The filename on the local machine
  \param remote_file The filename on the remote machine; if not present, the same as local_file
*/
void CFTPClient::Upload(const QString &local_file, const QString &remote_file)
{
   if (!SetupLocal(local_file, false)) {
     SetState(stFailed, 800);
     return;
   }

   if (remote_file.isNull())
     RemoteFileName = local_file;
   else
     RemoteFileName = remote_file;

   CurrentCommand = cmdUpload;
   if (m_Passive) {
     outputbuffer = "PASV\r\n";
     Send(); // That's all. We await the 227 response
   }
   else {
     if (!SetupDataActive()) {
       qDebug("CFTPClient::Upload() SetupData() failed:.");
       SetState(stFailed);
       CurrentCommand = cmdNop;
     }
   }
}


void CFTPClient::SetTypeAscii()
{
   m_TransferMethod = tmAscii;
   CurrentCommand = cmdSetType;
   outputbuffer = "TYPE A\r\n";
   Send();
}

void CFTPClient::SetTypeBinary()
{
   m_TransferMethod = tmBinary;
   CurrentCommand = cmdSetType;
   outputbuffer = "TYPE I\r\n";
   Send();
}

/** \brief Change to remote directory on server
*/
void CFTPClient::ChangeDir(const QString &new_dir)
{
   CurrentCommand = cmdChangeDir;
   outputbuffer = "CWD " + new_dir + "\r\n";
   Send();
}


/** \brief List contents of remote directory.

   This command will retrieve the list of filenames of the directory
   on the remote server. The found files are emitted one by one through
   \ref ListDirEntry(const QString &filename).

   Note: make sure you set the transfer type to ASCII first.
*/
void CFTPClient::ListDir()
{
   CurrentCommand = cmdListDir;
   if (m_Passive) {
     outputbuffer = "PASV\r\n";
     Send(); // That's all. We await the 227 response
   }
   else {
     if (!SetupDataActive()) {
       qDebug("CFTPClient::ListDir() SetupData() failed.");
       SetState(stFailed);
       CurrentCommand = cmdNop;
     }
   }
}


/** \brief Rename a file on the server
  \param from The existing filename
  \param to The new filename

  This command will instruct the FTP server to rename a file on the
  remote filesystem. A rename across directories (effectively moving
  the file) may be possible. A rename across filesystems most likely
  is not.
*/
void CFTPClient::Rename(const QString &from, const QString &to)
{
   CurrentCommand = cmdRename;
   outputbuffer = "RNFR " + from + "\r\n";
   outputbuffer += "RNTO " + to + "\r\n";
   Send();
}



void CFTPClient::Logout()
{
   CurrentCommand = cmdLogout;
   outputbuffer = "QUIT\r\n";
   Send();
}

QString CFTPClient::GetErrorString(int result) const
{
   QString res = tr("unknown");

   switch(result) {
     case 110: res = tr("Restart marker"); break;
     case 120: res = tr("Service momentarely unavailable."); break;
   };
   return res;
}

