/*****************************************************************************
* $CAMITK_LICENCE_BEGIN$
*
* CamiTK - Computer Assisted Medical Intervention ToolKit
* (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
*
* Visit http://camitk.imag.fr for more information
*
* This file is part of CamiTK.
*
* CamiTK is free software: you can redistribute it and/or modify
* it under the terms of the GNU Lesser General Public License version 3
* only, as published by the Free Software Foundation.
*
* CamiTK is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU Lesser General Public License version 3 for more details.
*
* You should have received a copy of the GNU Lesser General Public License
* version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
*
* $CAMITK_LICENCE_END$
****************************************************************************/


#include "MeshComponent.h"
#include "InteractiveViewer.h"
#include "Property.h"
#include <Application.h>

//-- Qt stuff
#include <QMessageBox>
#include <QTextStream>
#include <QFileInfo>
#include <QTableView>

//-- vtk stuff
#include <vtkPointSet.h>
#include <vtkUnstructuredGrid.h>
#include <vtkTetra.h>
#include <vtkHexahedron.h>
#include <vtkWedge.h>
#include <vtkPyramid.h>
#include <vtkCellArray.h>
#include <vtkGenericCell.h>
#include <vtkSelection.h>
#include <vtkSelectionNode.h>
#include <vtkCellData.h>
#include <vtkPointData.h>
#include <vtkDataSetAttributes.h>
#include <vtkExtractSelection.h>
#include <vtkDataSetMapper.h>
#include <vtkProperty.h>

namespace camitk {

// -------------------- constructor --------------------
MeshComponent::MeshComponent ( const QString & file ) throw ( AbortException ) : Component ( file, "Mesh", Component::GEOMETRY ) {
    init();
}

MeshComponent::MeshComponent ( vtkSmartPointer<vtkPointSet> aPointSet, const QString &name ) : Component ( "", name, Component::GEOMETRY ) {
    init();
    initRepresentation ( aPointSet );
    setModified();
}

MeshComponent::MeshComponent ( Component *parentComponent, vtkSmartPointer<vtkPointSet> aPointSet, const QString &name ) : Component ( parentComponent, name, Component::GEOMETRY ) {
    init();
    initRepresentation ( aPointSet );
}

// -------------------- destructor --------------------
MeshComponent::~MeshComponent() {
    if (selectionView != NULL)
        delete selectionView;
    if (dataView != NULL)
        delete dataView;
}

void MeshComponent::init() {
    pickedCellId = -1;
    pickedPointId = -1;

    selectionModel = new MeshSelectionModel ( this );

    // create selection view
    selectionView = new QTableView();
    selectionView->setObjectName ( "Selection" );
    selectionView->setSelectionBehavior(QAbstractItemView::SelectRows);
    selectionView->setModel ( selectionModel );
    selectionView->setSelectionMode(QAbstractItemView::SingleSelection);

    // create data view
    dataModel = new MeshDataModel( this );
    dataView = new QTableView();
    dataView->setObjectName ( "Data" );
    dataView->setSelectionBehavior(QAbstractItemView::SelectRows);
    dataView->setModel ( dataModel );
    dataView->setSelectionMode(QAbstractItemView::SingleSelection);

    connect(selectionView->selectionModel(), SIGNAL(currentChanged(const QModelIndex&, const QModelIndex&)), this, SLOT(changeSelectedSelection(const QModelIndex&, const QModelIndex&)));

    // selection
    currentSelection = vtkSmartPointer<vtkSelection>::New();
    vtkSmartPointer<vtkSelectionNode> selectedPoints = vtkSmartPointer<vtkSelectionNode>::New();
    selectedPoints->SetFieldType ( vtkSelectionNode::POINT );
    selectedPoints->SetContentType ( vtkSelectionNode::INDICES );
    vtkSmartPointer<vtkSelectionNode> selectedCells = vtkSmartPointer<vtkSelectionNode>::New();
    selectedCells->SetFieldType ( vtkSelectionNode::CELL );
    selectedCells->SetContentType ( vtkSelectionNode::INDICES );
    selectionList ["Selected Points"] = selectedPoints;
    selectionList ["Selected Cells"] = selectedCells;
    selectionModel->refresh();
}

// -------------------- initRepresentation --------------------
void MeshComponent::initRepresentation ( vtkSmartPointer<vtkPointSet> originalPointSet ) {
    // if there is no point set yet, just do nothing
    if ( originalPointSet == NULL )
        return;
    // else replace the point set
    if ( myGeometry ) {
        myGeometry->setPointSet ( originalPointSet );
    } else {
        myGeometry = new Geometry ( this->getName(), originalPointSet );
    }
    myGeometry->setMeshWorldTransform(getTransformFromWorld());

    // add it in the InteractiveViewer (automatically)
    setVisibility ( InteractiveViewer::get3DViewer(), true );

    vtkSmartPointer<vtkExtractSelection> selectionExtractor = vtkSmartPointer<vtkExtractSelection>::New();
    vtkSmartPointer<vtkActor> selectionActor = vtkSmartPointer<vtkActor>::New();
    vtkSmartPointer<vtkDataSetMapper> selectionMapper = vtkSmartPointer<vtkDataSetMapper>::New();

    //-- selection
    selectionExtractor->SetInputConnection ( 0, this->getDataPort() );
    selectionExtractor->SetInput ( 1, currentSelection );

    selectionMapper->SetInputConnection(selectionExtractor->GetOutputPort());

    selectionActor->SetPickable ( false );
    selectionActor->GetProperty()->SetRepresentationToSurface();
    selectionActor->GetProperty()->SetLineWidth ( 5 );
    selectionActor->GetProperty()->SetColor ( 1,0,0 );
    selectionActor->GetProperty()->SetPointSize(5);
    selectionActor->GetProperty()->SetOpacity ( 0.2 );
    selectionActor->SetMapper(selectionMapper);

    addProp("Selection",selectionActor);

    // initialize the dynamic properties
    initDynamicProperties();
}

// -------------------- cellPicked --------------------
void MeshComponent::cellPicked ( vtkIdType cellId, bool ) {
    pickedCellId = cellId;
}

// -------------------- pointPicked --------------------
void MeshComponent::pointPicked ( vtkIdType pointId, bool ) {
    pickedPointId = pointId;
}

// -------------------- getPickedCellId --------------------
vtkIdType MeshComponent::getPickedCellId() {
    return pickedCellId;
}

// -------------------- getPickedPointId --------------------
vtkIdType MeshComponent::getPickedPointId() {
    return pickedPointId;
}

// -------------------- initDynamicProperties --------------------
void MeshComponent::initDynamicProperties() {
    vtkIdType count = 0;

    if ( getPointSet() !=NULL )
        count = getPointSet()->GetNumberOfPoints();
    Property *nbPoints = new Property ( "Number Of Points", QVariant ( QString ( "%1" ).arg ( count ) ), "Number of 3D Points composing the geometry", "" );
    nbPoints->setReadOnly ( true );
    addProperty ( nbPoints );

    if ( getPointSet() !=NULL )
        count = getPointSet()->GetNumberOfCells();
    Property *nbCells = new Property ( "Number Of Cells", QVariant ( QString ( "%1" ).arg ( count ) ), "Number of Cells composing the geometry", "" );
    nbCells->setReadOnly ( true );
    addProperty ( nbCells );

    if ( getPointSet() !=NULL ) {
        // add a dynamic property to manage the surface color
        // setProperty("position point #1", QVector3D(1.0,0.0,0.0));
        vtkSmartPointer<vtkGenericCell> cell = vtkGenericCell::New();
        std::map<unsigned char, int> elementsMap;
        std::map<unsigned char, int>::iterator elementsMapIt;

        for ( int i=0; i<getPointSet()->GetNumberOfCells(); i++ ) {
            getPointSet()->GetCell ( i, cell );

            if ( !elementsMap.count ( cell->GetCellType() ) )
                elementsMap[ cell->GetCellType()] = 0;
            elementsMap[ cell->GetCellType() ]++;

        }

        // the list of all possible cell types is defined in VTKCellType enum of the VTKCellType class
        for ( elementsMapIt = elementsMap.begin(); elementsMapIt != elementsMap.end(); elementsMapIt++ ) {
            Property *cellProp;
            switch ( elementsMapIt->first ) {
            case VTK_EMPTY_CELL:
                cellProp = new Property ( "Empty Cells", elementsMapIt->second, tr("Number Of Empty Cells"), "" );
                break;
            case VTK_VERTEX:
                cellProp = new Property ( "Vertex", elementsMapIt->second, tr("Number Of Vertex Cells"), "" );
                break;
            case VTK_POLY_VERTEX:
                cellProp = new Property ( "Edges", elementsMapIt->second, tr("Number Of Edge Cells"), "" );
                break;
            case VTK_LINE :
                cellProp = new Property ( "Lines", elementsMapIt->second, tr("Number Of Line Cells"), "" );
                break;
            case VTK_POLY_LINE:
                cellProp = new Property ( "Polylines", elementsMapIt->second, tr("Number Of Polylines Cells"), "" );
                break;
            case VTK_TRIANGLE :
                cellProp = new Property ( "Triangles", elementsMapIt->second, tr("Number Of Triangle Cells"), "" );
                break;
            case VTK_TRIANGLE_STRIP:
                cellProp = new Property ( "Triangle Strips", elementsMapIt->second, tr("Number Of Triangle Strip Cells"), "" );
                break;
            case VTK_POLYGON:
                cellProp = new Property ( "Polygons", elementsMapIt->second, tr("Number Of Polygon Cells"), "" );
                break;
            case VTK_PIXEL:
                cellProp = new Property ( "Pixels", elementsMapIt->second, tr("Number Of Pixel Cells"), "" );
                break;
            case VTK_QUAD:
                cellProp = new Property ( "Quads", elementsMapIt->second, tr("Number Of Quad Cells"), "" );
                break;
            case VTK_TETRA :
                cellProp = new Property ( "Tetrahedra", elementsMapIt->second, tr("Number Of Tetrahedral Cells"), "" );
                break;
            case VTK_VOXEL:
                cellProp = new Property ( "Voxels", elementsMapIt->second, tr("Number Of Voxel Cells"), "" );
                break;
            case VTK_HEXAHEDRON :
                cellProp = new Property ( "Hexahedra", elementsMapIt->second, tr("Number Of Hexahedral Cells"), "" );
                break;
            case VTK_WEDGE :
                cellProp = new Property ( "Wedges", elementsMapIt->second, tr("Number Of Wedge Cells"), "" );
                break;
            case VTK_PYRAMID :
                cellProp = new Property ( "Pyramids", elementsMapIt->second, tr("Number Of Pyramid Cells"), "" );
                break;
            case VTK_PENTAGONAL_PRISM:
                cellProp = new Property ( "Pentagonal Prisms", elementsMapIt->second, tr("Number Of Pentagonal Prism Cells"), "" );
                break;
            case VTK_HEXAGONAL_PRISM:
                cellProp = new Property ( "Hexagonal Prisms", elementsMapIt->second, tr("Number Of Hexagonal Prism Cells"), "" );
                break;
            default :
                cellProp = new Property ( "Others", elementsMapIt->second, tr("Number Of <i>Other Type Of Cells</i>. <br/>It can be quadratic isoparametric cells, Cubic isoparametric cells, <br/>convex group of points, higher order cells in parametric form, <br/>higher order cells (see VTKCellType enum for more information)"), "" );
                break;
            }
            cellProp->setReadOnly ( true );
            addProperty ( cellProp );
        }
    }

    unsigned long memUsage = 0;
    if ( getPointSet() !=NULL )
        memUsage = getPointSet()->GetActualMemorySize();
    Property *memoryUsage = new Property ( "Size In Memory", QVariant ( QString ( "%1" ).arg ( memUsage ) ), tr("Actual size of the data in kilobytes. <br/>This number is valid only after the pipeline has updated. <br/>The memory size returned is guaranteed to be greater than or <br/>equal to the memory required to represent the data<br/> (e.g., extra space in arrays, etc. are not included in the return value)."), "Kb" );
    memoryUsage->setReadOnly ( true );
    addProperty ( memoryUsage );
}

// -------------------- addSelection --------------------
void MeshComponent::addSelection ( const QString& name, int fieldType, int contentType ) {
    vtkSmartPointer<vtkSelectionNode> sel = vtkSmartPointer<vtkSelectionNode>::New();
    sel->SetFieldType ( fieldType );
    sel->SetContentType ( contentType );
    selectionList [name] = sel;
    selectionModel->refresh();
}

// -------------------- addToSelection --------------------
void MeshComponent::addToSelection ( const QString& name, vtkSmartPointer< vtkAbstractArray > array ) {
    if ( !selectionList.contains ( name ) )
        return;
    selectionList [name]->SetSelectionList ( array );
    selectionModel->updateSelection(name);
    selectionView->selectionModel()->setCurrentIndex(selectionModel->index(selectionList.keys().indexOf(name), 0),QItemSelectionModel::ClearAndSelect | QItemSelectionModel::Rows);
}

// -------------------- getNumberOfSelections --------------------
int MeshComponent::getNumberOfSelections() const {
    return selectionList.size();
}

// -------------------- getSelection --------------------
vtkSmartPointer<vtkSelectionNode> MeshComponent::getSelection ( const QString& name ) const {
    if ( !selectionList.contains ( name ) )
        return NULL;
    else
        return selectionList [name];
}

// -------------------- getSelectionAt --------------------
vtkSmartPointer<vtkSelectionNode> MeshComponent::getSelectionAt ( int i ) const {
    return selectionList.values().at(i);
}

// -------------------- getSelectionList --------------------
const QMap< QString, vtkSmartPointer< vtkSelectionNode > >& MeshComponent::getSelectionList() const {
    return selectionList;
}

// -------------------- removeSelection --------------------
void MeshComponent::removeSelection ( const QString& name ) {
    if ( !selectionList.contains ( name ) )
        return;
    selectionList.remove ( name );
    selectionModel->refresh();
}

// -------------------- changeSelectedSelection --------------------
void MeshComponent::changeSelectedSelection(const QModelIndex& current, const QModelIndex& previous) {
    currentSelection->RemoveAllNodes();

    if (current.isValid())
        currentSelection->AddNode(selectionList[selectionModel->data(selectionModel->index(current.row(), 0)).toString()]);

    // Select the PropertyExplorer 'Selection' tab for the currently selected component
    if (!Application::getSelectedComponents().isEmpty()) {
        Component* currentComponent = Application::getSelectedComponents().last();
        currentComponent->setIndexOfPropertyExplorerTab(1);
    }

    this->refresh();
    return;
}

// -------------------- addDataArray --------------------
void MeshComponent::addDataArray ( const QString& name, vtkSmartPointer< vtkDataArray > data ) {
    data->SetName ( name.toStdString().c_str() );
    dataList[name] = data;
}

// -------------------- removeDataArray --------------------
void MeshComponent::removeDataArray ( const QString& name ) {
    dataList.remove ( name );
    bool refresh = false;
    if ( this->getPointSet()->GetPointData()->HasArray ( name.toStdString().c_str() ) ) {
        this->getPointSet()->GetPointData()->RemoveArray ( name.toStdString().c_str() );
        refresh = true;
    }
    if ( this->getPointSet()->GetCellData()->HasArray ( name.toStdString().c_str() ) ) {
        this->getPointSet()->GetCellData()->RemoveArray ( name.toStdString().c_str() );
        refresh = true;
    }
    if ( refresh )
        dataModel->refresh();
}

// -------------------- linkDataToPoints --------------------
void MeshComponent::linkDataToPoints ( const QString& name ) {
    this->getPointSet()->GetPointData()->AddArray ( dataList[name] );
    // set the new array as active
    switch ( dataList[name]->GetNumberOfComponents() ) {
    case 1 :
        getPointSet()->GetPointData()->SetActiveScalars ( name.toStdString().c_str() );
        break;
    case 3 :
        getPointSet()->GetPointData()->SetActiveVectors ( name.toStdString().c_str() );
        break;
    case 9 :
        getPointSet()->GetPointData()->SetActiveTensors ( name.toStdString().c_str() );
        break;
    default :
        break;
    }
    dataModel->refresh();
}

// -------------------- linkDataToCells --------------------
void MeshComponent::linkDataToCells ( const QString& name ) {
    this->getPointSet()->GetCellData()->AddArray ( dataList[name] );
    // set the new array as active
    switch ( dataList[name]->GetNumberOfComponents() ) {
    case 1 :
        getPointSet()->GetCellData()->SetActiveScalars ( name.toStdString().c_str() );
        break;
    case 3 :
        getPointSet()->GetCellData()->SetActiveVectors ( name.toStdString().c_str() );
        break;
    case 9 :
        getPointSet()->GetCellData()->SetActiveTensors ( name.toStdString().c_str() );
        break;
    default :
        break;
    }
    dataModel->refresh();
}

// -------------------- addPointData --------------------
void MeshComponent::addPointData ( const QString& name, int dataType, vtkSmartPointer< vtkDataArray > data ) {
    addDataArray ( name, data );
    linkDataToPoints ( name );
}

// -------------------- addCellData --------------------
void MeshComponent::addCellData ( const QString& name, int dataType, vtkSmartPointer< vtkDataArray > data ) {
    addDataArray ( name, data );
    linkDataToCells ( name );
}

// -------------------- getNumberOfPropertyWidget --------------------
unsigned int MeshComponent::getNumberOfPropertyWidget() {
    return 3;
}

// -------------------- getPropertyWidgetAt --------------------
QWidget* MeshComponent::getPropertyWidgetAt ( unsigned int i, QWidget* parent ) {
    switch ( i ) {
    case 0 :
        return this->getPropertyWidget();
        break;
    case 1 :
        return selectionView;
        break;
    case 2 :
        return dataView;
        break;
    default :
        return NULL;
    }
}

}
