#!
#! @ingroup group_sdk_cmake
#!
#! macro camitk_application simplify writing a CMakeLists.txt for CamiTK application extension
#!
#! The name of the application is automatically the name of the directory from where this macro
#! is called prefixed with "camitk-"
#!
#! On Unix additional system resources can also be automatically installed or configured: man pages,
#! desktop file and XPM icons.
#! - Man pages should be resources/camitk-appdirname.1 or resources/camitk-appdirname.1.in (in this
#! case, it will be configured first).
#! - Desktop file should be in resources/camitk-appdirname.desktop (refer to
#!    http://standards.freedesktop.org/desktop-entry-spec/desktop-entry-spec-latest.html and
#!    http://standards.freedesktop.org/menu-spec/1.0/apa.html for valid category details)
#! - XPM icons should be found in resources/camitk-appdirname.xpm (an icon can also be used in
#!   to set the application icon (use setWindowIcon(QIcon(":/resources/camitk-appdirname.xpm")) as well)
#!
#! To write a simple man page, please refer to SDK application examples. If you need to
#! generate a simple manpage or analyze the command line arguments, the CamiTK team recommands
#! to use Code Synthesis CLI configuration (see http://codesynthesis.com/projects/cli)
#! 
#! usage:
#! \code
#! camitk_application(
#!              [DISABLED]
#!              [NEEDS_QT_MODULES]
#!              [NEEDS_ITK]
#!              [NEEDS_PYTHON]
#!              [NEEDS_CEP_LIBRARIES CEPLib1 CEPLib12 ...]
#!              [NEEDS_COMPONENT_EXTENSION component1 component2 ...]
#!              [NEEDS_ACTION_EXTENSION ation1 action2 ...]
#!              [DEFINES flag1 flag2 ...]
#!              [CXX_FLAGS flag1 flag2 ...]
#!              [ADDITIONAL_SOURCES source.cxx source.cpp ...]
#!              [INCLUDE_DIRECTORIES dir1 dir2 ...]
#!              [EXTERNAL_LIBRARIES lib1 lib2 ...]
#!              [EXTRA_TRANSLATE_LANGUAGE]
#!     )
#! \endcode
#!
#! \param DISABLED                      means this is a default application is NOT to be compiled automatically
#! \param NEEDS_QT_MODULES              add this if your library depends on Qt modules, such as QtCore, QtGui, QtMultimedia, QtNetwork, QtOpenGL, QtScript, QtScriptTools, QtSql, QtSvg, QtWebkit, QtXml, QtXmlPatterns, QtDeclarative
#! \param NEEDS_ITK                     means this application requires ITK to be compiled / run.
#! \param NEEDS_CEP_LIBRARIES           list of needed camitk CEP libraries
#! \param NEEDS_COMPONENT_EXTENSION     list of needed component extensions
#! \param NEEDS_ACTION_EXTENSION        list of needed component extensions
#! \param DEFINES                       list of define flags to add at compilation time
#! \param CXX_FLAGS                 = list of compiler flags to add (such as warning levels (-Wall ...)).
#! \param ADDITIONAL_SOURCES            list of additional sources (that cannot be automatically found by gather_headers_and_sources macro)
#! \param CEP_NAME                      specify the CEP_NAME, which is used to categorized the application for packaging purpose
#!                                      No CEP_NAME provided will result in default categorization (generic application).
#! \param DESCRIPTION                   Simple description of the application. Used for packaging presentation for instance.
#! \param INCLUDE_DIRECTORIES           additional include directories
#! \param EXTERNAL_LIBRARIES            external libraries to add to the link command
#! \param EXTRA_TRANSLATE_LANGUAGE      Additionnal extra language to translate the application
macro(camitk_application)

    #########################################################################
    #                                                                       #
    #   ARGUMENTS PARSING                                                   #
    #                                                                       #
    #   * Use a macro to create the CMAKE variables according to the        #
    #     provided options as input.                                        #
    #                                                                       #
    #########################################################################

    get_directory_name(${CMAKE_CURRENT_SOURCE_DIR} APPLICATION_NAME)

    parse_arguments(${APPLICATION_NAME_CMAKE}
        "NEEDS_CEP_LIBRARIES;NEEDS_TOOL;NEEDS_COMPONENT_EXTENSION;NEEDS_ACTION_EXTENSION;DEFINES;CXX_FLAGS;ADDITIONAL_SOURCES;CEP_NAME;DESCRIPTION;EXTERNAL_LIBRARIES;INCLUDE_DIRECTORIES;EXTRA_TRANSLATE_LANGUAGE"  # possible lists
        "DISABLED;NEEDS_QT_MODULES;NEEDS_ITK;NEEDS_QTXML;NEEDS_XSD;NEEDS_XERCESC;NEEDS_PYTHON" # possible options
        ${ARGN}
    )
    
    
    
    #########################################################################
    #                                                                       #
    #   CREATE CMAKE VARIABLES                                              #
    #                                                                       #
    #   * Create required and useful CMake variables for the macro         #
    #                                                                       #
    #########################################################################

    # CMAKE CACHE VARIABLE
    # if it is the first cmake run, create the application variable with a correct initial value    
    if(NOT APPLICATION_${APPLICATION_NAME_CMAKE}_INTERNAL)
        # add option to enable/disable this extension and set it to true by default
        # Building the extension can be disabled by giving the argument DISABLED to the macro
        # or by passing the flag -D${TYPE_EXTENSION_CMAKE}_${APPLICATION_NAME_CMAKE}_DISABLED:BOOL=TRUE
        if(${APPLICATION_NAME_CMAKE}_DISABLED)
            set(APPLICATION_${APPLICATION_NAME_CMAKE}_ENABLED FALSE)
        else()
            set(APPLICATION_${APPLICATION_NAME_CMAKE}_ENABLED TRUE)
        endif()
        set(APPLICATION_${APPLICATION_NAME_CMAKE} ${APPLICATION_${APPLICATION_NAME_CMAKE}_ENABLED} CACHE BOOL "Build application ${APPLICATION_NAME}")
        set(APPLICATION_${APPLICATION_NAME_CMAKE}_INTERNAL TRUE CACHE INTERNAL "Is variable APPLICATION_${APPLICATION_NAME} already created?")
    endif()
    
    # APPLICATION TARGET NAME
    if (PACKAGING_NSIS)
        # NSIS requires that cpack component names do not feature space or "-" characters
        set(APPLICATION_TARGET_NAME application_${APPLICATION_NAME})
    else()
        set(APPLICATION_TARGET_NAME application-${APPLICATION_NAME})
    endif()
        
  
  
    # if this extension is enabled, do everything needed
    # otherwise... do nothing
    if (APPLICATION_${APPLICATION_NAME_CMAKE})
    
        message(STATUS "Building application ${APPLICATION_NAME}")
        
        
        
        #########################################################################
        #                                                                       #
        #   INCLUDE DIRECTORIES                                                 #
        #                                                                       #
        #   * Include basic directories where to look header files              #
        #   * Include also additional user provided directories                 #
        #   * These directories are used for compilation step                   #
        #                                                                       #
        #########################################################################
        # BASIC DIRECTORIES
        include_directories(${CAMITK_INCLUDE_DIRECTORIES}) 
        include_directories(${CMAKE_CURRENT_BINARY_DIR})
        include_directories(${CMAKE_CURRENT_SOURCE_DIR})
        
        # USER INPUT DIRECTORIES
        include_directories(${${APPLICATION_NAME_CMAKE}_INCLUDE_DIRECTORIES})
        
        
    
        #########################################################################
        #                                                                       #
        #   GATHER RESSOURCES                                                   #
        #                                                                       #
        #   * get all the headers (.h) and source files (.cpp) of the project   #
        #   * create the needed Qt files (using moc and uic)                    #
        #                                                                       #
        #########################################################################
        
        # get all headers, sources and do what is needed for Qt
        # one need to do this just before the add_library so that all defines, include directories and link directories
        # are set properly (gather_headers_and_sources include the call to Qt moc and uic)
        gather_headers_and_sources(${APPLICATION_NAME})
                
        
        
        #########################################################################
        #                                                                       #
        #   ADDITIONAL KNOWN LIBRARY DEPENDENCIES                               #
        #                                                                       #
        #   * Look for specific library needed                                  #
        #   * Specific libraries are specified as option with the               #
        #     NEEDS_LIBRARY syntax (see macro syntax for more options)          #
        #   * Backward compatibility : Warn user if using old NEEDS_TOOL syntax #
        #                                                                       #
        #########################################################################
        
        # Looking for ITK
        set(ITK_LIBRARIES "")
        if(${APPLICATION_NAME_CMAKE}_NEEDS_ITK)
            find_package(ITK REQUIRED)
            if(ITK_FOUND)
                include(${ITK_USE_FILE})
                set(ITK_VERSION ${ITK_VERSION_MAJOR}.${ITK_VERSION_MINOR}.${ITK_VERSION_PATCH}) #ITK_VERSION is not always set
                message(STATUS "${APPLICATION_TARGET_NAME}: Found ITK version ${ITK_VERSION}")
                
                # Now select the ITK libraries according to the ITK version found.
                # ITK 3.x.x (mostly 3.20.1)
                if ((${ITK_VERSION} VERSION_GREATER "3") AND (${ITK_VERSION} VERSION_LESS "4")) 
                    if (MSVC) # Visual Studio compilers
                        if(MSVC90 OR MSVC10) #ITK 3.x only works with VS 2008 / 2010
                            set(ITK_LIBRARIES
                                    debug ITKAlgorithms${CAMITK_DEBUG_POSTFIX}
                                    optimized ITKAlgorithms
                                    debug ITKBasicFilters${CAMITK_DEBUG_POSTFIX}
                                    optimized ITKBasicFilters
                                    debug ITKCommon${CAMITK_DEBUG_POSTFIX}
                                    optimized ITKCommon
                                    debug ITKIO${CAMITK_DEBUG_POSTFIX}
                                    optimized ITKIO
                                    debug ITKNrrdIO${CAMITK_DEBUG_POSTFIX}.lib
                                    debug itkgdcm${CAMITK_DEBUG_POSTFIX}.lib
                                    debug itkjpeg12${CAMITK_DEBUG_POSTFIX}.lib
                                    debug itkjpeg16${CAMITK_DEBUG_POSTFIX}.lib
                                    debug itkopenjpeg${CAMITK_DEBUG_POSTFIX}.lib
                                    debug itkpng${CAMITK_DEBUG_POSTFIX}.lib
                                    debug itktiff${CAMITK_DEBUG_POSTFIX}.lib
                                    debug itkjpeg8${CAMITK_DEBUG_POSTFIX}.lib
                                    debug ITKSpatialObject${CAMITK_DEBUG_POSTFIX}.lib
                                    debug itkvnl_inst${CAMITK_DEBUG_POSTFIX}.lib
                                    debug itkvnl_algo${CAMITK_DEBUG_POSTFIX}.lib
                                    debug itkv3p_netlib${CAMITK_DEBUG_POSTFIX}.lib
                                    debug itkvnl${CAMITK_DEBUG_POSTFIX}.lib
                                    debug itkvcl${CAMITK_DEBUG_POSTFIX}.lib
                                    debug itkv3p_lsqr${CAMITK_DEBUG_POSTFIX}.lib
                                    debug ITKMetaIO${CAMITK_DEBUG_POSTFIX}.lib
                                    debug itksys${CAMITK_DEBUG_POSTFIX}.lib
                                    debug ITKDICOMParser${CAMITK_DEBUG_POSTFIX}.lib
                                    debug ITKEXPAT${CAMITK_DEBUG_POSTFIX}.lib
                                    debug ITKniftiio${CAMITK_DEBUG_POSTFIX}.lib
                                    debug ITKznz${CAMITK_DEBUG_POSTFIX}.lib
                                    debug itkzlib${CAMITK_DEBUG_POSTFIX}.lib
                                    debug snmpapi.lib
                                    debug rpcrt4.lib
                                    debug ws2_32.lib
                                    debug comctl32.lib
                                    debug wsock32.lib
                                    debug opengl32.lib
                                )
                        elseif(MSVC11 OR MSVC12) # Visual Studion 2012/2013
                            message(FATAL_ERROR "It seems you have ITK version 3.x installed. This one is not compatible with Visual Studio 2012 or 2013. Please install ITK 4.3.x or 4.5.x instead")
                        endif()
                    elseif(MINGW)
                        message(FATAL_ERROR "CamiTK does not support ITK 3.x libraries for MinGW. You may file a feature request: https://bugzilla-timc.imag.fr/")
                    else() #APPLE OR UNIX
                        set(ITK_LIBRARIES ITKBasicFilters ITKCommon ITKIO ITKAlgorithms)
                    endif() 
                    
                # ITK 4.3.x (mostly 4.3.2)    
                elseif((${ITK_VERSION} VERSION_GREATER "4") AND (${ITK_VERSION} VERSION_LESS "4.4"))
                    if (MSVC) # Visual Studio compilers
                        if(MSVC90 OR MSVC10) #ITK 3.x only works with VS 2008 / 2010
                            message(FATAL_ERROR "ITK version ${ITK_VERSION} is not compatible with Visual Studio 2008 or 2010. Please install ITK version 4.3.x or 4.5.x")
                        elseif(MSVC11 OR MSVC12) # Visual Studion 2012/2013 
                            link_directories(${ITK_DIR}/../..)
                            set(ITK_LIBRARIES
                                # Debug libraries
                                debug ITKCommon-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKNrrdIO-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkgdcmCommon-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkjpeg-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkopenjpeg-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkpng-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itktiff-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKSpatialObjects-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkvnl_algo-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkv3p_netlib-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkvnl-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkvcl-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkv3p_lsqr-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKMetaIO-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itksys-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKDICOMParser-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKEXPAT-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKniftiio-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKznz-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkzlib-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug snmpapi.lib
                                debug rpcrt4.lib
                                debug ws2_32.lib
                                debug comctl32.lib
                                debug wsock32.lib
                                debug opengl32.lib
                                debug ITKIOGIPL-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIONIFTI-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIONRRD-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOPNG-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkgdcmDICT-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkgdcmDSED-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkgdcmIOD-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkgdcmjpeg8-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkgdcmjpeg12-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkgdcmjpeg16-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkgdcmMSFF-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKgiftiio-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkhdf5_cpp-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkhdf5-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKMetaIO-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOBioRad-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOStimulate-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOTIFF-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOVTK-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOPNG-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKBiasCorrection-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKBioCell-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKFEM-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOBMP-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOCSV-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOGDCM-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOGE-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOGIPL-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOHDF5-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOImageBase-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOIPL-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOJPEG-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOLSM-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOMesh-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOMeta-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOSiemens-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOSpatialObjects-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOTransformBase-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOTransformHDF5-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOTransformInsightLegacy-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOTransformMatlab-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKIOXML-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKKLMRegionGrowing-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKLabelMap-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKMesh-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKMetaIO-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkNetlibSlatec-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkopenjpeg-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKOptimizers-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKOptimizersv4-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKPath-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkpng-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKPolynomials-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKQuadEdgeMesh-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKReview-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKStatistics-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkv3p_lsqr-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkv3p_netlib-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug itkvcl-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKVideoCore-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKVideoIO-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKVTK-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKVNLInstantiation-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                debug ITKWatersheds-4.3${CAMITK_DEBUG_POSTFIX}.lib
                                # Release librariesdebug
                                optimized ITKCommon-4.3.lib
                                optimized ITKNrrdIO-4.3.lib
                                optimized itkgdcmCommon-4.3.lib
                                optimized itkjpeg-4.3.lib
                                optimized itkopenjpeg-4.3.lib
                                optimized itkpng-4.3.lib
                                optimized itktiff-4.3.lib
                                optimized ITKSpatialObjects-4.3.lib
                                optimized itkvnl_algo-4.3.lib
                                optimized itkv3p_netlib-4.3.lib
                                optimized itkvnl-4.3.lib
                                optimized itkvcl-4.3.lib
                                optimized itkv3p_lsqr-4.3.lib
                                optimized ITKMetaIO-4.3.lib
                                optimized itksys-4.3.lib
                                optimized ITKDICOMParser-4.3.lib
                                optimized ITKEXPAT-4.3.lib
                                optimized ITKniftiio-4.3.lib
                                optimized ITKznz-4.3.lib
                                optimized itkzlib-4.3.lib
                                optimized snmpapi.lib
                                optimized rpcrt4.lib
                                optimized ws2_32.lib
                                optimized comctl32.lib
                                optimized wsock32.lib
                                optimized opengl32.lib
                                optimized ITKIOGIPL-4.3.lib
                                optimized ITKIONIFTI-4.3.lib
                                optimized ITKIONRRD-4.3.lib
                                optimized ITKIOPNG-4.3.lib
                                optimized itkgdcmDICT-4.3.lib
                                optimized itkgdcmDSED-4.3.lib
                                optimized itkgdcmIOD-4.3.lib
                                optimized itkgdcmjpeg8-4.3.lib
                                optimized itkgdcmjpeg12-4.3.lib
                                optimized itkgdcmjpeg16-4.3.lib
                                optimized itkgdcmMSFF-4.3.lib
                                optimized ITKgiftiio-4.3.lib
                                optimized itkhdf5_cpp-4.3.lib
                                optimized itkhdf5-4.3.lib
                                optimized ITKMetaIO-4.3.lib
                                optimized ITKIOBioRad-4.3.lib
                                optimized ITKIOStimulate-4.3.lib
                                optimized ITKIOTIFF-4.3.lib
                                optimized ITKIOVTK-4.3.lib
                                optimized ITKIOPNG-4.3.lib
                                optimized ITKBiasCorrection-4.3.lib
                                optimized ITKBioCell-4.3.lib
                                optimized ITKFEM-4.3.lib
                                optimized ITKIOBMP-4.3.lib
                                optimized ITKIOCSV-4.3.lib
                                optimized ITKIOGDCM-4.3.lib
                                optimized ITKIOGE-4.3.lib
                                optimized ITKIOGIPL-4.3.lib
                                optimized ITKIOHDF5-4.3.lib
                                optimized ITKIOImageBase-4.3.lib
                                optimized ITKIOIPL-4.3.lib
                                optimized ITKIOJPEG-4.3.lib
                                optimized ITKIOLSM-4.3.lib
                                optimized ITKIOMesh-4.3.lib
                                optimized ITKIOMeta-4.3.lib
                                optimized ITKIOSiemens-4.3.lib
                                optimized ITKIOSpatialObjects-4.3.lib
                                optimized ITKIOTransformBase-4.3.lib
                                optimized ITKIOTransformHDF5-4.3.lib
                                optimized ITKIOTransformInsightLegacy-4.3.lib
                                optimized ITKIOTransformMatlab-4.3.lib
                                optimized ITKIOXML-4.3.lib
                                optimized ITKKLMRegionGrowing-4.3.lib
                                optimized ITKLabelMap-4.3.lib
                                optimized ITKMesh-4.3.lib
                                optimized ITKMetaIO-4.3.lib
                                optimized itkNetlibSlatec-4.3.lib
                                optimized itkopenjpeg-4.3.lib
                                optimized ITKOptimizers-4.3.lib
                                optimized ITKOptimizersv4-4.3.lib
                                optimized ITKPath-4.3.lib
                                optimized itkpng-4.3.lib
                                optimized ITKPolynomials-4.3.lib
                                optimized ITKQuadEdgeMesh-4.3.lib
                                optimized ITKReview-4.3.lib
                                optimized ITKStatistics-4.3.lib
                                optimized itkv3p_lsqr-4.3.lib
                                optimized itkv3p_netlib-4.3.lib
                                optimized itkvcl-4.3.lib
                                optimized ITKVideoCore-4.3.lib
                                optimized ITKVideoIO-4.3.lib
                                optimized ITKVTK-4.3.lib
                                optimized ITKVNLInstantiation-4.3.lib
                                optimized ITKWatersheds-4.3.lib
                            )
                        endif()
                    elseif(MINGW OR APPLE)
                        link_directories(${ITK_DIR}/../..)
                        set(ITK_LIBRARIES
                            ITKCommon-4.3
                            ITKNrrdIO-4.3
                            itkgdcmCommon-4.3
                            itkjpeg-4.3
                            itkopenjpeg-4.3
                            itkpng-4.3
                            itktiff-4.3
                            ITKSpatialObjects-4.3
                            itkvnl_algo-4.3
                            itkv3p_netlib-4.3
                            itkvnl-4.3
                            itkvcl-4.3
                            itkv3p_lsqr-4.3
                            ITKMetaIO-4.3
                            itksys-4.3
                            ITKDICOMParser-4.3
                            ITKEXPAT-4.3
                            ITKniftiio-4.3
                            ITKznz-4.3
                            itkzlib-4.3
                            snmpapi
                            rpcrt4
                            ws2_32
                            comctl32
                            wsock32
                            opengl32
                            ITKIOGIPL-4.3
                            ITKIONIFTI-4.3
                            ITKIONRRD-4.3
                            ITKIOPNG-4.3
                            itkgdcmDICT-4.3
                            itkgdcmDSED-4.3
                            itkgdcmIOD-4.3
                            itkgdcmjpeg8-4.3
                            itkgdcmjpeg12-4.3
                            itkgdcmjpeg16-4.3
                            itkgdcmMSFF-4.3
                            ITKgiftiio-4.3
                            itkhdf5_cpp-4.3
                            itkhdf5-4.3
                            ITKMetaIO-4.3
                            ITKIOBioRad-4.3
                            ITKIOStimulate-4.3
                            ITKIOTIFF-4.3
                            ITKIOVTK-4.3
                            ITKIOPNG-4.3
                            ITKBiasCorrection-4.3
                            ITKBioCell-4.3
                            ITKFEM-4.3
                            ITKIOBMP-4.3
                            ITKIOCSV-4.3
                            ITKIOGDCM-4.3
                            ITKIOGE-4.3
                            ITKIOGIPL-4.3
                            ITKIOHDF5-4.3
                            ITKIOImageBase-4.3
                            ITKIOIPL-4.3
                            ITKIOJPEG-4.3
                            ITKIOLSM-4.3
                            ITKIOMesh-4.3
                            ITKIOMeta-4.3
                            ITKIOSiemens-4.3
                            ITKIOSpatialObjects-4.3
                            ITKIOTransformBase-4.3
                            ITKIOTransformHDF5-4.3
                            ITKIOTransformInsightLegacy-4.3
                            ITKIOTransformMatlab-4.3
                            ITKIOXML-4.3
                            ITKKLMRegionGrowing-4.3
                            ITKLabelMap-4.3
                            ITKMesh-4.3
                            ITKMetaIO-4.3
                            itkNetlibSlatec-4.3
                            itkopenjpeg-4.3
                            ITKOptimizers-4.3
                            ITKOptimizersv4-4.3
                            ITKPath-4.3
                            itkpng-4.3
                            ITKPolynomials-4.3
                            ITKQuadEdgeMesh-4.3
                            ITKReview-4.3
                            ITKStatistics-4.3
                            itkv3p_lsqr-4.3
                            itkv3p_netlib-4.3
                            itkvcl-4.3
                            ITKVideoCore-4.3
                            ITKVideoIO-4.3
                            ITKVTK-4.3
                            ITKVNLInstantiation-4.3
                            ITKWatersheds-4.3
                        )
                    else() #Linux
                        set(ITK_LIBRARIES ITKCommon-4.3)
                    endif()
                
                # ITK 4.5.x (mostly 4.5.1)    
                elseif((${ITK_VERSION} VERSION_GREATER "4.4") AND (${ITK_VERSION} VERSION_LESS "4.7"))
                    if(MSVC)
                        message(FATAL_ERROR "ITK version 4.5.x is not supported yet on Visual Studio compiler. You may file a feature request: https://bugzilla-timc.imag.fr/")
                    elseif(MINGW)
                        message(FATAL_ERROR "ITK version 4.5.x is not supported yet for MinGW compiler. You may file a feature request: https://bugzilla-timc.imag.fr/")
                    elseif(APPLE)
                        set(ITK_LIBRARIES
                                ITKBiasCorrection
                                ITKBioCell
                                ITKCommon
                                ITKDICOMParser
                                ITKEXPAT
                                ITKFEM
                                ITKGPUAnisotropicSmoothing
                                ITKGPUCommon
                                ITKGPUFiniteDifference
                                ITKGPUImageFilterBase
                                ITKGPUPDEDeformable
                                ITKGPUSmoothing
                                ITKGPUThresholding
                                ITKIOBMP
                                ITKIOBioRad
                                ITKIOCSV
                                ITKIOGDCM
                                ITKIOGE
                                ITKIOGIPL
                                ITKIOHDF5
                                ITKIOIPL
                                ITKIOImageBase
                                ITKIOJPEG
                                ITKIOLSM
                                ITKIOMesh
                                ITKIOMeta
                                ITKIONIFTI
                                ITKIONRRD
                                ITKIOPNG
                                ITKIOSiemens
                                ITKIOSpatialObjects
                                ITKIOStimulate
                                ITKIOTIFF
                                ITKIOTransformBase
                                ITKIOTransformHDF5
                                ITKIOTransformInsightLegacy
                                ITKIOTransformMatlab
                                ITKIOVTK
                                ITKIOXML
                                ITKKLMRegionGrowing
                                ITKLabelMap
                                ITKMesh
                                ITKMetaIO
                                ITKNrrdIO
                                ITKOptimizers
                                ITKOptimizersv4
                                ITKPath
                                ITKPolynomials
                                ITKQuadEdgeMesh
                                ITKReview
                                ITKSpatialObjects
                                ITKStatistics
                                ITKVNLInstantiation
                                ITKVTK
                                ITKVideoCore
                                ITKVideoIO
                                ITKWatersheds
                                ITKgiftiio
                                ITKniftiio
                                ITKznz
                                itkNetlibSlatec
                                itkdouble-conversion
                                itkhdf5
                                itkhdf5_cpp
                                itkjpeg
                                itkopenjpeg
                                itkpng
                                itksys
                                itktiff
                                itkv3p_lsqr
                                itkv3p_netlib
                                itkvcl
                                itkvnl
                                itkvnl_algo
                                itkzlib
                            )
                    else() #Linux
                        message(FATAL_ERROR "ITK version 4.5.x is not supported yet for Linux platform. You may file a feature request: https://bugzilla-timc.imag.fr/")
                    endif()
                    
                # ITK version > 4.6 (mostly ITK 5.x)
                else()
                    message(FATAL_ERROR "CamiTK does not support yet ITK version 5.x. You may file a feature request: https://bugzilla-timc.imag.fr/")
                endif()
            endif()
        endif() #NEEDS ITK

        # XERCES-C
        set(XERCESC_LIBRARIES)
        if(${APPLICATION_NAME_CMAKE}_NEEDS_XERCESC)
          # XercesC is required
          find_package(XercesC REQUIRED)
          if (XERCESC_FOUND)
            include_directories(${XERCESC_INCLUDE_DIR})
            set(XERCESC_LIBRARIES ${XERCESC_LIBRARY})
          else()
            # most probably win32 or crosscompiling
            message(FATAL_ERROR "${APPLICATION_NAME}: xerces-c required. Please provide Xerces-C path.")
          endif()
        endif()

        # XSD
        if(${APPLICATION_NAME_CMAKE}_NEEDS_XSD)
            # XercesC is required
            find_package(XercesC REQUIRED)
            if (XERCESC_FOUND)
                include_directories(${XERCESC_INCLUDE_DIR})
                set(XERCESC_LIBRARIES ${XERCESC_LIBRARY})
                find_package(XSD REQUIRED)
                include_directories(${XSD_INCLUDE_DIR})
            else()
                # most probably win32 or crosscompiling
                message(FATAL_ERROR "${APPLICATION_NAME}: xerces-c required because of XSD cxx, please set XERCESC_INCLUDE_DIR")
            endif()
        endif()

        # PYTHON
        set(PYTHON_LIBRARIES "")
        if(${APPLICATION_NAME_CMAKE}_NEEDS_PYTHON)
            message(STATUS "Python needed by ${APPLICATION_NAME}")
            find_package(PythonLibs 2.7 REQUIRED)
            if(PYTHONLIBS_FOUND)
                message(STATUS "Python found (needed by ${APPLICATION_NAME})")
                include_directories(${PYTHON_INCLUDE_DIRS})
                # PYTHON_LIRARIES is automatically and correctly set by find_package
            else()
                message(FATAL_ERROR "Python (2.7 or above) is required by ${APPLICATION_NAME} : please add your python installation dir to your PATH environment variable")
            endif()
        endif()
        
        # QT MODULES
        set(QT_MODULES_LIBRARIES)
        if(${APPLICATION_NAME_CMAKE}_NEEDS_QT_MODULES)
            find_package(Qt4 "4.7" COMPONENTS QtCore QtGui QtXml QtXmlPatterns QtWebkit QtOpenGL QtScript QtSQL QtNetwork REQUIRED)
            if(NOT QT4_FOUND)
                message(SEND_ERROR "${APPLICATION_NAME} : Failed to find Qt 4.7 or greater. This is needed by CamiTK.")
            endif()
        
            # CMake will add the Qt modules specific include directories
            include(${QT_USE_FILE})
            # Qt additional modules are required for linking
            set(QT_MODULES_LIBRARIES ${QT_LIBRARIES})
        endif()
       
        # EXTERNAL LIBRARIES
        set(EXTERNAL_LIBRARIES)
        if(${APPLICATION_NAME_CMAKE}_EXTERNAL_LIBRARIES)
            foreach(EXTERNAL_LIBRARY ${${APPLICATION_NAME_CMAKE}_EXTERNAL_LIBRARIES})
                if (MSVC) # TODO We have to decide how to handle debug version of external libraries
                    set(EXTERNAL_LIBRARIES ${EXTERNAL_LIBRARIES} 
                                           debug ${EXTERNAL_LIBRARY}${CAMITK_DEBUG_POSTFIX}
                                           optimized ${EXTERNAL_LIBRARY}
                    )
                else()
                    set(EXTERNAL_LIBRARIES ${EXTERNAL_LIBRARIES} ${EXTERNAL_LIBRARY})
                endif()
            endforeach()
        endif()
        
        # BACKWARD COMPATIBILITY
        # NEEDS_TOOLS option
        # TODO CAMITK_DEPRECATED. This option is marked as deprecated. It is to be removed in CamiTK 4.0
        # check deprecated (and warn)
        # TODO CAMITK_DEPRECATED. This option is marked as deprecated. It is to be removed in CamiTK 4.0
        if(${APPLICATION_NAME_CMAKE}_NEEDS_TOOL)
            message(WARNING "Warning: ${CMAKE_CURRENT_SOURCE_DIR}/CMakeLists.txt: camitk_application macro option NEEDS_TOOL is deprecated, please use NEEDS_CEP_LIBRARIES instead..")
            set(CEP_LIBRARIES ${CEP_LIBRARIES} ${${APPLICATION_NAME_CMAKE}_NEEDS_TOOL})
        endif()
        
        
        
        #########################################################################
        #                                                                       #
        #   LINK DIRECTORIES                                                    #
        #                                                                       #
        #   * Link directories are used to indicate the compiler where          #
        #     to look for folder containing libraries to link with.             #
        #                                                                       #
        #########################################################################
        # CAMITK BASIC LIB DIRECTORIES
        link_directories(${CAMITK_LINK_DIRECTORIES})
        
        

        #########################################################################
        #                                                                       #
        #   TARGET COMPILATION  DEFINITION                                      #
        #                                                                       #
        #   * Additional sources files to consider at compilation (.cpp)        #
        #   * CMake project target definition                                   #
        #                                                                       #
        #########################################################################
        # EXTERNAL SOURCES
        set(${APPLICATION_NAME}_SOURCES ${${APPLICATION_NAME}_SOURCES} ${${APPLICATION_NAME_CMAKE}_ADDITIONAL_SOURCES})
        
        # APPLE BUNDLE DEFINITIONS
        if(APPLE)
            set(MACOSX_BUNDLE_INFO_STRING "${APPLICATION_TARGET_NAME} ${CAMITK_SHORT_VERSION_STRING}")
            set(MACOSX_BUNDLE_BUNDLE_VERSION "${APPLICATION_TARGET_NAME} ${CAMITK_SHORT_VERSION_STRING}")
            set(MACOSX_BUNDLE_LONG_VERSION_STRING "${APPLICATION_TARGET_NAME} ${CAMITK_SHORT_VERSION_STRING}")
            set(MACOSX_BUNDLE_SHORT_VERSION_STRING "${CAMITK_SHORT_VERSION_STRING}")
            set(MACOSX_BUNDLE_COPYRIGHT "2014 UJF")
            set(MACOSX_BUNDLE_ICON_FILE "${CMAKE_CURRENT_SOURCE_DIR}/resources/camitk-${APPLICATION_NAME}.icns")
            set(MACOSX_BUNDLE_BUNDLE_NAME "${APPLICATION_TARGET_NAME}")

            set(MACOSX_BUNDLE_RESOURCES "${CMAKE_CURRENT_BINARY_DIR}/${APPLICATION_TARGET_NAME}.app/Contents/Resources")
            set(MACOSX_BUNDLE_ICON "${MACOSX_BUNDLE_ICON_FILE}")
            execute_process(COMMAND ${CMAKE_COMMAND} -E make_directory ${MACOSX_BUNDLE_RESOURCES})
            execute_process(COMMAND ${CMAKE_COMMAND} -E copy_if_different ${MACOSX_BUNDLE_ICON} ${MACOSX_BUNDLE_RESOURCES})
        endif()
         
        # CMAKE TARGET DEFINITION
#        add_executable(${APPLICATION_TARGET_NAME} MACOSX_BUNDLE ${${APPLICATION_NAME}_SOURCES})
        add_executable(${APPLICATION_TARGET_NAME} ${${APPLICATION_NAME}_SOURCES})
        
        
        
        #########################################################################
        #                                                                       #
        #   CAMITK ACTION / COMPONENT / LIBRARIES DEPENDENCIES                  #
        #                                                                       #
        #   * Look for action / component / libraries dependencies              #
        #   * Specific actions / components / libraries are specified as option #
        #     with the NEEDS_ACTION/COMPONENT_EXTENSION/CEP_LIBRARIES syntax    #
        #   * Add dependencies to library-camitkcore and the testing            #
        #     action/component if test are runned on it                         #                                                                                       
        #                                                                       #
        #########################################################################
        
        # 1st CAMITKCORE LIBRARY DEPENDENCY
        add_dependencies(${APPLICATION_TARGET_NAME} ${CAMITK_CORE_TARGET_LIB_NAME})

        # 2nd COMPONENTS DEPENDENCIES
        if(${APPLICATION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION)
            set(COMPONENTS_DEPENDENCY_LIST "") #use for generating the project.xml file
            foreach(COMPONENT_NEEDED ${${APPLICATION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION})
                # include directories from build, camitk (local or global install).
                include_directories(${CAMITK_BUILD_INCLUDE_DIR}/components/${COMPONENT_NEEDED})
                include_directories(${CAMITK_INCLUDE_DIR}/components/${COMPONENT_NEEDED})
                # file dependency
                if (MSVC)
                    set(COMPONENT_EXTENSION_LIBRARIES ${COMPONENT_EXTENSION_LIBRARIES}
                                                    debug ${CAMITK_BUILD_PRIVATE_LIB_DIR}/components/${COMPONENT_NEEDED}${CAMITK_DEBUG_POSTFIX}
                                                    optimized ${COMPONENT_NEEDED}
                    )
                else()
                    set(COMPONENT_EXTENSION_LIBRARIES ${COMPONENT_EXTENSION_LIBRARIES} ${COMPONENT_NEEDED})
                endif()
                # CMake / CDash dependencies
                if(PACKAGING_NSIS)
                    add_dependencies(${APPLICATION_TARGET_NAME} component_${COMPONENT_NEEDED})
                else()
                    add_dependencies(${APPLICATION_TARGET_NAME} component-${COMPONENT_NEEDED})
                    set(COMPONENTS_DEPENDENCY_LIST ${COMPONENTS_DEPENDENCY_LIST} component-${COMPONENT_NEEDED})
                endif()
            endforeach()
        endif()

        # 3rd ACTIONS DEPENDENCIES
        if(${APPLICATION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION)
            set(ACTIONS_DEPENDENCY_LIST "") #use for generating the project.xml file
            foreach(ACTION_NEEDED ${${APPLICATION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION})
                # include directories from build, camitk (local or global install).
                include_directories(${CAMITK_BUILD_INCLUDE_DIR}/actions/${ACTION_NEEDED})
                include_directories(${CAMITK_INCLUDE_DIR}/actions/${ACTION_NEEDED})
                # file dependency
                if (MSVC)
                    set(ACTION_EXTENSION_LIBRARIES ${ACTION_EXTENSION_LIBRARIES}
                                                   debug ${CAMITK_BUILD_PRIVATE_LIB_DIR}/actions/${ACTION_NEEDED}${CAMITK_DEBUG_POSTFIX}
                                                   optimized ${ACTION_NEEDED}
                    )
                else()
                    set(ACTION_EXTENSION_LIBRARIES ${ACTION_EXTENSION_LIBRARIES} ${ACTION_NEEDED})
                endif()
                # CMake / CDash dependencies
                if (PACKAGING_NSIS)
                    add_dependencies(${APPLICATION_TARGET_NAME} action_${ACTION_NEEDED})
                else()
                    add_dependencies(${APPLICATION_TARGET_NAME} action-${ACTION_NEEDED})
                    set(ACTIONS_DEPENDENCY_LIST ${ACTIONS_DEPENDENCY_LIST} action-${ACTION_NEEDED})
                endif()
            endforeach()
        endif()

        # 4th CEP LIBRARIES DEPENDENCIES
        if(${APPLICATION_NAME_CMAKE}_NEEDS_CEP_LIBRARIES)
            set(CEP_LIBRARIES_DEPENDENCY_LIST "") #use for generating the project.xml file
            foreach(CEP_LIBRARY_NEEDED ${${APPLICATION_NAME_CMAKE}_NEEDS_CEP_LIBRARIES})
                 # include directories from build, camitk (local or global install).
                include_directories(${CAMITK_BUILD_INCLUDE_DIR}/libraries/${CEP_LIBRARY_NEEDED})
                include_directories(${CAMITK_INCLUDE_DIR}/libraries/${CEP_LIBRARY_NEEDED})
                # file dependency
                if (MSVC)
                        set(CEP_LIBRARIES ${CEP_LIBRARIES}  debug ${CEP_LIBRARY_NEEDED}${CAMITK_DEBUG_POSTFIX}
                                                            optimized ${CEP_LIBRARY_NEEDED}
                        )
                else()
                    if(PACKAGING_NSIS)
                        set(CEP_LIBRARIES ${CEP_LIBRARIES} ${CEP_LIBRARY_NEEDED})
                    else()
                        set(CEP_LIBRARIES ${CEP_LIBRARIES} ${CEP_LIBRARY_NEEDED})
                    endif()
                endif()
                # CMake / CDash dependencies
                if (PACKAGING_NSIS)
                    add_dependencies(${APPLICATION_TARGET_NAME} library_${CEP_LIBRARY_NEEDED})
                else()
                    add_dependencies(${APPLICATION_TARGET_NAME} library-${CEP_LIBRARY_NEEDED})
                    set(CEP_LIBRARIES_DEPENDENCY_LIST ${CEP_LIBRARIES_DEPENDENCY_LIST} library-${CEP_LIBRARY_NEEDED})
                endif()
            endforeach()
        endif()
            
        

        #########################################################################
        #                                                                       #
        #   COMPILATION FLAG                                                    #
        #                                                                       #
        #   * Flags are options to give to the compiler                         #
        #   * Add user input flags                                              #
        #   * Add platform specific flags                                       #
        #                                                                       #
        #########################################################################

        # USER INPUT COMPILER FLAG
        if(${APPLICATION_NAME_CMAKE}_DEFINES)
          foreach (FLAG ${${APPLICATION_NAME_CMAKE}_DEFINES})
            add_definitions(-D${FLAG})
          endforeach()
        endif()
        
        # USER INPUT CUSTOM COMPILER FLAG
        if(${APPLICATION_NAME_CMAKE}_CXX_FLAGS)
          foreach (FLAG ${${APPLICATION_NAME_CMAKE}_CXX_FLAGS})
            set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} ${FLAG}")
          endforeach()
        endif()
       
        
        
        #########################################################################
        #                                                                       #
        #   LINKING                                                             #
        #                                                                       #
        #   * Linking is the last stage of compilation                          #
        #   * Indicate what libraries to use for linking the target             #
        #                                                                       #
        #########################################################################
        # LINKING LIBRARIES
        target_link_libraries(${APPLICATION_TARGET_NAME} ${CAMITK_CORE_LIBRARIES} ${CAMITK_LIBRARIES}  ${COMPONENT_EXTENSION_LIBRARIES} ${ACTION_EXTENSION_LIBRARIES} ${CEP_LIBRARIES} ${QT_MODULES_LIBRARIES} ${XERCESC_LIBRARY} ${ITK_LIBRARIES} ${PYTHON_LIBRARIES} ${EXTERNAL_LIBRARIES})

        

        #########################################################################
        #                                                                       #
        #   OUTPUT                                                              #
        #                                                                       #
        #   * Define the output directory (location and name)                   #
        #   * Define the output name of the library                             #
        #   * Add ${CAMITK_DEBUG_POSTFIX} suffix to Debug MSVC built libraries  #
        #                                                                       #
        #########################################################################
      
        # OUTPUT NAME
        if (MSVC)
            set_target_properties(${APPLICATION_TARGET_NAME}
                                  PROPERTIES OUTPUT_NAME camitk-${APPLICATION_NAME}
                                  DEBUG_POSTFIX ${CAMITK_DEBUG_POSTFIX}
            )
        else()
            set_target_properties(${APPLICATION_TARGET_NAME}
                                  PROPERTIES OUTPUT_NAME camitk-${APPLICATION_NAME}
            )
        endif()

        # OUTPUT DIRECTORY LOCATION
        # always in bin folder
        set_target_properties(${APPLICATION_TARGET_NAME} PROPERTIES RUNTIME_OUTPUT_DIRECTORY ${CAMITK_BUILD_BIN_DIR}
                                                                    RUNTIME_OUTPUT_DIRECTORY_DEBUG ${CAMITK_BUILD_BIN_DIR}
                                                                    RUNTIME_OUTPUT_DIRECTORY_RELEASE ${CAMITK_BUILD_BIN_DIR}
        )    
    
        #########################################################################
        #                                                                       #
        #   INSTALLATION                                                        #
        #                                                                       #
        #   * When installing the project, header files (.h) and test data are  #
        #     copied into a installation folder to determine.                   #
        #   * Indicate in this section, where to install your project and which #
        #     files to copy into that folder (during local/global installation) #
        #                                                                       #
        #########################################################################

        # FOLDER INSTALLATION
        # Indicate where to install the application
#		if (NOT APPLE)
			install(TARGETS ${APPLICATION_TARGET_NAME}
            		RUNTIME DESTINATION bin
               		COMPONENT ${APPLICATION_TARGET_NAME}
        	)
 #   	else ()
#		set (PLUGINS QtCore QtGui QtXml QtXmlPatterns)
#		    include(DeployQt4)
#		    install_qt4_executable("bin/camitk-${APPLICATION_NAME}.app" "${PLUGINS}")
#		endif()
    
        
        
        #########################################################################
        #                                                                       #
        #   CDASH SUBPROJECT DESCRIPTION                                        #
        #                                                                       #
        #   * Update the XML descriton of the subprojects dependenicies         #
        #     for CDash.                                                        #
        #                                                                       #
        #########################################################################       
        # CDASH XML SUBPROJECTS DESCRIPTION UPDATE
        if(NOT ((APPLICATION_TARGET_NAME STREQUAL "application-testactions") OR (APPLICATION_TARGET_NAME STREQUAL "application-testcomponents")))
            # We will add manually the CDash dependencies of the testing applications
            # See CamiTKSubProjectValidate.cmake
            camitk_sub_project_add(APPLICATION ${APPLICATION_TARGET_NAME} DEPENDENCIES library-camitkcore ${COMPONENTS_DEPENDENCY_LIST} ${ACTIONS_DEPENDENCY_LIST} ${CEP_LIBRARIES_DEPENDENCY_LIST})
        endif()
        
		
        
		#########################################################################
        #                                                                       #
        #   TRANSLATION				                                            #
        #                                                                       #
		#   * CAMITK_TRANSLATIONS contains the list of language to translate 	#
		#	the QString to.														#
		#                                                                       #
        #   * Create the translate.pro file which contains 4 sections:			#
		#		- HEADERS: 		list of .h/.hpp files to look for tr("") QString#		
		#		- SOURCES: 		list of .cpp files to look for tr("") QString	#
		#		- FORMS: 		list of .ui files to look for tr("") QString	#			
		#		- TRANSLATIONS: list of .ts files which use CAMITK_TRANSLATIONS #
		#			to define each .ts file 									#
        #  																		#
		#	* Execute lupdate program to update the .ts files with new QString	#
		#  		found.															#
		#                                                                       #
		#   * Execute lrelease program to create .qm files (binary equivalent of#
		#       .ts files														#
		#                                                                       #
		#   * Create translate.qrc which contains the list of .qm files.		#
		#   * Create the flags.qrc file which contains the list of .png flags	#
		# 		images															#
		#                                                                       #
        #########################################################################
		if(${APPLICATION_NAME_CMAKE}_EXTRA_TRANSLATE_LANGUAGE)
			camitk_translate(EXTRA_LANGUAGE ${${APPLICATION_NAME_CMAKE}_EXTRA_TRANSLATE_LANGUAGE})
		else()
			camitk_translate()
		endif()
		
		
        #########################################################################
        #                                                                       #
        #   PACKAGING CATEGORIZATION                                            #
        #                                                                       #
        #   * On Windows, when building a package (win32 installer), the        #
        #     install shield wizard proposes you to select which component      #
        #     to install.                                                       #
        #   * Each component to install has a short description following its   #
        #     name to understand its role.                                      #
        #   * This section deals with the categorization and the description    #
        #     of the component in this installer.                               #
        #                                                                       #
        #########################################################################
    
        # WINDOWS INSTALLER CATEGORIZATION
        if(${APPLICATION_NAME_CMAKE}_CEP_NAME)
            if (${APPLICATION_NAME_CMAKE}_CEP_NAME MATCHES "SDK")
                # The default SDK extensions are categorized as "required" and are not "unselectable" by the user at installation time
                cpack_add_component(${APPLICATION_TARGET_NAME}
                                    DISPLAY_NAME ${APPLICATION_TARGET_NAME}
                                    DESCRIPTION ${${APPLICATION_NAME_CMAKE}_DESCRIPTION}
                                    REQUIRED
                                    GROUP SDK
                                    )
            
            else()
                # Extension is selectable for installation in the wizard of the installer
                cpack_add_component(${APPLICATION_TARGET_NAME}
                                    DISPLAY_NAME ${APPLICATION_TARGET_NAME}
                                    DESCRIPTION ${${APPLICATION_NAME_CMAKE}_DESCRIPTION}
                                    GROUP ${${APPLICATION_NAME_CMAKE}_CEP_NAME}
                                    )
            endif()
        else()
            # Extension if not categorized for packaging presentation
            cpack_add_component(${APPLICATION_TARGET_NAME}
                                DISPLAY_NAME ${APPLICATION_TARGET_NAME}
                                DESCRIPTION ${${APPLICATION_NAME_CMAKE}_DESCRIPTION}
                                )
        
        endif()

        # additional unix system files/resources
        if(UNIX)
            # Man page installation
            set(${APPLICATION_NAME_CMAKE}_MAN_PAGE ${CMAKE_CURRENT_SOURCE_DIR}/resources/camitk-${APPLICATION_NAME}.1)
            if (EXISTS ${${APPLICATION_NAME_CMAKE}_MAN_PAGE})
                message(STATUS "Found man page for ${APPLICATION_TARGET_NAME}")
                install(FILES ${${APPLICATION_NAME_CMAKE}_MAN_PAGE}
                        DESTINATION ${CAMITK_APPLICATION_MAN_INSTALL_DIR}
                )
            else()
                # check if .in exists
                if (EXISTS ${${APPLICATION_NAME_CMAKE}_MAN_PAGE}.in)
                    message(STATUS "Found man page configuration file for ${APPLICATION_TARGET_NAME}")
                    install(CODE "message(STATUS \"Updating ${APPLICATION_NAME} man page (version ${CAMITK_VERSION_MAJOR}.${CAMITK_VERSION_MINOR} as of ${CURRENT_DATE})\")")
                    # set variable in sub-cmake shell
                    install(CODE "set(APPLICATION_NAME ${APPLICATION_NAME})")
                    install(CODE "set(CURRENT_DATE ${CURRENT_DATE})")
                    install(CODE "set(CAMITK_VERSION_MAJOR ${CAMITK_VERSION_MAJOR})")
                    install(CODE "set(CAMITK_VERSION_MINOR ${CAMITK_VERSION_MINOR})")
                    # remove previous version
                    install(CODE "execute_process(COMMAND ${CMAKE_COMMAND} -E remove -f ${CMAKE_CURRENT_BINARY_DIR}/camitk-${APPLICATION_NAME}.1)")
                    # configure current version
                    install(CODE "configure_file(${CMAKE_CURRENT_SOURCE_DIR}/resources/camitk-${APPLICATION_NAME}.1.in ${CMAKE_CURRENT_BINARY_DIR}/camitk-${APPLICATION_NAME}.1 @ONLY)")
                    # install file
                    install(FILES ${CMAKE_CURRENT_BINARY_DIR}/camitk-${APPLICATION_NAME}.1 DESTINATION ${CAMITK_APPLICATION_MAN_INSTALL_DIR})
                endif()
            endif()

            # Desktop file
            set(${APPLICATION_NAME_CMAKE}_DESKTOP ${CMAKE_CURRENT_SOURCE_DIR}/resources/camitk-${APPLICATION_NAME}.desktop)
            if (EXISTS ${${APPLICATION_NAME_CMAKE}_DESKTOP})
                message(STATUS "Found desktop file for ${APPLICATION_TARGET_NAME}")
                install(FILES ${${APPLICATION_NAME_CMAKE}_DESKTOP} DESTINATION ${CAMITK_APPLICATION_DESKTOP_INSTALL_DIR})
            endif()

            # application pixmap
            set(${APPLICATION_NAME_CMAKE}_PIXMAP ${CMAKE_CURRENT_SOURCE_DIR}/resources/camitk-${APPLICATION_NAME}.xpm)
            if (EXISTS ${${APPLICATION_NAME_CMAKE}_PIXMAP})
                message(STATUS "Found pixmap for ${APPLICATION_TARGET_NAME}")
                install(FILES ${${APPLICATION_NAME_CMAKE}_PIXMAP} DESTINATION ${CAMITK_APPLICATION_PIXMAP_INSTALL_DIR})
            endif()
            
        endif()

    endif() #APPLICATION_${APPLICATION_NAME_CMAKE}

endmacro()
