/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include "PickedPixelMap.h"
#include <ImageComponent.h>
#include <QFileInfo>
#include <QMessageBox>
#include <QTextStream>

using namespace camitk;

//-------------------------- Constructor ---------------------
PickedPixelMap::PickedPixelMap (ImageComponent *imageComp) {
    image=imageComp;
    pixelIndexList = new QList<QVector3D>();
}

//------------- Destructor -------------
PickedPixelMap::~PickedPixelMap() {}

//---------------- addPixel --------------------
void PickedPixelMap::addPixel() {
    // get the last picked pixel index
    int i, j, k;
    image->getLastPixelPicked (&i, &j, &k);
    QVector3D lastIndex = QVector3D (i,j,k);

    // check validity
    if (i != -1 && j != -1 && k != -1) {
        //insert the new point in the list
        pixelIndexList->insert (pixelIndexList->size(),lastIndex);
    }
}

//------------------------ convertIndexToCoordinates ------------------------
QVector3D PickedPixelMap::convertIndexToCoordinates (int i, int j, int k) {
    QVector3D realCoord;

    // get the size of voxel and the origin
    QVector3D voxelSize   = image->getVoxelSize();
    QVector3D imageOrigin = image->getImageOrigin();

    // transform pixel index to 3D real coordinates
    // (index pixel + 0.5) * size of voxel + imageOrigin position = real coordinates
    realCoord.setX ((i + 0.5)*voxelSize.x() + imageOrigin.x());
    realCoord.setY ((j + 0.5)*voxelSize.y() + imageOrigin.y());
    realCoord.setZ ((k + 0.5)*voxelSize.z() + imageOrigin.z());

    return realCoord;
}


QVector3D PickedPixelMap::convertCoordinatesToIndex(double x, double y, double z) {
    QVector3D index;

    // get the size of voxel and the origin
    QVector3D voxelSize = image->getVoxelSize();
    QVector3D imageOrigin = image->getImageOrigin();

    // transform 3D real coordinates to index
    // real coords = (index + 0.5) * voxelSize + imageOrigin
    // index = (real coords - imageOrigin) / voxelSize - 0.5
    index.setX((int) ((x - imageOrigin.x()) / voxelSize.x() - 0.5));
    index.setY((int) ((y - imageOrigin.y()) / voxelSize.y() - 0.5));
    index.setZ((int) ((z - imageOrigin.z()) / voxelSize.z() - 0.5));

    return index;

}

//-------------------- getCoordIndexList -----------------------
QList<QVector3D>* PickedPixelMap::getCoordIndexList() {
    QList<QVector3D> *coord = new QList<QVector3D>();

    for (int i = 0; i<pixelIndexList->size(); i++) {
        // For each point in the list of index, coordinates are calculated
        coord->insert (i,convertIndexToCoordinates (pixelIndexList->value (i).x(),pixelIndexList->value (i).y(),pixelIndexList->value (i).z()));
    }

    return coord;
}

QList<double> * PickedPixelMap::getPixelValueList() {
    QList<double> * values = new QList<double>();

    for (int i = 0; i < pixelIndexList->size(); i++) {
        // NOTE if this was a colored image, this would be the red value (the last parameter
        // of GetScalarComponentAsDouble is the index of the color component)
        double val = image->getImageData()->GetScalarComponentAsDouble(pixelIndexList->value(i).x(),
                     pixelIndexList->value(i).y(),
                     pixelIndexList->value(i).z(),
                     0);
        values->insert(i, val);
    }

    return values;
}
//----------------------- getCoordIndexList --------------------
QList<QVector3D>* PickedPixelMap::getPixelIndexList() {
    return pixelIndexList;
}

//-------------- modifyPixel -----------------------------------
void PickedPixelMap::modifyPixel (int index) {
    // get the last picked pixel index
    int i, j, k;
    image->getLastPixelPicked (&i, &j, &k);
    QVector3D lastIndex = QVector3D (i,j,k);

    // check validity
    if (i != -1 && j != -1 && k != -1 && (index >=0 && index< pixelIndexList->size())) {
        pixelIndexList->replace (index,lastIndex);
    }
}

//------------------ removePixel ----------------------------
void PickedPixelMap::removePixel (int index) {
    if (index >=0 && index< pixelIndexList->size()) {
        pixelIndexList->removeAt (index);
    }
}

//------------------ savePixelList ---------------------
void PickedPixelMap::savePixelList (QString fileName) {
    //-- prepare filename (add .csv if needed)
    QFileInfo fileNameInfo (fileName);

    if (fileNameInfo.suffix() != "csv")
        fileName += ".csv";

    QVector3D pix2Coord;
    // write the points in a csv file
    // the form is : index (int) | i (int) | j (int) | k (int) | x (double) | y (double) | z (double) |
    ofstream myFile;

    // open the file to write
    char separator=',';
    myFile.open (fileName.toStdString().c_str());
    myFile << "index" << separator << "i" << separator << "j" << separator << "k" << separator << "x" <<separator << "y" << separator << "z" <<endl;

    if (pixelIndexList != NULL) {
        for (int i = 0; i < pixelIndexList->size(); i++) {
            pix2Coord = convertIndexToCoordinates (pixelIndexList->value (i).x(),pixelIndexList->value (i).y(),pixelIndexList->value (i).z());
            myFile << i << separator;
            myFile << (int) pixelIndexList->value (i).x() << separator << (int) pixelIndexList->value (i).y() << separator << (int) pixelIndexList->value (i).z() << separator;
            myFile << pix2Coord.x() << separator << pix2Coord.y() << separator << pix2Coord.z() << endl;
        }
    }

    // close after
    myFile.close();
}

void PickedPixelMap::openPixelList (QString fileName) {

    // Read the input file
    QFile file(fileName);

    if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
        QMessageBox::warning(NULL, "Warning !", "Could not open file " + fileName + "\n, Sorry...");
        return;
    }

    if (pixelIndexList==NULL) {
        pixelIndexList = new QList<QVector3D>();
    }

    QTextStream in(&file);

    while (! in.atEnd()) {
        int i, j, k;

        // We process the file line by line
        QString line = in.readLine();

        if (! line.isEmpty()) {
            QStringList liste = line.split(",");
            QString s = liste.at(0);

            if ( (s != QString("index")) && (liste.size() == 7)) {
                s = liste.at(1);
                i = s.toInt();
                s = liste.at(2);
                j = s.toInt();
                s = liste.at(3);
                k = s.toInt();
                s = liste.at(4);

                s = liste.at(5);

                s = liste.at(6);


                //insert the new point in the list
                QVector3D lastIndex = QVector3D (i,j,k);
                pixelIndexList->insert (pixelIndexList->size(),lastIndex);
            }
        }
    }
}


//-----------------getImage-------------------
ImageComponent* PickedPixelMap::getImage() {
    return image;
}

void PickedPixelMap::resetPixelList() {
    // Empty the list if it is not already empty
    if ((pixelIndexList != NULL) && !(pixelIndexList->isEmpty())) {
        pixelIndexList->erase(pixelIndexList->begin(), pixelIndexList->end());
    }

    if (pixelIndexList == NULL)
        pixelIndexList = new QList<QVector3D>();
}

void PickedPixelMap::changePixelIndexList(QList<QVector3D> *liste) {
    resetPixelList();

    for (int i = 0 ; i < liste->size(); i++) {
        pixelIndexList->append(liste->at(i));
    }
}

void PickedPixelMap::changePixelIndexListFromRealCoordinates(QList<QVector3D> *listeOfRealCoords) {
    resetPixelList();

    for (int i = 0; i < listeOfRealCoords->size(); i++) {
        QVector3D index = convertCoordinatesToIndex(listeOfRealCoords->at(i).x(),
                          listeOfRealCoords->at(i).y(),
                          listeOfRealCoords->at(i).z());
        pixelIndexList->append(index);
    }
}

