/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include <pml/PhysicalModel.h>
#include <pml/MultiComponent.h>

#include "LoadsManager.h"
#include "PMManagerDC.h"


#include "PMManagerDCPopup.h"
#include "LoadsSimulation.h"

#include <Application.h>
using namespace camitk;

#include <lml/Loads.h>

#include <SliderTextWidget.h>

#include <QFileDialog>
#include <QMessageBox>
#include <QSettings>
#include <QInputDialog>

// ---------------------- constructor  ----------------------------
PMManagerDCPopup::PMManagerDCPopup(camitk::Component *dc, QWidget *parent) : camitk::ComponentPopupMenu(dc, parent) {

    myLM = dynamic_cast<PMManagerDC *>(myComponent)->getLoadsManager();

    addAction("Simulation...", SLOT(simulation()), true);
    addSeparator();
    addAction("Open Loads...", SLOT(openLoads()));

    // -- recent loads
    // change this number if you want more than 5 recent loads
    maxNrOfRecents = 5;
    recentLoads = new QMenu("Open Recents...");

    // get the recent loads from settings
    nrOfRecents = 0;
    recentLoadFiles.clear();
    QSettings & settings = Application::getSettings();
    settings.beginGroup("physicalModelComponent");
    int size = settings.beginReadArray("recentLoads");

    for (int i = 0; i < size; ++i) {
        settings.setArrayIndex(i);
        recentLoadFiles.push_back(settings.value("location").toString());
    }

    nrOfRecents = recentLoadFiles.size();

    settings.endArray();
    settings.endGroup();
    updateRecentLoads();

// add it to the menu and update the popup (and slots)
    addMenu(recentLoads);

    //-- other load operations
    QMenu *loadOperations = new QMenu("Loads");
    save = loadOperations->addAction("Save Loads", this, SLOT(saveLoads()));
    saveAs = loadOperations->addAction("Save Loads As...", this, SLOT(saveAsLoads()));
    close = loadOperations->addAction("Close Loads", this, SLOT(closeLoads()));
    loadOperations->addSeparator();
    edit = loadOperations->addAction("Edit Loads...", this, SLOT(editLoads()));
    preview = loadOperations->addAction("Preview Loads...", this, SLOT(previewLoads()));
    showLoads = loadOperations->addAction("Show Loads", this, SLOT(displayLoads()));
    showLoads->setCheckable(true);

    addMenu(loadOperations);
    loadOperations->setEnabled(true);

    addSeparator();

    addAction("Open Reference PML...", SLOT(openReferencePML()));

    QActionGroup *atomDataGrp = new QActionGroup(this);
    connect(atomDataGrp, SIGNAL(selected(QAction*)), this, SLOT(atomDataDisplay(QAction*)));

    hideAtomData = new QAction(tr("Hide Atom Data"), atomDataGrp);
    hideAtomData->setCheckable(true);
    hideAtomData->setChecked(true);
    QMenu::addAction(hideAtomData);
    showAddOnAtomData = new QAction(tr("Show Add-On Atom Data"), atomDataGrp);
    showAddOnAtomData->setCheckable(true);
    QMenu::addAction(showAddOnAtomData);
    showDisplacements = new QAction(tr("Show Displacements"), atomDataGrp);
    showDisplacements->setCheckable(true);
    QMenu::addAction(showDisplacements);
    showDistances = new QAction(tr("Show Distances"), atomDataGrp);
    showDistances->setCheckable(true);
    QMenu::addAction(showDistances);
    showRENError = new QAction(tr("Show R.E.N Error"), atomDataGrp);
    showRENError->setCheckable(true);
    QMenu::addAction(showRENError);

    fixAtomDataScale = addAction("Set Atom Data Scale Bounds...", SLOT(setAtomDataScale()));
    fixAtomDataScale->setCheckable(true);

    addSeparator();

//TODO fix import/export  addAction("Import From VTK...", SLOT(vtkImport()));
}

// ---------------------- updateMenuActions ----------------------------
void PMManagerDCPopup::updateMenuActions() {
    // if loads are modified we could save, and save as...
    save->setEnabled(myLM->isModified());

    // default: nothing is enabled, but open
    close->setEnabled(false);
    saveAs->setEnabled(false);
    edit->setEnabled(false);
    preview->setEnabled(false);
    showLoads->setEnabled(false);

    // if there are some loads, then we can close, saveAs, edit and preview
    Loads *l = myLM->getLoads();

    if (l && l->numberOfLoads() > 0) {
        close->setEnabled(true);
        saveAs->setEnabled(true);
        edit->setEnabled(true);
        preview->setEnabled(true);
        showLoads->setEnabled(true);
        // if the loads are displayed in 3D, check the menu item
        showLoads->setChecked(myLM->displayLoads());
    }

    // update the atom-data display
    switch (myLM->getAtomDataDisplay()) {
    case LoadsManager::NONE:
        hideAtomData->setChecked(true);
        break;
    case LoadsManager::ADD_ON:
        showAddOnAtomData->setChecked(true);
        break;
    case LoadsManager::DISPLACEMENTS:
        showDisplacements->setChecked(true);
        break;
    case LoadsManager::DISTANCES:
        showDistances->setChecked(true);
        break;
    case LoadsManager::RELATIVE_ENERGY_NORM_ERROR:
        showRENError->setChecked(true);
        break;
    }

}

// ---------------------- updateRecentLoads ----------------------------
void PMManagerDCPopup::updateRecentLoads() {
    // remove the last items
    if (nrOfRecents == 0) {
        recentLoads->setEnabled(false);
        return;
    }

    recentLoads->setEnabled(true);

    // clear all
    recentLoads->clear();

    // now create the whole list of recent documents and connect them to the unique slot
    for (unsigned int i = 0; i < nrOfRecents && i < maxNrOfRecents; i++) {
        QString menuName = QFileInfo(recentLoadFiles[i]).baseName();
        QAction * recentAction = recentLoads->addAction(menuName);
        connect(recentAction, SIGNAL(triggered()), this, SLOT(openRecentLoad()));
        // set the data (that will automatically be used in the slot to open the corresponding file)
        recentAction->setData(recentLoadFiles[i]);
        recentAction->setStatusTip(recentLoadFiles[i]);
    }
}

// ------------------------------ addRecentLoads -------------------------------
void PMManagerDCPopup::addRecentLoads(QString f) {
    recentLoadFiles.removeAll(f);
    recentLoadFiles.push_front(f);
    saveRecentLoads();
    nrOfRecents = recentLoadFiles.size();
}

// ------------------------------ saveRecentLoads -------------------------------
void PMManagerDCPopup::saveRecentLoads() {
    QSettings & settings = Application::getSettings();

    settings.beginGroup("physicalModelComponent");
    settings.beginWriteArray("recentLoads");

    unsigned int i = 0;
    QStringList::Iterator it = recentLoadFiles.begin();

    while (i < maxNrOfRecents && it != recentLoadFiles.end()) {
        settings.setArrayIndex(i);
        settings.setValue("location", (*it));
        ++it;
        ++i;
    }

    settings.endArray();
    settings.endGroup();
}

// ---------------------- openLoads ----------------------------
void PMManagerDCPopup::openLoads() {
    QDir lastOpenedDirectory;

    if (recentLoadFiles.size() == 0)
        lastOpenedDirectory = QFileInfo(myComponent->getFileName()).absoluteDir();
    else
        lastOpenedDirectory = QFileInfo(recentLoadFiles[0]).absoluteDir();

    // open a file open dialog
    QString fileName = QFileDialog::getOpenFileName(NULL, "Open a LML Document..." , lastOpenedDirectory.absolutePath(), tr("LML (*.lml)"));

    if (!fileName.isNull()) {
        openLoads(QFileInfo(fileName).absoluteFilePath());
    }

}

// ---------------------- openRecentLoad ----------------------------
void PMManagerDCPopup::openRecentLoad() {
    QAction *action = qobject_cast<QAction *>(sender());

    if (action)
        openLoads(action->data().toString());
}

// ---------------------- openLoads ----------------------------
void PMManagerDCPopup::openLoads(QString fileName) {
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
    // give the filename to the load manager
    myLM->open(fileName);
    // update the view
    myLM->updateLoadsDisplay();
    QApplication::restoreOverrideCursor();

    // update the popup
    addRecentLoads(fileName);
    updateRecentLoads();
    updateMenuActions();
}

// ---------------------- saveLoads ----------------------------
void PMManagerDCPopup::saveLoads() {
    myLM->save();
    updateMenuActions();
}

// ---------------------- saveAsLoads ----------------------------
void PMManagerDCPopup::saveAsLoads() {
    QDir lastOpenedDirectory;

    if (recentLoadFiles.size() == 0)
        lastOpenedDirectory = QFileInfo(myComponent->getFileName()).absoluteDir();
    else
        lastOpenedDirectory = QFileInfo(recentLoadFiles.last()).absoluteDir();


    QString newFileName = QFileDialog::getSaveFileName(NULL, "Save LML Document", lastOpenedDirectory.absolutePath(), tr("LML (*.lml)"));
    if (!newFileName.isNull())
        myLM->saveAs(newFileName);
    updateMenuActions();
}

// ---------------------- closeLoads ----------------------------
void PMManagerDCPopup::closeLoads() {
    myLM->close();
    updateMenuActions();
}

// ---------------------- displayLoads ----------------------------
void PMManagerDCPopup::displayLoads() {
    myLM->setDisplayLoads(showLoads->isChecked());
    updateMenuActions();
}

// ---------------------- previewLoads ----------------------------
void PMManagerDCPopup::previewLoads() {
    myLM->previewLoads();
    updateMenuActions();
}

// ---------------------- editLoads ----------------------------
void PMManagerDCPopup::editLoads() {
    myLM->editLoads();
    updateMenuActions();
}

// ---------------------- simulation ----------------------------
void PMManagerDCPopup::simulation() {
    myLM->simulation();
}

/* TODO fix import/export
// ---------------------- vtkImport ----------------------------
void PMManagerDCPopup::vtkImport() {
  QDir lastOpenedDirectory = QFileInfo(myComponent->getFileName()).absoluteDir();

  QString vtkFileName = QFileDialog::getOpenFileName(NULL, "Import VTK File...", lastOpenedDirectory.absolutePath(), tr("Vtk (*.vtk)"));

  if (!vtkFileName.isNull()) {
    PhysicalModel *pm = dynamic_cast<PMManagerDC *>(myComponent)->vtkToPhysicalModel(vtkFileName);

    // change the extension
    QFileInfo vtkFile(vtkFileName);
    QString pmlFile = QFileInfo(vtkFile.absoluteDir(), vtkFile.baseName() + ".pml").absoluteFilePath();

    // export
    ofstream outputFile(pmlFile.toStdString().c_str());
    pm->xmlPrint(outputFile, true);

    QMessageBox::information(NULL, "VTK file saved as PML", vtkFileName + " saved as physical model in " + pmlFile);
  }
}
*/

// ---------------------- atomDataDisplay ----------------------------
void PMManagerDCPopup::atomDataDisplay(QAction *selectedAction) {
    if (selectedAction == hideAtomData)
        myLM->setAtomDataDisplay(LoadsManager::NONE);
    else if (selectedAction == showAddOnAtomData)
        myLM->setAtomDataDisplay(LoadsManager::ADD_ON);
    else if (selectedAction == showDisplacements)
        myLM->setAtomDataDisplay(LoadsManager::DISPLACEMENTS);
    else if (selectedAction == showDistances) {
        // if there is yet not any physical model get a file name
        if (myLM->getReferencePM() == NULL)
            openReferencePML();

        if (myLM->getReferencePM() != NULL)
            myLM->setAtomDataDisplay(LoadsManager::DISTANCES);
        else
            hideAtomData->setChecked(true);
    } else if (selectedAction == showRENError) {
        // if there is yet not any physical model get a file name
        if (myLM->getReferencePM() == NULL)
            openReferencePML();

        if (myLM->getReferencePM() != NULL)
            myLM->setAtomDataDisplay(LoadsManager::RELATIVE_ENERGY_NORM_ERROR);
        else
            hideAtomData->setChecked(true);
    }
}

// ---------------------- openReferencePML ----------------------------
void PMManagerDCPopup::openReferencePML() {
    QDir lastOpenedDirectory = QFileInfo(myComponent->getFileName()).absoluteDir();

    QString pmldiffFileName = QFileDialog::getOpenFileName(NULL, "Open PML Reference Document", lastOpenedDirectory.absolutePath(), tr("PML (*.pml)"));

    if (!pmldiffFileName.isNull())
        myLM->setReferencePML(pmldiffFileName);
}

// ---------------------- setAtomDataScale ----------------------------
void PMManagerDCPopup::setAtomDataScale() {
    if (!myLM->getUserConstrainedAtomDataScale()) {
        double min = QInputDialog::getDouble(NULL,"Min Value", "Enter min value for atom data scale:", 0.0);
        double max = QInputDialog::getDouble(NULL,"Max Value", "Enter max value for atom data scale:", 0.0);
        myLM->updateAtomDataScale(min,max);
        myLM->userConstrainedAtomDataScale(true);
    } else {
        myLM->userConstrainedAtomDataScale(false);
    }
    fixAtomDataScale->setChecked(myLM->getUserConstrainedAtomDataScale());
}

