/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef DICOMTREESERIECOMPONENT_H
#define DICOMTREESERIECOMPONENT_H

// CamiTK includes
#include <ImageComponent.h>
class DicomStudyComponent;
namespace camitk {
class ImageOrientationHelper;
}

// Qt includes
#include <QObject>

// GDCM includes
class vtkGDCMImageReader;

/**
 * @ingroup group_cepimaging_components_dicomdirectory
 *
 * @brief
 * This component allows you to read DICOM images files of a single SERIE of a given STUDY
 *
 * One SERIE of images is the result of one medical examination using one MODALITY.
 *
 * This component works on images acquired of a same SERIE for a given STUDY
 * This SERIE of images will be represented as a 3D volumic image (ImageComponent)
 * DICOM images tags are also displayed as Property in the PropertyExplorer (DICOM tab).
 *
 * @note
 * Requires GDCM library in order to work correctly.
 * This component does not support 2D Images serie (GDCM limitation)
 *
 */
class DicomTreeSerieComponent : public camitk::ImageComponent {

    Q_OBJECT

public:
    /**
     * Build a ImageComponent representing the volumic image of the acquistion by a specific MODALITY.
     * @param parent the parent DicomStudyComponent that contains all the SERIES.
     * @param path: the image slices of a same SERIE for a given STUDY.
     **/
    DicomTreeSerieComponent(const std::vector < std::string > & fileNames);

    /// Default Destructor
    virtual ~DicomTreeSerieComponent();

private:

    /**
     * @brief The image reader using GDCM that allows to read DICOM image as vtkImageData
     *
     * @note This class allows to make the bridge between GDCM and CamiTK.
     */
    vtkGDCMImageReader* imageReader;

    /**
     * @brief Update the LUT of the image by reading the good information from the DICOM headers.
     *
     * @note This method would have to be removed as soons as the CamiTK LUT is updated and working.
     */
    void updateLUT();

    /**
     * @brief Read the direct cosines angles from the input image
     * @param fileNames the SERIE slices
     * @return the corresponding ImageOrientation, rounded modulo 90 degrees
     */
    camitk::ImageOrientationHelper::PossibleImageOrientations readDirectCosinesAngle(const std::vector< std::string > & fileNames) const;

    /**
     * @brief Round the cosine input to the closest value between -1, 0 or 1
     * @param value the input cosine value
     * @return the rounded cosine value -1, 0 or 1
     */
    double roundCosine(const double & value) const;

    /**
     * @brief Translate the image data to the frame origin and store the opposite translate as a frame of the component
     * This allows to get the image frame in RAI origin of the image. Else, picking won't work.
     * @return the imageData translated.
     */
    vtkSmartPointer<vtkImageData> translate(vtkSmartPointer<vtkImageData> imageData, double x, double y, double z);


};

#endif // DICOMTREESERIECOMPONENT_H

