/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include "DicomItkComponentExtension.h"
#include "DicomItkComponent.h"

#include <itkGDCMSeriesFileNames.h>
#include <itkImageFileReader.h>
#include <itkGDCMImageIO.h>
#include <itkImageSeriesReader.h>

#include <Log.h>
using namespace camitk;

#include <QMessageBox>
#include <QtGui/QDialog>
#include <QDialogButtonBox>
#include <QtGui/QVBoxLayout>
#include <QGridLayout>
#include <QtGui/QDialogButtonBox>
#include <QList>
#include <QLabel>
#include <QTextStream>
#include <QString>
#include <stdio.h>
#include <iostream>

// Definition of tag keys (from dcmtk: http://support.dcmtk.org/docs/dcdeftag_8h-source.html)
// Could also be retrieve from http://www.sno.phy.queensu.ca/~phil/exiftool/TagNames/DICOM.html
#define ComponentM_Modality                             "0008|0060"
#define ComponentM_ProtocolName                         "0018|1030"
#define ComponentM_SeriesDescription                    "0008|103e"
#define ComponentM_Rows                                 "0028|0010"
#define ComponentM_Columns                              "0028|0011"
#define ComponentM_NumberOfTemporalPositions            "0020|0105"
#define ComponentM_TemporalPositionIdientifier          "0020|0100"
#define ComponentM_BitsAllocated                        "0028|0100"
#define ComponentM_BitsStored                           "0028|0101"
#define ComponentM_HighBit                              "0028|0102"
#define ComponentM_StudyDescription                     "0008|1030"


// --------------- declare the plugin -------------------
Q_EXPORT_PLUGIN2(dicomitk, DicomItkComponentExtension);


// If nothing works, get gdcm 2.0: http://apps.sourceforge.net/mediawiki/gdcm/index.php?title=ITK_USE_SYSTEM_GDCM

// --------------- getName -------------------
QString DicomItkComponentExtension::getName() const {
    return "Dicom Component";
}

// --------------- getDescription -------------------
QString DicomItkComponentExtension::getDescription() const {
    return "Manage Dicom directory in <b>CamiTK</b>";
}

// --------------- hasDataDirectory -------------------
bool DicomItkComponentExtension::hasDataDirectory() const {
    return true;
}


#include <QDateTime>
#include <Log.h>
// --------------- open -------------------
Component * DicomItkComponentExtension::open(const QString & directoryName) throw (AbortException) {
    int nVol = 0;
    QString name;
    QString nameVol;
    DicomItkComponent * res = NULL;

    try {
        findAllDicomSeries(directoryName);
        chooseDicomSeriesIds();
        std::map<SomeDicomInfo*, FileNamesContainerType>::iterator it;
        for(it = theSeries.begin(); it != theSeries.end();  it++) {
            if (it->first->openIt->isChecked()) {
                name.clear();
                QTextStream(&name) << it->first->protocolName << "_" << nVol;
                if ((it->first->numberOfTemporalPositions.isEmpty())    ||
                        (it->first->numberOfTemporalPositions.toInt() == 1) ||
                        !(it->first->desInterlace->isChecked()))
                    res = new DicomItkComponent(directoryName, it->second, name);
                else { // In case of temporal series, the volume are stored interlaced
                    int nbVols = it->first->numberOfTemporalPositions.toInt();
                    FileNamesContainerType allFiles = it->second;
                    for (int volumeNo = 0; volumeNo < nbVols; volumeNo++) {
                        FileNamesContainerType volumeFiles;
                        nameVol.clear();
                        QTextStream(&nameVol) << name << "_" << volumeNo;
                        // Fill in volumeFiles with one out of nbVols file names
                        for (unsigned int n = volumeNo; n < allFiles.size(); n+= nbVols)
                            volumeFiles.push_back(allFiles[n]);

                        QDateTime start = QDateTime::currentDateTime();


                        res = new DicomItkComponent(directoryName, volumeFiles, nameVol);

                    }
                }
                nVol ++;
            }
        }

    } catch(AbortException e) {
        QMessageBox * msg = new QMessageBox();
        msg->setText(QString(e.what()));
        msg->show();
        return NULL;
    }

    theSeries.clear();

    return res;
}



bool DicomItkComponentExtension::findAllDicomSeries(const QString & directoryName) {
    std::string modalityTag, protocolNameTag, seriesDescriptionTag, rowsTag, columnsTag;
    std::string slicesTag, numberOfTemporalPositionsTag, bitsAllocatedTag, highBitTag, studyDescriptionTag;

    // To add information to series identifyers
    // We must actually read the data series to get acces to the tags.
    // Code Snippet from ITK: \itk-3.10.1\Examples\IO\DicomSeriesReadPrintTags.cxx
    // (or The ITK Software Guide Chapter 7, 7.12.6 "Printing Dicom Tags From a Series" (p303))

    typedef signed short PixelType;
    const unsigned int   Dimension = 3;
    typedef itk::Image<PixelType, Dimension>     ImageType;
    typedef itk::ImageSeriesReader< ImageType >  ReaderType;
    typedef itk::GDCMImageIO                     ImageIOType;
    typedef itk::GDCMSeriesFileNames             NamesGeneratorType;
    typedef itk::SerieUIDContainer               SeriesIdContainerType;
    typedef itk::MetaDataDictionary              DictionaryType;
    typedef itk::MetaDataObject< std::string >   MetaDataStringType;


    ImageIOType::Pointer        dicomIO       = ImageIOType::New();
    ReaderType::Pointer         reader        = ReaderType::New();
    reader->SetImageIO( dicomIO );

    // Find series/volumes identifiers
    NamesGeneratorType::Pointer nameGenerator = NamesGeneratorType::New();
    nameGenerator->SetUseSeriesDetails(true);
    nameGenerator->SetDirectory(directoryName.toStdString());
    // Vector containing all series identifiers
    SeriesIdContainerType seriesUID = SeriesIdContainerType(nameGenerator->GetSeriesUIDs());
    // For each series identifier,
    //    - find the corresponding filenames
    //    - read those filenames and find tags to inform user about the series
    SeriesIdContainerType::const_iterator it = seriesUID.begin();

    while (it != seriesUID.end()) {
        // Obtain Filenames and dicom tags for each series
        FilenamesContainerType filenames = nameGenerator->GetFileNames((*it));
        reader->SetFileNames( filenames );
        // ITK have to actually read the files...
        reader->Update();

        // Write the all the dictionary info available on the console
        // See Code Snippet from ITK: \itk-3.10.1\Examples\IO\DicomSeriesReadPrintTags.cxx
        // (or The ITK Software Guide Chapter 7, 7.12.6 "Printing Dicom Tags From a Series" (p303)
        const  DictionaryType & dictionary = dicomIO->GetMetaDataDictionary();
        DictionaryType::ConstIterator itr = dictionary.Begin();
        DictionaryType::ConstIterator end = dictionary.End();
        // Write all availaible dicom tags for each series in the console
        while( itr != end ) {
            itk::MetaDataObjectBase::Pointer  entry = itr->second;
            MetaDataStringType::Pointer entryvalue = dynamic_cast<MetaDataStringType *>( entry.GetPointer() ) ;
            if( entryvalue ) {
                std::string tagkey   = itr->first;
                std::string tagname;
                itk::GDCMImageIO::GetLabelFromTag(tagkey, tagname);
                std::string tagvalue = entryvalue->GetMetaDataObjectValue();
                CAMITK_INFO("DicomComponentExtension", "findAllDicomSeries", tagkey << " | " << tagname << " => " << tagvalue);
            }
            ++itr;
        }

        // Store needed info in a SomeDicomInfo structure
        SomeDicomInfo * info = new SomeDicomInfo();

        dicomIO->GetValueFromTag(ComponentM_Modality,  modalityTag);
        dicomIO->GetValueFromTag(ComponentM_ProtocolName, protocolNameTag);
        dicomIO->GetValueFromTag(ComponentM_SeriesDescription, seriesDescriptionTag);
        dicomIO->GetValueFromTag(ComponentM_Rows, rowsTag);
        dicomIO->GetValueFromTag(ComponentM_Columns, columnsTag);
        dicomIO->GetValueFromTag(ComponentM_NumberOfTemporalPositions, numberOfTemporalPositionsTag);
        dicomIO->GetValueFromTag(ComponentM_BitsAllocated, bitsAllocatedTag);
        dicomIO->GetValueFromTag(ComponentM_HighBit, highBitTag);
        dicomIO->GetValueFromTag(ComponentM_StudyDescription, studyDescriptionTag);

        info->modality = QString(modalityTag.c_str());
        info->protocolName = QString(protocolNameTag.c_str());
        info->seriesDescription = QString(seriesDescriptionTag.c_str());
        info->rows = QString(rowsTag.c_str());
        info->columns = QString(columnsTag.c_str());
        QTextStream(&info->slices) << filenames.size();
        info->numberOfTemporalPositions = QString(numberOfTemporalPositionsTag.c_str());
        info->bitsAllocated = QString(bitsAllocatedTag.c_str());
        info->highBit = QString(highBitTag.c_str());
        info->studyDescription = QString(studyDescriptionTag.c_str());

        // Keep the filenames and the info in the map for this series
        theSeries[info] = filenames;

        //seriesFilenames.push_back(filenames);
        //// Keep info for this series
        //seriesInfos.push_back(info);

        it++;
    }

    return true;
}


bool DicomItkComponentExtension::chooseDicomSeriesIds() throw (AbortException) {
    // If no volume -> error
    if (theSeries.empty())
        throw new AbortException(std::string("No dicom volume series found in this directory\n"));

    // If there is only one volume in the directory, don't bother to ask user, just open it
    if (theSeries.size() == 1) {
        theSeries.begin()->first->openIt = new QCheckBox();
        theSeries.begin()->first->openIt->setChecked(true);
        return true;
    }


    // Draw a nice Qt dialog to ask the user the series he/she wants to open
    QDialog * seriesChoiceDialog = new QDialog();
    QVBoxLayout * v_layout = new QVBoxLayout();

    // Little tex explaining what is happening
    QLabel * explaination = new QLabel();
    QString explaination_str;
    QTextStream(& explaination_str) << "There are several dicom image volumes in this directory<br/>";
    QTextStream(& explaination_str) << "Please choose the volume(s) series you want <tt>CamiTK</tt> to open.<br/>";
    explaination->setTextFormat(Qt::RichText);
    explaination->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
    explaination->setFont(QFont("Helvetica [Cronyx]", 12, QFont::Bold));
    explaination->setText(explaination_str);
    v_layout->addWidget(explaination);

    int seriesNo = 1;
    QGridLayout * seriesNameLayout = new QGridLayout();
//  seriesNameLayout->setColStretch(0, 1);
//  seriesNameLayout->setColStretch(1, 2);
    std::map<SomeDicomInfo *, FileNamesContainerType>::iterator it;
    for (it = theSeries.begin(); it != theSeries.end(); it++) {
        // Checkbox for the user to choose or not the series
        it->first->openIt = new QCheckBox();
        it->first->openIt->setChecked(true);

        // Defining the description of the seires
        QString description;

        QTextStream(&description) << "Series no " << seriesNo << "<ul>";
        QTextStream(&description) << "<li><i> modality:           </i>" << it->first->modality           << "</li>";
        QTextStream(&description) << "<li><i> protocol name:      </i>" << it->first->protocolName       << "</li>";
        QTextStream(&description) << "<li><i> series description: </i>" << it->first->seriesDescription  << "</li>";
        QTextStream(&description) << "<li><i> study  description: </i>" << it->first->studyDescription   << "</li>";
        QTextStream(&description) << "<li><i> image size: </i>" << it->first->rows << "x";
        QTextStream(&description) << it->first->columns << "x" << it->first->slices << "</li>";
        QTextStream(&description) << "</ul><br/>";

        CAMITK_INFO("DicomComponentExtension", "chooseDicomSeriesIds", "Files for " << seriesNo << "/" << it->first->modality.toStdString() << "/" << it->first->protocolName.toStdString() << "/" << it->first->seriesDescription.toStdString() << "/" << it->first->rows.toStdString() << "x" << it->first->columns.toStdString() << "x" << it->first->slices.toStdString() << ":");
        for (FileNamesContainerType::iterator itFile = it->second.begin(); itFile!=it->second.end(); itFile++) {
            CAMITK_INFO("DicomComponentExtension", "chooseDicomSeriesIds", (*itFile) << " ");
        }

        QLabel      * descriptionLabel   = new QLabel();
        descriptionLabel->setTextFormat(Qt::RichText);
        descriptionLabel->setText(description);

        // De-interlace?
        it->first->desInterlace = new QCheckBox();
        it->first->desInterlace->setChecked(false);

        // Putting everything in a nice layout
        seriesNameLayout->addWidget(it->first->openIt, seriesNo - 1, 0);
        seriesNameLayout->addWidget(descriptionLabel,  seriesNo - 1, 1);

        if (!(it->first->numberOfTemporalPositions.isEmpty()) &&
                (it->first->numberOfTemporalPositions.toInt() > 1)) {
            it->first->desInterlace->setChecked(true);
            it->first->desInterlace->setText("de-interlace?");
            seriesNameLayout->addWidget(it->first->desInterlace, seriesNo - 1, 2);
        }

        seriesNo ++;
    }

    v_layout->addLayout(seriesNameLayout);

    // Ok and cancel buttons
    QDialogButtonBox *buttonBox = new QDialogButtonBox(QDialogButtonBox::Ok |QDialogButtonBox::Cancel);
    v_layout->addWidget(buttonBox);

    seriesChoiceDialog->setModal(Qt::ApplicationModal);
    seriesChoiceDialog->setLayout(v_layout);

    // Connect Ok and cancel buttons
    connect(buttonBox, SIGNAL(accepted()), seriesChoiceDialog, SLOT(accept()));
    connect(buttonBox, SIGNAL(rejected()), seriesChoiceDialog, SLOT(reject()));

    // Show the window and manage the cancel case
    if (seriesChoiceDialog->exec() != QDialog::Accepted)
        throw AbortException("Operation canceled\n");

    return true;
}


QStringList DicomItkComponentExtension::getFileExtensions()  const {
    QStringList ext;
    ext << "[directory]";
    return ext;
}
