/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
#include "LaplacianSharpening.h"

#include <Application.h>
#include <ItkProgressObserver.h>
#include <QMessageBox>
#include <QString>
#include <QTextStream>
#include <itkImageToVTKImageFilter.h>
#include <itkVTKImageToImageFilter.h>
#include <itkLaplacianSharpeningImageFilter.h>

using namespace camitk;


// --------------- constructor -------------------
LaplacianSharpening::LaplacianSharpening(ActionExtension * extension) : Action(extension) {
    // Setting name, description and input component
    setName("Laplacian Sharpening");
    setDescription("<br>This filter <b> <i>sharpens an image using a Laplacian filter</i> </b>.<br> \
				   Laplacian Sharpening highlights regions of rapid intensity change and therefore highlights or enhances the edges. The result is an image that appears more in focus.<br/>");
    setComponent("ImageComponent");

    // Setting classification family and tags
    this->setFamily("ITK Filter");
    this->addTag("Derivative");
    this->addTag("Edge Detection");
    this->addTag("Contours");
    this->addTag("Sharpen");

    // Setting the widget containing the parameters
    actionWidget = NULL;

}

// --------------- destructor -------------------
LaplacianSharpening::~LaplacianSharpening() {
}

// --------------- apply -------------------
Action::ApplyStatus LaplacianSharpening::apply() {
    foreach (Component *comp, getTargets()) {
        ImageComponent * input = dynamic_cast<ImageComponent *> ( comp );
        process(input);
    }
    return SUCCESS;
}

void LaplacianSharpening::process(ImageComponent * comp) {
    // ITK filter implementation using templates
    vtkSmartPointer<vtkImageData> inputImage = comp->getImageData();
    vtkSmartPointer<vtkImageData> outputImage = implementProcess (inputImage);
    QString newName;
    QTextStream(&newName) << comp->getName() << "_sharpened";
    new ImageComponent(outputImage, newName);
    Application::refresh();

}

#include "LaplacianSharpening.impl"

// ITK filter implementation
template <class InputPixelType, class OutputPixelType, const int dim>
vtkSmartPointer<vtkImageData> LaplacianSharpening::itkProcess(vtkSmartPointer<vtkImageData> img) {
    vtkSmartPointer<vtkImageData> result = vtkSmartPointer<vtkImageData>::New();

    // --------------------- Filters declaration and creation ----------------------
    // Define ITK input and output image types with respect to the instanciation
    //    types of the tamplate.
    typedef itk::Image< InputPixelType,  dim > InputImageType;
    typedef itk::Image< OutputPixelType, dim > OutputImageType;

    // Convert the image from CamiTK in VTK format to ITK format to use ITK filters.
    typedef itk::VTKImageToImageFilter<InputImageType> vtkToItkFilterType;
    typename vtkToItkFilterType::Pointer vtkToItkFilter = vtkToItkFilterType::New();

    // Declare and create your own private ITK filter here...
    typedef itk::LaplacianSharpeningImageFilter<InputImageType, OutputImageType> LaplacianSharpeningFilterType;
    typename LaplacianSharpeningFilterType::Pointer laplacianSharpeningFilter = LaplacianSharpeningFilterType::New();

    // In the same way, once the image is filtered, we need to convert it again to
    // VTK format to give it to CamiTK.
    typedef itk::ImageToVTKImageFilter<OutputImageType> itkToVtkFilterType;
    typename itkToVtkFilterType::Pointer itkToVtkFilter = itkToVtkFilterType::New();
// ------------------------- WRITE YOUR CODE HERE ----------------------------------

    // To update CamiTK progress bar while filtering, add an ITK observer to the filters.
    ItkProgressObserver::Pointer observer = ItkProgressObserver::New();
    // ITK observers generally give values between 0 and 1, and CamiTK progress bar
    //    wants values between 0 and 100...
    observer->SetCoef(100.0);

    // --------------------- Plug filters and parameters ---------------------------
    // From VTK to ITK
    vtkToItkFilter->SetInput(img);
    laplacianSharpeningFilter->SetInput(vtkToItkFilter->GetOutput());
    laplacianSharpeningFilter->AddObserver(itk::ProgressEvent(), observer);

    // From ITK to VTK
    itkToVtkFilter->SetInput(laplacianSharpeningFilter->GetOutput());

    // --------------------- Actually execute all filters parts --------------------
    itkToVtkFilter->Update();
    observer->Reset();

    // --------------------- Create and return a copy (the filters will be deleted)--
    vtkSmartPointer<vtkImageData> resultImage = itkToVtkFilter->GetOutput();
    int extent[6];
    resultImage->GetExtent(extent);
    result->SetExtent(extent);
    result->DeepCopy(resultImage);
    result->Update();

    // Set CamiTK progress bar back to zero (the processing filter is over)
    observer->Reset();

    return result;
}

