/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

// -- Core stuff
#include "Application.h"
#include "Core.h"
#include "Log.h"
#include "MainWindow.h"
#include "ExtensionManager.h"
#include "Action.h"

// -- QT stuff
#include <QSettings>
#include <QMessageBox>

// -- VTK stuff (only for updating progress bar via vtk filters)
#include <vtkAlgorithm.h>

// see http://doc.qt.nokia.com/latest/resources.html (bottom)
inline void initIcons() {
    Q_INIT_RESOURCE(CamiTKIcons);
}

namespace camitk {

// the main window (static, unique instance, verifies singleton)
MainWindow * Application::mainWindow = NULL;
QSettings Application::settings(QSettings::IniFormat, QSettings::UserScope, "TIMC-IMAG", QString(Core::version).remove(QChar(' ')));
QList<QFileInfo> Application::recentDocuments;
QDir Application::lastUsedDirectory;
int Application::maxRecentDocuments = 0;
QString Application::name=Core::version;

// ----------------- constructor --------------------
Application::Application(QString name, int & theArgc, char ** theArgv, bool autoloadExtensions): QApplication(theArgc, theArgv)  {
    this->name = name;
    argc = theArgc;
    argv = theArgv;
    mainWindow = NULL;

    initIcons(); // sometimes needed to get the icon when compiled in static mode

    // recommanded (see exec() comment in QApplication)
    connect(this, SIGNAL(aboutToQuit()), SLOT(quitting()));
    connect(this, SIGNAL(lastWindowClosed()), SLOT(quitting()));

    // init log file if needed
#ifdef LOG_IN_FILE
    Log::openLogFile();
#endif

    // once the log and the redirection is setup, only now one can try to load the extensions
    if (autoloadExtensions) {
	ExtensionManager::autoload();
    }

    // initialize recent/lastUsedDirectory documents from the settings
    settings.beginGroup ( name + ".Application" );

    // max memorized recent documents
    maxRecentDocuments = settings.value( "maxRecentDocuments", 10).toInt();

    // the recent documents
    QStringList recentDoc = settings.value ( "recentDocuments" ).toStringList();
    recentDocuments.clear();
    foreach ( QString fileName, recentDoc ) {
        recentDocuments.append ( fileName );
    }

    // the last used directory
    QDir defaultDir ( Core::getTestDataDir() );
    lastUsedDirectory = settings.value ( "lastUsedDirectory", defaultDir.absolutePath() ).toString();

    settings.endGroup();

}

// ----------------- destructor --------------------
Application::~Application() {
    // do not use the destructor to clean or free resources, but quitting()
}

// ----------------- destructor --------------------
QString Application::getName() {
    return name;
}

// ----------------- quitting --------------------
void Application::quitting() {
    // this is connect to the aboutToQuit signal from QApplication
    // it should contain all the code that frees the resources

    // delete all actions (they are instanciated when the extension is loaded)
    ExtensionManager::unloadAllActionExtensions();

#ifdef LOG_IN_FILE
    Log::closeLogFile();
#endif
}

// ----------------- setMainWindow --------------------
void Application::setMainWindow(MainWindow* mw) {
    if (mw==NULL)
        mainWindow = new MainWindow(name);
    else
        mainWindow = mw;

    // check for console option
    bool useConsole = true;
    int i = 0;
    while (i < argc && useConsole) {
        useConsole = !(strcmp(argv[i], "--noConsole") == 0) && !(strcmp(argv[i], "-n") == 0);
        i++;
    }

    mainWindow->redirectToConsole(useConsole);

}


// ----------------- getMainWindow --------------------
MainWindow* Application::getMainWindow() {
    if (!mainWindow)
        dynamic_cast<Application*>(qApp)->setMainWindow(NULL);

    return mainWindow;
}

// ----------------- getSettings --------------------
QSettings& Application::getSettings() {
    return settings;
}

// ----------------- exec --------------------
int Application::exec() {
    if (!mainWindow)
        dynamic_cast<Application*>(qApp)->setMainWindow(NULL);

    mainWindow->aboutToShow();
    mainWindow->show();

    return QApplication::exec();
}

// ----------------- refresh --------------------
void Application::refresh() {
    mainWindow->refresh();
}

// ----------------- showStatusBarMessage --------------------
void Application::showStatusBarMessage(QString msg, int timeout) {
    QStatusBar * statusBar = mainWindow->statusBar();

    if (statusBar) {
        statusBar->showMessage(msg, timeout);
    }
    else {
        CAMITK_INFO("Application", "showStatusBarMessage", msg.toStdString() << std::endl);
    }
}

// ----------------- resetProgressBar --------------------
void Application::resetProgressBar() {
    QProgressBar * progress = mainWindow->getProgressBar();

    if (progress) {
        progress->setValue(0);
    }
}

// ----------------- setProgressBarValue --------------------
void Application::setProgressBarValue(int value) {
    QProgressBar * progress = mainWindow->getProgressBar();

    if (progress) {
        progress->setValue(value);
    }

}

// ----------------- vtkProgressFunction --------------------
void Application::vtkProgressFunction(vtkObject* caller, long unsigned int eventId, void* clientData, void* callData) {
    QProgressBar * progress = mainWindow->getProgressBar();
    vtkAlgorithm* filter = static_cast<vtkAlgorithm*>(caller);
    int progressVal = filter->GetProgress() * 100;

    if (progress) {
        progress->setValue(progressVal);
    }
}

// ----------------- addRecentDocument --------------------
void Application::addRecentDocument(QFileInfo filename) {
    recentDocuments.removeOne ( filename );
    recentDocuments.append ( filename );

    // update the last used dir
    lastUsedDirectory = recentDocuments.last().absoluteDir();

    // save settings (the last 10 recent files by default)
    settings.beginGroup (  name + ".Application" );

    // max memorized recent documents
    settings.setValue( "maxRecentDocuments", maxRecentDocuments);

    // save all up to maxRecentDocuments
    int firstOpened = recentDocuments.size() - maxRecentDocuments;

    if ( firstOpened < 0 )
        firstOpened = 0;

    QStringList recentDoc;

    for ( int i = firstOpened; i < recentDocuments.size(); i++ )
        recentDoc.append ( recentDocuments[i].absoluteFilePath() );

    settings.setValue ( "recentDocuments", recentDoc );

    // last used directory
    settings.setValue ( "lastUsedDirectory", lastUsedDirectory.absolutePath() );
    settings.endGroup();
}

// ----------------- getLastUsedDirectory --------------------
const QDir Application::getLastUsedDirectory() {
    return lastUsedDirectory;
}

// ----------------- setLastUsedDirectory --------------------
void Application::setLastUsedDirectory(QDir last) {
    lastUsedDirectory = last;
}

// ----------------- getRecentDocuments --------------------
const QList< QFileInfo > Application::getRecentDocuments() {
    return recentDocuments;
}

// ----------------- getMaxRecentDocuments --------------------
const int Application::getMaxRecentDocuments() {
    return maxRecentDocuments;
}


// -------------------- open --------------------
Component * Application::open(const QString & fileName) {
    // set waiting cursor
    setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    Component *comp = NULL;

    // -- Get the corresponding extension... (compatible format)
    QString extension = QFileInfo(fileName).completeSuffix();

    // -- ask the plugin instance to create the Component instance
    ComponentExtension * cp = ExtensionManager::getComponentExtensions().value(extension);

    // -- check the validity of the plugin
    if (!cp) {
        QMessageBox::warning(NULL, "ComponentExtension Opening Error...", "Cannot find the appropriate component plugin for opening:<br>" + fileName + " (extension " + extension + ")<p>To solve this problem, make sure that: <ul><li>A corresponding valid plugin is present in " + Core::getComponentDir() + "</li><li>Your application called <tt>ComponentExtension::autoloadExtensions()</tt> before calling <tt>ComponentExtension::open(...)</tt></li></ul>");
        return comp;
    }

    // -- ask the plugin to create the top level component
    try {
        // open using transformed path so that anything looking like C:\\Dir1\\Dir2\\foo.zzz is unified to a C:/Dir1/Dir2/foo.zz
        comp = cp->open(QFileInfo(fileName).absoluteFilePath());

        // -- notify the application
        if ( comp == NULL ) {
            showStatusBarMessage( tr ( "Error loading file:" ) + fileName );
        }
        else {
            // add the document to the recent list
            addRecentDocument(fileName);
            showStatusBarMessage( tr ( QString("File " + fileName + " successfuly loaded").toStdString().c_str() ) );
        }

        // restore the normal cursor/progress bar
        restoreOverrideCursor();
        resetProgressBar();

        // refresh all viewers
        refresh();

        return comp;
    }
    catch (AbortException & e) {
        QMessageBox::warning(NULL, "Opening aborted...", "Extension: " + cp->getName() + "<br>Error: cannot open file " + fileName + "<br>Reason:<br>" + e.what());
        return NULL;
    }
    catch (std::exception& e) {
        QMessageBox::warning(NULL, "Opening aborted...", "Extension " + cp->getName() + "<br>Error: cannot open file " + fileName + "<br>External:<br>" + "This exception was not generated directly by the extension,<br>but by one of its dependency.<br>Reason:<br>" + e.what());
        return NULL;
    }
    catch (...) {
        QMessageBox::warning(NULL, "Opening aborted...", "Extension " + cp->getName() + "<br>Error: cannot open file " + fileName + "<br>Unknown Reason:<br>" + "This exception was not generated directly by the extension,<br>but by one of its dependency.");
        return NULL;
    }

}

// -------------------- openDirectory --------------------
Component * Application::openDirectory(const QString & dirName, const QString & pluginName) {
    // set waiting cursor
    setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    ComponentExtension *cp = ExtensionManager::getDataDirectoryComponents().value(pluginName);

    if (cp != NULL) {
        // Ask the plugin instance to create the Component instance
        Component *comp = NULL;

        try {
            comp = cp->open(QDir(dirName).absolutePath());

            // -- notify the application
            if ( comp == NULL ) {
                showStatusBarMessage( tr ( "Error loading directory:" ) + dirName );
            }
            else {
                showStatusBarMessage( tr ( QString("Directory " + dirName + " successfuly loaded").toStdString().c_str() ) );
            }

            // restore the normal cursor/progress bar
            restoreOverrideCursor();
            resetProgressBar();

            // refresh all viewers
            refresh();
            return comp;
        }
        catch (AbortException & e) {
            QMessageBox::warning(NULL, "Opening aborted...", "Extension: " + cp->getName() + "<br>Error: cannot open file " + dirName + "<br>Reason:<br>" + e.what());
            return NULL;
        }
        catch (...) {
            QMessageBox::warning(NULL, "Opening aborted...", "Extension " + cp->getName() + " Error: cannot open directory " + dirName + "<br>Unknown Reason:<br>" + "This exception was not generated directly by the extension,<br>but by one of its dependency.");
            delete comp;
            return NULL;
        }
    }
    else {
        QMessageBox::warning(NULL, "ComponentExtension Opening Error...", "Cannot find the appropriate component plugin for opening directory:<br>" + dirName
                             + "<p>To solve this problem, make sure that:<ul>"
                             +" <li>A corresponding valid plugin is present in " + Core::getComponentDir() + "</li>"
                             + "<li>And your application is initialized with the autoloadExtensions option;</li>"
                             +" <li>Or your correctly registered your component in the CamiTK settings</li></ul>");
    }

    // restore the normal cursor/progress bar
    restoreOverrideCursor();

    return NULL;
}

// -------------------------- close ------------------------------
bool Application::close ( Component *comp ) {
    int keyPressed = QMessageBox::Discard;
    bool saveOk = true;
    QString compName = comp->getName();

    // check if the top-level component needs to be saved
    if ( comp->getModified() ) {
        // dManager has changed, propose to save it
        keyPressed = QMessageBox::warning ( NULL, "Closing...", "Component \"" + compName + "\" has been modified.\nDo you want to save your change before closing?", QMessageBox::Save | QMessageBox::Discard | QMessageBox::Cancel, QMessageBox::Save);

        // Do we have to save or not?
        if ( keyPressed == QMessageBox::Save )
            saveOk = save ( comp );		
    }

    // Do we have to close or not?
    if ( saveOk && keyPressed != QMessageBox::Cancel ) {
        // delete the data
        delete comp;
        // refresh all viewers
        refresh();

        showStatusBarMessage( compName + tr ( " successfuly closed..." ));
        return true;
    }
    else {
        // return that the close was cancelled
        showStatusBarMessage( tr ( "Close cancelled..." ));
        return false;
    }

}

// -------------------- save --------------------
bool Application::save(Component* component) {

    // no name -> save as
    if (component->getFileName().isEmpty()) {
        return (getAction("Save As")->trigger()==Action::SUCCESS);
    }

    // Get the saving extension...
    QString extension = QFileInfo(component->getFileName()).completeSuffix();

    // Get the extension instance that can save the component instance (compatible format)
    ComponentExtension * cp = ExtensionManager::getComponentExtensions().value(extension);

    if (!cp) {
        QMessageBox::warning(NULL, "Saving Error...", "Cannot find the appropriate component plugin for saving component:<br/>\"" + component->getName() + "\" in file:<br/>" + component->getFileName() + " (extension " + extension + ")"
                             + "<p>To solve this problem, make sure that:<ul>"
                             +" <li>A corresponding valid plugin is present in " + Core::getComponentDir() + "</li>"
                             + "<li>And your application is initialized with the autoloadExtensions option;</li>"
                             +" <li>Or your correctly registered your component in the CamiTK settings</li></ul>");
        return false;
    }
    else {
        if (cp->save(component)) {
            // update the last used dir
            setLastUsedDirectory(QFileInfo(component->getFileName()).absoluteDir());
            showStatusBarMessage( component->getName() + tr ( " successfuly saved..." ));
            return true;
        }
        else {
            return false;
        }
    }
}


// -------------------- getActionMap --------------------
QMap<QString, Action*> & Application::getActionMap() {
    static QMap<QString, Action*> actionMap;

    return actionMap;
}

// -------------------- getActions --------------------
const ActionList Application::getActions() {
    return getActionMap().values();
}

// -------------------- registerAllActions --------------------
int Application::registerAllActions(ActionExtension* ext) {
    int registered = 0;
    
    foreach(Action *action, ext->getActions()) {
        // check if an action with same name was not already registered
        if (getActionMap().contains(action->getName())) {
            CAMITK_ERROR("ExtensionManager", "loadExtension", "Cannot add action: " << action->getName().toStdString()
                         << "(extension: " << action->getExtensionName().toStdString()
                         << ", family: " << action->getFamily().toStdString()
                         << ", description:\"" << action->getDescription().toStdString()
                         << "\"): extension of same name already registered by extension \"" << getAction(action->getName())->getExtensionName().toStdString() << "\"");
        }
        else {
            getActionMap().insert(action->getName(), action);
	    registered++;
	}
    }
    return registered;
}

// ---------------- actionLessThan ----------------
bool actionLessThan(const camitk::Action* a1, const camitk::Action * a2) {
    // This method is needed by qsort in the sort method to sort action by name
    return a1->getName() < a2->getName();
}

// ---------------- sort ----------------
ActionList Application::sort(ActionSet actionSet) {
    // sort actions by name
    ActionList actionList = actionSet.toList();
    qSort(actionList.begin(), actionList.end(), actionLessThan);

    return actionList;
}

// ---------------- getAction ----------------
Action * Application::getAction(QString name) {
    return getActionMap().value(name);
}

// ---------------- getActions ----------------
ActionList Application::getActions(Component *component) {
    ActionSet actions;

    if (component) {
        QStringList componentHierarchy = component->getHierarchy();
        foreach (Action * currentAct, Application::getActions()) {
            if (componentHierarchy.contains(currentAct->getComponent()))
                actions.insert(currentAct);
        }
    }
    else {
        foreach (Action * currentAct, Application::getActions()) {
            if (currentAct->getComponent().isEmpty()) {
                actions.insert(currentAct);
            }
        }
    }

    return sort(actions);
}

// ---------------- getActions ----------------
ActionList Application::getActions(ComponentList cList) {
    // if this is an empty list, return all action not based on a component
    if (cList.size() == 0) {
        return getActions(NULL);
    }
    else {
        ActionSet actions;
        foreach (Component * currentComp, cList) {
            actions += getActions(currentComp).toSet();
        }
        return sort(actions);
    }
}

// ---------------- getActions ----------------
ActionList Application::getActions(ComponentList selComp, QString tag) {
    // first build the list of possible actions for cList
    ActionList possibleActions = getActions(selComp);

    // now check possibleActions considering the tag value
    ActionList actions;
    foreach(Action * action, possibleActions) {
        if (action->getTag().contains(tag)) {
            actions.append(action);
        }
    }

    // sort and return
    qSort(actions.begin(), actions.end(), actionLessThan);
    return actions;

}

// ---------- isAlive ----------
bool Application::isAlive(Component *comp) {
  return getAllComponents().contains(comp);
}

// -------------------- hasModified --------------------
bool Application::hasModified() {
  // look for a top level component that has been modified
  int i = 0;
  while (i < getTopLevelComponents().size() && !getTopLevelComponents()[i]->getModified())
    i++;

  return(i < getTopLevelComponents().size());
}

// -------------------- addComponent --------------------
void Application::addComponent(Component *comp) {
  getAllComponentList().append(comp);
  if (comp->getParentComponent() == NULL)
    // this a top level component
    getTopLevelComponentList().append(comp);
}

// -------------------- removeComponent --------------------
void Application::removeComponent(Component *comp) {
  getAllComponentList().removeAll(comp);
  getTopLevelComponentList().removeAll(comp);
  getSelectedComponentList().removeAll(comp);
}

// -------------------- getTopLevelComponentList --------------------
ComponentList & Application::getTopLevelComponentList() {
  static ComponentList topLevelComponents;

  return topLevelComponents;
}

// -------------------- getAllComponentList --------------------
ComponentList & Application::getAllComponentList() {
  static ComponentList allComponents;

  return allComponents;
}

// -------------------- getSelectedComponentList --------------------
ComponentList & Application::getSelectedComponentList() {
  static ComponentList selectedComponents;

  return selectedComponents;
}

// -------------------- getTopLevelComponents --------------------
const ComponentList & Application::getTopLevelComponents() {
  return getTopLevelComponentList();
}

// -------------------- getAllComponents --------------------
const ComponentList & Application::getAllComponents() {
  return getAllComponentList();
}

// -------------------- getSelectedComponents --------------------
const ComponentList & Application::getSelectedComponents() {
  return getSelectedComponentList();
}

// -------------------- setSelected --------------------
void Application::setSelected(Component* component, bool isSelected) {
  // in case the component is selected again, put it in the end
  // therefore the correct processing is :
  // 1. remove component from the list
  // 2. if isSelected is true, append
  getSelectedComponentList().removeAll(component);
  if (isSelected)
      getSelectedComponentList().append(component);
}

// -------------------- clearSelectedComponents --------------------
void Application::clearSelectedComponents() {
  foreach(Component *comp, getSelectedComponentList()) {
    comp->setSelected(false);
  }
  getSelectedComponentList().clear();
}

}
