/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
#include "ExtensionGenerator.h"
#include "ActionGenerator.h"
#include "ComponentGenerator.h"

#include <Core.h>

#include <iostream>


#include <QMap>
#include <QUrl>
#include <QRegExp>
#include <QFileInfo>
#include <QTextStream>
#include <QFileDialog>
#include <QMessageBox>
#include <QDesktopServices>


ExtensionGenerator::ExtensionGenerator(QString xmlFileName, QString devDirectoryName) throw (QString) {
    // Should be called first !!
    setXmlFileName(xmlFileName);

    // TODO manage exception (despite Qt...).
    std::string xmlFileStr = this->xmlFileName.canonicalFilePath().toStdString();
    this->theExtension = extension(xmlFileStr, xml_schema::flags::dont_validate);

    QString theType(theExtension->type()->c_str());
    if (theType == QString("action"))
        type = ACTION;
    else
        if (theType == QString("application"))
            type = APPLICATION;
        else
            if (theType == QString("component"))
                type = COMPONENT;
            else
                if (theType == QString("viewer"))
                    type = VIEWER;
                else
                    type = UNKNOWN;

    setDevDirectoryName(devDirectoryName);


}

void ExtensionGenerator::setDevDirectoryName(QString devDirectoryName) throw (QString) {
    devDirectoryName = devDirectoryName + "/";
    QFileInfo devDir(devDirectoryName);
    if ( ! devDir.isDir()) {
        QString msg = "Exception from extension generation:\nThe path " + devDirectoryName + " is not a directory\n";
        throw (msg);
    }
    this->devDirectoryName = devDir.absoluteDir();

    // Create a sub-folder where to put the extension sources
    QString extensionClassName = this->theExtension->extensionClass().c_str();
    QString newDir = this->devDirectoryName.absolutePath() + extensionClassName.toLower();
    this->devDirectoryName.mkdir(extensionClassName.toLower());
	this->srcDir.cd(this->devDirectoryName.absolutePath());
	this->srcDir.cd(extensionClassName.toLower());

	// Create a build directory
	this->devDirectoryName.mkdir("build");

	// Create a sub-folder for testdata
	if (type == COMPONENT) {
		createTestDataFiles();
	}

}

void ExtensionGenerator::createTestDataFiles() throw (QString) {
	QDir testDataDir;
	testDataDir.cd(this->devDirectoryName.absolutePath());
	testDataDir.mkdir("testdata");
	testDataDir.cd("testdata");
	Components::componentExtension_sequence listOfExtensions = this->theExtension->components()->componentExtension();
	Components::componentExtension_iterator iterator;
	for (iterator = listOfExtensions.begin(); iterator < listOfExtensions.end(); iterator++) {
		QString extension(iterator->name()->c_str());
		QFileInfo testFileName;
		testFileName.setFile(testDataDir, QString("test." + extension));
		std::cout << "creating test file " << testFileName.absoluteFilePath().toStdString() << std::endl;
		QFile testFile(testFileName.absoluteFilePath());
		if (!testFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
			QString msg = "Exception from test file generation: \n    Cannot write on file " + testFile.fileName() + "\n";
			throw (msg);
		}
		QTextStream out(&testFile);
		out << "This is a test file for the extension " << extension << endl;
		out << "It does not contain anything yet... " << endl;
		testFile.close();
	}

}

void ExtensionGenerator::setXmlFileName(QString xmlFileName) throw (QString) {
    QFileInfo xmlFile(xmlFileName);

    if ((! xmlFile.exists()) || (! xmlFile.isFile())) {
        QString msg = "Exception from extension generation: \n     The file " + xmlFileName +  " does not exist or is not a file...\n";
        throw (msg);
    }
    this->xmlFileName = xmlFile;

}

void ExtensionGenerator::generateCMakeLists() throw (QString) {
	// Copying FindCamiTK.cmake
	std::cout << "Copying FindCamiTK.cmake..." << std::endl;
    QDir cMakeModuleDir = camitk::Core::getTestDataDir();
    cMakeModuleDir.cdUp();
    cMakeModuleDir.cd("cmake");
    QFile findCamiTK(cMakeModuleDir.canonicalPath()+"/"+"FindCamiTK.cmake");
    std::cout << "Copying " << findCamiTK.fileName().toStdString() << std::endl;
    findCamiTK.copy(this->srcDir.canonicalPath() + "/" + "FindCamiTK.cmake");

    std::cout << "Generating CMakeLists file... " << std::endl;

    QString extensionClassName(this->theExtension->extensionClass().c_str());

	// Checking for dependancies    
	QMap<QString, QString> neededLibs;
    QStringList neededComponents;
    QStringList neededTools;
    if (! theExtension->external().empty() ) {
        for (Extension::external_const_iterator it = theExtension->external().begin();
                it < theExtension->external().end(); it++) {

            QString type = QString((*it).type().c_str());
            if (type == QString("component"))
                neededComponents.push_back(QString((*it).name().c_str()));
            else
                if (type == QString("library"))
                    neededLibs.insert(QString((*it).name().c_str()), ((*it).additional().present() ? QString((*it).additional().get().c_str()) : QString("")));
                else
                    if (type == QString("tool"))
                        neededTools.push_back(QString((*it).name().c_str()));
		}
	}

    // Building CMakeLists.txt
    QFileInfo cmakeFileName;
    cmakeFileName.setFile(this->srcDir, "CMakeLists.txt");
    std::cout << "writing in " << cmakeFileName.absoluteFilePath().toStdString() << std::endl;
    QFile cmakeLists(cmakeFileName.absoluteFilePath());
    if (!cmakeLists.open(QIODevice::WriteOnly | QIODevice::Text)) {
        QString msg = "Exception from extension generation: \n    Cannot write on file " + cmakeLists.fileName() + "\n";
        throw (msg);
    }
    QTextStream out(&cmakeLists);

	QFile initFile(":/resources/CMakeLists.txt.in");
	initFile.open(QIODevice::ReadOnly | QIODevice::Text);
	QTextStream in(&initFile);

	QString text;
	do {
		text = in.readLine();
		text.replace(QRegExp("@EXTENSIONNAME@"), extensionClassName.toLower());
		text.replace(QRegExp("@EXTTYPELOW@"), QString(theExtension->type()->c_str()));
		text.replace(QRegExp("@EXTTYPEHIGH@"), QString(theExtension->type()->c_str()).toUpper());

		if (text.contains(QRegExp("@NEEDEDLIBS@"))) {
			foreach(QString lib, neededLibs.keys()) {
	            out << "NEEDS_" << lib << endl;
				if (neededLibs[lib] != QString(""))
					out << "LIBRARIES " << neededLibs[lib] << endl;
			}
		} else if (text.contains(QRegExp("@NEEDEDTOOLS@"))) {
			if (! neededTools.isEmpty()) {
				out << "NEEDS_TOOL ";
				for (QStringList::const_iterator it = neededTools.begin(); it < neededTools.end(); it++)
					out << (*it) << " ";
				out << endl;
			}
		} else if (text.contains(QRegExp("@NEEDEDCOMPEXT@"))) {
			if (! neededComponents.isEmpty()) {
				out << "NEEDS_COMPONENT_EXTENSIONS " ;
				for (QStringList::const_iterator it = neededComponents.begin(); it < neededComponents.end(); it ++)
					out << (*it) << " ";
				out << endl;
			}

		} else {
			out << text << endl;
		}
	} while (! text.isNull());

	cmakeLists.close();
	initFile.close();
}

void ExtensionGenerator::generateExtensionFiles() throw (QString) {
    switch (type) {
    case ACTION:
        this->generateActionExtensionFiles();
        break;
    case COMPONENT:
        this->generateComponentExtensionFiles();
        break;
    default:
        QString msg = "Exception from extenion generation: \n     The type of extension: " + QString(theExtension->type()->c_str()) + "is not implemented yet, sorry...\n";
        throw (msg);
    }
}

void ExtensionGenerator::generateInternFilesList() throw (QString) {
    switch (type) {
    case ACTION:
        this->generateActionsList();
        break;
    case COMPONENT:
        this->generateComponentsList();
        break;
    default:
        QString msg = "Exception from extenion generation: \n     The type of extension: " + QString(theExtension->type()->c_str()) + "is not implemented yet, sorry...\n";
        throw (msg);
    }
}


void ExtensionGenerator::generateInternFiles() throw (QString) {
    switch (type) {
    case ACTION:
        this->generateActionFiles();
        break;
    case COMPONENT:
        this->generateComponentFiles();
        break;
    default:
        QString msg = "Exception from extenion generation: \n     The type of extension: " + QString(theExtension->type()->c_str()) + "is not implemented yet, sorry...\n";
        throw (msg);
    }
}

void ExtensionGenerator::generateActionExtensionFiles() throw(QString) {
    std::cout << "Generating extension file... " << std::endl;
    QString extensionClassName(this->theExtension->extensionClass().c_str());

    // Generating .h
	QFile initHFile(":/resources/ActionExtension.h.in");
	initHFile.open(QIODevice::ReadOnly | QIODevice::Text);
	QTextStream inh(&initHFile);


    QFileInfo extFileHPath;
    extFileHPath.setFile(this->srcDir, extensionClassName + ".h");
    QFile extFileH(extFileHPath.absoluteFilePath());
    if (! extFileH.open(QIODevice::WriteOnly | QIODevice::Text)) {
        QString msg = "Exception from extension generation: \n    Cannot write on file " + extFileHPath.fileName() + "\n";
        throw (msg);
    }
    QTextStream outh(&extFileH);

    QString headdef = extensionClassName.toUpper();
	QString text;
	do {
		text = inh.readLine();
		text.replace(QRegExp("@HEADDEF@"), headdef);
		text.replace(QRegExp("@EXTENSIONCLASSNAME@"), extensionClassName);
		text.replace(QRegExp("@EXTENSIONNAME@"), QString(this->theExtension->name().c_str()));
		text.replace(QRegExp("@EXTENSIONDESCRIPTION@"), QString(this->theExtension->description().c_str()));

		outh << text << endl;
	} while (! text.isNull());

	extFileH.close();
	initHFile.close();


    // Generating .cpp
	QFile initCFile(":/resources/ActionExtension.cpp.in");
	initCFile.open(QIODevice::ReadOnly | QIODevice::Text);
	QTextStream inc(&initCFile);

    QFileInfo extFileCPath;
    extFileCPath.setFile(this->srcDir, extensionClassName + ".cpp");
    QFile extFileC(extFileCPath.absoluteFilePath());
    if (! extFileC.open(QIODevice::WriteOnly | QIODevice::Text)) {
        QString msg = "Exception from extension generation: \n    Cannot write on file " + extFileCPath.fileName() + "\n";
        throw (msg);
    }
    QTextStream outc(&extFileC);

	std::vector<QString>::iterator it;
	do {
		text = inc.readLine();
		text.replace(QRegExp("@EXTENSIONCLASSNAME@"), extensionClassName);
		text.replace(QRegExp("@EXTENSIONCLASSNAMELOW@"), extensionClassName.toLower());

		if (text.contains(QRegExp("@INCLUDEACTIONHEADERS@"))) {
			for (it = generatedClassList.begin(); it < generatedClassList.end(); it++) {
				outc << "#include \"" << (*it) << ".h\"" << endl;
			}
		}
		else if (text.contains(QRegExp("@BEGIN_REGISTERACTIONS@"))) {
			text = inc.readLine();
			int nbActions = generatedClassList.size();
			int actionIndex = 0;
			if (nbActions > 0) {
				QString * registerOneAction = new QString[nbActions];
				for (actionIndex = 0; actionIndex < nbActions; actionIndex++)
					registerOneAction[actionIndex] = "";

				while (! text.contains(QRegExp("@END_REGISTERACTIONS@"))) {
					actionIndex = 0;
					for (it = generatedClassList.begin(); it < generatedClassList.end(); it++) {
						QString  textTmp = text;
						textTmp.replace(QRegExp("@ACTIONNAME@"), (*it));
						registerOneAction[actionIndex].append(textTmp).append("\n");
						actionIndex++;
					}
					text = inc.readLine();
				}
				for (actionIndex = 0; actionIndex < nbActions; actionIndex++) {
					outc << registerOneAction[actionIndex] << endl;
				}
			}
			else {
				while (! text.contains(QRegExp("@ENDREGISTERACTIONS@"))) {
					text = inc.readLine();
				}
				outc << "// You should register your actions here." << endl;
			}
		}
		else {
			outc << text << endl;
		}
	} while (! text.isNull());

	extFileC.close();
	initCFile.close();
}

void ExtensionGenerator::generateComponentExtensionFiles() throw (QString) {
    std::cout << "Generating extension file... " << std::endl;
    QString extensionClassName(this->theExtension->extensionClass().c_str());

    // Generating .h
	QFile initHFile(":/resources/ComponentExtension.h.in");
	initHFile.open(QIODevice::ReadOnly | QIODevice::Text);
	QTextStream inh(&initHFile);

    QFileInfo extFileHPath;
    extFileHPath.setFile(this->srcDir, extensionClassName + ".h");
    QFile extFileH(extFileHPath.absoluteFilePath());
    if (! extFileH.open(QIODevice::WriteOnly | QIODevice::Text)) {
        QString msg = "Exception from extension generation: \n    Cannot write on file " + extFileHPath.fileName() + "\n";
        throw (msg);
    }
    QTextStream outh(&extFileH);

    QString headdef = extensionClassName.toUpper();
	QString text;
	do {
		text = inh.readLine();
		text.replace(QRegExp("@HEADDEF@"), headdef);
		text.replace(QRegExp("@EXTENSIONCLASSNAME@"), extensionClassName);
		text.replace(QRegExp("@EXTENSIONNAME@"), QString(this->theExtension->name().c_str()));
		text.replace(QRegExp("@EXTENSIONDESCRIPTION@"), QString(this->theExtension->description().c_str()));

		outh << text << endl;
	} while (! text.isNull());

	extFileH.close();
	initHFile.close();

    // Generating .cpp
	// Find top level component
	QString topLevelComponentName = "YourTopLevelComponent";
    for (Components::componentFile_const_iterator it = theExtension->components().get().componentFile().begin(); it < theExtension->components().get().componentFile().end(); it ++) {
		if ((*it).TopLevel() == true) {
            QFileInfo componentFileInfo(this->xmlFileName.path(), QString((*it).file().get().c_str()));
            std::auto_ptr<Component> comp = coreschema::component(componentFileInfo.absoluteFilePath().toStdString(), xml_schema::flags::dont_validate);
            topLevelComponentName = QString(comp->classNames().componentClass().c_str());
		}
	}

	QFile initCFile(":/resources/ComponentExtension.cpp.in");
	initCFile.open(QIODevice::ReadOnly | QIODevice::Text);
	QTextStream inc(&initCFile);

    QFileInfo extFileCPath;
    extFileCPath.setFile(this->srcDir, extensionClassName + ".cpp");
    QFile extFileC(extFileCPath.absoluteFilePath());
    if (! extFileC.open(QIODevice::WriteOnly | QIODevice::Text)) {
        QString msg = "Exception from extension generation: \n    Cannot write on file " + extFileCPath.fileName() + "\n";
        throw (msg);
    }
    QTextStream outc(&extFileC);

	std::vector<QString>::iterator it;
	do {
		text = inc.readLine();
		text.replace(QRegExp("@EXTENSIONCLASSNAME@"), extensionClassName);
		text.replace(QRegExp("@EXTENSIONCLASSNAMELOW@"), extensionClassName.toLower());
		text.replace(QRegExp("@TOPLEVELCOMPONENTCLASSNAME@"), topLevelComponentName);

		if (text.contains(QRegExp("@INCLUDECOMPONENTSHEADERS@"))) {
			outc << "#include \"" << topLevelComponentName << ".h\"" << endl;
		}
		else if (text.contains(QRegExp("@EXTENSIONSDEF@"))) {
			for (Components::componentExtension_const_iterator it = theExtension->components().get().componentExtension().begin(); it < theExtension->components().get().componentExtension().end(); it ++)
				outc << "    ext << \"" << (*it).name().get().c_str() << "\";" << endl;

		}
		else {
			outc << text << endl;
		}
	} while (! text.isNull());

	extFileC.close();
	initCFile.close();



 }

void ExtensionGenerator::generateActionsList() throw (QString) {
    Actions theActions = this->theExtension->actions().get();
    Actions::actionFile_sequence actionSeq = theActions.actionFile();
    Actions::actionFile_const_iterator it;
    for (it = actionSeq.begin(); it < actionSeq.end(); it++) {
        QString actionFileName = (*it).c_str();
        // TODO: correct this !
        QFileInfo actionFileInfo(this->xmlFileName.path(), actionFileName);
        this->elementsList.push_back(actionFileInfo);
    }
}

void ExtensionGenerator::generateComponentsList() throw (QString) {
    Components theComponents = this->theExtension->components().get();
    for (Components::componentFile_const_iterator it = theComponents.componentFile().begin();
            it < theComponents.componentFile().end(); it++) {
        QFileInfo componentFileInfo(this->xmlFileName.path(), QString((*it).file().get().c_str()));
//            QFileInfo componentFileInfo(QDir("../../xml/"), QString((*it).file().get().c_str()));
        this->elementsList.push_back(componentFileInfo);
    }
}

void ExtensionGenerator::generateActionFiles() throw (QString) {
    for (std::vector<QFileInfo>::const_iterator it = elementsList.begin();
            it < elementsList.end(); it ++) {
        QString actionClassName;
        ActionGenerator::generateActionFiles((*it).filePath(), srcDir.canonicalPath(), &actionClassName);
        generatedClassList.push_back(actionClassName);
    }
}

void ExtensionGenerator::generateComponentFiles() throw (QString) {
    for (std::vector<QFileInfo>::const_iterator it = elementsList.begin();
            it < elementsList.end(); it ++) {
        QString componentClassName;
        ComponentGenerator::generateComponentFiles((*it).filePath(), srcDir.canonicalPath(), &componentClassName);
        generatedClassList.push_back(componentClassName);
    }
}

void ExtensionGenerator::writeReport() {
	QDir workingDir(devDirectoryName);
	workingDir.mkdir("doc");
	workingDir.cd("doc");
    QString extensionClassName(this->theExtension->extensionClass().c_str());
	QString workingDirectory(workingDir.absolutePath());
	QString devDirectory(devDirectoryName.absolutePath());
	QString typeStr = "";
	QString typeDir = "";
	QString extension = "";
	switch (type) {
		case ACTION:
			typeStr = "Action";
			typeDir = "actions";
			break;
		case COMPONENT:
			typeStr = "Component";
			typeDir = "components";
			break;
		default:
			typeStr = "???";
			typeDir = "???";
			break;
	}

    QFileInfo htmlFileInfo;
    htmlFileInfo.setFile(workingDirectory, extensionClassName + ".html");
    QFile htmlFile(htmlFileInfo.absoluteFilePath());
    if (! htmlFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
        QString msg = "Exception from extension html help file generation \n    Cannot write on file " + htmlFileInfo.absoluteFilePath() + "\n";
        throw msg;
    }
    QTextStream out(&htmlFile);

	QFile initFile(":/resources/ExtensionReadMe.html.in");
	initFile.open(QIODevice::ReadOnly | QIODevice::Text);
	QTextStream in(&initFile);

	QString text;
	do {
		text = in.readLine();
		text.replace(QRegExp("@EXTTYPE@"), typeStr);
		text.replace(QRegExp("@EXTTYPEDIR@"), typeDir);
		text.replace(QRegExp("@WORKDIR@"), devDirectory);
		text.replace(QRegExp("@EXTENSIONDIR@"), extensionClassName.toLower());

		if (text.contains(QRegExp("@IF_ACTION@"))) {
			if ( type != ACTION) {
				do {
					text = in.readLine();
				} while (! text.contains(QRegExp("@ENDIF_ACTION@")));
			}
		}
		else if (text.contains(QRegExp("@ENDIF_ACTION@"))){
			// Go to the next line.
		}
		else if (text.contains(QRegExp("@IF_COMPONENT@"))) {
			if ( type != COMPONENT) {
				do {
					text = in.readLine();
				} while (! text.contains("@ENDIF_COMPONENT@"));
			}
		} 
		else if (text.contains("@ENDIF_COMPONENT@")) {
			// Go to the next line...
		}

		else {
			out << text << endl;
		}
	} while (! text.isNull());

	htmlFile.close();
	initFile.close();

}

void ExtensionGenerator::showReport() {
    QString extensionClassName(this->theExtension->extensionClass().c_str());

	this->devDirectoryName.cd("doc");

    QFileInfo htmlFileInfo;
    htmlFileInfo.setFile(devDirectoryName, extensionClassName + ".html");

    QDesktopServices::openUrl(QUrl(htmlFileInfo.absoluteFilePath()));    

	this->devDirectoryName.cd("..");

}



bool ExtensionGenerator::generateExtensionFiles(QString xmlFileName, QString devDirectoryName) {


    try {
        ExtensionGenerator * generator = new ExtensionGenerator(xmlFileName, devDirectoryName);

        generator->generateInternFilesList();
        generator->generateInternFiles();

        generator->generateCMakeLists();
        generator->generateExtensionFiles();

		QMessageBox::warning(NULL, "CamiTK Extension Wizard", "The files corresponding to your extension have been created.\nYou will now be redirected to the help page via your browser.\n\nThank you for your interest for CamiTK.\n");

		generator->writeReport();
		generator->showReport();

    }
    catch (QString msg) {
        std::cout << msg.toStdString() << std::endl;
        std::cout << "Could not generate extension files, sorry..." << std::endl;
        return false;
    }


    return true;
}

