/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
/**
 * @file GridTopology.cpp
 * @brief Class implementation for 3D grid mesh creation
 * @author Mathieu BAILET
 * @date July 5 2012
 */

#include "GridTopology.h"
#include <MeshComponent.h>
#include <Log.h>
#include <Application.h>
#include <MainWindow.h>

#include <vtkCell.h>
#include <vtkPoints.h>
#include <vtkSmartPointer.h>
#include <vtkHexahedron.h>
#include <vtkLine.h>
#include <vtkCellArray.h>

GridTopology::GridTopology(ActionExtension *extension) : Action(extension) {
    setName("Grid Topology");
    setDescription("Build a regular grid");
    setComponent(""); // empty component: it does not need any data to be triggered
    setFamily("Mesh Processing");
    addTag("Build Grid");

    setProperty("Tetrahedralize", false);
    setProperty("Network", false);
    setProperty("Dimension X", 1);
    setProperty("Dimension Y", 1);
    setProperty("Dimension Z", 1);
    setProperty("Position X", 0.);
    setProperty("Position Y", 0.);
    setProperty("Position Z", 0.);
    setProperty("Width", 1.);
    setProperty("Height", 1.);
    setProperty("Depth", 1.);
}

GridTopology::~GridTopology() {
}

Action::ApplyStatus GridTopology::apply() {
    int dimX = property("Dimension X").toInt();
    int dimY = property("Dimension Y").toInt();
    int dimZ = property("Dimension Z").toInt();
    double posX = property("Position X").toDouble();
    double posY = property("Position Y").toDouble();
    double posZ = property("Position Z").toDouble();
    double width = property("Width").toDouble();
    double height = property("Height").toDouble();
    double depth = property("Depth").toDouble();
    double stepX = width/dimX;
    double stepY = height/dimY;
    double stepZ = depth/dimZ;

    double pt[3];

    vtkSmartPointer<vtkPoints> points = vtkSmartPointer<vtkPoints>::New();

    for (int i=0; i<=dimX; i++)
        for (int j=0; j<=dimY; j++)
            for (int k=0; k<=dimZ; k++) {
                pt[0] = posX + i * stepX;
                pt[1] = posY + j * stepY;
                pt[2] = posZ + k * stepZ;
                points->InsertNextPoint(pt);
            }

    vtkSmartPointer<vtkUnstructuredGrid> grid = vtkSmartPointer<vtkUnstructuredGrid>::New();
    grid->SetPoints(points);


    if (property("Network").toBool()) {
        buildNetwork(grid);
    }
    else {
        buildGrid(grid);

    }

    grid->Update();

    new MeshComponent(grid, "Grid_" + QString::number(dimX)+"x"+QString::number(dimY)+"x"+QString::number(dimZ));
    
    Application::refresh();
    Application::resetProgressBar();
    Application::showStatusBarMessage("");

    return SUCCESS;
}

void GridTopology::buildGrid(vtkSmartPointer<vtkUnstructuredGrid> grid) {
    vtkSmartPointer<vtkCellArray> elements = vtkSmartPointer<vtkCellArray>::New();
    vtkSmartPointer<vtkHexahedron> element = vtkSmartPointer<vtkHexahedron>::New();

    int dimX = property("Dimension X").toInt();
    int dimY = property("Dimension Y").toInt();
    int dimZ = property("Dimension Z").toInt();

    for (int i=1; i<=dimX; i++)
        for (int j=1; j<=dimY; j++)
            for (int k=1; k<=dimZ; k++) {
                

                element->GetPointIds()->SetId (0, (((i-1)*(dimY+1))+j-1)*(dimZ+1)+k-1);
                element->GetPointIds()->SetId (1, (((i-1)*(dimY+1))+j)*(dimZ+1)+k-1);
                element->GetPointIds()->SetId (2, ((i*(dimY+1))+j)*(dimZ+1)+k-1);
                element->GetPointIds()->SetId (3, ((i*(dimY+1))+(j-1))*(dimZ+1)+k-1);
                element->GetPointIds()->SetId (4, (((i-1)*(dimY+1))+j-1)*(dimZ+1)+k);
                element->GetPointIds()->SetId (5, (((i-1)*(dimY+1))+j)*(dimZ+1)+k);
                element->GetPointIds()->SetId (6, ((i*(dimY+1))+j)*(dimZ+1)+k);
                element->GetPointIds()->SetId (7, ((i*(dimY+1))+(j-1))*(dimZ+1)+k);

                elements->InsertNextCell ( element );
            }

    //grid->Allocate(dimX*dimY*dimZ);
    grid->SetCells(VTK_HEXAHEDRON, elements);
}

void GridTopology::buildNetwork(vtkSmartPointer<vtkUnstructuredGrid> grid) {
    vtkSmartPointer<vtkCellArray> elements = vtkSmartPointer<vtkCellArray>::New();
    vtkSmartPointer<vtkLine> element = vtkSmartPointer<vtkLine>::New();

    int dimX = property("Dimension X").toInt();
    int dimY = property("Dimension Y").toInt();
    int dimZ = property("Dimension Z").toInt();

    for (int i=1; i<=dimX; i++)
        for (int j=0; j<=dimY; j++)
            for (int k=0; k<=dimZ; k++) {
                element->GetPointIds()->SetId (0, (((i-1)*( dimY + 1 ))+j)*( dimZ + 1 )+k);
                element->GetPointIds()->SetId (1, ((i*( dimY + 1 ))+j)*( dimZ + 1 )+k);
                elements->InsertNextCell ( element );
            }

    for (int i=0; i<=dimX; i++)
        for (int j=1; j<=dimY; j++)
            for (int k=0; k<=dimZ; k++) {
                element->GetPointIds()->SetId (0, ( ( i * ( dimY + 1 ) ) + j - 1 ) * ( dimZ + 1 ) +k );
                element->GetPointIds()->SetId (1, ( ( i * ( dimY + 1 ) ) + j ) * ( dimZ + 1 ) + k );
                elements->InsertNextCell ( element );
            }

    for (int i=0; i<=dimX; i++)
        for (int j=0; j<=dimY; j++)
            for (int k=1; k<=dimZ; k++) {
                element->GetPointIds()->SetId (0, ((i*( dimY + 1 ))+j)*( dimZ + 1 )+k-1);
                element->GetPointIds()->SetId (1, ((i*( dimY + 1 ))+j)*( dimZ + 1 )+k);
                elements->InsertNextCell ( element );
            }

    for (int i=1; i<=dimX; i++)
        for (int j=1; j<=dimY; j++)
            for (int k=1; k<=dimZ; k++) {
                element->GetPointIds()->SetId (0, (((i-1)*( dimY + 1 ))+j-1)*( dimZ + 1 )+k-1);
                element->GetPointIds()->SetId (1, ((i*( dimY + 1 ))+j)*( dimZ + 1 )+k);
                elements->InsertNextCell ( element );
            }

    for (int i=1; i<=dimX; i++)
        for (int j=1; j<=dimY; j++)
            for (int k=1; k<=dimZ; k++) {
                element->GetPointIds()->SetId (0, (((i-1)*( dimY + 1 ))+j)*( dimZ + 1 )+k);
                element->GetPointIds()->SetId (1, ((i*( dimY + 1 ))+j-1)*( dimZ + 1 )+k-1);
                elements->InsertNextCell ( element );
            }

    for (int i=1; i<=dimX; i++)
        for (int j=1; j<=dimY; j++)
            for (int k=1; k<=dimZ; k++) {
                element->GetPointIds()->SetId (0, (((i-1)*( dimY + 1 ))+j-1)*( dimZ + 1 )+k);
                element->GetPointIds()->SetId (1, ((i*( dimY + 1 ))+j)*( dimZ + 1 )+k-1);
                elements->InsertNextCell ( element );
            }

    for (int i=1; i<=dimX; i++)
        for (int j=1; j<=dimY; j++)
            for (int k=1; k<=dimZ; k++) {
                element->GetPointIds()->SetId (0, (((i-1)*( dimY + 1 ))+j)*( dimZ + 1 )+k-1);
                element->GetPointIds()->SetId (1, ((i*( dimY + 1 ))+j-1)*( dimZ + 1 )+k);
                elements->InsertNextCell ( element );
            }
            
            
            grid->SetCells(VTK_LINE, elements);
}