(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** Graphical interface. *)

open Chat_types
open Chat_proto

module C = Chat_com
module G = Chat_global

let filename_max_length = 30

let ko = 1024.0
let mo = ko *. ko

(** To pretty-print a file size (int) *)
let size_of_int size =
  let f = float_of_int size in
  if f > mo then
    Printf.sprintf "%.1fM" (f /. mo)
  else
    if f > ko then
      Printf.sprintf "%.1fk" (f /. ko)
    else
      string_of_int size

class file_progress title file size f_stop =
  let s_file =
    let len = String.length file in
    if len > filename_max_length then
      "..."^
      (String.sub file (len - filename_max_length) filename_max_length)
    else
      file
  in
  let format_string = s_file^" : %p\\% of "^(size_of_int size) in
  object(self)
    inherit Chat_gui_base.progress ()

    method progress done_size =
      wpb#set_value (float_of_int done_size);
      while Glib.Main.pending () do
	ignore (Glib.Main.iteration false)
      done

    method message s = 
      wpb#set_format_string (s_file^" : "^s)

    initializer
      ignore (wb_cancel#connect#clicked f_stop);
      win#set_title title;
      wpb#set_show_text true;
      wpb#set_format_string format_string;
      wpb#configure ~current: 0.0 ~min: 0.0 ~max: (float_of_int size);
      win#show ()
  end

type dialog_type = 
    Single of id * host * port
  | Room of id * (id * host * port) list

class dialog typ_dial =
  let id, host, port = 
    match typ_dial with
      Single (i,h,p) -> (i,h,p)
    | Room (n,_) -> n,"",0
  in
  object (self)
    inherit Chat_gui_base.dialog ()
	
    val mutable name =
      match typ_dial with
	Single (id, h, p) -> Printf.sprintf "%s @ %s:%d" id h p
      |	Room (name, _) -> name

    method name =  name
    method id = id
    method host = host
    method port = port

    method send s =
      match typ_dial with
	Single (i,h,p) ->
	  (
	   try C.send i (h, p) (Message s) 
	   with Failure s -> Chat_messages.verbose s
	  )
      |	Room (name, people) ->
	  List.iter
	    (fun (i,h,p) ->
	      try C.send i (h, p) (RoomMessage (self#name, people, s))
	      with Failure s -> Chat_messages.verbose s
	    )
	    people

    method handle_message source_id mes =
      let col = Chat_misc.color_of_name source_id in
      wt_dialog#insert ~foreground: col source_id;
      wt_dialog#insert (" : "^mes^"\n");
      wt_dialog#set_position (wt_dialog#length - 1)

    initializer
      let return () = 
	let s = wt_input#get_chars 0 wt_input#length in
	let len = String.length s in
	let s2 = 
	  if len <= 0 then s
	  else
	    match s.[0] with
	      '\n' -> String.sub s 1 (len - 1)
	    | _ -> s
	and col = Chat_misc.color_of_name G.conf#id in
	self#send s2;
	wt_dialog#insert 
	  ~foreground: col G.conf#id;
	wt_dialog#insert (" : "^s2^"\n") ;
	wt_input#delete_text ~start: 0 ~stop: wt_input#length
	  
      in
      Okey.add wt_input ~mods: [] GdkKeysyms._Return return;
      Okey.add_list wt_input ~mods: [`CONTROL]
	[GdkKeysyms._c; GdkKeysyms._C]
	box#destroy;
      Okey.add_list wt_dialog ~mods: [`CONTROL] 
	[GdkKeysyms._c; GdkKeysyms._C]
	box#destroy;
      Okey.add_list wt_input ~mods: [`CONTROL] 
	[GdkKeysyms._l; GdkKeysyms._L]
	wb_show_hide#clicked;
      Okey.add_list wt_dialog ~mods: [`CONTROL] 
	[GdkKeysyms._l; GdkKeysyms._L]
	wb_show_hide#clicked;

      (
       if G.conf#dialog_buttons then
	 (
	  let wb_send = GButton.button 
	      ~label: Chat_messages.send
	      ~packing: (hbox_buttons#pack ~expand: true ~padding:2)
	      ()
	  in
	  ignore (wb_send#connect#clicked return);
	  let wb_close = GButton.button 
	      ~label: Chat_messages.close
	      ~packing: (hbox_buttons#pack ~expand: true ~padding:2)
	      ()
	  in
	  ignore (wb_close#connect#clicked box#destroy)
	 )
       else
	 ()
      );

      match typ_dial with
	Single _ -> 
	  wb_show_hide#misc#hide ();
	  wscroll_people#misc#hide ()
      |	Room (name, people) ->
	  wscroll_people#misc#hide ();
	  let show = ref false in
	  ignore (wb_show_hide#connect#clicked
		    (fun () -> 
		      show := not !show;
		      if !show then
			wscroll_people#misc#show ()
		      else
			wscroll_people#misc#hide ()));
	  List.iter
	    (fun (i,h,p) ->
	      ignore (wlist_people#append
			[i ; h ; string_of_int p]))
	    people;
	  GToolbox.autosize_clist wlist_people
  end

(** Liste des dialogs ouverts *)
let dialogs = ref ([] : (GWindow.window * dialog) list)

(** Liste des rooms ouvertes *)
let room_dialogs = ref ([] : (GWindow.window * dialog) list)

(** Remove the dialog with the given id from the list of dialogs. *)
let remove_dialog typ_dial =
  match typ_dial with
    Single (id,host,port) ->
      dialogs := List.filter
	  (fun (_,d) -> not (G.pred (id,host,port) (d#id, d#host, d#port)))
	  !dialogs
  | Room (name, people) ->
      room_dialogs := List.filter
	  (fun (_,d) -> d#name <> name)
	  !room_dialogs


(** Find the window and dialog with the given id. It
   it was not found, create it and add it to the list of dialogs.*)
let get_dialog ?(show=true) typ_dial =
  try
    match typ_dial with
      Single (id,host,port) ->
	let (w,d) = List.find 
	    (fun (w,d) -> G.pred (id,host,port) (d#id, d#host, d#port))
	    !dialogs 
	in
	d#wt_input#misc#grab_focus ();
	if show then w#show () ;
	d
    | Room (name, people) ->
	let (w,d) = List.find (fun (_,d) -> d#name = name) !room_dialogs in
	d#wt_input#misc#grab_focus ();
	if show then w#show () ;
	d
  with
    Not_found ->
      let window = GWindow.window ~kind: `DIALOG ~width: 300 ~height: 200 ~title: "" () in
      ignore (window#connect#destroy (fun () -> remove_dialog typ_dial));
      let dialog = new dialog typ_dial in
      window#set_title dialog#name;
      ignore (dialog#box#connect#destroy window#destroy);
      window#add dialog#box#coerce;
      if show then window#show ();
      (
       match typ_dial with
	 Single _ -> dialogs := (window, dialog) :: ! dialogs
       | Room _ -> room_dialogs := (window, dialog) :: ! room_dialogs
      );
      dialog#wt_input#misc#grab_focus ();
      dialog

class gui =
  object (self)
    inherit Chat_gui_base.gui ()

    val mutable selected_people = []
    val mutable people = []

    method update =
      wlist#clear ();
      people <- Hashtbl.fold 
	  (fun (h,p) (s,id) acc -> (id,h,p) :: acc) 
	  G.connections [];
      List.iter
	(fun (id,host,port) ->
	  let temp = 
	    not (List.exists (G.pred (id, host, port)) G.conf#people)
	  in
	  ignore (wlist#append ["" ; id ; host ; (string_of_int port) ; 
				 Chat_messages.yes_or_no temp]
		 );
	  let state = Connected in
	  let color,pix = 
	    match state, temp with
	      Connected, true -> 
		(G.conf#color_connected_temp,
		 Chat_icons.create_gdk_pixmap Chat_icons.connected)
	    | Connected, false -> 
		(G.conf#color_connected,
		 Chat_icons.create_gdk_pixmap Chat_icons.connected)
	    | Not_connected, _ -> 
		(G.conf#color_not_connected,
		 Chat_icons.create_gdk_pixmap Chat_icons.not_connected)
	  in
	  wlist#set_cell ~pixmap: pix (wlist#rows - 1) 0 ;
	  wlist#set_row ~foreground: (`NAME color) (wlist#rows - 1)
	)
	people;
      GToolbox.autosize_clist wlist;
      selected_people <- []

    method open_dialog (id, host, port) =
      ignore (get_dialog (Single (id, host, port)))

    method open_room room_name people =
      ignore (get_dialog (Room (room_name, people)))

    method open_dialog_for_selected_people =
      match selected_people with
	[] -> ()
      |	[p] -> self#open_dialog p
      |	l -> 
	  match GToolbox.input_string 
	      ~title: Chat_messages.m_open_dialog_for_selected_people
	      (Chat_messages.room_name^": ") 
	  with
	    None -> ()
	  | Some name ->
	      let c = G.conf in
	      self#open_room name ((c#id, Unix.gethostname (), c#port) :: l)


    initializer
      ignore (itemQuit#connect#activate box#destroy);
      ignore (itemOpenDialog#connect#activate 
		(fun () -> self#open_dialog_for_selected_people));
      ignore (itemAbout#connect#activate
		(fun () -> 
		  GToolbox.message_box 
		    Chat_messages.m_about
		    Chat_messages.software_about)
	     );


      let maybe_double_click (ev : GdkEvent.Button.t) = 
	let t = GdkEvent.get_type ev in
	match t with
	  `TWO_BUTTON_PRESS -> itemOpenDialog#activate ()
	| _ -> ()
      in

      let f_select ~row ~column ~event =
        try 
	  let (id,host,port) as p = List.nth people row in
	  if List.exists 
	      (fun (i,h,p) -> G.pred (id,host,port) (i,h,p)) 
	      selected_people 
	  then
	    ()
	  else
	    selected_people <- p :: selected_people ;
	  match event with
	    None -> ()
	  | Some ev -> maybe_double_click ev
        with _ -> ()
      in
      let f_unselect ~row ~column ~event =
        try 
	  let (id, host, port) = List.nth people row  in
	  selected_people <- List.filter
	      (fun (i,h,p) -> not (G.pred (id,host,port) (i,h,p)))
	      selected_people;
	  match event with
	    None -> ()
	  | Some ev -> maybe_double_click ev
        with _ -> ()
      in
      (* connect the select and deselect events *) 
      ignore (wlist#connect#select_row f_select) ;
      ignore (wlist#connect#unselect_row f_unselect) ;


  end
