<?php
/* SVN FILE: $Id: magic_db.test.php 7296 2008-06-27 09:09:03Z gwoo $ */
/**
 * MagicDb test
 *
 * PHP versions 4 and 5
 *
 * CakePHP(tm) :  Rapid Development Framework <http://www.cakephp.org/>
 * Copyright 2005-2008, Cake Software Foundation, Inc.
 *								1785 E. Sahara Avenue, Suite 490-204
 *								Las Vegas, Nevada 89104
 *
 * Licensed under The MIT License
 * Redistributions of files must retain the above copyright notice.
 *
 * @filesource
 * @copyright		Copyright 2005-2008, Cake Software Foundation, Inc.
 * @link			http://www.cakefoundation.org/projects/info/cakephp CakePHP(tm) Project
 * @package			cake
 * @subpackage		cake.cake.libs
 * @since			CakePHP(tm) v 1.2.0
 * @version			$Revision: 7296 $
 * @modifiedby		$LastChangedBy: gwoo $
 * @lastmodified	$Date: 2008-06-27 05:09:03 -0400 (Fri, 27 Jun 2008) $
 * @license			http://www.opensource.org/licenses/mit-license.php The MIT License
 */

uses('magic_db', 'object');
/**
 * The test class for the MagicDb classA
 *
 * @package		cake.tests
 * @subpackage	cake.tests.cases.libs
 */
class MagicDbTest extends UnitTestCase {
/**
 * The MagicDb instance to be tested
 *
 * @var object
 * @access public
 */
	var $Db = null;

/**
 * Sets up a MagicDb class instance for testing
 *
 * @access public
 */
	function setUp() {
		$this->Db =& new MagicDb();
	}

/**
 * MagicDb::analyze should properly detect the file type and output additional info as requested.
 *
 * @access public
 */
	function testAnalyze() {
		$r = $this->Db->read(MagicDbTestData::get('magic.db'));
		$this->assertTrue($r === true);

		$r = $this->Db->analyze(array());
		$this->assertTrue($r === false);

		$r = $this->Db->analyze(WWW_ROOT.'img'.DS.'cake.icon.gif');
		// TODO: Check several serialized file samples for accurate detection
	}

/**
 * MagicDb::read should properly read MagicDb databases from .php-/.db-files and plain data arguments passed in and return false if the file wasn't found or
 * if the readed data did not validate.
 *
 * @access public
 */
	function testRead() {
		$this->Db->db = array();

		$r = $this->Db->read(true);
		$this->assertTrue($r === false);
		$r = $this->Db->read(5);
		$this->assertTrue($r === false);

		$this->Db->db = array('a');
		$r = $this->Db->read(array('foo' => 'bar'));
		$this->assertTrue($r === false);
		$this->assertTrue($this->Db->db === array('a'));

		$magicDb = array('header' => array(), 'database' => array());
		$r = $this->Db->read($magicDb);
		$this->assertTrue($r === true);
		$this->assertTrue($this->Db->db === $magicDb);

		// @TODO: Test parsing an actual magic.db file

		$r = $this->Db->read('does-not-exist.db');
		$this->assertTrue($r === false);
		$this->assertTrue($this->Db->db === $magicDb);

		if (file_exists(VENDORS.'magic.php')) {
			$r = $this->Db->read(VENDORS.'magic.php');
			$this->assertTrue($r === true);
			$this->assertTrue($this->Db->db === array('header' => array(), 'database' => array()));
		}

		$r = $this->Db->read(MagicDbTestData::get('magic.snippet.db'));
		$this->assertTrue($r === true);
	}

/**
 * MagicDb::toArray should either return the MagicDb::db property, or the parsed array data if a magic.db dump is passed in as the first argument
 *
 * @access public
 */
	function testToArray() {
		$this->Db->db = array();

		$r = $this->Db->toArray();
		$this->assertTrue($r === array());
		$this->Db->db = array('foo' => 'bar');
		$r = $this->Db->toArray();
		$this->assertTrue($r === array('foo' => 'bar'));

		$r = $this->Db->toArray(array('yeah'));
		$this->assertTrue($r === array('yeah'));

		$r = $this->Db->toArray('foo');
		$this->assertTrue($r === array());

		$r = $this->Db->toArray(MagicDbTestData::get('magic.snippet.db'));
		$this->assertTrue($r === MagicDbTestData::get('magic.snippet.db.result'));
	}

/**
 * The MagicDb::validates function should return if the array passed to it or the local db property contains a valid MagicDb record set
 *
 * @access public
 */
	function testValidates() {
		$r = $this->Db->validates(array());
		$this->assertTrue($r === false);

		$r = $this->Db->validates(array('header' => true, 'database' => true));
		$this->assertTrue($r === false);
		$magicDb = array('header' => array(), 'database' => array());
		$r = $this->Db->validates($magicDb);
		$this->assertTrue($r === true);

		$this->Db->db = array();
		$r = $this->Db->validates();
		$this->assertTrue($r === false);

		$this->Db->db = $magicDb;
		$r = $this->Db->validates();
		$this->assertTrue($r === true);
	}
}

/**
 * Test data holding object for MagicDb tests
 *
 * @package		cake.tests
 * @subpackage	cake.tests.cases.libs
 */
/**
 * MagicDbTestData class
 * 
 * @package              cake
 * @subpackage           cake.tests.cases.libs
 */
class MagicDbTestData extends Object {
/**
 * Base64 encoded data
 *
 * @var array
 * @access public
 */
	var $data = array(
		'magic.snippet.db' => 'IyBGSUxFX0lEIERCDQojIERhdGU6MjAwNS0wMy0yOQ0KIyBTb3VyY2U6aHR0cDovL3d3dy5tYWdpY2RiLm9yZw0KDQojIE1hZ2ljIElEIGZvciBXb3JkcGVyZmVjdCBmaWxlcy4NCiMgU3VibWl0dGVkIG9uIDIwMDQtMDItMTQgYnkgQ2FybA0KMAlzdHJpbmcJXFx4RkZXUEMJW2ZpZD0wMDAwMDEwMDgtMDAtMDAwMDAwMTtleHQ9O21pbWU9O11Xb3JkcGVyZmVjdCBoZWxwIGZpbGUNCiY5CWJ5dGUJMHgwMgkNCj4xMCBieXRlCXgJLCB2ZXJzaW9uICVkDQo+MTEJYnl0ZQl4CS4lZA0KDQojIE1hZ2ljIElEIGZvciBXb3JkcGVyZmVjdCBmaWxlcy4NCiMgU3VibWl0dGVkIG9uIDIwMDQtMDItMTQgYnkgQ2FybA0KMAlzdHJpbmcJXFx4RkZXUEMJW2ZpZD0wMDAwMDEwMDgtMDAtMDAwMDAwMTtleHQ9O21pbWU9O11Xb3JkcGVyZmVjdCBhcHBsaWNhdGlvbiByZXNvdXJjZSBsaWJyYXJ5DQomOQlieXRlCTUxCQ0KPjEwCWJ5dGUJeAksIHZlcnNpb24gJWQNCj4xMQlieXRlCXgJLiVkDQoNCiMgTWFnaWMgSUQgZm9yIFdvcmRwZXJmZWN0IGZpbGVzLg0KIyBTdWJtaXR0ZWQgb24gMjAwNC0wMi0xNCBieSBDYXJsDQowCXN0cmluZwlcXHhGRldQQwlbZmlkPTAwMDAwMTAwOC0wMC0wMDAwMDAxO2V4dD07bWltZT07XVdvcmRwZXJmZWN0IGJsb2NrIGZpbGUNCiY5CWJ5dGUJMTMJDQo+MTAJYnl0ZQl4CSwgdmVyc2lvbiAlZA0KPjExCWJ5dGUJeAkuJWQ=',
		'magic.snippet.db.result' => 'YToyOntzOjY6ImhlYWRlciI7YToyOntzOjQ6IkRhdGUiO3M6MTA6IjIwMDUtMDMtMjkiO3M6NjoiU291cmNlIjtzOjIyOiJodHRwOi8vd3d3Lm1hZ2ljZGIub3JnIjt9czo4OiJkYXRhYmFzZSI7YToyOntpOjA7YTo0OntpOjA7YTo0OntpOjA7czoxOiIwIjtpOjE7czo2OiJzdHJpbmciO2k6MjtzOjg6IlxceEZGV1BDIjtpOjM7czo1OToiW2ZpZD0wMDAwMDEwMDgtMDAtMDAwMDAwMTtleHQ9O21pbWU9O11Xb3JkcGVyZmVjdCBoZWxwIGZpbGUiO31pOjE7YTo0OntpOjA7czoyOiImOSI7aToxO3M6NDoiYnl0ZSI7aToyO3M6NDoiMHgwMiI7aTozO3M6MDoiIjt9aToyO2E6Mzp7aTowO3M6ODoiPjEwIGJ5dGUiO2k6MTtzOjE6IngiO2k6MjtzOjEyOiIsIHZlcnNpb24gJWQiO31pOjM7YTo0OntpOjA7czozOiI+MTEiO2k6MTtzOjQ6ImJ5dGUiO2k6MjtzOjE6IngiO2k6MztzOjM6Ii4lZCI7fX1pOjE7YTo0OntpOjA7YTo0OntpOjA7czoxOiIwIjtpOjE7czo2OiJzdHJpbmciO2k6MjtzOjg6IlxceEZGV1BDIjtpOjM7czo3ODoiW2ZpZD0wMDAwMDEwMDgtMDAtMDAwMDAwMTtleHQ9O21pbWU9O11Xb3JkcGVyZmVjdCBhcHBsaWNhdGlvbiByZXNvdXJjZSBsaWJyYXJ5Ijt9aToxO2E6NDp7aTowO3M6MjoiJjkiO2k6MTtzOjQ6ImJ5dGUiO2k6MjtzOjI6IjUxIjtpOjM7czowOiIiO31pOjI7YTo0OntpOjA7czozOiI+MTAiO2k6MTtzOjQ6ImJ5dGUiO2k6MjtzOjE6IngiO2k6MztzOjEyOiIsIHZlcnNpb24gJWQiO31pOjM7YTo0OntpOjA7czozOiI+MTEiO2k6MTtzOjQ6ImJ5dGUiO2k6MjtzOjE6IngiO2k6MztzOjM6Ii4lZCI7fX19fQ==',
		'magic.db' => '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'
	);

/**
 * Returns the test data for a given key
 *
 * @param string $key
 * @access public
 **/
	function get($key) {
/**
 * data property
 * 
 * @var array
 * @access public
 */
		static $data = array();

		if (empty($data)) {
			$vars = get_class_vars(__CLASS__);
			foreach ($vars['data'] as $k => $v) {
				$data[$k] = base64_decode($v);
				if (preg_match('/^[ais]:[0-9]+/', $data[$k])) {
					$data[$k] = unserialize($data[$k]);
				}
			}
		}

		if (!isset($data[$key])) {
			return false;
		}
		return $data[$key];
	}
}
?>